/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The properties that describe a container group resource. You can update all properties of a container group
 * definition properties. Updates to a container group definition are saved as new versions.
 * </p>
 * <p>
 * <b>Used with:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
 * >CreateContainerGroupDefinition</a>
 * </p>
 * <p>
 * <b>Returned by:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeContainerGroupDefinition.html"
 * >DescribeContainerGroupDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListContainerGroupDefinitions.html"
 * >ListContainerGroupDefinitions</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateContainerGroupDefinition.html"
 * >UpdateContainerGroupDefinition</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerGroupDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerGroupDefinition.Builder, ContainerGroupDefinition> {
    private static final SdkField<String> CONTAINER_GROUP_DEFINITION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ContainerGroupDefinitionArn")
            .getter(getter(ContainerGroupDefinition::containerGroupDefinitionArn))
            .setter(setter(Builder::containerGroupDefinitionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerGroupDefinitionArn")
                    .build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(ContainerGroupDefinition::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(ContainerGroupDefinition::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ContainerGroupDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CONTAINER_GROUP_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerGroupType").getter(getter(ContainerGroupDefinition::containerGroupTypeAsString))
            .setter(setter(Builder::containerGroupType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerGroupType").build())
            .build();

    private static final SdkField<Integer> TOTAL_MEMORY_LIMIT_MEBIBYTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("TotalMemoryLimitMebibytes")
            .getter(getter(ContainerGroupDefinition::totalMemoryLimitMebibytes))
            .setter(setter(Builder::totalMemoryLimitMebibytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalMemoryLimitMebibytes").build())
            .build();

    private static final SdkField<Double> TOTAL_VCPU_LIMIT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("TotalVcpuLimit").getter(getter(ContainerGroupDefinition::totalVcpuLimit))
            .setter(setter(Builder::totalVcpuLimit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalVcpuLimit").build()).build();

    private static final SdkField<GameServerContainerDefinition> GAME_SERVER_CONTAINER_DEFINITION_FIELD = SdkField
            .<GameServerContainerDefinition> builder(MarshallingType.SDK_POJO)
            .memberName("GameServerContainerDefinition")
            .getter(getter(ContainerGroupDefinition::gameServerContainerDefinition))
            .setter(setter(Builder::gameServerContainerDefinition))
            .constructor(GameServerContainerDefinition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerContainerDefinition")
                    .build()).build();

    private static final SdkField<List<SupportContainerDefinition>> SUPPORT_CONTAINER_DEFINITIONS_FIELD = SdkField
            .<List<SupportContainerDefinition>> builder(MarshallingType.LIST)
            .memberName("SupportContainerDefinitions")
            .getter(getter(ContainerGroupDefinition::supportContainerDefinitions))
            .setter(setter(Builder::supportContainerDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportContainerDefinitions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SupportContainerDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(SupportContainerDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> VERSION_NUMBER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("VersionNumber").getter(getter(ContainerGroupDefinition::versionNumber))
            .setter(setter(Builder::versionNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionNumber").build()).build();

    private static final SdkField<String> VERSION_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionDescription").getter(getter(ContainerGroupDefinition::versionDescription))
            .setter(setter(Builder::versionDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionDescription").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ContainerGroupDefinition::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(ContainerGroupDefinition::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CONTAINER_GROUP_DEFINITION_ARN_FIELD, CREATION_TIME_FIELD, OPERATING_SYSTEM_FIELD, NAME_FIELD,
            CONTAINER_GROUP_TYPE_FIELD, TOTAL_MEMORY_LIMIT_MEBIBYTES_FIELD, TOTAL_VCPU_LIMIT_FIELD,
            GAME_SERVER_CONTAINER_DEFINITION_FIELD, SUPPORT_CONTAINER_DEFINITIONS_FIELD, VERSION_NUMBER_FIELD,
            VERSION_DESCRIPTION_FIELD, STATUS_FIELD, STATUS_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String containerGroupDefinitionArn;

    private final Instant creationTime;

    private final String operatingSystem;

    private final String name;

    private final String containerGroupType;

    private final Integer totalMemoryLimitMebibytes;

    private final Double totalVcpuLimit;

    private final GameServerContainerDefinition gameServerContainerDefinition;

    private final List<SupportContainerDefinition> supportContainerDefinitions;

    private final Integer versionNumber;

    private final String versionDescription;

    private final String status;

    private final String statusReason;

    private ContainerGroupDefinition(BuilderImpl builder) {
        this.containerGroupDefinitionArn = builder.containerGroupDefinitionArn;
        this.creationTime = builder.creationTime;
        this.operatingSystem = builder.operatingSystem;
        this.name = builder.name;
        this.containerGroupType = builder.containerGroupType;
        this.totalMemoryLimitMebibytes = builder.totalMemoryLimitMebibytes;
        this.totalVcpuLimit = builder.totalVcpuLimit;
        this.gameServerContainerDefinition = builder.gameServerContainerDefinition;
        this.supportContainerDefinitions = builder.supportContainerDefinitions;
        this.versionNumber = builder.versionNumber;
        this.versionDescription = builder.versionDescription;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to an Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the
     * resource across all Amazon Web Services Regions. Format is
     * <code>arn:aws:gamelift:[region]::containergroupdefinition/[container group definition name]:[version]</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to an
     *         Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource
     *         across all Amazon Web Services Regions. Format is
     *         <code>arn:aws:gamelift:[region]::containergroupdefinition/[container group definition name]:[version]</code>
     *         .
     */
    public final String containerGroupDefinitionArn() {
        return containerGroupDefinitionArn;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The platform that all containers in the container group definition run on.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK 5.x, and then deploy
     * to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link ContainerOperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The platform that all containers in the container group definition run on.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see ContainerOperatingSystem
     */
    public final ContainerOperatingSystem operatingSystem() {
        return ContainerOperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * The platform that all containers in the container group definition run on.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK 5.x, and then deploy
     * to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link ContainerOperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The platform that all containers in the container group definition run on.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see ContainerOperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * <p>
     * A descriptive identifier for the container group definition. The name value is unique in an Amazon Web Services
     * Region.
     * </p>
     * 
     * @return A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
     *         Services Region.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of container group. Container group type determines how Amazon GameLift deploys the container group on
     * each fleet instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerGroupType} will return {@link ContainerGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #containerGroupTypeAsString}.
     * </p>
     * 
     * @return The type of container group. Container group type determines how Amazon GameLift deploys the container
     *         group on each fleet instance.
     * @see ContainerGroupType
     */
    public final ContainerGroupType containerGroupType() {
        return ContainerGroupType.fromValue(containerGroupType);
    }

    /**
     * <p>
     * The type of container group. Container group type determines how Amazon GameLift deploys the container group on
     * each fleet instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #containerGroupType} will return {@link ContainerGroupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #containerGroupTypeAsString}.
     * </p>
     * 
     * @return The type of container group. Container group type determines how Amazon GameLift deploys the container
     *         group on each fleet instance.
     * @see ContainerGroupType
     */
    public final String containerGroupTypeAsString() {
        return containerGroupType;
    }

    /**
     * <p>
     * The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in the
     * group share these resources.
     * </p>
     * <p>
     * You can set a limit for each container definition in the group. If individual containers have limits, this total
     * value must be greater than any individual container's memory limit.
     * </p>
     * 
     * @return The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in
     *         the group share these resources. </p>
     *         <p>
     *         You can set a limit for each container definition in the group. If individual containers have limits,
     *         this total value must be greater than any individual container's memory limit.
     */
    public final Integer totalMemoryLimitMebibytes() {
        return totalMemoryLimitMebibytes;
    }

    /**
     * <p>
     * The amount of vCPU units on a fleet instance to allocate for the container group (1 vCPU is equal to 1024 CPU
     * units). All containers in the group share these resources. You can set a limit for each container definition in
     * the group. If individual containers have limits, this total value must be equal to or greater than the sum of the
     * limits for each container in the group.
     * </p>
     * 
     * @return The amount of vCPU units on a fleet instance to allocate for the container group (1 vCPU is equal to 1024
     *         CPU units). All containers in the group share these resources. You can set a limit for each container
     *         definition in the group. If individual containers have limits, this total value must be equal to or
     *         greater than the sum of the limits for each container in the group.
     */
    public final Double totalVcpuLimit() {
        return totalVcpuLimit;
    }

    /**
     * <p>
     * The definition for the game server container in this group. This property is used only when the container group
     * type is <code>GAME_SERVER</code>. This container definition specifies a container image with the game server
     * build.
     * </p>
     * 
     * @return The definition for the game server container in this group. This property is used only when the container
     *         group type is <code>GAME_SERVER</code>. This container definition specifies a container image with the
     *         game server build.
     */
    public final GameServerContainerDefinition gameServerContainerDefinition() {
        return gameServerContainerDefinition;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportContainerDefinitions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportContainerDefinitions() {
        return supportContainerDefinitions != null && !(supportContainerDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of definitions for support containers in this group. A container group definition might have zero support
     * container definitions. Support container can be used in any type of container group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportContainerDefinitions} method.
     * </p>
     * 
     * @return The set of definitions for support containers in this group. A container group definition might have zero
     *         support container definitions. Support container can be used in any type of container group.
     */
    public final List<SupportContainerDefinition> supportContainerDefinitions() {
        return supportContainerDefinitions;
    }

    /**
     * <p>
     * Indicates the version of a particular container group definition. This number is incremented automatically when
     * you update a container group definition. You can view, update, or delete individual versions or the entire
     * container group definition.
     * </p>
     * 
     * @return Indicates the version of a particular container group definition. This number is incremented
     *         automatically when you update a container group definition. You can view, update, or delete individual
     *         versions or the entire container group definition.
     */
    public final Integer versionNumber() {
        return versionNumber;
    }

    /**
     * <p>
     * An optional description that was provided for a container group definition update. Each version can have a unique
     * description.
     * </p>
     * 
     * @return An optional description that was provided for a container group definition update. Each version can have
     *         a unique description.
     */
    public final String versionDescription() {
        return versionDescription;
    }

    /**
     * <p>
     * Current status of the container group definition resource. Values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
     * defined in the group. While in this state, the resource can't be used to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in the
     * group. You can use a container group definition in this status to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
     * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
     * failed status will be deleted within a few minutes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerGroupDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the container group definition resource. Values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that
     *         are defined in the group. While in this state, the resource can't be used to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined
     *         in the group. You can use a container group definition in this status to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For
     *         more details on the cause of the failure, see <code>StatusReason</code>. A container group definition
     *         resource in failed status will be deleted within a few minutes.
     *         </p>
     *         </li>
     * @see ContainerGroupDefinitionStatus
     */
    public final ContainerGroupDefinitionStatus status() {
        return ContainerGroupDefinitionStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the container group definition resource. Values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
     * defined in the group. While in this state, the resource can't be used to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in the
     * group. You can use a container group definition in this status to create a container fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
     * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
     * failed status will be deleted within a few minutes.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ContainerGroupDefinitionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the container group definition resource. Values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that
     *         are defined in the group. While in this state, the resource can't be used to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined
     *         in the group. You can use a container group definition in this status to create a container fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For
     *         more details on the cause of the failure, see <code>StatusReason</code>. A container group definition
     *         resource in failed status will be deleted within a few minutes.
     *         </p>
     *         </li>
     * @see ContainerGroupDefinitionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about a container group definition that's in <code>FAILED</code> status. Possible reasons
     * include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete the
     * failed resource and call <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
     * >CreateContainerGroupDefinition</a>again.
     * </p>
     * </li>
     * <li>
     * <p>
     * An access-denied message means that you don't have permissions to access the container image on ECR. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html.html"> IAM
     * permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>ImageUri</code> value for at least one of the containers in the container group definition was invalid
     * or not found in the current Amazon Web Services account.
     * </p>
     * </li>
     * <li>
     * <p>
     * At least one of the container images referenced in the container group definition exceeds the allowed size. For
     * size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon GameLift endpoints
     * and quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * At least one of the container images referenced in the container group definition uses a different operating
     * system than the one defined for the container group.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Additional information about a container group definition that's in <code>FAILED</code> status. Possible
     *         reasons include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete
     *         the failed resource and call <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
     *         >CreateContainerGroupDefinition</a>again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An access-denied message means that you don't have permissions to access the container image on ECR. See
     *         <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html.html"> IAM
     *         permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>ImageUri</code> value for at least one of the containers in the container group definition was
     *         invalid or not found in the current Amazon Web Services account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         At least one of the container images referenced in the container group definition exceeds the allowed
     *         size. For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon
     *         GameLift endpoints and quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         At least one of the container images referenced in the container group definition uses a different
     *         operating system than the one defined for the container group.
     *         </p>
     *         </li>
     */
    public final String statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerGroupDefinitionArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(containerGroupTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalMemoryLimitMebibytes());
        hashCode = 31 * hashCode + Objects.hashCode(totalVcpuLimit());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerContainerDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportContainerDefinitions() ? supportContainerDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(versionNumber());
        hashCode = 31 * hashCode + Objects.hashCode(versionDescription());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerGroupDefinition)) {
            return false;
        }
        ContainerGroupDefinition other = (ContainerGroupDefinition) obj;
        return Objects.equals(containerGroupDefinitionArn(), other.containerGroupDefinitionArn())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(name(), other.name())
                && Objects.equals(containerGroupTypeAsString(), other.containerGroupTypeAsString())
                && Objects.equals(totalMemoryLimitMebibytes(), other.totalMemoryLimitMebibytes())
                && Objects.equals(totalVcpuLimit(), other.totalVcpuLimit())
                && Objects.equals(gameServerContainerDefinition(), other.gameServerContainerDefinition())
                && hasSupportContainerDefinitions() == other.hasSupportContainerDefinitions()
                && Objects.equals(supportContainerDefinitions(), other.supportContainerDefinitions())
                && Objects.equals(versionNumber(), other.versionNumber())
                && Objects.equals(versionDescription(), other.versionDescription())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerGroupDefinition").add("ContainerGroupDefinitionArn", containerGroupDefinitionArn())
                .add("CreationTime", creationTime()).add("OperatingSystem", operatingSystemAsString()).add("Name", name())
                .add("ContainerGroupType", containerGroupTypeAsString())
                .add("TotalMemoryLimitMebibytes", totalMemoryLimitMebibytes()).add("TotalVcpuLimit", totalVcpuLimit())
                .add("GameServerContainerDefinition", gameServerContainerDefinition())
                .add("SupportContainerDefinitions", hasSupportContainerDefinitions() ? supportContainerDefinitions() : null)
                .add("VersionNumber", versionNumber()).add("VersionDescription", versionDescription())
                .add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerGroupDefinitionArn":
            return Optional.ofNullable(clazz.cast(containerGroupDefinitionArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ContainerGroupType":
            return Optional.ofNullable(clazz.cast(containerGroupTypeAsString()));
        case "TotalMemoryLimitMebibytes":
            return Optional.ofNullable(clazz.cast(totalMemoryLimitMebibytes()));
        case "TotalVcpuLimit":
            return Optional.ofNullable(clazz.cast(totalVcpuLimit()));
        case "GameServerContainerDefinition":
            return Optional.ofNullable(clazz.cast(gameServerContainerDefinition()));
        case "SupportContainerDefinitions":
            return Optional.ofNullable(clazz.cast(supportContainerDefinitions()));
        case "VersionNumber":
            return Optional.ofNullable(clazz.cast(versionNumber()));
        case "VersionDescription":
            return Optional.ofNullable(clazz.cast(versionDescription()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ContainerGroupDefinitionArn", CONTAINER_GROUP_DEFINITION_ARN_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("OperatingSystem", OPERATING_SYSTEM_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("ContainerGroupType", CONTAINER_GROUP_TYPE_FIELD);
        map.put("TotalMemoryLimitMebibytes", TOTAL_MEMORY_LIMIT_MEBIBYTES_FIELD);
        map.put("TotalVcpuLimit", TOTAL_VCPU_LIMIT_FIELD);
        map.put("GameServerContainerDefinition", GAME_SERVER_CONTAINER_DEFINITION_FIELD);
        map.put("SupportContainerDefinitions", SUPPORT_CONTAINER_DEFINITIONS_FIELD);
        map.put("VersionNumber", VERSION_NUMBER_FIELD);
        map.put("VersionDescription", VERSION_DESCRIPTION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("StatusReason", STATUS_REASON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerGroupDefinition, T> g) {
        return obj -> g.apply((ContainerGroupDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerGroupDefinition> {
        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to an
         * Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource across
         * all Amazon Web Services Regions. Format is
         * <code>arn:aws:gamelift:[region]::containergroupdefinition/[container group definition name]:[version]</code>.
         * </p>
         * 
         * @param containerGroupDefinitionArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        an Amazon GameLift <code>ContainerGroupDefinition</code> resource. It uniquely identifies the resource
         *        across all Amazon Web Services Regions. Format is
         *        <code>arn:aws:gamelift:[region]::containergroupdefinition/[container group definition name]:[version]</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerGroupDefinitionArn(String containerGroupDefinitionArn);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The platform that all containers in the container group definition run on.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK 5.x, and then
         * deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The platform that all containers in the container group definition run on.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK
         *        5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see ContainerOperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerOperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * The platform that all containers in the container group definition run on.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK 5.x, and then
         * deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The platform that all containers in the container group definition run on.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x, first update the game server build to server SDK
         *        5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see ContainerOperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerOperatingSystem
         */
        Builder operatingSystem(ContainerOperatingSystem operatingSystem);

        /**
         * <p>
         * A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
         * Services Region.
         * </p>
         * 
         * @param name
         *        A descriptive identifier for the container group definition. The name value is unique in an Amazon Web
         *        Services Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of container group. Container group type determines how Amazon GameLift deploys the container group
         * on each fleet instance.
         * </p>
         * 
         * @param containerGroupType
         *        The type of container group. Container group type determines how Amazon GameLift deploys the container
         *        group on each fleet instance.
         * @see ContainerGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupType
         */
        Builder containerGroupType(String containerGroupType);

        /**
         * <p>
         * The type of container group. Container group type determines how Amazon GameLift deploys the container group
         * on each fleet instance.
         * </p>
         * 
         * @param containerGroupType
         *        The type of container group. Container group type determines how Amazon GameLift deploys the container
         *        group on each fleet instance.
         * @see ContainerGroupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupType
         */
        Builder containerGroupType(ContainerGroupType containerGroupType);

        /**
         * <p>
         * The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers in the
         * group share these resources.
         * </p>
         * <p>
         * You can set a limit for each container definition in the group. If individual containers have limits, this
         * total value must be greater than any individual container's memory limit.
         * </p>
         * 
         * @param totalMemoryLimitMebibytes
         *        The amount of memory (in MiB) on a fleet instance to allocate for the container group. All containers
         *        in the group share these resources. </p>
         *        <p>
         *        You can set a limit for each container definition in the group. If individual containers have limits,
         *        this total value must be greater than any individual container's memory limit.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalMemoryLimitMebibytes(Integer totalMemoryLimitMebibytes);

        /**
         * <p>
         * The amount of vCPU units on a fleet instance to allocate for the container group (1 vCPU is equal to 1024 CPU
         * units). All containers in the group share these resources. You can set a limit for each container definition
         * in the group. If individual containers have limits, this total value must be equal to or greater than the sum
         * of the limits for each container in the group.
         * </p>
         * 
         * @param totalVcpuLimit
         *        The amount of vCPU units on a fleet instance to allocate for the container group (1 vCPU is equal to
         *        1024 CPU units). All containers in the group share these resources. You can set a limit for each
         *        container definition in the group. If individual containers have limits, this total value must be
         *        equal to or greater than the sum of the limits for each container in the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalVcpuLimit(Double totalVcpuLimit);

        /**
         * <p>
         * The definition for the game server container in this group. This property is used only when the container
         * group type is <code>GAME_SERVER</code>. This container definition specifies a container image with the game
         * server build.
         * </p>
         * 
         * @param gameServerContainerDefinition
         *        The definition for the game server container in this group. This property is used only when the
         *        container group type is <code>GAME_SERVER</code>. This container definition specifies a container
         *        image with the game server build.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerContainerDefinition(GameServerContainerDefinition gameServerContainerDefinition);

        /**
         * <p>
         * The definition for the game server container in this group. This property is used only when the container
         * group type is <code>GAME_SERVER</code>. This container definition specifies a container image with the game
         * server build.
         * </p>
         * This is a convenience method that creates an instance of the {@link GameServerContainerDefinition.Builder}
         * avoiding the need to create one manually via {@link GameServerContainerDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GameServerContainerDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #gameServerContainerDefinition(GameServerContainerDefinition)}.
         * 
         * @param gameServerContainerDefinition
         *        a consumer that will call methods on {@link GameServerContainerDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameServerContainerDefinition(GameServerContainerDefinition)
         */
        default Builder gameServerContainerDefinition(
                Consumer<GameServerContainerDefinition.Builder> gameServerContainerDefinition) {
            return gameServerContainerDefinition(GameServerContainerDefinition.builder()
                    .applyMutation(gameServerContainerDefinition).build());
        }

        /**
         * <p>
         * The set of definitions for support containers in this group. A container group definition might have zero
         * support container definitions. Support container can be used in any type of container group.
         * </p>
         * 
         * @param supportContainerDefinitions
         *        The set of definitions for support containers in this group. A container group definition might have
         *        zero support container definitions. Support container can be used in any type of container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportContainerDefinitions(Collection<SupportContainerDefinition> supportContainerDefinitions);

        /**
         * <p>
         * The set of definitions for support containers in this group. A container group definition might have zero
         * support container definitions. Support container can be used in any type of container group.
         * </p>
         * 
         * @param supportContainerDefinitions
         *        The set of definitions for support containers in this group. A container group definition might have
         *        zero support container definitions. Support container can be used in any type of container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportContainerDefinitions(SupportContainerDefinition... supportContainerDefinitions);

        /**
         * <p>
         * The set of definitions for support containers in this group. A container group definition might have zero
         * support container definitions. Support container can be used in any type of container group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.SupportContainerDefinition.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.gamelift.model.SupportContainerDefinition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.SupportContainerDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #supportContainerDefinitions(List<SupportContainerDefinition>)}.
         * 
         * @param supportContainerDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.SupportContainerDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportContainerDefinitions(java.util.Collection<SupportContainerDefinition>)
         */
        Builder supportContainerDefinitions(Consumer<SupportContainerDefinition.Builder>... supportContainerDefinitions);

        /**
         * <p>
         * Indicates the version of a particular container group definition. This number is incremented automatically
         * when you update a container group definition. You can view, update, or delete individual versions or the
         * entire container group definition.
         * </p>
         * 
         * @param versionNumber
         *        Indicates the version of a particular container group definition. This number is incremented
         *        automatically when you update a container group definition. You can view, update, or delete individual
         *        versions or the entire container group definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionNumber(Integer versionNumber);

        /**
         * <p>
         * An optional description that was provided for a container group definition update. Each version can have a
         * unique description.
         * </p>
         * 
         * @param versionDescription
         *        An optional description that was provided for a container group definition update. Each version can
         *        have a unique description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionDescription(String versionDescription);

        /**
         * <p>
         * Current status of the container group definition resource. Values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
         * defined in the group. While in this state, the resource can't be used to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in
         * the group. You can use a container group definition in this status to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
         * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
         * failed status will be deleted within a few minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the container group definition resource. Values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images
         *        that are defined in the group. While in this state, the resource can't be used to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are
         *        defined in the group. You can use a container group definition in this status to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource.
         *        For more details on the cause of the failure, see <code>StatusReason</code>. A container group
         *        definition resource in failed status will be deleted within a few minutes.
         *        </p>
         *        </li>
         * @see ContainerGroupDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupDefinitionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the container group definition resource. Values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images that are
         * defined in the group. While in this state, the resource can't be used to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are defined in
         * the group. You can use a container group definition in this status to create a container fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource. For more
         * details on the cause of the failure, see <code>StatusReason</code>. A container group definition resource in
         * failed status will be deleted within a few minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the container group definition resource. Values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COPYING</code> -- Amazon GameLift is in the process of making copies of all container images
         *        that are defined in the group. While in this state, the resource can't be used to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READY</code> -- Amazon GameLift has copied the registry images for all containers that are
         *        defined in the group. You can use a container group definition in this status to create a container
         *        fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> -- Amazon GameLift failed to create a valid container group definition resource.
         *        For more details on the cause of the failure, see <code>StatusReason</code>. A container group
         *        definition resource in failed status will be deleted within a few minutes.
         *        </p>
         *        </li>
         * @see ContainerGroupDefinitionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContainerGroupDefinitionStatus
         */
        Builder status(ContainerGroupDefinitionStatus status);

        /**
         * <p>
         * Additional information about a container group definition that's in <code>FAILED</code> status. Possible
         * reasons include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An internal issue prevented Amazon GameLift from creating the container group definition resource. Delete the
         * failed resource and call <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
         * >CreateContainerGroupDefinition</a>again.
         * </p>
         * </li>
         * <li>
         * <p>
         * An access-denied message means that you don't have permissions to access the container image on ECR. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html.html"> IAM
         * permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>ImageUri</code> value for at least one of the containers in the container group definition was
         * invalid or not found in the current Amazon Web Services account.
         * </p>
         * </li>
         * <li>
         * <p>
         * At least one of the container images referenced in the container group definition exceeds the allowed size.
         * For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html"> Amazon GameLift
         * endpoints and quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * At least one of the container images referenced in the container group definition uses a different operating
         * system than the one defined for the container group.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statusReason
         *        Additional information about a container group definition that's in <code>FAILED</code> status.
         *        Possible reasons include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        An internal issue prevented Amazon GameLift from creating the container group definition resource.
         *        Delete the failed resource and call <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
         *        >CreateContainerGroupDefinition</a>again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An access-denied message means that you don't have permissions to access the container image on ECR.
         *        See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-iam-policy-examples.html.html">
         *        IAM permission examples</a> for help setting up required IAM permissions for Amazon GameLift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>ImageUri</code> value for at least one of the containers in the container group definition
         *        was invalid or not found in the current Amazon Web Services account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        At least one of the container images referenced in the container group definition exceeds the allowed
         *        size. For size limits, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">
         *        Amazon GameLift endpoints and quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        At least one of the container images referenced in the container group definition uses a different
         *        operating system than the one defined for the container group.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);
    }

    static final class BuilderImpl implements Builder {
        private String containerGroupDefinitionArn;

        private Instant creationTime;

        private String operatingSystem;

        private String name;

        private String containerGroupType;

        private Integer totalMemoryLimitMebibytes;

        private Double totalVcpuLimit;

        private GameServerContainerDefinition gameServerContainerDefinition;

        private List<SupportContainerDefinition> supportContainerDefinitions = DefaultSdkAutoConstructList.getInstance();

        private Integer versionNumber;

        private String versionDescription;

        private String status;

        private String statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerGroupDefinition model) {
            containerGroupDefinitionArn(model.containerGroupDefinitionArn);
            creationTime(model.creationTime);
            operatingSystem(model.operatingSystem);
            name(model.name);
            containerGroupType(model.containerGroupType);
            totalMemoryLimitMebibytes(model.totalMemoryLimitMebibytes);
            totalVcpuLimit(model.totalVcpuLimit);
            gameServerContainerDefinition(model.gameServerContainerDefinition);
            supportContainerDefinitions(model.supportContainerDefinitions);
            versionNumber(model.versionNumber);
            versionDescription(model.versionDescription);
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getContainerGroupDefinitionArn() {
            return containerGroupDefinitionArn;
        }

        public final void setContainerGroupDefinitionArn(String containerGroupDefinitionArn) {
            this.containerGroupDefinitionArn = containerGroupDefinitionArn;
        }

        @Override
        public final Builder containerGroupDefinitionArn(String containerGroupDefinitionArn) {
            this.containerGroupDefinitionArn = containerGroupDefinitionArn;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(ContainerOperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getContainerGroupType() {
            return containerGroupType;
        }

        public final void setContainerGroupType(String containerGroupType) {
            this.containerGroupType = containerGroupType;
        }

        @Override
        public final Builder containerGroupType(String containerGroupType) {
            this.containerGroupType = containerGroupType;
            return this;
        }

        @Override
        public final Builder containerGroupType(ContainerGroupType containerGroupType) {
            this.containerGroupType(containerGroupType == null ? null : containerGroupType.toString());
            return this;
        }

        public final Integer getTotalMemoryLimitMebibytes() {
            return totalMemoryLimitMebibytes;
        }

        public final void setTotalMemoryLimitMebibytes(Integer totalMemoryLimitMebibytes) {
            this.totalMemoryLimitMebibytes = totalMemoryLimitMebibytes;
        }

        @Override
        public final Builder totalMemoryLimitMebibytes(Integer totalMemoryLimitMebibytes) {
            this.totalMemoryLimitMebibytes = totalMemoryLimitMebibytes;
            return this;
        }

        public final Double getTotalVcpuLimit() {
            return totalVcpuLimit;
        }

        public final void setTotalVcpuLimit(Double totalVcpuLimit) {
            this.totalVcpuLimit = totalVcpuLimit;
        }

        @Override
        public final Builder totalVcpuLimit(Double totalVcpuLimit) {
            this.totalVcpuLimit = totalVcpuLimit;
            return this;
        }

        public final GameServerContainerDefinition.Builder getGameServerContainerDefinition() {
            return gameServerContainerDefinition != null ? gameServerContainerDefinition.toBuilder() : null;
        }

        public final void setGameServerContainerDefinition(GameServerContainerDefinition.BuilderImpl gameServerContainerDefinition) {
            this.gameServerContainerDefinition = gameServerContainerDefinition != null ? gameServerContainerDefinition.build()
                    : null;
        }

        @Override
        public final Builder gameServerContainerDefinition(GameServerContainerDefinition gameServerContainerDefinition) {
            this.gameServerContainerDefinition = gameServerContainerDefinition;
            return this;
        }

        public final List<SupportContainerDefinition.Builder> getSupportContainerDefinitions() {
            List<SupportContainerDefinition.Builder> result = SupportContainerDefinitionListCopier
                    .copyToBuilder(this.supportContainerDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSupportContainerDefinitions(
                Collection<SupportContainerDefinition.BuilderImpl> supportContainerDefinitions) {
            this.supportContainerDefinitions = SupportContainerDefinitionListCopier.copyFromBuilder(supportContainerDefinitions);
        }

        @Override
        public final Builder supportContainerDefinitions(Collection<SupportContainerDefinition> supportContainerDefinitions) {
            this.supportContainerDefinitions = SupportContainerDefinitionListCopier.copy(supportContainerDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportContainerDefinitions(SupportContainerDefinition... supportContainerDefinitions) {
            supportContainerDefinitions(Arrays.asList(supportContainerDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportContainerDefinitions(
                Consumer<SupportContainerDefinition.Builder>... supportContainerDefinitions) {
            supportContainerDefinitions(Stream.of(supportContainerDefinitions)
                    .map(c -> SupportContainerDefinition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getVersionNumber() {
            return versionNumber;
        }

        public final void setVersionNumber(Integer versionNumber) {
            this.versionNumber = versionNumber;
        }

        @Override
        public final Builder versionNumber(Integer versionNumber) {
            this.versionNumber = versionNumber;
            return this;
        }

        public final String getVersionDescription() {
            return versionDescription;
        }

        public final void setVersionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
        }

        @Override
        public final Builder versionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ContainerGroupDefinitionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        @Override
        public ContainerGroupDefinition build() {
            return new ContainerGroupDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
