/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon GameLift fleet of game hosting resources. Attributes differ based on the fleet's compute type, as
 * follows:
 * </p>
 * <ul>
 * <li>
 * <p>
 * EC2 fleet attributes identify a <code>Build</code> resource (for fleets with customer game server builds) or a
 * <code>Script</code> resource (for Realtime Servers fleets).
 * </p>
 * </li>
 * <li>
 * <p>
 * Amazon GameLift Anywhere fleets have an abbreviated set of attributes, because most fleet configurations are set
 * directly on the fleet's computes. Attributes include fleet identifiers and descriptive properties,
 * creation/termination time, and fleet status.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Returned by:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes">https
 * ://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeFleetAttributes</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FleetAttributes implements SdkPojo, Serializable, ToCopyableBuilder<FleetAttributes.Builder, FleetAttributes> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(FleetAttributes::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetArn").getter(getter(FleetAttributes::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<String> FLEET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetType").getter(getter(FleetAttributes::fleetTypeAsString)).setter(setter(Builder::fleetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetType").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(FleetAttributes::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(FleetAttributes::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(FleetAttributes::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(FleetAttributes::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> TERMINATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TerminationTime").getter(getter(FleetAttributes::terminationTime))
            .setter(setter(Builder::terminationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(FleetAttributes::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> BUILD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildId").getter(getter(FleetAttributes::buildId)).setter(setter(Builder::buildId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildId").build()).build();

    private static final SdkField<String> BUILD_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BuildArn").getter(getter(FleetAttributes::buildArn)).setter(setter(Builder::buildArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BuildArn").build()).build();

    private static final SdkField<String> SCRIPT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScriptId").getter(getter(FleetAttributes::scriptId)).setter(setter(Builder::scriptId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScriptId").build()).build();

    private static final SdkField<String> SCRIPT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScriptArn").getter(getter(FleetAttributes::scriptArn)).setter(setter(Builder::scriptArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScriptArn").build()).build();

    private static final SdkField<String> SERVER_LAUNCH_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerLaunchPath").getter(getter(FleetAttributes::serverLaunchPath))
            .setter(setter(Builder::serverLaunchPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerLaunchPath").build()).build();

    private static final SdkField<String> SERVER_LAUNCH_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerLaunchParameters").getter(getter(FleetAttributes::serverLaunchParameters))
            .setter(setter(Builder::serverLaunchParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerLaunchParameters").build())
            .build();

    private static final SdkField<List<String>> LOG_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LogPaths")
            .getter(getter(FleetAttributes::logPaths))
            .setter(setter(Builder::logPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogPaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEW_GAME_SESSION_PROTECTION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NewGameSessionProtectionPolicy")
            .getter(getter(FleetAttributes::newGameSessionProtectionPolicyAsString))
            .setter(setter(Builder::newGameSessionProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewGameSessionProtectionPolicy")
                    .build()).build();

    private static final SdkField<String> OPERATING_SYSTEM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperatingSystem").getter(getter(FleetAttributes::operatingSystemAsString))
            .setter(setter(Builder::operatingSystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingSystem").build()).build();

    private static final SdkField<ResourceCreationLimitPolicy> RESOURCE_CREATION_LIMIT_POLICY_FIELD = SdkField
            .<ResourceCreationLimitPolicy> builder(MarshallingType.SDK_POJO)
            .memberName("ResourceCreationLimitPolicy")
            .getter(getter(FleetAttributes::resourceCreationLimitPolicy))
            .setter(setter(Builder::resourceCreationLimitPolicy))
            .constructor(ResourceCreationLimitPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCreationLimitPolicy")
                    .build()).build();

    private static final SdkField<List<String>> METRIC_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MetricGroups")
            .getter(getter(FleetAttributes::metricGroups))
            .setter(setter(Builder::metricGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> STOPPED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("StoppedActions")
            .getter(getter(FleetAttributes::stoppedActionsAsStrings))
            .setter(setter(Builder::stoppedActionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceRoleArn").getter(getter(FleetAttributes::instanceRoleArn))
            .setter(setter(Builder::instanceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRoleArn").build()).build();

    private static final SdkField<CertificateConfiguration> CERTIFICATE_CONFIGURATION_FIELD = SdkField
            .<CertificateConfiguration> builder(MarshallingType.SDK_POJO).memberName("CertificateConfiguration")
            .getter(getter(FleetAttributes::certificateConfiguration)).setter(setter(Builder::certificateConfiguration))
            .constructor(CertificateConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateConfiguration").build())
            .build();

    private static final SdkField<String> COMPUTE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComputeType").getter(getter(FleetAttributes::computeTypeAsString)).setter(setter(Builder::computeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeType").build()).build();

    private static final SdkField<AnywhereConfiguration> ANYWHERE_CONFIGURATION_FIELD = SdkField
            .<AnywhereConfiguration> builder(MarshallingType.SDK_POJO).memberName("AnywhereConfiguration")
            .getter(getter(FleetAttributes::anywhereConfiguration)).setter(setter(Builder::anywhereConfiguration))
            .constructor(AnywhereConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnywhereConfiguration").build())
            .build();

    private static final SdkField<String> INSTANCE_ROLE_CREDENTIALS_PROVIDER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceRoleCredentialsProvider")
            .getter(getter(FleetAttributes::instanceRoleCredentialsProviderAsString))
            .setter(setter(Builder::instanceRoleCredentialsProvider))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRoleCredentialsProvider")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD,
            FLEET_ARN_FIELD, FLEET_TYPE_FIELD, INSTANCE_TYPE_FIELD, DESCRIPTION_FIELD, NAME_FIELD, CREATION_TIME_FIELD,
            TERMINATION_TIME_FIELD, STATUS_FIELD, BUILD_ID_FIELD, BUILD_ARN_FIELD, SCRIPT_ID_FIELD, SCRIPT_ARN_FIELD,
            SERVER_LAUNCH_PATH_FIELD, SERVER_LAUNCH_PARAMETERS_FIELD, LOG_PATHS_FIELD, NEW_GAME_SESSION_PROTECTION_POLICY_FIELD,
            OPERATING_SYSTEM_FIELD, RESOURCE_CREATION_LIMIT_POLICY_FIELD, METRIC_GROUPS_FIELD, STOPPED_ACTIONS_FIELD,
            INSTANCE_ROLE_ARN_FIELD, CERTIFICATE_CONFIGURATION_FIELD, COMPUTE_TYPE_FIELD, ANYWHERE_CONFIGURATION_FIELD,
            INSTANCE_ROLE_CREDENTIALS_PROVIDER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String fleetArn;

    private final String fleetType;

    private final String instanceType;

    private final String description;

    private final String name;

    private final Instant creationTime;

    private final Instant terminationTime;

    private final String status;

    private final String buildId;

    private final String buildArn;

    private final String scriptId;

    private final String scriptArn;

    private final String serverLaunchPath;

    private final String serverLaunchParameters;

    private final List<String> logPaths;

    private final String newGameSessionProtectionPolicy;

    private final String operatingSystem;

    private final ResourceCreationLimitPolicy resourceCreationLimitPolicy;

    private final List<String> metricGroups;

    private final List<String> stoppedActions;

    private final String instanceRoleArn;

    private final CertificateConfiguration certificateConfiguration;

    private final String computeType;

    private final AnywhereConfiguration anywhereConfiguration;

    private final String instanceRoleCredentialsProvider;

    private FleetAttributes(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.fleetType = builder.fleetType;
        this.instanceType = builder.instanceType;
        this.description = builder.description;
        this.name = builder.name;
        this.creationTime = builder.creationTime;
        this.terminationTime = builder.terminationTime;
        this.status = builder.status;
        this.buildId = builder.buildId;
        this.buildArn = builder.buildArn;
        this.scriptId = builder.scriptId;
        this.scriptArn = builder.scriptArn;
        this.serverLaunchPath = builder.serverLaunchPath;
        this.serverLaunchParameters = builder.serverLaunchParameters;
        this.logPaths = builder.logPaths;
        this.newGameSessionProtectionPolicy = builder.newGameSessionProtectionPolicy;
        this.operatingSystem = builder.operatingSystem;
        this.resourceCreationLimitPolicy = builder.resourceCreationLimitPolicy;
        this.metricGroups = builder.metricGroups;
        this.stoppedActions = builder.stoppedActions;
        this.instanceRoleArn = builder.instanceRoleArn;
        this.certificateConfiguration = builder.certificateConfiguration;
        this.computeType = builder.computeType;
        this.anywhereConfiguration = builder.anywhereConfiguration;
        this.instanceRoleCredentialsProvider = builder.instanceRoleCredentialsProvider;
    }

    /**
     * <p>
     * A unique identifier for the fleet.
     * </p>
     * 
     * @return A unique identifier for the fleet.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * that is assigned to a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all
     * Regions. Format is <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>
     * . In a GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
     *         Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
     *         <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
     *         GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
     */
    public final String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     * > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fleetType} will
     * return {@link FleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fleetTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     *         > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
     * @see FleetType
     */
    public final FleetType fleetType() {
        return FleetType.fromValue(fleetType);
    }

    /**
     * <p>
     * Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     * > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fleetType} will
     * return {@link FleetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fleetTypeAsString}.
     * </p>
     * 
     * @return Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
     *         > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
     * @see FleetType
     */
    public final String fleetTypeAsString() {
        return fleetType;
    }

    /**
     * <p>
     * The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of each
     * instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for detailed
     * descriptions. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of
     *         each instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for
     *         detailed descriptions. This attribute is used with fleets where <code>ComputeType</code> is
     *         <code>EC2</code>.
     * @see EC2InstanceType
     */
    public final EC2InstanceType instanceType() {
        return EC2InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of each
     * instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
     * href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for detailed
     * descriptions. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link EC2InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of
     *         each instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
     *         href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for
     *         detailed descriptions. This attribute is used with fleets where <code>ComputeType</code> is
     *         <code>EC2</code>.
     * @see EC2InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * A human-readable description of the fleet.
     * </p>
     * 
     * @return A human-readable description of the fleet.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time
     *         as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant terminationTime() {
        return terminationTime;
    }

    /**
     * <p>
     * Current status of the fleet. Possible fleet statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
     * instances is set to 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install scripts,
     * and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the Amazon
     * GameLift service.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE -- The fleet is now ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING -- Hosts are responding to a delete fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * TERMINATED -- The fleet no longer exists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the fleet. Possible fleet statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
     *         instances is set to 1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
     *         scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the
     *         Amazon GameLift service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE -- The fleet is now ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING -- Hosts are responding to a delete fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TERMINATED -- The fleet no longer exists.
     *         </p>
     *         </li>
     * @see FleetStatus
     */
    public final FleetStatus status() {
        return FleetStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the fleet. Possible fleet statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
     * instances is set to 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install scripts,
     * and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the Amazon
     * GameLift service.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE -- The fleet is now ready to host game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING -- Hosts are responding to a delete fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * TERMINATED -- The fleet no longer exists.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FleetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the fleet. Possible fleet statuses include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
     *         instances is set to 1.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
     *         scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the
     *         Amazon GameLift service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE -- The fleet is now ready to host game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING -- Hosts are responding to a delete fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TERMINATED -- The fleet no longer exists.
     *         </p>
     *         </li>
     * @see FleetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A unique identifier for the build resource that is deployed on instances in this fleet. This attribute is used
     * with fleets where <code>ComputeType</code> is "EC2".
     * </p>
     * 
     * @return A unique identifier for the build resource that is deployed on instances in this fleet. This attribute is
     *         used with fleets where <code>ComputeType</code> is "EC2".
     */
    public final String buildId() {
        return buildId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the Amazon GameLift build resource that is deployed on instances in this fleet. In a GameLift
     * build ARN, the resource ID matches the <code>BuildId</code> value. This attribute is used with fleets where
     * <code>ComputeType</code> is "EC2".
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         Amazon GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN, the
     *         resource ID matches the <code>BuildId</code> value. This attribute is used with fleets where
     *         <code>ComputeType</code> is "EC2".
     */
    public final String buildArn() {
        return buildArn;
    }

    /**
     * <p>
     * A unique identifier for the Realtime script resource that is deployed on instances in this fleet. This attribute
     * is used with fleets where <code>ComputeType</code> is "EC2".
     * </p>
     * 
     * @return A unique identifier for the Realtime script resource that is deployed on instances in this fleet. This
     *         attribute is used with fleets where <code>ComputeType</code> is "EC2".
     */
    public final String scriptId() {
        return scriptId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * associated with the GameLift script resource that is deployed on instances in this fleet. In a GameLift script
     * ARN, the resource ID matches the <code>ScriptId</code> value.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
     *         GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the
     *         resource ID matches the <code>ScriptId</code> value.
     */
    public final String scriptArn() {
        return scriptArn;
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/RuntimeConfiguration.html"
     * >RuntimeConfiguration</a>. Requests that use this parameter continue to be valid.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/RuntimeConfiguration.html"
     *         >RuntimeConfiguration</a>. Requests that use this parameter continue to be valid.
     */
    public final String serverLaunchPath() {
        return serverLaunchPath;
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's runtime
     * configuration. Requests that use this parameter continue to be valid.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
     *         runtime configuration. Requests that use this parameter continue to be valid.
     */
    public final String serverLaunchParameters() {
        return serverLaunchParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the LogPaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasLogPaths() {
        return logPaths != null && !(logPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon GameLift server
     * API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
     * >Server API Reference</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLogPaths} method.
     * </p>
     * 
     * @return <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon GameLift
     *         server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
     *         >Server API Reference</a>.
     */
    public final List<String> logPaths() {
        return logPaths;
    }

    /**
     * <p>
     * The type of game session protection to set on all new instances that are started in the fleet. This attribute is
     * used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated during
     * a scale-down event.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return The type of game session protection to set on all new instances that are started in the fleet. This
     *         attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
     *         during a scale-down event.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final ProtectionPolicy newGameSessionProtectionPolicy() {
        return ProtectionPolicy.fromValue(newGameSessionProtectionPolicy);
    }

    /**
     * <p>
     * The type of game session protection to set on all new instances that are started in the fleet. This attribute is
     * used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated during
     * a scale-down event.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #newGameSessionProtectionPolicy} will return {@link ProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #newGameSessionProtectionPolicyAsString}.
     * </p>
     * 
     * @return The type of game session protection to set on all new instances that are started in the fleet. This
     *         attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
     *         during a scale-down event.
     *         </p>
     *         </li>
     * @see ProtectionPolicy
     */
    public final String newGameSessionProtectionPolicyAsString() {
        return newGameSessionProtectionPolicy;
    }

    /**
     * <p>
     * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS of
     * the build or script that is deployed on this fleet. This attribute is used with fleets where
     * <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and then
     * deploy to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The operating system of the fleet's computing resources. A fleet's operating system is determined by the
     *         OS of the build or script that is deployed on this fleet. This attribute is used with fleets where
     *         <code>ComputeType</code> is <code>EC2</code>.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see OperatingSystem
     */
    public final OperatingSystem operatingSystem() {
        return OperatingSystem.fromValue(operatingSystem);
    }

    /**
     * <p>
     * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS of
     * the build or script that is deployed on this fleet. This attribute is used with fleets where
     * <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <note>
     * <p>
     * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted on
     * AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and then
     * deploy to AL2023 instances. See <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html"> Migrate to
     * Amazon GameLift server SDK version 5.</a>
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operatingSystem}
     * will return {@link OperatingSystem#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operatingSystemAsString}.
     * </p>
     * 
     * @return The operating system of the fleet's computing resources. A fleet's operating system is determined by the
     *         OS of the build or script that is deployed on this fleet. This attribute is used with fleets where
     *         <code>ComputeType</code> is <code>EC2</code>.</p> <note>
     *         <p>
     *         Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
     *         href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
     *         hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK
     *         5.x, and then deploy to AL2023 instances. See <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
     *         Migrate to Amazon GameLift server SDK version 5.</a>
     *         </p>
     * @see OperatingSystem
     */
    public final String operatingSystemAsString() {
        return operatingSystem;
    }

    /**
     * Returns the value of the ResourceCreationLimitPolicy property for this object.
     * 
     * @return The value of the ResourceCreationLimitPolicy property for this object.
     */
    public final ResourceCreationLimitPolicy resourceCreationLimitPolicy() {
        return resourceCreationLimitPolicy;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricGroups property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricGroups() {
        return metricGroups != null && !(metricGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view aggregated
     * metrics for fleets that are in a metric group. A fleet can be included in only one metric group at a time. This
     * attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricGroups} method.
     * </p>
     * 
     * @return Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
     *         aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
     *         group at a time. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
     */
    public final List<String> metricGroups() {
        return metricGroups;
    }

    /**
     * <p>
     * A list of fleet activity that has been suspended using <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html">StopFleetActions</a>.
     * This includes fleet auto-scaling. This attribute is used with fleets where <code>ComputeType</code> is
     * <code>EC2</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStoppedActions} method.
     * </p>
     * 
     * @return A list of fleet activity that has been suspended using <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
     *         >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
     *         <code>ComputeType</code> is <code>EC2</code>.
     */
    public final List<FleetAction> stoppedActions() {
        return FleetActionListCopier.copyStringToEnum(stoppedActions);
    }

    /**
     * For responses, this returns true if the service returned a value for the StoppedActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStoppedActions() {
        return stoppedActions != null && !(stoppedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of fleet activity that has been suspended using <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html">StopFleetActions</a>.
     * This includes fleet auto-scaling. This attribute is used with fleets where <code>ComputeType</code> is
     * <code>EC2</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStoppedActions} method.
     * </p>
     * 
     * @return A list of fleet activity that has been suspended using <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
     *         >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
     *         <code>ComputeType</code> is <code>EC2</code>.
     */
    public final List<String> stoppedActionsAsStrings() {
        return stoppedActions;
    }

    /**
     * <p>
     * A unique identifier for an IAM role that manages access to your Amazon Web Services services. With an instance
     * role ARN set, any application that runs on an instance in this fleet can assume the role, including install
     * scripts, server processes, and daemons (background processes). Create a role or look up a role's ARN by using the
     * <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the Amazon Web Services Management Console.
     * Learn more about using on-box credentials for your game servers at <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Access
     * external resources from a game server</a>. This attribute is used with fleets where <code>ComputeType</code> is
     * <code>EC2</code>.
     * </p>
     * 
     * @return A unique identifier for an IAM role that manages access to your Amazon Web Services services. With an
     *         instance role ARN set, any application that runs on an instance in this fleet can assume the role,
     *         including install scripts, server processes, and daemons (background processes). Create a role or look up
     *         a role's ARN by using the <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the Amazon
     *         Web Services Management Console. Learn more about using on-box credentials for your game servers at <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
     *         Access external resources from a game server</a>. This attribute is used with fleets where
     *         <code>ComputeType</code> is <code>EC2</code>.
     */
    public final String instanceRoleArn() {
        return instanceRoleArn;
    }

    /**
     * <p>
     * Determines whether a TLS/SSL certificate is generated for a fleet. This feature must be enabled when creating the
     * fleet. All instances in a fleet share the same certificate.
     * </p>
     * 
     * @return Determines whether a TLS/SSL certificate is generated for a fleet. This feature must be enabled when
     *         creating the fleet. All instances in a fleet share the same certificate.
     */
    public final CertificateConfiguration certificateConfiguration() {
        return certificateConfiguration;
    }

    /**
     * <p>
     * The type of compute resource used to host your game servers. You can use your own compute resources with Amazon
     * GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeType} will
     * return {@link ComputeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeAsString}.
     * </p>
     * 
     * @return The type of compute resource used to host your game servers. You can use your own compute resources with
     *         Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
     * @see ComputeType
     */
    public final ComputeType computeType() {
        return ComputeType.fromValue(computeType);
    }

    /**
     * <p>
     * The type of compute resource used to host your game servers. You can use your own compute resources with Amazon
     * GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #computeType} will
     * return {@link ComputeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #computeTypeAsString}.
     * </p>
     * 
     * @return The type of compute resource used to host your game servers. You can use your own compute resources with
     *         Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
     * @see ComputeType
     */
    public final String computeTypeAsString() {
        return computeType;
    }

    /**
     * <p>
     * A set of attributes that are specific to an Anywhere fleet.
     * </p>
     * 
     * @return A set of attributes that are specific to an Anywhere fleet.
     */
    public final AnywhereConfiguration anywhereConfiguration() {
        return anywhereConfiguration;
    }

    /**
     * <p>
     * Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
     * <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game server
     * executable to communicate with other Amazon Web Services resources. This property is used only when the game
     * server is integrated with the server SDK version 5.x. For more information about using shared credentials, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Communicate
     * with other Amazon Web Services resources from your fleets</a>. This attribute is used with fleets where
     * <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceRoleCredentialsProvider} will return
     * {@link InstanceRoleCredentialsProvider#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceRoleCredentialsProviderAsString}.
     * </p>
     * 
     * @return Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
     *         <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game
     *         server executable to communicate with other Amazon Web Services resources. This property is used only
     *         when the game server is integrated with the server SDK version 5.x. For more information about using
     *         shared credentials, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
     *         Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with
     *         fleets where <code>ComputeType</code> is <code>EC2</code>.
     * @see InstanceRoleCredentialsProvider
     */
    public final InstanceRoleCredentialsProvider instanceRoleCredentialsProvider() {
        return InstanceRoleCredentialsProvider.fromValue(instanceRoleCredentialsProvider);
    }

    /**
     * <p>
     * Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
     * <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game server
     * executable to communicate with other Amazon Web Services resources. This property is used only when the game
     * server is integrated with the server SDK version 5.x. For more information about using shared credentials, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Communicate
     * with other Amazon Web Services resources from your fleets</a>. This attribute is used with fleets where
     * <code>ComputeType</code> is <code>EC2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceRoleCredentialsProvider} will return
     * {@link InstanceRoleCredentialsProvider#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceRoleCredentialsProviderAsString}.
     * </p>
     * 
     * @return Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
     *         <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game
     *         server executable to communicate with other Amazon Web Services resources. This property is used only
     *         when the game server is integrated with the server SDK version 5.x. For more information about using
     *         shared credentials, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
     *         Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with
     *         fleets where <code>ComputeType</code> is <code>EC2</code>.
     * @see InstanceRoleCredentialsProvider
     */
    public final String instanceRoleCredentialsProviderAsString() {
        return instanceRoleCredentialsProvider;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(fleetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(terminationTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(buildId());
        hashCode = 31 * hashCode + Objects.hashCode(buildArn());
        hashCode = 31 * hashCode + Objects.hashCode(scriptId());
        hashCode = 31 * hashCode + Objects.hashCode(scriptArn());
        hashCode = 31 * hashCode + Objects.hashCode(serverLaunchPath());
        hashCode = 31 * hashCode + Objects.hashCode(serverLaunchParameters());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogPaths() ? logPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(newGameSessionProtectionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operatingSystemAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCreationLimitPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricGroups() ? metricGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStoppedActions() ? stoppedActionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(computeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(anywhereConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(instanceRoleCredentialsProviderAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FleetAttributes)) {
            return false;
        }
        FleetAttributes other = (FleetAttributes) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(fleetArn(), other.fleetArn())
                && Objects.equals(fleetTypeAsString(), other.fleetTypeAsString())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(name(), other.name())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(terminationTime(), other.terminationTime())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(buildId(), other.buildId())
                && Objects.equals(buildArn(), other.buildArn()) && Objects.equals(scriptId(), other.scriptId())
                && Objects.equals(scriptArn(), other.scriptArn()) && Objects.equals(serverLaunchPath(), other.serverLaunchPath())
                && Objects.equals(serverLaunchParameters(), other.serverLaunchParameters())
                && hasLogPaths() == other.hasLogPaths() && Objects.equals(logPaths(), other.logPaths())
                && Objects.equals(newGameSessionProtectionPolicyAsString(), other.newGameSessionProtectionPolicyAsString())
                && Objects.equals(operatingSystemAsString(), other.operatingSystemAsString())
                && Objects.equals(resourceCreationLimitPolicy(), other.resourceCreationLimitPolicy())
                && hasMetricGroups() == other.hasMetricGroups() && Objects.equals(metricGroups(), other.metricGroups())
                && hasStoppedActions() == other.hasStoppedActions()
                && Objects.equals(stoppedActionsAsStrings(), other.stoppedActionsAsStrings())
                && Objects.equals(instanceRoleArn(), other.instanceRoleArn())
                && Objects.equals(certificateConfiguration(), other.certificateConfiguration())
                && Objects.equals(computeTypeAsString(), other.computeTypeAsString())
                && Objects.equals(anywhereConfiguration(), other.anywhereConfiguration())
                && Objects.equals(instanceRoleCredentialsProviderAsString(), other.instanceRoleCredentialsProviderAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FleetAttributes").add("FleetId", fleetId()).add("FleetArn", fleetArn())
                .add("FleetType", fleetTypeAsString()).add("InstanceType", instanceTypeAsString())
                .add("Description", description()).add("Name", name()).add("CreationTime", creationTime())
                .add("TerminationTime", terminationTime()).add("Status", statusAsString()).add("BuildId", buildId())
                .add("BuildArn", buildArn()).add("ScriptId", scriptId()).add("ScriptArn", scriptArn())
                .add("ServerLaunchPath", serverLaunchPath()).add("ServerLaunchParameters", serverLaunchParameters())
                .add("LogPaths", hasLogPaths() ? logPaths() : null)
                .add("NewGameSessionProtectionPolicy", newGameSessionProtectionPolicyAsString())
                .add("OperatingSystem", operatingSystemAsString())
                .add("ResourceCreationLimitPolicy", resourceCreationLimitPolicy())
                .add("MetricGroups", hasMetricGroups() ? metricGroups() : null)
                .add("StoppedActions", hasStoppedActions() ? stoppedActionsAsStrings() : null)
                .add("InstanceRoleArn", instanceRoleArn()).add("CertificateConfiguration", certificateConfiguration())
                .add("ComputeType", computeTypeAsString()).add("AnywhereConfiguration", anywhereConfiguration())
                .add("InstanceRoleCredentialsProvider", instanceRoleCredentialsProviderAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "FleetType":
            return Optional.ofNullable(clazz.cast(fleetTypeAsString()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "TerminationTime":
            return Optional.ofNullable(clazz.cast(terminationTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "BuildId":
            return Optional.ofNullable(clazz.cast(buildId()));
        case "BuildArn":
            return Optional.ofNullable(clazz.cast(buildArn()));
        case "ScriptId":
            return Optional.ofNullable(clazz.cast(scriptId()));
        case "ScriptArn":
            return Optional.ofNullable(clazz.cast(scriptArn()));
        case "ServerLaunchPath":
            return Optional.ofNullable(clazz.cast(serverLaunchPath()));
        case "ServerLaunchParameters":
            return Optional.ofNullable(clazz.cast(serverLaunchParameters()));
        case "LogPaths":
            return Optional.ofNullable(clazz.cast(logPaths()));
        case "NewGameSessionProtectionPolicy":
            return Optional.ofNullable(clazz.cast(newGameSessionProtectionPolicyAsString()));
        case "OperatingSystem":
            return Optional.ofNullable(clazz.cast(operatingSystemAsString()));
        case "ResourceCreationLimitPolicy":
            return Optional.ofNullable(clazz.cast(resourceCreationLimitPolicy()));
        case "MetricGroups":
            return Optional.ofNullable(clazz.cast(metricGroups()));
        case "StoppedActions":
            return Optional.ofNullable(clazz.cast(stoppedActionsAsStrings()));
        case "InstanceRoleArn":
            return Optional.ofNullable(clazz.cast(instanceRoleArn()));
        case "CertificateConfiguration":
            return Optional.ofNullable(clazz.cast(certificateConfiguration()));
        case "ComputeType":
            return Optional.ofNullable(clazz.cast(computeTypeAsString()));
        case "AnywhereConfiguration":
            return Optional.ofNullable(clazz.cast(anywhereConfiguration()));
        case "InstanceRoleCredentialsProvider":
            return Optional.ofNullable(clazz.cast(instanceRoleCredentialsProviderAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FleetId", FLEET_ID_FIELD);
        map.put("FleetArn", FLEET_ARN_FIELD);
        map.put("FleetType", FLEET_TYPE_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("TerminationTime", TERMINATION_TIME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("BuildId", BUILD_ID_FIELD);
        map.put("BuildArn", BUILD_ARN_FIELD);
        map.put("ScriptId", SCRIPT_ID_FIELD);
        map.put("ScriptArn", SCRIPT_ARN_FIELD);
        map.put("ServerLaunchPath", SERVER_LAUNCH_PATH_FIELD);
        map.put("ServerLaunchParameters", SERVER_LAUNCH_PARAMETERS_FIELD);
        map.put("LogPaths", LOG_PATHS_FIELD);
        map.put("NewGameSessionProtectionPolicy", NEW_GAME_SESSION_PROTECTION_POLICY_FIELD);
        map.put("OperatingSystem", OPERATING_SYSTEM_FIELD);
        map.put("ResourceCreationLimitPolicy", RESOURCE_CREATION_LIMIT_POLICY_FIELD);
        map.put("MetricGroups", METRIC_GROUPS_FIELD);
        map.put("StoppedActions", STOPPED_ACTIONS_FIELD);
        map.put("InstanceRoleArn", INSTANCE_ROLE_ARN_FIELD);
        map.put("CertificateConfiguration", CERTIFICATE_CONFIGURATION_FIELD);
        map.put("ComputeType", COMPUTE_TYPE_FIELD);
        map.put("AnywhereConfiguration", ANYWHERE_CONFIGURATION_FIELD);
        map.put("InstanceRoleCredentialsProvider", INSTANCE_ROLE_CREDENTIALS_PROVIDER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FleetAttributes, T> g) {
        return obj -> g.apply((FleetAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FleetAttributes> {
        /**
         * <p>
         * A unique identifier for the fleet.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for the fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to a
         * Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions. Format is
         * <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a GameLift
         * fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) that is assigned to
         *        a Amazon GameLift fleet resource and uniquely identifies it. ARNs are unique across all Regions.
         *        Format is
         *        <code>arn:aws:gamelift:&lt;region&gt;::fleet/fleet-a1234567-b8c9-0d1e-2fa3-b45c6d7e8912</code>. In a
         *        GameLift fleet ARN, the resource ID matches the <code>FleetId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         * > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
         * </p>
         * 
         * @param fleetType
         *        Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         *        > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is
         *        created.
         * @see FleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetType
         */
        Builder fleetType(String fleetType);

        /**
         * <p>
         * Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         * > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is created.
         * </p>
         * 
         * @param fleetType
         *        Indicates whether the fleet uses On-Demand or Spot instances. For more information, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-ec2-instances.html#gamelift-ec2-instances-spot"
         *        > On-Demand versus Spot Instances</a>. This fleet property can't be changed after the fleet is
         *        created.
         * @see FleetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetType
         */
        Builder fleetType(FleetType fleetType);

        /**
         * <p>
         * The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of each
         * instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
         * href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for detailed
         * descriptions. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of
         *        each instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for
         *        detailed descriptions. This attribute is used with fleets where <code>ComputeType</code> is
         *        <code>EC2</code>.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of each
         * instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
         * href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for detailed
         * descriptions. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type that the fleet uses. Instance type determines the computing resources of
         *        each instance in the fleet, including CPU, memory, storage, and networking capacity. See <a
         *        href="http://aws.amazon.com/ec2/instance-types/">Amazon Elastic Compute Cloud Instance Types</a> for
         *        detailed descriptions. This attribute is used with fleets where <code>ComputeType</code> is
         *        <code>EC2</code>.
         * @see EC2InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EC2InstanceType
         */
        Builder instanceType(EC2InstanceType instanceType);

        /**
         * <p>
         * A human-readable description of the fleet.
         * </p>
         * 
         * @param description
         *        A human-readable description of the fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a fleet. Fleet names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param terminationTime
         *        A time stamp indicating when this data object was terminated. Format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationTime(Instant terminationTime);

        /**
         * <p>
         * Current status of the fleet. Possible fleet statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
         * instances is set to 1.
         * </p>
         * </li>
         * <li>
         * <p>
         * DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
         * scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the Amazon
         * GameLift service.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE -- The fleet is now ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING -- Hosts are responding to a delete fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * TERMINATED -- The fleet no longer exists.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the fleet. Possible fleet statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet.
         *        Desired instances is set to 1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
         *        scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet
         *        instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the
         *        Amazon GameLift service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE -- The fleet is now ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING -- Hosts are responding to a delete fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TERMINATED -- The fleet no longer exists.
         *        </p>
         *        </li>
         * @see FleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the fleet. Possible fleet statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet. Desired
         * instances is set to 1.
         * </p>
         * </li>
         * <li>
         * <p>
         * DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
         * scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the Amazon
         * GameLift service.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE -- The fleet is now ready to host game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING -- Hosts are responding to a delete fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * TERMINATED -- The fleet no longer exists.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the fleet. Possible fleet statuses include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        NEW -- A new fleet resource has been defined and Amazon GameLift has started creating the fleet.
         *        Desired instances is set to 1.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DOWNLOADING/VALIDATING/BUILDING -- Amazon GameLift is download the game server build, running install
         *        scripts, and then validating the build files. When complete, Amazon GameLift launches a fleet
         *        instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVATING -- Amazon GameLift is launching a game server process and testing its connectivity with the
         *        Amazon GameLift service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE -- The fleet is now ready to host game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR -- An error occurred when downloading, validating, building, or activating the fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING -- Hosts are responding to a delete fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TERMINATED -- The fleet no longer exists.
         *        </p>
         *        </li>
         * @see FleetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetStatus
         */
        Builder status(FleetStatus status);

        /**
         * <p>
         * A unique identifier for the build resource that is deployed on instances in this fleet. This attribute is
         * used with fleets where <code>ComputeType</code> is "EC2".
         * </p>
         * 
         * @param buildId
         *        A unique identifier for the build resource that is deployed on instances in this fleet. This attribute
         *        is used with fleets where <code>ComputeType</code> is "EC2".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildId(String buildId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the Amazon
         * GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN, the resource ID
         * matches the <code>BuildId</code> value. This attribute is used with fleets where <code>ComputeType</code> is
         * "EC2".
         * </p>
         * 
         * @param buildArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        Amazon GameLift build resource that is deployed on instances in this fleet. In a GameLift build ARN,
         *        the resource ID matches the <code>BuildId</code> value. This attribute is used with fleets where
         *        <code>ComputeType</code> is "EC2".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buildArn(String buildArn);

        /**
         * <p>
         * A unique identifier for the Realtime script resource that is deployed on instances in this fleet. This
         * attribute is used with fleets where <code>ComputeType</code> is "EC2".
         * </p>
         * 
         * @param scriptId
         *        A unique identifier for the Realtime script resource that is deployed on instances in this fleet. This
         *        attribute is used with fleets where <code>ComputeType</code> is "EC2".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scriptId(String scriptId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         * GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the resource
         * ID matches the <code>ScriptId</code> value.
         * </p>
         * 
         * @param scriptArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) associated with the
         *        GameLift script resource that is deployed on instances in this fleet. In a GameLift script ARN, the
         *        resource ID matches the <code>ScriptId</code> value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scriptArn(String scriptArn);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/RuntimeConfiguration.html"
         * >RuntimeConfiguration</a>. Requests that use this parameter continue to be valid.
         * </p>
         * 
         * @param serverLaunchPath
         *        <b>This parameter is no longer used.</b> Server launch paths are now defined using the fleet's <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/RuntimeConfiguration.html">
         *        RuntimeConfiguration</a>. Requests that use this parameter continue to be valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverLaunchPath(String serverLaunchPath);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's runtime
         * configuration. Requests that use this parameter continue to be valid.
         * </p>
         * 
         * @param serverLaunchParameters
         *        <b>This parameter is no longer used.</b> Server launch parameters are now defined using the fleet's
         *        runtime configuration. Requests that use this parameter continue to be valid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverLaunchParameters(String serverLaunchParameters);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon GameLift
         * server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         * >Server API Reference</a>.
         * </p>
         * 
         * @param logPaths
         *        <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon
         *        GameLift server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in
         *        the <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         *        >Server API Reference</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPaths(Collection<String> logPaths);

        /**
         * <p>
         * <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon GameLift
         * server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in the <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         * >Server API Reference</a>.
         * </p>
         * 
         * @param logPaths
         *        <b>This parameter is no longer used.</b> Game session log paths are now defined using the Amazon
         *        GameLift server API <code>ProcessReady()</code> <code>logParameters</code>. See more information in
         *        the <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api-ref.html#gamelift-sdk-server-api-ref-dataypes-process"
         *        >Server API Reference</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logPaths(String... logPaths);

        /**
         * <p>
         * The type of game session protection to set on all new instances that are started in the fleet. This attribute
         * is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
         * during a scale-down event.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        The type of game session protection to set on all new instances that are started in the fleet. This
         *        attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be
         *        terminated during a scale-down event.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy);

        /**
         * <p>
         * The type of game session protection to set on all new instances that are started in the fleet. This attribute
         * is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be terminated
         * during a scale-down event.
         * </p>
         * </li>
         * </ul>
         * 
         * @param newGameSessionProtectionPolicy
         *        The type of game session protection to set on all new instances that are started in the fleet. This
         *        attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>NoProtection</b> -- The game session can be terminated during a scale-down event.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FullProtection</b> -- If the game session is in an <code>ACTIVE</code> status, it cannot be
         *        terminated during a scale-down event.
         *        </p>
         *        </li>
         * @see ProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProtectionPolicy
         */
        Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy);

        /**
         * <p>
         * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS
         * of the build or script that is deployed on this fleet. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and
         * then deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The operating system of the fleet's computing resources. A fleet's operating system is determined by
         *        the OS of the build or script that is deployed on this fleet. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server
         *        SDK 5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(String operatingSystem);

        /**
         * <p>
         * The operating system of the fleet's computing resources. A fleet's operating system is determined by the OS
         * of the build or script that is deployed on this fleet. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * <note>
         * <p>
         * Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         * href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are hosted
         * on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server SDK 5.x, and
         * then deploy to AL2023 instances. See <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         * Migrate to Amazon GameLift server SDK version 5.</a>
         * </p>
         * </note>
         * 
         * @param operatingSystem
         *        The operating system of the fleet's computing resources. A fleet's operating system is determined by
         *        the OS of the build or script that is deployed on this fleet. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.</p> <note>
         *        <p>
         *        Amazon Linux 2 (AL2) will reach end of support on 6/30/2025. See more details in the <a
         *        href="https://aws.amazon.com/amazon-linux-2/faqs/">Amazon Linux 2 FAQs</a>. For game servers that are
         *        hosted on AL2 and use Amazon GameLift server SDK 4.x., first update the game server build to server
         *        SDK 5.x, and then deploy to AL2023 instances. See <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-serversdk5-migration.html">
         *        Migrate to Amazon GameLift server SDK version 5.</a>
         *        </p>
         * @see OperatingSystem
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperatingSystem
         */
        Builder operatingSystem(OperatingSystem operatingSystem);

        /**
         * Sets the value of the ResourceCreationLimitPolicy property for this object.
         *
         * @param resourceCreationLimitPolicy
         *        The new value for the ResourceCreationLimitPolicy property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCreationLimitPolicy(ResourceCreationLimitPolicy resourceCreationLimitPolicy);

        /**
         * Sets the value of the ResourceCreationLimitPolicy property for this object.
         *
         * This is a convenience method that creates an instance of the {@link ResourceCreationLimitPolicy.Builder}
         * avoiding the need to create one manually via {@link ResourceCreationLimitPolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceCreationLimitPolicy.Builder#build()} is called
         * immediately and its result is passed to {@link #resourceCreationLimitPolicy(ResourceCreationLimitPolicy)}.
         * 
         * @param resourceCreationLimitPolicy
         *        a consumer that will call methods on {@link ResourceCreationLimitPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceCreationLimitPolicy(ResourceCreationLimitPolicy)
         */
        default Builder resourceCreationLimitPolicy(Consumer<ResourceCreationLimitPolicy.Builder> resourceCreationLimitPolicy) {
            return resourceCreationLimitPolicy(ResourceCreationLimitPolicy.builder().applyMutation(resourceCreationLimitPolicy)
                    .build());
        }

        /**
         * <p>
         * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         * aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric group at
         * a time. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param metricGroups
         *        Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         *        aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
         *        group at a time. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(Collection<String> metricGroups);

        /**
         * <p>
         * Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         * aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric group at
         * a time. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param metricGroups
         *        Name of a metric group that metrics for this fleet are added to. In Amazon CloudWatch, you can view
         *        aggregated metrics for fleets that are in a metric group. A fleet can be included in only one metric
         *        group at a time. This attribute is used with fleets where <code>ComputeType</code> is <code>EC2</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricGroups(String... metricGroups);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         * >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         *        >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(Collection<String> stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         * >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         *        >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActionsWithStrings(String... stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         * >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         *        >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(Collection<FleetAction> stoppedActions);

        /**
         * <p>
         * A list of fleet activity that has been suspended using <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         * >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         * <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param stoppedActions
         *        A list of fleet activity that has been suspended using <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_StopFleetActions.html"
         *        >StopFleetActions</a>. This includes fleet auto-scaling. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedActions(FleetAction... stoppedActions);

        /**
         * <p>
         * A unique identifier for an IAM role that manages access to your Amazon Web Services services. With an
         * instance role ARN set, any application that runs on an instance in this fleet can assume the role, including
         * install scripts, server processes, and daemons (background processes). Create a role or look up a role's ARN
         * by using the <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the Amazon Web Services
         * Management Console. Learn more about using on-box credentials for your game servers at <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html"> Access
         * external resources from a game server</a>. This attribute is used with fleets where <code>ComputeType</code>
         * is <code>EC2</code>.
         * </p>
         * 
         * @param instanceRoleArn
         *        A unique identifier for an IAM role that manages access to your Amazon Web Services services. With an
         *        instance role ARN set, any application that runs on an instance in this fleet can assume the role,
         *        including install scripts, server processes, and daemons (background processes). Create a role or look
         *        up a role's ARN by using the <a href="https://console.aws.amazon.com/iam/">IAM dashboard</a> in the
         *        Amazon Web Services Management Console. Learn more about using on-box credentials for your game
         *        servers at <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         *        Access external resources from a game server</a>. This attribute is used with fleets where
         *        <code>ComputeType</code> is <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRoleArn(String instanceRoleArn);

        /**
         * <p>
         * Determines whether a TLS/SSL certificate is generated for a fleet. This feature must be enabled when creating
         * the fleet. All instances in a fleet share the same certificate.
         * </p>
         * 
         * @param certificateConfiguration
         *        Determines whether a TLS/SSL certificate is generated for a fleet. This feature must be enabled when
         *        creating the fleet. All instances in a fleet share the same certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateConfiguration(CertificateConfiguration certificateConfiguration);

        /**
         * <p>
         * Determines whether a TLS/SSL certificate is generated for a fleet. This feature must be enabled when creating
         * the fleet. All instances in a fleet share the same certificate.
         * </p>
         * This is a convenience method that creates an instance of the {@link CertificateConfiguration.Builder}
         * avoiding the need to create one manually via {@link CertificateConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CertificateConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #certificateConfiguration(CertificateConfiguration)}.
         * 
         * @param certificateConfiguration
         *        a consumer that will call methods on {@link CertificateConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateConfiguration(CertificateConfiguration)
         */
        default Builder certificateConfiguration(Consumer<CertificateConfiguration.Builder> certificateConfiguration) {
            return certificateConfiguration(CertificateConfiguration.builder().applyMutation(certificateConfiguration).build());
        }

        /**
         * <p>
         * The type of compute resource used to host your game servers. You can use your own compute resources with
         * Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
         * </p>
         * 
         * @param computeType
         *        The type of compute resource used to host your game servers. You can use your own compute resources
         *        with Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
         * @see ComputeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeType
         */
        Builder computeType(String computeType);

        /**
         * <p>
         * The type of compute resource used to host your game servers. You can use your own compute resources with
         * Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
         * </p>
         * 
         * @param computeType
         *        The type of compute resource used to host your game servers. You can use your own compute resources
         *        with Amazon GameLift Anywhere or use Amazon EC2 instances with managed Amazon GameLift.
         * @see ComputeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComputeType
         */
        Builder computeType(ComputeType computeType);

        /**
         * <p>
         * A set of attributes that are specific to an Anywhere fleet.
         * </p>
         * 
         * @param anywhereConfiguration
         *        A set of attributes that are specific to an Anywhere fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anywhereConfiguration(AnywhereConfiguration anywhereConfiguration);

        /**
         * <p>
         * A set of attributes that are specific to an Anywhere fleet.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnywhereConfiguration.Builder} avoiding
         * the need to create one manually via {@link AnywhereConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnywhereConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #anywhereConfiguration(AnywhereConfiguration)}.
         * 
         * @param anywhereConfiguration
         *        a consumer that will call methods on {@link AnywhereConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anywhereConfiguration(AnywhereConfiguration)
         */
        default Builder anywhereConfiguration(Consumer<AnywhereConfiguration.Builder> anywhereConfiguration) {
            return anywhereConfiguration(AnywhereConfiguration.builder().applyMutation(anywhereConfiguration).build());
        }

        /**
         * <p>
         * Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
         * <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game server
         * executable to communicate with other Amazon Web Services resources. This property is used only when the game
         * server is integrated with the server SDK version 5.x. For more information about using shared credentials,
         * see <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         * Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with fleets
         * where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param instanceRoleCredentialsProvider
         *        Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
         *        <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game
         *        server executable to communicate with other Amazon Web Services resources. This property is used only
         *        when the game server is integrated with the server SDK version 5.x. For more information about using
         *        shared credentials, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         *        Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with
         *        fleets where <code>ComputeType</code> is <code>EC2</code>.
         * @see InstanceRoleCredentialsProvider
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleCredentialsProvider
         */
        Builder instanceRoleCredentialsProvider(String instanceRoleCredentialsProvider);

        /**
         * <p>
         * Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
         * <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game server
         * executable to communicate with other Amazon Web Services resources. This property is used only when the game
         * server is integrated with the server SDK version 5.x. For more information about using shared credentials,
         * see <a href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         * Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with fleets
         * where <code>ComputeType</code> is <code>EC2</code>.
         * </p>
         * 
         * @param instanceRoleCredentialsProvider
         *        Indicates that fleet instances maintain a shared credentials file for the IAM role defined in
         *        <code>InstanceRoleArn</code>. Shared credentials allow applications that are deployed with the game
         *        server executable to communicate with other Amazon Web Services resources. This property is used only
         *        when the game server is integrated with the server SDK version 5.x. For more information about using
         *        shared credentials, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-resources.html">
         *        Communicate with other Amazon Web Services resources from your fleets</a>. This attribute is used with
         *        fleets where <code>ComputeType</code> is <code>EC2</code>.
         * @see InstanceRoleCredentialsProvider
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleCredentialsProvider
         */
        Builder instanceRoleCredentialsProvider(InstanceRoleCredentialsProvider instanceRoleCredentialsProvider);
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String fleetArn;

        private String fleetType;

        private String instanceType;

        private String description;

        private String name;

        private Instant creationTime;

        private Instant terminationTime;

        private String status;

        private String buildId;

        private String buildArn;

        private String scriptId;

        private String scriptArn;

        private String serverLaunchPath;

        private String serverLaunchParameters;

        private List<String> logPaths = DefaultSdkAutoConstructList.getInstance();

        private String newGameSessionProtectionPolicy;

        private String operatingSystem;

        private ResourceCreationLimitPolicy resourceCreationLimitPolicy;

        private List<String> metricGroups = DefaultSdkAutoConstructList.getInstance();

        private List<String> stoppedActions = DefaultSdkAutoConstructList.getInstance();

        private String instanceRoleArn;

        private CertificateConfiguration certificateConfiguration;

        private String computeType;

        private AnywhereConfiguration anywhereConfiguration;

        private String instanceRoleCredentialsProvider;

        private BuilderImpl() {
        }

        private BuilderImpl(FleetAttributes model) {
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            fleetType(model.fleetType);
            instanceType(model.instanceType);
            description(model.description);
            name(model.name);
            creationTime(model.creationTime);
            terminationTime(model.terminationTime);
            status(model.status);
            buildId(model.buildId);
            buildArn(model.buildArn);
            scriptId(model.scriptId);
            scriptArn(model.scriptArn);
            serverLaunchPath(model.serverLaunchPath);
            serverLaunchParameters(model.serverLaunchParameters);
            logPaths(model.logPaths);
            newGameSessionProtectionPolicy(model.newGameSessionProtectionPolicy);
            operatingSystem(model.operatingSystem);
            resourceCreationLimitPolicy(model.resourceCreationLimitPolicy);
            metricGroups(model.metricGroups);
            stoppedActionsWithStrings(model.stoppedActions);
            instanceRoleArn(model.instanceRoleArn);
            certificateConfiguration(model.certificateConfiguration);
            computeType(model.computeType);
            anywhereConfiguration(model.anywhereConfiguration);
            instanceRoleCredentialsProvider(model.instanceRoleCredentialsProvider);
        }

        public final String getFleetId() {
            return fleetId;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final String getFleetType() {
            return fleetType;
        }

        public final void setFleetType(String fleetType) {
            this.fleetType = fleetType;
        }

        @Override
        public final Builder fleetType(String fleetType) {
            this.fleetType = fleetType;
            return this;
        }

        @Override
        public final Builder fleetType(FleetType fleetType) {
            this.fleetType(fleetType == null ? null : fleetType.toString());
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(EC2InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getTerminationTime() {
            return terminationTime;
        }

        public final void setTerminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
        }

        @Override
        public final Builder terminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FleetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getBuildId() {
            return buildId;
        }

        public final void setBuildId(String buildId) {
            this.buildId = buildId;
        }

        @Override
        public final Builder buildId(String buildId) {
            this.buildId = buildId;
            return this;
        }

        public final String getBuildArn() {
            return buildArn;
        }

        public final void setBuildArn(String buildArn) {
            this.buildArn = buildArn;
        }

        @Override
        public final Builder buildArn(String buildArn) {
            this.buildArn = buildArn;
            return this;
        }

        public final String getScriptId() {
            return scriptId;
        }

        public final void setScriptId(String scriptId) {
            this.scriptId = scriptId;
        }

        @Override
        public final Builder scriptId(String scriptId) {
            this.scriptId = scriptId;
            return this;
        }

        public final String getScriptArn() {
            return scriptArn;
        }

        public final void setScriptArn(String scriptArn) {
            this.scriptArn = scriptArn;
        }

        @Override
        public final Builder scriptArn(String scriptArn) {
            this.scriptArn = scriptArn;
            return this;
        }

        public final String getServerLaunchPath() {
            return serverLaunchPath;
        }

        public final void setServerLaunchPath(String serverLaunchPath) {
            this.serverLaunchPath = serverLaunchPath;
        }

        @Override
        public final Builder serverLaunchPath(String serverLaunchPath) {
            this.serverLaunchPath = serverLaunchPath;
            return this;
        }

        public final String getServerLaunchParameters() {
            return serverLaunchParameters;
        }

        public final void setServerLaunchParameters(String serverLaunchParameters) {
            this.serverLaunchParameters = serverLaunchParameters;
        }

        @Override
        public final Builder serverLaunchParameters(String serverLaunchParameters) {
            this.serverLaunchParameters = serverLaunchParameters;
            return this;
        }

        public final Collection<String> getLogPaths() {
            if (logPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return logPaths;
        }

        public final void setLogPaths(Collection<String> logPaths) {
            this.logPaths = StringListCopier.copy(logPaths);
        }

        @Override
        public final Builder logPaths(Collection<String> logPaths) {
            this.logPaths = StringListCopier.copy(logPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logPaths(String... logPaths) {
            logPaths(Arrays.asList(logPaths));
            return this;
        }

        public final String getNewGameSessionProtectionPolicy() {
            return newGameSessionProtectionPolicy;
        }

        public final void setNewGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(String newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy = newGameSessionProtectionPolicy;
            return this;
        }

        @Override
        public final Builder newGameSessionProtectionPolicy(ProtectionPolicy newGameSessionProtectionPolicy) {
            this.newGameSessionProtectionPolicy(newGameSessionProtectionPolicy == null ? null : newGameSessionProtectionPolicy
                    .toString());
            return this;
        }

        public final String getOperatingSystem() {
            return operatingSystem;
        }

        public final void setOperatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
        }

        @Override
        public final Builder operatingSystem(String operatingSystem) {
            this.operatingSystem = operatingSystem;
            return this;
        }

        @Override
        public final Builder operatingSystem(OperatingSystem operatingSystem) {
            this.operatingSystem(operatingSystem == null ? null : operatingSystem.toString());
            return this;
        }

        public final ResourceCreationLimitPolicy.Builder getResourceCreationLimitPolicy() {
            return resourceCreationLimitPolicy != null ? resourceCreationLimitPolicy.toBuilder() : null;
        }

        public final void setResourceCreationLimitPolicy(ResourceCreationLimitPolicy.BuilderImpl resourceCreationLimitPolicy) {
            this.resourceCreationLimitPolicy = resourceCreationLimitPolicy != null ? resourceCreationLimitPolicy.build() : null;
        }

        @Override
        public final Builder resourceCreationLimitPolicy(ResourceCreationLimitPolicy resourceCreationLimitPolicy) {
            this.resourceCreationLimitPolicy = resourceCreationLimitPolicy;
            return this;
        }

        public final Collection<String> getMetricGroups() {
            if (metricGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return metricGroups;
        }

        public final void setMetricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
        }

        @Override
        public final Builder metricGroups(Collection<String> metricGroups) {
            this.metricGroups = MetricGroupListCopier.copy(metricGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricGroups(String... metricGroups) {
            metricGroups(Arrays.asList(metricGroups));
            return this;
        }

        public final Collection<String> getStoppedActions() {
            if (stoppedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return stoppedActions;
        }

        public final void setStoppedActions(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
        }

        @Override
        public final Builder stoppedActionsWithStrings(Collection<String> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copy(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActionsWithStrings(String... stoppedActions) {
            stoppedActionsWithStrings(Arrays.asList(stoppedActions));
            return this;
        }

        @Override
        public final Builder stoppedActions(Collection<FleetAction> stoppedActions) {
            this.stoppedActions = FleetActionListCopier.copyEnumToString(stoppedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stoppedActions(FleetAction... stoppedActions) {
            stoppedActions(Arrays.asList(stoppedActions));
            return this;
        }

        public final String getInstanceRoleArn() {
            return instanceRoleArn;
        }

        public final void setInstanceRoleArn(String instanceRoleArn) {
            this.instanceRoleArn = instanceRoleArn;
        }

        @Override
        public final Builder instanceRoleArn(String instanceRoleArn) {
            this.instanceRoleArn = instanceRoleArn;
            return this;
        }

        public final CertificateConfiguration.Builder getCertificateConfiguration() {
            return certificateConfiguration != null ? certificateConfiguration.toBuilder() : null;
        }

        public final void setCertificateConfiguration(CertificateConfiguration.BuilderImpl certificateConfiguration) {
            this.certificateConfiguration = certificateConfiguration != null ? certificateConfiguration.build() : null;
        }

        @Override
        public final Builder certificateConfiguration(CertificateConfiguration certificateConfiguration) {
            this.certificateConfiguration = certificateConfiguration;
            return this;
        }

        public final String getComputeType() {
            return computeType;
        }

        public final void setComputeType(String computeType) {
            this.computeType = computeType;
        }

        @Override
        public final Builder computeType(String computeType) {
            this.computeType = computeType;
            return this;
        }

        @Override
        public final Builder computeType(ComputeType computeType) {
            this.computeType(computeType == null ? null : computeType.toString());
            return this;
        }

        public final AnywhereConfiguration.Builder getAnywhereConfiguration() {
            return anywhereConfiguration != null ? anywhereConfiguration.toBuilder() : null;
        }

        public final void setAnywhereConfiguration(AnywhereConfiguration.BuilderImpl anywhereConfiguration) {
            this.anywhereConfiguration = anywhereConfiguration != null ? anywhereConfiguration.build() : null;
        }

        @Override
        public final Builder anywhereConfiguration(AnywhereConfiguration anywhereConfiguration) {
            this.anywhereConfiguration = anywhereConfiguration;
            return this;
        }

        public final String getInstanceRoleCredentialsProvider() {
            return instanceRoleCredentialsProvider;
        }

        public final void setInstanceRoleCredentialsProvider(String instanceRoleCredentialsProvider) {
            this.instanceRoleCredentialsProvider = instanceRoleCredentialsProvider;
        }

        @Override
        public final Builder instanceRoleCredentialsProvider(String instanceRoleCredentialsProvider) {
            this.instanceRoleCredentialsProvider = instanceRoleCredentialsProvider;
            return this;
        }

        @Override
        public final Builder instanceRoleCredentialsProvider(InstanceRoleCredentialsProvider instanceRoleCredentialsProvider) {
            this.instanceRoleCredentialsProvider(instanceRoleCredentialsProvider == null ? null : instanceRoleCredentialsProvider
                    .toString());
            return this;
        }

        @Override
        public FleetAttributes build() {
            return new FleetAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
