/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration for a container that runs your game server executable. This definition includes container
 * configuration, resources, and start instructions. Use this data type when creating or updating a game server
 * container group definition. For properties of a deployed container, see <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameServerContainerDefinition.html"
 * >GameServerContainerDefinition</a>. A game server container is automatically considered essential; if an essential
 * container fails, the entire container group restarts.
 * </p>
 * <p>
 * <b>Use with: </b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_CreateContainerGroupDefinition.html"
 * >CreateContainerGroupDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateContainerGroupDefinition.html"
 * >UpdateContainerGroupDefinition</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServerContainerDefinitionInput implements SdkPojo, Serializable,
        ToCopyableBuilder<GameServerContainerDefinitionInput.Builder, GameServerContainerDefinitionInput> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerName").getter(getter(GameServerContainerDefinitionInput::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<List<ContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<ContainerDependency>> builder(MarshallingType.LIST)
            .memberName("DependsOn")
            .getter(getter(GameServerContainerDefinitionInput::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerMountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<ContainerMountPoint>> builder(MarshallingType.LIST)
            .memberName("MountPoints")
            .getter(getter(GameServerContainerDefinitionInput::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerMountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerMountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerEnvironment>> ENVIRONMENT_OVERRIDE_FIELD = SdkField
            .<List<ContainerEnvironment>> builder(MarshallingType.LIST)
            .memberName("EnvironmentOverride")
            .getter(getter(GameServerContainerDefinitionInput::environmentOverride))
            .setter(setter(Builder::environmentOverride))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentOverride").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerEnvironment> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerEnvironment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(GameServerContainerDefinitionInput::imageUri))
            .setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<ContainerPortConfiguration> PORT_CONFIGURATION_FIELD = SdkField
            .<ContainerPortConfiguration> builder(MarshallingType.SDK_POJO).memberName("PortConfiguration")
            .getter(getter(GameServerContainerDefinitionInput::portConfiguration)).setter(setter(Builder::portConfiguration))
            .constructor(ContainerPortConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PortConfiguration").build()).build();

    private static final SdkField<String> SERVER_SDK_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerSdkVersion").getter(getter(GameServerContainerDefinitionInput::serverSdkVersion))
            .setter(setter(Builder::serverSdkVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerSdkVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_NAME_FIELD,
            DEPENDS_ON_FIELD, MOUNT_POINTS_FIELD, ENVIRONMENT_OVERRIDE_FIELD, IMAGE_URI_FIELD, PORT_CONFIGURATION_FIELD,
            SERVER_SDK_VERSION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final List<ContainerDependency> dependsOn;

    private final List<ContainerMountPoint> mountPoints;

    private final List<ContainerEnvironment> environmentOverride;

    private final String imageUri;

    private final ContainerPortConfiguration portConfiguration;

    private final String serverSdkVersion;

    private GameServerContainerDefinitionInput(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.dependsOn = builder.dependsOn;
        this.mountPoints = builder.mountPoints;
        this.environmentOverride = builder.environmentOverride;
        this.imageUri = builder.imageUri;
        this.portConfiguration = builder.portConfiguration;
        this.serverSdkVersion = builder.serverSdkVersion;
    }

    /**
     * <p>
     * A string that uniquely identifies the container definition within a container group.
     * </p>
     * 
     * @return A string that uniquely identifies the container definition within a container group.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Establishes dependencies between this container and the status of other containers in the same container group. A
     * container can have dependencies on multiple different containers.
     * </p>
     * <p>
     * You can use dependencies to establish a startup/shutdown sequence across the container group. For example, you
     * might specify that <i>ContainerB</i> has a <code>START</code> dependency on <i>ContainerA</i>. This dependency
     * means that <i>ContainerB</i> can't start until after <i>ContainerA</i> has started. This dependency is reversed
     * on shutdown, which means that <i>ContainerB</i> must shut down before <i>ContainerA</i> can shut down.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return Establishes dependencies between this container and the status of other containers in the same container
     *         group. A container can have dependencies on multiple different containers. </p>
     *         <p>
     *         You can use dependencies to establish a startup/shutdown sequence across the container group. For
     *         example, you might specify that <i>ContainerB</i> has a <code>START</code> dependency on
     *         <i>ContainerA</i>. This dependency means that <i>ContainerB</i> can't start until after <i>ContainerA</i>
     *         has started. This dependency is reversed on shutdown, which means that <i>ContainerB</i> must shut down
     *         before <i>ContainerA</i> can shut down.
     */
    public final List<ContainerDependency> dependsOn() {
        return dependsOn;
    }

    /**
     * For responses, this returns true if the service returned a value for the MountPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A mount point that binds a path inside the container to a file or directory on the host system and lets it access
     * the file or directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMountPoints} method.
     * </p>
     * 
     * @return A mount point that binds a path inside the container to a file or directory on the host system and lets
     *         it access the file or directory.
     */
    public final List<ContainerMountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentOverride property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentOverride() {
        return environmentOverride != null && !(environmentOverride instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of environment variables to pass to the container on startup. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentOverride} method.
     * </p>
     * 
     * @return A set of environment variables to pass to the container on startup. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     *         >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<ContainerEnvironment> environmentOverride() {
        return environmentOverride;
    }

    /**
     * <p>
     * The location of the container image to deploy to a container fleet. Provide an image in an Amazon Elastic
     * Container Registry public or private repository. The repository must be in the same Amazon Web Services account
     * and Amazon Web Services Region where you're creating the container group definition. For limits on image size,
     * see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift endpoints and
     * quotas</a>. You can use any of the following image URI formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Image ID and digest: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The location of the container image to deploy to a container fleet. Provide an image in an Amazon Elastic
     *         Container Registry public or private repository. The repository must be in the same Amazon Web Services
     *         account and Amazon Web Services Region where you're creating the container group definition. For limits
     *         on image size, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift
     *         endpoints and quotas</a>. You can use any of the following image URI formats: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Image ID and digest:
     *         <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
     *         </p>
     *         </li>
     */
    public final String imageUri() {
        return imageUri;
    }

    /**
     * <p>
     * A set of ports that Amazon GameLift can assign to processes in the container. Processes, must be assigned a
     * container port to accept inbound traffic connections. For example, a game server process requires a container
     * port to allow game clients to connect to it. Container ports aren't directly accessed by inbound traffic.
     * Instead, Amazon GameLift maps container ports to externally accessible connection ports (see the container fleet
     * property <code>ConnectionPortRange</code>).
     * </p>
     * 
     * @return A set of ports that Amazon GameLift can assign to processes in the container. Processes, must be assigned
     *         a container port to accept inbound traffic connections. For example, a game server process requires a
     *         container port to allow game clients to connect to it. Container ports aren't directly accessed by
     *         inbound traffic. Instead, Amazon GameLift maps container ports to externally accessible connection ports
     *         (see the container fleet property <code>ConnectionPortRange</code>).
     */
    public final ContainerPortConfiguration portConfiguration() {
        return portConfiguration;
    }

    /**
     * <p>
     * The Amazon GameLift server SDK version that the game server is integrated with. Only game servers using 5.2.0 or
     * higher are compatible with container fleets.
     * </p>
     * 
     * @return The Amazon GameLift server SDK version that the game server is integrated with. Only game servers using
     *         5.2.0 or higher are compatible with container fleets.
     */
    public final String serverSdkVersion() {
        return serverSdkVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentOverride() ? environmentOverride() : null);
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(portConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(serverSdkVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServerContainerDefinitionInput)) {
            return false;
        }
        GameServerContainerDefinitionInput other = (GameServerContainerDefinitionInput) obj;
        return Objects.equals(containerName(), other.containerName()) && hasDependsOn() == other.hasDependsOn()
                && Objects.equals(dependsOn(), other.dependsOn()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints())
                && hasEnvironmentOverride() == other.hasEnvironmentOverride()
                && Objects.equals(environmentOverride(), other.environmentOverride())
                && Objects.equals(imageUri(), other.imageUri()) && Objects.equals(portConfiguration(), other.portConfiguration())
                && Objects.equals(serverSdkVersion(), other.serverSdkVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameServerContainerDefinitionInput").add("ContainerName", containerName())
                .add("DependsOn", hasDependsOn() ? dependsOn() : null)
                .add("MountPoints", hasMountPoints() ? mountPoints() : null)
                .add("EnvironmentOverride", hasEnvironmentOverride() ? environmentOverride() : null).add("ImageUri", imageUri())
                .add("PortConfiguration", portConfiguration()).add("ServerSdkVersion", serverSdkVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "DependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        case "MountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "EnvironmentOverride":
            return Optional.ofNullable(clazz.cast(environmentOverride()));
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "PortConfiguration":
            return Optional.ofNullable(clazz.cast(portConfiguration()));
        case "ServerSdkVersion":
            return Optional.ofNullable(clazz.cast(serverSdkVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ContainerName", CONTAINER_NAME_FIELD);
        map.put("DependsOn", DEPENDS_ON_FIELD);
        map.put("MountPoints", MOUNT_POINTS_FIELD);
        map.put("EnvironmentOverride", ENVIRONMENT_OVERRIDE_FIELD);
        map.put("ImageUri", IMAGE_URI_FIELD);
        map.put("PortConfiguration", PORT_CONFIGURATION_FIELD);
        map.put("ServerSdkVersion", SERVER_SDK_VERSION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GameServerContainerDefinitionInput, T> g) {
        return obj -> g.apply((GameServerContainerDefinitionInput) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServerContainerDefinitionInput> {
        /**
         * <p>
         * A string that uniquely identifies the container definition within a container group.
         * </p>
         * 
         * @param containerName
         *        A string that uniquely identifies the container definition within a container group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * Establishes dependencies between this container and the status of other containers in the same container
         * group. A container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. For example,
         * you might specify that <i>ContainerB</i> has a <code>START</code> dependency on <i>ContainerA</i>. This
         * dependency means that <i>ContainerB</i> can't start until after <i>ContainerA</i> has started. This
         * dependency is reversed on shutdown, which means that <i>ContainerB</i> must shut down before
         * <i>ContainerA</i> can shut down.
         * </p>
         * 
         * @param dependsOn
         *        Establishes dependencies between this container and the status of other containers in the same
         *        container group. A container can have dependencies on multiple different containers. </p>
         *        <p>
         *        You can use dependencies to establish a startup/shutdown sequence across the container group. For
         *        example, you might specify that <i>ContainerB</i> has a <code>START</code> dependency on
         *        <i>ContainerA</i>. This dependency means that <i>ContainerB</i> can't start until after
         *        <i>ContainerA</i> has started. This dependency is reversed on shutdown, which means that
         *        <i>ContainerB</i> must shut down before <i>ContainerA</i> can shut down.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<ContainerDependency> dependsOn);

        /**
         * <p>
         * Establishes dependencies between this container and the status of other containers in the same container
         * group. A container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. For example,
         * you might specify that <i>ContainerB</i> has a <code>START</code> dependency on <i>ContainerA</i>. This
         * dependency means that <i>ContainerB</i> can't start until after <i>ContainerA</i> has started. This
         * dependency is reversed on shutdown, which means that <i>ContainerB</i> must shut down before
         * <i>ContainerA</i> can shut down.
         * </p>
         * 
         * @param dependsOn
         *        Establishes dependencies between this container and the status of other containers in the same
         *        container group. A container can have dependencies on multiple different containers. </p>
         *        <p>
         *        You can use dependencies to establish a startup/shutdown sequence across the container group. For
         *        example, you might specify that <i>ContainerB</i> has a <code>START</code> dependency on
         *        <i>ContainerA</i>. This dependency means that <i>ContainerB</i> can't start until after
         *        <i>ContainerA</i> has started. This dependency is reversed on shutdown, which means that
         *        <i>ContainerB</i> must shut down before <i>ContainerA</i> can shut down.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(ContainerDependency... dependsOn);

        /**
         * <p>
         * Establishes dependencies between this container and the status of other containers in the same container
         * group. A container can have dependencies on multiple different containers.
         * </p>
         * <p>
         * You can use dependencies to establish a startup/shutdown sequence across the container group. For example,
         * you might specify that <i>ContainerB</i> has a <code>START</code> dependency on <i>ContainerA</i>. This
         * dependency means that <i>ContainerB</i> can't start until after <i>ContainerA</i> has started. This
         * dependency is reversed on shutdown, which means that <i>ContainerB</i> must shut down before
         * <i>ContainerA</i> can shut down.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder#build()} is called
         * immediately and its result is passed to {@link #dependsOn(List<ContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<ContainerDependency>)
         */
        Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * 
         * @param mountPoints
         *        A mount point that binds a path inside the container to a file or directory on the host system and
         *        lets it access the file or directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<ContainerMountPoint> mountPoints);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * 
         * @param mountPoints
         *        A mount point that binds a path inside the container to a file or directory on the host system and
         *        lets it access the file or directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(ContainerMountPoint... mountPoints);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder#build()} is called
         * immediately and its result is passed to {@link #mountPoints(List<ContainerMountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(java.util.Collection<ContainerMountPoint>)
         */
        Builder mountPoints(Consumer<ContainerMountPoint.Builder>... mountPoints);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environmentOverride
         *        A set of environment variables to pass to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentOverride(Collection<ContainerEnvironment> environmentOverride);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environmentOverride
         *        A set of environment variables to pass to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentOverride(ContainerEnvironment... environmentOverride);

        /**
         * <p>
         * A set of environment variables to pass to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder#build()} is called
         * immediately and its result is passed to {@link #environmentOverride(List<ContainerEnvironment>)}.
         * 
         * @param environmentOverride
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environmentOverride(java.util.Collection<ContainerEnvironment>)
         */
        Builder environmentOverride(Consumer<ContainerEnvironment.Builder>... environmentOverride);

        /**
         * <p>
         * The location of the container image to deploy to a container fleet. Provide an image in an Amazon Elastic
         * Container Registry public or private repository. The repository must be in the same Amazon Web Services
         * account and Amazon Web Services Region where you're creating the container group definition. For limits on
         * image size, see <a href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift
         * endpoints and quotas</a>. You can use any of the following image URI formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Image ID and digest: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param imageUri
         *        The location of the container image to deploy to a container fleet. Provide an image in an Amazon
         *        Elastic Container Registry public or private repository. The repository must be in the same Amazon Web
         *        Services account and Amazon Web Services Region where you're creating the container group definition.
         *        For limits on image size, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/gamelift.html">Amazon GameLift endpoints and
         *        quotas</a>. You can use any of the following image URI formats: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Image ID only: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Image ID and digest:
         *        <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]@[digest]</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Image ID and tag: <code>[AWS account].dkr.ecr.[AWS region].amazonaws.com/[repository ID]:[tag]</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * A set of ports that Amazon GameLift can assign to processes in the container. Processes, must be assigned a
         * container port to accept inbound traffic connections. For example, a game server process requires a container
         * port to allow game clients to connect to it. Container ports aren't directly accessed by inbound traffic.
         * Instead, Amazon GameLift maps container ports to externally accessible connection ports (see the container
         * fleet property <code>ConnectionPortRange</code>).
         * </p>
         * 
         * @param portConfiguration
         *        A set of ports that Amazon GameLift can assign to processes in the container. Processes, must be
         *        assigned a container port to accept inbound traffic connections. For example, a game server process
         *        requires a container port to allow game clients to connect to it. Container ports aren't directly
         *        accessed by inbound traffic. Instead, Amazon GameLift maps container ports to externally accessible
         *        connection ports (see the container fleet property <code>ConnectionPortRange</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portConfiguration(ContainerPortConfiguration portConfiguration);

        /**
         * <p>
         * A set of ports that Amazon GameLift can assign to processes in the container. Processes, must be assigned a
         * container port to accept inbound traffic connections. For example, a game server process requires a container
         * port to allow game clients to connect to it. Container ports aren't directly accessed by inbound traffic.
         * Instead, Amazon GameLift maps container ports to externally accessible connection ports (see the container
         * fleet property <code>ConnectionPortRange</code>).
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerPortConfiguration.Builder}
         * avoiding the need to create one manually via {@link ContainerPortConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerPortConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #portConfiguration(ContainerPortConfiguration)}.
         * 
         * @param portConfiguration
         *        a consumer that will call methods on {@link ContainerPortConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portConfiguration(ContainerPortConfiguration)
         */
        default Builder portConfiguration(Consumer<ContainerPortConfiguration.Builder> portConfiguration) {
            return portConfiguration(ContainerPortConfiguration.builder().applyMutation(portConfiguration).build());
        }

        /**
         * <p>
         * The Amazon GameLift server SDK version that the game server is integrated with. Only game servers using 5.2.0
         * or higher are compatible with container fleets.
         * </p>
         * 
         * @param serverSdkVersion
         *        The Amazon GameLift server SDK version that the game server is integrated with. Only game servers
         *        using 5.2.0 or higher are compatible with container fleets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverSdkVersion(String serverSdkVersion);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private List<ContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerMountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerEnvironment> environmentOverride = DefaultSdkAutoConstructList.getInstance();

        private String imageUri;

        private ContainerPortConfiguration portConfiguration;

        private String serverSdkVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServerContainerDefinitionInput model) {
            containerName(model.containerName);
            dependsOn(model.dependsOn);
            mountPoints(model.mountPoints);
            environmentOverride(model.environmentOverride);
            imageUri(model.imageUri);
            portConfiguration(model.portConfiguration);
            serverSdkVersion(model.serverSdkVersion);
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final List<ContainerDependency.Builder> getDependsOn() {
            List<ContainerDependency.Builder> result = ContainerDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<ContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<ContainerDependency> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(ContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> ContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerMountPoint.Builder> getMountPoints() {
            List<ContainerMountPoint.Builder> result = ContainerMountPointListCopier.copyToBuilder(this.mountPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMountPoints(Collection<ContainerMountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = ContainerMountPointListCopier.copyFromBuilder(mountPoints);
        }

        @Override
        public final Builder mountPoints(Collection<ContainerMountPoint> mountPoints) {
            this.mountPoints = ContainerMountPointListCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(ContainerMountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<ContainerMountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> ContainerMountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerEnvironment.Builder> getEnvironmentOverride() {
            List<ContainerEnvironment.Builder> result = ContainerEnvironmentListCopier.copyToBuilder(this.environmentOverride);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironmentOverride(Collection<ContainerEnvironment.BuilderImpl> environmentOverride) {
            this.environmentOverride = ContainerEnvironmentListCopier.copyFromBuilder(environmentOverride);
        }

        @Override
        public final Builder environmentOverride(Collection<ContainerEnvironment> environmentOverride) {
            this.environmentOverride = ContainerEnvironmentListCopier.copy(environmentOverride);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentOverride(ContainerEnvironment... environmentOverride) {
            environmentOverride(Arrays.asList(environmentOverride));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentOverride(Consumer<ContainerEnvironment.Builder>... environmentOverride) {
            environmentOverride(Stream.of(environmentOverride).map(c -> ContainerEnvironment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getImageUri() {
            return imageUri;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final ContainerPortConfiguration.Builder getPortConfiguration() {
            return portConfiguration != null ? portConfiguration.toBuilder() : null;
        }

        public final void setPortConfiguration(ContainerPortConfiguration.BuilderImpl portConfiguration) {
            this.portConfiguration = portConfiguration != null ? portConfiguration.build() : null;
        }

        @Override
        public final Builder portConfiguration(ContainerPortConfiguration portConfiguration) {
            this.portConfiguration = portConfiguration;
            return this;
        }

        public final String getServerSdkVersion() {
            return serverSdkVersion;
        }

        public final void setServerSdkVersion(String serverSdkVersion) {
            this.serverSdkVersion = serverSdkVersion;
        }

        @Override
        public final Builder serverSdkVersion(String serverSdkVersion) {
            this.serverSdkVersion = serverSdkVersion;
            return this;
        }

        @Override
        public GameServerContainerDefinitionInput build() {
            return new GameServerContainerDefinitionInput(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
