/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a potential game session placement, including the full details of the original placement request and the
 * current status.
 * </p>
 * <note>
 * <p>
 * If the game session placement status is <code>PENDING</code>, the properties for game session ID/ARN, region, IP
 * address/DNS, and port aren't final. A game session is not active and ready to accept players until placement status
 * reaches <code>FULFILLED</code>. When the placement is in <code>PENDING</code> status, Amazon GameLift may attempt to
 * place a game session multiple times before succeeding. With each attempt it creates a <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameSession"
 * >https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameSession</a> object and updates this placement
 * object with the new game session properties.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameSessionPlacement implements SdkPojo, Serializable,
        ToCopyableBuilder<GameSessionPlacement.Builder, GameSessionPlacement> {
    private static final SdkField<String> PLACEMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlacementId").getter(getter(GameSessionPlacement::placementId)).setter(setter(Builder::placementId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementId").build()).build();

    private static final SdkField<String> GAME_SESSION_QUEUE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionQueueName").getter(getter(GameSessionPlacement::gameSessionQueueName))
            .setter(setter(Builder::gameSessionQueueName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionQueueName").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(GameSessionPlacement::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<GameProperty>> GAME_PROPERTIES_FIELD = SdkField
            .<List<GameProperty>> builder(MarshallingType.LIST)
            .memberName("GameProperties")
            .getter(getter(GameSessionPlacement::gameProperties))
            .setter(setter(Builder::gameProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAXIMUM_PLAYER_SESSION_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaximumPlayerSessionCount")
            .getter(getter(GameSessionPlacement::maximumPlayerSessionCount)).setter(setter(Builder::maximumPlayerSessionCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumPlayerSessionCount").build())
            .build();

    private static final SdkField<String> GAME_SESSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionName").getter(getter(GameSessionPlacement::gameSessionName))
            .setter(setter(Builder::gameSessionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionName").build()).build();

    private static final SdkField<String> GAME_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionId").getter(getter(GameSessionPlacement::gameSessionId))
            .setter(setter(Builder::gameSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionId").build()).build();

    private static final SdkField<String> GAME_SESSION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionArn").getter(getter(GameSessionPlacement::gameSessionArn))
            .setter(setter(Builder::gameSessionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionArn").build()).build();

    private static final SdkField<String> GAME_SESSION_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionRegion").getter(getter(GameSessionPlacement::gameSessionRegion))
            .setter(setter(Builder::gameSessionRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionRegion").build()).build();

    private static final SdkField<List<PlayerLatency>> PLAYER_LATENCIES_FIELD = SdkField
            .<List<PlayerLatency>> builder(MarshallingType.LIST)
            .memberName("PlayerLatencies")
            .getter(getter(GameSessionPlacement::playerLatencies))
            .setter(setter(Builder::playerLatencies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerLatencies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlayerLatency> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlayerLatency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(GameSessionPlacement::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(GameSessionPlacement::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddress").getter(getter(GameSessionPlacement::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(GameSessionPlacement::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(GameSessionPlacement::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<List<PlacedPlayerSession>> PLACED_PLAYER_SESSIONS_FIELD = SdkField
            .<List<PlacedPlayerSession>> builder(MarshallingType.LIST)
            .memberName("PlacedPlayerSessions")
            .getter(getter(GameSessionPlacement::placedPlayerSessions))
            .setter(setter(Builder::placedPlayerSessions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacedPlayerSessions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlacedPlayerSession> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlacedPlayerSession::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GAME_SESSION_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GameSessionData").getter(getter(GameSessionPlacement::gameSessionData))
            .setter(setter(Builder::gameSessionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionData").build()).build();

    private static final SdkField<String> MATCHMAKER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MatchmakerData").getter(getter(GameSessionPlacement::matchmakerData))
            .setter(setter(Builder::matchmakerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchmakerData").build()).build();

    private static final SdkField<PriorityConfigurationOverride> PRIORITY_CONFIGURATION_OVERRIDE_FIELD = SdkField
            .<PriorityConfigurationOverride> builder(MarshallingType.SDK_POJO)
            .memberName("PriorityConfigurationOverride")
            .getter(getter(GameSessionPlacement::priorityConfigurationOverride))
            .setter(setter(Builder::priorityConfigurationOverride))
            .constructor(PriorityConfigurationOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PriorityConfigurationOverride")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PLACEMENT_ID_FIELD,
            GAME_SESSION_QUEUE_NAME_FIELD, STATUS_FIELD, GAME_PROPERTIES_FIELD, MAXIMUM_PLAYER_SESSION_COUNT_FIELD,
            GAME_SESSION_NAME_FIELD, GAME_SESSION_ID_FIELD, GAME_SESSION_ARN_FIELD, GAME_SESSION_REGION_FIELD,
            PLAYER_LATENCIES_FIELD, START_TIME_FIELD, END_TIME_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, PORT_FIELD,
            PLACED_PLAYER_SESSIONS_FIELD, GAME_SESSION_DATA_FIELD, MATCHMAKER_DATA_FIELD, PRIORITY_CONFIGURATION_OVERRIDE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String placementId;

    private final String gameSessionQueueName;

    private final String status;

    private final List<GameProperty> gameProperties;

    private final Integer maximumPlayerSessionCount;

    private final String gameSessionName;

    private final String gameSessionId;

    private final String gameSessionArn;

    private final String gameSessionRegion;

    private final List<PlayerLatency> playerLatencies;

    private final Instant startTime;

    private final Instant endTime;

    private final String ipAddress;

    private final String dnsName;

    private final Integer port;

    private final List<PlacedPlayerSession> placedPlayerSessions;

    private final String gameSessionData;

    private final String matchmakerData;

    private final PriorityConfigurationOverride priorityConfigurationOverride;

    private GameSessionPlacement(BuilderImpl builder) {
        this.placementId = builder.placementId;
        this.gameSessionQueueName = builder.gameSessionQueueName;
        this.status = builder.status;
        this.gameProperties = builder.gameProperties;
        this.maximumPlayerSessionCount = builder.maximumPlayerSessionCount;
        this.gameSessionName = builder.gameSessionName;
        this.gameSessionId = builder.gameSessionId;
        this.gameSessionArn = builder.gameSessionArn;
        this.gameSessionRegion = builder.gameSessionRegion;
        this.playerLatencies = builder.playerLatencies;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.port = builder.port;
        this.placedPlayerSessions = builder.placedPlayerSessions;
        this.gameSessionData = builder.gameSessionData;
        this.matchmakerData = builder.matchmakerData;
        this.priorityConfigurationOverride = builder.priorityConfigurationOverride;
    }

    /**
     * <p>
     * A unique identifier for a game session placement.
     * </p>
     * 
     * @return A unique identifier for a game session placement.
     */
    public final String placementId() {
        return placementId;
    }

    /**
     * <p>
     * A descriptive label that is associated with game session queue. Queue names must be unique within each Region.
     * </p>
     * 
     * @return A descriptive label that is associated with game session queue. Queue names must be unique within each
     *         Region.
     */
    public final String gameSessionQueueName() {
        return gameSessionQueueName;
    }

    /**
     * <p>
     * Current status of the game session placement request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties are not
     * yet final.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now final.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The placement request was canceled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
     * resubmit the placement request as needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common reasons
     * are the game session terminated before the placement process was completed, or an unexpected internal error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionPlacementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session placement request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties
     *         are not yet final.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now
     *         final.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The placement request was canceled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
     *         can resubmit the placement request as needed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common
     *         reasons are the game session terminated before the placement process was completed, or an unexpected
     *         internal error.
     *         </p>
     *         </li>
     * @see GameSessionPlacementState
     */
    public final GameSessionPlacementState status() {
        return GameSessionPlacementState.fromValue(status);
    }

    /**
     * <p>
     * Current status of the game session placement request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties are not
     * yet final.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now final.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The placement request was canceled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
     * resubmit the placement request as needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common reasons
     * are the game session terminated before the placement process was completed, or an unexpected internal error.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionPlacementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session placement request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties
     *         are not yet final.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now
     *         final.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The placement request was canceled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
     *         can resubmit the placement request as needed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common
     *         reasons are the game session terminated before the placement process was completed, or an unexpected
     *         internal error.
     *         </p>
     *         </li>
     * @see GameSessionPlacementState
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the GameProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGameProperties() {
        return gameProperties != null && !(gameProperties instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of key-value pairs that can store custom data in a game session. For example:
     * <code>{"Key": "difficulty", "Value": "novice"}</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGameProperties} method.
     * </p>
     * 
     * @return A set of key-value pairs that can store custom data in a game session. For example:
     *         <code>{"Key": "difficulty", "Value": "novice"}</code>.
     */
    public final List<GameProperty> gameProperties() {
        return gameProperties;
    }

    /**
     * <p>
     * The maximum number of players that can be connected simultaneously to the game session.
     * </p>
     * 
     * @return The maximum number of players that can be connected simultaneously to the game session.
     */
    public final Integer maximumPlayerSessionCount() {
        return maximumPlayerSessionCount;
    }

    /**
     * <p>
     * A descriptive label that is associated with a game session. Session names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a game session. Session names do not need to be unique.
     */
    public final String gameSessionName() {
        return gameSessionName;
    }

    /**
     * <p>
     * A unique identifier for the game session. This value isn't final until placement status is <code>FULFILLED</code>
     * .
     * </p>
     * 
     * @return A unique identifier for the game session. This value isn't final until placement status is
     *         <code>FULFILLED</code>.
     */
    public final String gameSessionId() {
        return gameSessionId;
    }

    /**
     * <p>
     * Identifier for the game session created by this placement request. This identifier is unique across all Regions.
     * This value isn't final until placement status is <code>FULFILLED</code>.
     * </p>
     * 
     * @return Identifier for the game session created by this placement request. This identifier is unique across all
     *         Regions. This value isn't final until placement status is <code>FULFILLED</code>.
     */
    public final String gameSessionArn() {
        return gameSessionArn;
    }

    /**
     * <p>
     * Name of the Region where the game session created by this placement request is running. This value isn't final
     * until placement status is <code>FULFILLED</code>.
     * </p>
     * 
     * @return Name of the Region where the game session created by this placement request is running. This value isn't
     *         final until placement status is <code>FULFILLED</code>.
     */
    public final String gameSessionRegion() {
        return gameSessionRegion;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlayerLatencies property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlayerLatencies() {
        return playerLatencies != null && !(playerLatencies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences when
     * connected to Amazon Web Services Regions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlayerLatencies} method.
     * </p>
     * 
     * @return A set of values, expressed in milliseconds, that indicates the amount of latency that a player
     *         experiences when connected to Amazon Web Services Regions.
     */
    public final List<PlayerLatency> playerLatencies() {
        return playerLatencies;
    }

    /**
     * <p>
     * Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix time as
     * milliseconds (for example <code>"1469498468.057"</code>).
     * </p>
     * 
     * @return Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix
     *         time as milliseconds (for example <code>"1469498468.057"</code>).
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * Time stamp indicating when this request was completed, canceled, or timed out.
     * </p>
     * 
     * @return Time stamp indicating when this request was completed, canceled, or timed out.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP address
     * and port number. This value isn't final until placement status is <code>FULFILLED</code>.
     * </p>
     * 
     * @return The IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP
     *         address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * The DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return The DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The port number for the game session. To connect to a Amazon GameLift game server, an app needs both the IP
     * address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
     * </p>
     * 
     * @return The port number for the game session. To connect to a Amazon GameLift game server, an app needs both the
     *         IP address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
     */
    public final Integer port() {
        return port;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlacedPlayerSessions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlacedPlayerSessions() {
        return placedPlayerSessions != null && !(placedPlayerSessions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of information on player sessions created in response to the game session placement request. These
     * player sessions are created only after a new game session is successfully placed (placement status is
     * <code>FULFILLED</code>). This information includes the player ID, provided in the placement request, and a
     * corresponding player session ID.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlacedPlayerSessions} method.
     * </p>
     * 
     * @return A collection of information on player sessions created in response to the game session placement request.
     *         These player sessions are created only after a new game session is successfully placed (placement status
     *         is <code>FULFILLED</code>). This information includes the player ID, provided in the placement request,
     *         and a corresponding player session ID.
     */
    public final List<PlacedPlayerSession> placedPlayerSessions() {
        return placedPlayerSessions;
    }

    /**
     * <p>
     * A set of custom game session properties, formatted as a single string value. This data is passed to a game server
     * process with a request to start a new game session. For more information, see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a game session</a>.
     * </p>
     * 
     * @return A set of custom game session properties, formatted as a single string value. This data is passed to a
     *         game server process with a request to start a new game session. For more information, see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a game session</a>.
     */
    public final String gameSessionData() {
        return gameSessionData;
    }

    /**
     * <p>
     * Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
     * identifies the matchmaking configuration used to create the match, and contains data on all players assigned to
     * the match, including player attributes and team assignments. For more details on matchmaker data, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
     * Data</a>.
     * </p>
     * 
     * @return Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
     *         identifies the matchmaking configuration used to create the match, and contains data on all players
     *         assigned to the match, including player attributes and team assignments. For more details on matchmaker
     *         data, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data"
     *         >Match Data</a>.
     */
    public final String matchmakerData() {
        return matchmakerData;
    }

    /**
     * <p>
     * A prioritized list of locations to use with a game session placement request and instructions on how to use it.
     * This list overrides a queue's prioritized location list for a single game session placement request only. The
     * list can include Amazon Web Services Regions, local zones, and custom locations (for Anywhere fleets). The
     * fallback strategy instructs Amazon GameLift to use the override list for the first placement attempt only or for
     * all placement attempts.
     * </p>
     * 
     * @return A prioritized list of locations to use with a game session placement request and instructions on how to
     *         use it. This list overrides a queue's prioritized location list for a single game session placement
     *         request only. The list can include Amazon Web Services Regions, local zones, and custom locations (for
     *         Anywhere fleets). The fallback strategy instructs Amazon GameLift to use the override list for the first
     *         placement attempt only or for all placement attempts.
     */
    public final PriorityConfigurationOverride priorityConfigurationOverride() {
        return priorityConfigurationOverride;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(placementId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionQueueName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasGameProperties() ? gameProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maximumPlayerSessionCount());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionName());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionArn());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionRegion());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlayerLatencies() ? playerLatencies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlacedPlayerSessions() ? placedPlayerSessions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionData());
        hashCode = 31 * hashCode + Objects.hashCode(matchmakerData());
        hashCode = 31 * hashCode + Objects.hashCode(priorityConfigurationOverride());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameSessionPlacement)) {
            return false;
        }
        GameSessionPlacement other = (GameSessionPlacement) obj;
        return Objects.equals(placementId(), other.placementId())
                && Objects.equals(gameSessionQueueName(), other.gameSessionQueueName())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasGameProperties() == other.hasGameProperties()
                && Objects.equals(gameProperties(), other.gameProperties())
                && Objects.equals(maximumPlayerSessionCount(), other.maximumPlayerSessionCount())
                && Objects.equals(gameSessionName(), other.gameSessionName())
                && Objects.equals(gameSessionId(), other.gameSessionId())
                && Objects.equals(gameSessionArn(), other.gameSessionArn())
                && Objects.equals(gameSessionRegion(), other.gameSessionRegion())
                && hasPlayerLatencies() == other.hasPlayerLatencies()
                && Objects.equals(playerLatencies(), other.playerLatencies()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(port(), other.port())
                && hasPlacedPlayerSessions() == other.hasPlacedPlayerSessions()
                && Objects.equals(placedPlayerSessions(), other.placedPlayerSessions())
                && Objects.equals(gameSessionData(), other.gameSessionData())
                && Objects.equals(matchmakerData(), other.matchmakerData())
                && Objects.equals(priorityConfigurationOverride(), other.priorityConfigurationOverride());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GameSessionPlacement").add("PlacementId", placementId())
                .add("GameSessionQueueName", gameSessionQueueName()).add("Status", statusAsString())
                .add("GameProperties", hasGameProperties() ? gameProperties() : null)
                .add("MaximumPlayerSessionCount", maximumPlayerSessionCount()).add("GameSessionName", gameSessionName())
                .add("GameSessionId", gameSessionId()).add("GameSessionArn", gameSessionArn())
                .add("GameSessionRegion", gameSessionRegion())
                .add("PlayerLatencies", hasPlayerLatencies() ? playerLatencies() : null).add("StartTime", startTime())
                .add("EndTime", endTime()).add("IpAddress", ipAddress() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DnsName", dnsName()).add("Port", port() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PlacedPlayerSessions", hasPlacedPlayerSessions() ? placedPlayerSessions() : null)
                .add("GameSessionData", gameSessionData()).add("MatchmakerData", matchmakerData())
                .add("PriorityConfigurationOverride", priorityConfigurationOverride()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlacementId":
            return Optional.ofNullable(clazz.cast(placementId()));
        case "GameSessionQueueName":
            return Optional.ofNullable(clazz.cast(gameSessionQueueName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "GameProperties":
            return Optional.ofNullable(clazz.cast(gameProperties()));
        case "MaximumPlayerSessionCount":
            return Optional.ofNullable(clazz.cast(maximumPlayerSessionCount()));
        case "GameSessionName":
            return Optional.ofNullable(clazz.cast(gameSessionName()));
        case "GameSessionId":
            return Optional.ofNullable(clazz.cast(gameSessionId()));
        case "GameSessionArn":
            return Optional.ofNullable(clazz.cast(gameSessionArn()));
        case "GameSessionRegion":
            return Optional.ofNullable(clazz.cast(gameSessionRegion()));
        case "PlayerLatencies":
            return Optional.ofNullable(clazz.cast(playerLatencies()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "PlacedPlayerSessions":
            return Optional.ofNullable(clazz.cast(placedPlayerSessions()));
        case "GameSessionData":
            return Optional.ofNullable(clazz.cast(gameSessionData()));
        case "MatchmakerData":
            return Optional.ofNullable(clazz.cast(matchmakerData()));
        case "PriorityConfigurationOverride":
            return Optional.ofNullable(clazz.cast(priorityConfigurationOverride()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("PlacementId", PLACEMENT_ID_FIELD);
        map.put("GameSessionQueueName", GAME_SESSION_QUEUE_NAME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("GameProperties", GAME_PROPERTIES_FIELD);
        map.put("MaximumPlayerSessionCount", MAXIMUM_PLAYER_SESSION_COUNT_FIELD);
        map.put("GameSessionName", GAME_SESSION_NAME_FIELD);
        map.put("GameSessionId", GAME_SESSION_ID_FIELD);
        map.put("GameSessionArn", GAME_SESSION_ARN_FIELD);
        map.put("GameSessionRegion", GAME_SESSION_REGION_FIELD);
        map.put("PlayerLatencies", PLAYER_LATENCIES_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("EndTime", END_TIME_FIELD);
        map.put("IpAddress", IP_ADDRESS_FIELD);
        map.put("DnsName", DNS_NAME_FIELD);
        map.put("Port", PORT_FIELD);
        map.put("PlacedPlayerSessions", PLACED_PLAYER_SESSIONS_FIELD);
        map.put("GameSessionData", GAME_SESSION_DATA_FIELD);
        map.put("MatchmakerData", MATCHMAKER_DATA_FIELD);
        map.put("PriorityConfigurationOverride", PRIORITY_CONFIGURATION_OVERRIDE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GameSessionPlacement, T> g) {
        return obj -> g.apply((GameSessionPlacement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameSessionPlacement> {
        /**
         * <p>
         * A unique identifier for a game session placement.
         * </p>
         * 
         * @param placementId
         *        A unique identifier for a game session placement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementId(String placementId);

        /**
         * <p>
         * A descriptive label that is associated with game session queue. Queue names must be unique within each
         * Region.
         * </p>
         * 
         * @param gameSessionQueueName
         *        A descriptive label that is associated with game session queue. Queue names must be unique within each
         *        Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueName(String gameSessionQueueName);

        /**
         * <p>
         * Current status of the game session placement request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties are
         * not yet final.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now final.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The placement request was canceled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
         * resubmit the placement request as needed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common
         * reasons are the game session terminated before the placement process was completed, or an unexpected internal
         * error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the game session placement request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session
         *        properties are not yet final.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now
         *        final.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The placement request was canceled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
         *        can resubmit the placement request as needed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session.
         *        Common reasons are the game session terminated before the placement process was completed, or an
         *        unexpected internal error.
         *        </p>
         *        </li>
         * @see GameSessionPlacementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionPlacementState
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the game session placement request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session properties are
         * not yet final.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now final.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The placement request was canceled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
         * resubmit the placement request as needed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session. Common
         * reasons are the game session terminated before the placement process was completed, or an unexpected internal
         * error.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the game session placement request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The placement request is in the queue waiting to be processed. Game session
         *        properties are not yet final.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FULFILLED</b> -- A new game session has been successfully placed. Game session properties are now
         *        final.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The placement request was canceled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
         *        can resubmit the placement request as needed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FAILED</b> -- Amazon GameLift is not able to complete the process of placing the game session.
         *        Common reasons are the game session terminated before the placement process was completed, or an
         *        unexpected internal error.
         *        </p>
         *        </li>
         * @see GameSessionPlacementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionPlacementState
         */
        Builder status(GameSessionPlacementState status);

        /**
         * <p>
         * A set of key-value pairs that can store custom data in a game session. For example:
         * <code>{"Key": "difficulty", "Value": "novice"}</code>.
         * </p>
         * 
         * @param gameProperties
         *        A set of key-value pairs that can store custom data in a game session. For example:
         *        <code>{"Key": "difficulty", "Value": "novice"}</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(Collection<GameProperty> gameProperties);

        /**
         * <p>
         * A set of key-value pairs that can store custom data in a game session. For example:
         * <code>{"Key": "difficulty", "Value": "novice"}</code>.
         * </p>
         * 
         * @param gameProperties
         *        A set of key-value pairs that can store custom data in a game session. For example:
         *        <code>{"Key": "difficulty", "Value": "novice"}</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(GameProperty... gameProperties);

        /**
         * <p>
         * A set of key-value pairs that can store custom data in a game session. For example:
         * <code>{"Key": "difficulty", "Value": "novice"}</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.GameProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #gameProperties(List<GameProperty>)}.
         * 
         * @param gameProperties
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.GameProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameProperties(java.util.Collection<GameProperty>)
         */
        Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties);

        /**
         * <p>
         * The maximum number of players that can be connected simultaneously to the game session.
         * </p>
         * 
         * @param maximumPlayerSessionCount
         *        The maximum number of players that can be connected simultaneously to the game session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount);

        /**
         * <p>
         * A descriptive label that is associated with a game session. Session names do not need to be unique.
         * </p>
         * 
         * @param gameSessionName
         *        A descriptive label that is associated with a game session. Session names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionName(String gameSessionName);

        /**
         * <p>
         * A unique identifier for the game session. This value isn't final until placement status is
         * <code>FULFILLED</code>.
         * </p>
         * 
         * @param gameSessionId
         *        A unique identifier for the game session. This value isn't final until placement status is
         *        <code>FULFILLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionId(String gameSessionId);

        /**
         * <p>
         * Identifier for the game session created by this placement request. This identifier is unique across all
         * Regions. This value isn't final until placement status is <code>FULFILLED</code>.
         * </p>
         * 
         * @param gameSessionArn
         *        Identifier for the game session created by this placement request. This identifier is unique across
         *        all Regions. This value isn't final until placement status is <code>FULFILLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionArn(String gameSessionArn);

        /**
         * <p>
         * Name of the Region where the game session created by this placement request is running. This value isn't
         * final until placement status is <code>FULFILLED</code>.
         * </p>
         * 
         * @param gameSessionRegion
         *        Name of the Region where the game session created by this placement request is running. This value
         *        isn't final until placement status is <code>FULFILLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionRegion(String gameSessionRegion);

        /**
         * <p>
         * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences
         * when connected to Amazon Web Services Regions.
         * </p>
         * 
         * @param playerLatencies
         *        A set of values, expressed in milliseconds, that indicates the amount of latency that a player
         *        experiences when connected to Amazon Web Services Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencies(Collection<PlayerLatency> playerLatencies);

        /**
         * <p>
         * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences
         * when connected to Amazon Web Services Regions.
         * </p>
         * 
         * @param playerLatencies
         *        A set of values, expressed in milliseconds, that indicates the amount of latency that a player
         *        experiences when connected to Amazon Web Services Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencies(PlayerLatency... playerLatencies);

        /**
         * <p>
         * A set of values, expressed in milliseconds, that indicates the amount of latency that a player experiences
         * when connected to Amazon Web Services Regions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.PlayerLatency.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.gamelift.model.PlayerLatency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.PlayerLatency.Builder#build()} is called immediately
         * and its result is passed to {@link #playerLatencies(List<PlayerLatency>)}.
         * 
         * @param playerLatencies
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.PlayerLatency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #playerLatencies(java.util.Collection<PlayerLatency>)
         */
        Builder playerLatencies(Consumer<PlayerLatency.Builder>... playerLatencies);

        /**
         * <p>
         * Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix time as
         * milliseconds (for example <code>"1469498468.057"</code>).
         * </p>
         * 
         * @param startTime
         *        Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix
         *        time as milliseconds (for example <code>"1469498468.057"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * Time stamp indicating when this request was completed, canceled, or timed out.
         * </p>
         * 
         * @param endTime
         *        Time stamp indicating when this request was completed, canceled, or timed out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP
         * address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
         * </p>
         * 
         * @param ipAddress
         *        The IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the
         *        IP address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * The DNS identifier assigned to the instance that is running the game session. Values have the following
         * format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        The DNS identifier assigned to the instance that is running the game session. Values have the
         *        following format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The port number for the game session. To connect to a Amazon GameLift game server, an app needs both the IP
         * address and port number. This value isn't final until placement status is <code>FULFILLED</code>.
         * </p>
         * 
         * @param port
         *        The port number for the game session. To connect to a Amazon GameLift game server, an app needs both
         *        the IP address and port number. This value isn't final until placement status is
         *        <code>FULFILLED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * A collection of information on player sessions created in response to the game session placement request.
         * These player sessions are created only after a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID, provided in the placement request, and a
         * corresponding player session ID.
         * </p>
         * 
         * @param placedPlayerSessions
         *        A collection of information on player sessions created in response to the game session placement
         *        request. These player sessions are created only after a new game session is successfully placed
         *        (placement status is <code>FULFILLED</code>). This information includes the player ID, provided in the
         *        placement request, and a corresponding player session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placedPlayerSessions(Collection<PlacedPlayerSession> placedPlayerSessions);

        /**
         * <p>
         * A collection of information on player sessions created in response to the game session placement request.
         * These player sessions are created only after a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID, provided in the placement request, and a
         * corresponding player session ID.
         * </p>
         * 
         * @param placedPlayerSessions
         *        A collection of information on player sessions created in response to the game session placement
         *        request. These player sessions are created only after a new game session is successfully placed
         *        (placement status is <code>FULFILLED</code>). This information includes the player ID, provided in the
         *        placement request, and a corresponding player session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placedPlayerSessions(PlacedPlayerSession... placedPlayerSessions);

        /**
         * <p>
         * A collection of information on player sessions created in response to the game session placement request.
         * These player sessions are created only after a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID, provided in the placement request, and a
         * corresponding player session ID.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.PlacedPlayerSession.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.PlacedPlayerSession#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.PlacedPlayerSession.Builder#build()} is called
         * immediately and its result is passed to {@link #placedPlayerSessions(List<PlacedPlayerSession>)}.
         * 
         * @param placedPlayerSessions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.PlacedPlayerSession.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placedPlayerSessions(java.util.Collection<PlacedPlayerSession>)
         */
        Builder placedPlayerSessions(Consumer<PlacedPlayerSession.Builder>... placedPlayerSessions);

        /**
         * <p>
         * A set of custom game session properties, formatted as a single string value. This data is passed to a game
         * server process with a request to start a new game session. For more information, see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a game session</a>.
         * </p>
         * 
         * @param gameSessionData
         *        A set of custom game session properties, formatted as a single string value. This data is passed to a
         *        game server process with a request to start a new game session. For more information, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a game session</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionData(String gameSessionData);

        /**
         * <p>
         * Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
         * identifies the matchmaking configuration used to create the match, and contains data on all players assigned
         * to the match, including player attributes and team assignments. For more details on matchmaker data, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
         * Data</a>.
         * </p>
         * 
         * @param matchmakerData
         *        Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string.
         *        It identifies the matchmaking configuration used to create the match, and contains data on all players
         *        assigned to the match, including player attributes and team assignments. For more details on
         *        matchmaker data, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-server.html#match-server-data">Match
         *        Data</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchmakerData(String matchmakerData);

        /**
         * <p>
         * A prioritized list of locations to use with a game session placement request and instructions on how to use
         * it. This list overrides a queue's prioritized location list for a single game session placement request only.
         * The list can include Amazon Web Services Regions, local zones, and custom locations (for Anywhere fleets).
         * The fallback strategy instructs Amazon GameLift to use the override list for the first placement attempt only
         * or for all placement attempts.
         * </p>
         * 
         * @param priorityConfigurationOverride
         *        A prioritized list of locations to use with a game session placement request and instructions on how
         *        to use it. This list overrides a queue's prioritized location list for a single game session placement
         *        request only. The list can include Amazon Web Services Regions, local zones, and custom locations (for
         *        Anywhere fleets). The fallback strategy instructs Amazon GameLift to use the override list for the
         *        first placement attempt only or for all placement attempts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priorityConfigurationOverride(PriorityConfigurationOverride priorityConfigurationOverride);

        /**
         * <p>
         * A prioritized list of locations to use with a game session placement request and instructions on how to use
         * it. This list overrides a queue's prioritized location list for a single game session placement request only.
         * The list can include Amazon Web Services Regions, local zones, and custom locations (for Anywhere fleets).
         * The fallback strategy instructs Amazon GameLift to use the override list for the first placement attempt only
         * or for all placement attempts.
         * </p>
         * This is a convenience method that creates an instance of the {@link PriorityConfigurationOverride.Builder}
         * avoiding the need to create one manually via {@link PriorityConfigurationOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PriorityConfigurationOverride.Builder#build()} is called
         * immediately and its result is passed to {@link #priorityConfigurationOverride(PriorityConfigurationOverride)}.
         * 
         * @param priorityConfigurationOverride
         *        a consumer that will call methods on {@link PriorityConfigurationOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #priorityConfigurationOverride(PriorityConfigurationOverride)
         */
        default Builder priorityConfigurationOverride(
                Consumer<PriorityConfigurationOverride.Builder> priorityConfigurationOverride) {
            return priorityConfigurationOverride(PriorityConfigurationOverride.builder()
                    .applyMutation(priorityConfigurationOverride).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String placementId;

        private String gameSessionQueueName;

        private String status;

        private List<GameProperty> gameProperties = DefaultSdkAutoConstructList.getInstance();

        private Integer maximumPlayerSessionCount;

        private String gameSessionName;

        private String gameSessionId;

        private String gameSessionArn;

        private String gameSessionRegion;

        private List<PlayerLatency> playerLatencies = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private String ipAddress;

        private String dnsName;

        private Integer port;

        private List<PlacedPlayerSession> placedPlayerSessions = DefaultSdkAutoConstructList.getInstance();

        private String gameSessionData;

        private String matchmakerData;

        private PriorityConfigurationOverride priorityConfigurationOverride;

        private BuilderImpl() {
        }

        private BuilderImpl(GameSessionPlacement model) {
            placementId(model.placementId);
            gameSessionQueueName(model.gameSessionQueueName);
            status(model.status);
            gameProperties(model.gameProperties);
            maximumPlayerSessionCount(model.maximumPlayerSessionCount);
            gameSessionName(model.gameSessionName);
            gameSessionId(model.gameSessionId);
            gameSessionArn(model.gameSessionArn);
            gameSessionRegion(model.gameSessionRegion);
            playerLatencies(model.playerLatencies);
            startTime(model.startTime);
            endTime(model.endTime);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            port(model.port);
            placedPlayerSessions(model.placedPlayerSessions);
            gameSessionData(model.gameSessionData);
            matchmakerData(model.matchmakerData);
            priorityConfigurationOverride(model.priorityConfigurationOverride);
        }

        public final String getPlacementId() {
            return placementId;
        }

        public final void setPlacementId(String placementId) {
            this.placementId = placementId;
        }

        @Override
        public final Builder placementId(String placementId) {
            this.placementId = placementId;
            return this;
        }

        public final String getGameSessionQueueName() {
            return gameSessionQueueName;
        }

        public final void setGameSessionQueueName(String gameSessionQueueName) {
            this.gameSessionQueueName = gameSessionQueueName;
        }

        @Override
        public final Builder gameSessionQueueName(String gameSessionQueueName) {
            this.gameSessionQueueName = gameSessionQueueName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GameSessionPlacementState status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<GameProperty.Builder> getGameProperties() {
            List<GameProperty.Builder> result = GamePropertyListCopier.copyToBuilder(this.gameProperties);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGameProperties(Collection<GameProperty.BuilderImpl> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copyFromBuilder(gameProperties);
        }

        @Override
        public final Builder gameProperties(Collection<GameProperty> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copy(gameProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(GameProperty... gameProperties) {
            gameProperties(Arrays.asList(gameProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties) {
            gameProperties(Stream.of(gameProperties).map(c -> GameProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaximumPlayerSessionCount() {
            return maximumPlayerSessionCount;
        }

        public final void setMaximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
        }

        @Override
        public final Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
            return this;
        }

        public final String getGameSessionName() {
            return gameSessionName;
        }

        public final void setGameSessionName(String gameSessionName) {
            this.gameSessionName = gameSessionName;
        }

        @Override
        public final Builder gameSessionName(String gameSessionName) {
            this.gameSessionName = gameSessionName;
            return this;
        }

        public final String getGameSessionId() {
            return gameSessionId;
        }

        public final void setGameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
        }

        @Override
        public final Builder gameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
            return this;
        }

        public final String getGameSessionArn() {
            return gameSessionArn;
        }

        public final void setGameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
        }

        @Override
        public final Builder gameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
            return this;
        }

        public final String getGameSessionRegion() {
            return gameSessionRegion;
        }

        public final void setGameSessionRegion(String gameSessionRegion) {
            this.gameSessionRegion = gameSessionRegion;
        }

        @Override
        public final Builder gameSessionRegion(String gameSessionRegion) {
            this.gameSessionRegion = gameSessionRegion;
            return this;
        }

        public final List<PlayerLatency.Builder> getPlayerLatencies() {
            List<PlayerLatency.Builder> result = PlayerLatencyListCopier.copyToBuilder(this.playerLatencies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlayerLatencies(Collection<PlayerLatency.BuilderImpl> playerLatencies) {
            this.playerLatencies = PlayerLatencyListCopier.copyFromBuilder(playerLatencies);
        }

        @Override
        public final Builder playerLatencies(Collection<PlayerLatency> playerLatencies) {
            this.playerLatencies = PlayerLatencyListCopier.copy(playerLatencies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencies(PlayerLatency... playerLatencies) {
            playerLatencies(Arrays.asList(playerLatencies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencies(Consumer<PlayerLatency.Builder>... playerLatencies) {
            playerLatencies(Stream.of(playerLatencies).map(c -> PlayerLatency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final List<PlacedPlayerSession.Builder> getPlacedPlayerSessions() {
            List<PlacedPlayerSession.Builder> result = PlacedPlayerSessionListCopier.copyToBuilder(this.placedPlayerSessions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPlacedPlayerSessions(Collection<PlacedPlayerSession.BuilderImpl> placedPlayerSessions) {
            this.placedPlayerSessions = PlacedPlayerSessionListCopier.copyFromBuilder(placedPlayerSessions);
        }

        @Override
        public final Builder placedPlayerSessions(Collection<PlacedPlayerSession> placedPlayerSessions) {
            this.placedPlayerSessions = PlacedPlayerSessionListCopier.copy(placedPlayerSessions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placedPlayerSessions(PlacedPlayerSession... placedPlayerSessions) {
            placedPlayerSessions(Arrays.asList(placedPlayerSessions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placedPlayerSessions(Consumer<PlacedPlayerSession.Builder>... placedPlayerSessions) {
            placedPlayerSessions(Stream.of(placedPlayerSessions).map(c -> PlacedPlayerSession.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getGameSessionData() {
            return gameSessionData;
        }

        public final void setGameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
        }

        @Override
        public final Builder gameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
            return this;
        }

        public final String getMatchmakerData() {
            return matchmakerData;
        }

        public final void setMatchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
        }

        @Override
        public final Builder matchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
            return this;
        }

        public final PriorityConfigurationOverride.Builder getPriorityConfigurationOverride() {
            return priorityConfigurationOverride != null ? priorityConfigurationOverride.toBuilder() : null;
        }

        public final void setPriorityConfigurationOverride(PriorityConfigurationOverride.BuilderImpl priorityConfigurationOverride) {
            this.priorityConfigurationOverride = priorityConfigurationOverride != null ? priorityConfigurationOverride.build()
                    : null;
        }

        @Override
        public final Builder priorityConfigurationOverride(PriorityConfigurationOverride priorityConfigurationOverride) {
            this.priorityConfigurationOverride = priorityConfigurationOverride;
            return this;
        }

        @Override
        public GameSessionPlacement build() {
            return new GameSessionPlacement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
