/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a support container in a container group. A support container might be in a game server container group or
 * a per-instance container group. Support containers don't run game server processes.
 * </p>
 * <p>
 * You can update a support container definition and deploy the updates to an existing fleet. When creating or updating
 * a game server container group definition, use the property <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_GameServerContainerDefinitionInput.html"
 * >GameServerContainerDefinitionInput</a>.
 * </p>
 * <p>
 * <b>Part of:</b> <a href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html">
 * ContainerGroupDefinition</a>
 * </p>
 * <p>
 * <b>Returned by:</b> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_DescribeContainerGroupDefinition.html"
 * >DescribeContainerGroupDefinition</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ListContainerGroupDefinitions.html"
 * >ListContainerGroupDefinitions</a>, <a
 * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_UpdateContainerGroupDefinition.html"
 * >UpdateContainerGroupDefinition</a>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SupportContainerDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<SupportContainerDefinition.Builder, SupportContainerDefinition> {
    private static final SdkField<String> CONTAINER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerName").getter(getter(SupportContainerDefinition::containerName))
            .setter(setter(Builder::containerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerName").build()).build();

    private static final SdkField<List<ContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<ContainerDependency>> builder(MarshallingType.LIST)
            .memberName("DependsOn")
            .getter(getter(SupportContainerDefinition::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerMountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<ContainerMountPoint>> builder(MarshallingType.LIST)
            .memberName("MountPoints")
            .getter(getter(SupportContainerDefinition::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerMountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerMountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ContainerEnvironment>> ENVIRONMENT_OVERRIDE_FIELD = SdkField
            .<List<ContainerEnvironment>> builder(MarshallingType.LIST)
            .memberName("EnvironmentOverride")
            .getter(getter(SupportContainerDefinition::environmentOverride))
            .setter(setter(Builder::environmentOverride))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnvironmentOverride").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ContainerEnvironment> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContainerEnvironment::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ESSENTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Essential").getter(getter(SupportContainerDefinition::essential)).setter(setter(Builder::essential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Essential").build()).build();

    private static final SdkField<ContainerHealthCheck> HEALTH_CHECK_FIELD = SdkField
            .<ContainerHealthCheck> builder(MarshallingType.SDK_POJO).memberName("HealthCheck")
            .getter(getter(SupportContainerDefinition::healthCheck)).setter(setter(Builder::healthCheck))
            .constructor(ContainerHealthCheck::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HealthCheck").build()).build();

    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(SupportContainerDefinition::imageUri)).setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<Integer> MEMORY_HARD_LIMIT_MEBIBYTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MemoryHardLimitMebibytes")
            .getter(getter(SupportContainerDefinition::memoryHardLimitMebibytes))
            .setter(setter(Builder::memoryHardLimitMebibytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryHardLimitMebibytes").build())
            .build();

    private static final SdkField<ContainerPortConfiguration> PORT_CONFIGURATION_FIELD = SdkField
            .<ContainerPortConfiguration> builder(MarshallingType.SDK_POJO).memberName("PortConfiguration")
            .getter(getter(SupportContainerDefinition::portConfiguration)).setter(setter(Builder::portConfiguration))
            .constructor(ContainerPortConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PortConfiguration").build()).build();

    private static final SdkField<String> RESOLVED_IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolvedImageDigest").getter(getter(SupportContainerDefinition::resolvedImageDigest))
            .setter(setter(Builder::resolvedImageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolvedImageDigest").build())
            .build();

    private static final SdkField<Double> VCPU_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Vcpu")
            .getter(getter(SupportContainerDefinition::vcpu)).setter(setter(Builder::vcpu))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Vcpu").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(CONTAINER_NAME_FIELD, DEPENDS_ON_FIELD, MOUNT_POINTS_FIELD, ENVIRONMENT_OVERRIDE_FIELD, ESSENTIAL_FIELD,
                    HEALTH_CHECK_FIELD, IMAGE_URI_FIELD, MEMORY_HARD_LIMIT_MEBIBYTES_FIELD, PORT_CONFIGURATION_FIELD,
                    RESOLVED_IMAGE_DIGEST_FIELD, VCPU_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String containerName;

    private final List<ContainerDependency> dependsOn;

    private final List<ContainerMountPoint> mountPoints;

    private final List<ContainerEnvironment> environmentOverride;

    private final Boolean essential;

    private final ContainerHealthCheck healthCheck;

    private final String imageUri;

    private final Integer memoryHardLimitMebibytes;

    private final ContainerPortConfiguration portConfiguration;

    private final String resolvedImageDigest;

    private final Double vcpu;

    private SupportContainerDefinition(BuilderImpl builder) {
        this.containerName = builder.containerName;
        this.dependsOn = builder.dependsOn;
        this.mountPoints = builder.mountPoints;
        this.environmentOverride = builder.environmentOverride;
        this.essential = builder.essential;
        this.healthCheck = builder.healthCheck;
        this.imageUri = builder.imageUri;
        this.memoryHardLimitMebibytes = builder.memoryHardLimitMebibytes;
        this.portConfiguration = builder.portConfiguration;
        this.resolvedImageDigest = builder.resolvedImageDigest;
        this.vcpu = builder.vcpu;
    }

    /**
     * <p>
     * The container definition identifier. Container names are unique within a container group definition.
     * </p>
     * 
     * @return The container definition identifier. Container names are unique within a container group definition.
     */
    public final String containerName() {
        return containerName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates that the container relies on the status of other containers in the same container group during its
     * startup and shutdown sequences. A container might have dependencies on multiple containers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return Indicates that the container relies on the status of other containers in the same container group during
     *         its startup and shutdown sequences. A container might have dependencies on multiple containers.
     */
    public final List<ContainerDependency> dependsOn() {
        return dependsOn;
    }

    /**
     * For responses, this returns true if the service returned a value for the MountPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A mount point that binds a path inside the container to a file or directory on the host system and lets it access
     * the file or directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMountPoints} method.
     * </p>
     * 
     * @return A mount point that binds a path inside the container to a file or directory on the host system and lets
     *         it access the file or directory.
     */
    public final List<ContainerMountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentOverride property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentOverride() {
        return environmentOverride != null && !(environmentOverride instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of environment variables that's passed to the container on startup. See the <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentOverride} method.
     * </p>
     * 
     * @return A set of environment variables that's passed to the container on startup. See the <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
     *         >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
     *         Reference</i>.
     */
    public final List<ContainerEnvironment> environmentOverride() {
        return environmentOverride;
    }

    /**
     * <p>
     * Indicates whether the container is vital to the container group. If an essential container fails, the entire
     * container group restarts.
     * </p>
     * 
     * @return Indicates whether the container is vital to the container group. If an essential container fails, the
     *         entire container group restarts.
     */
    public final Boolean essential() {
        return essential;
    }

    /**
     * <p>
     * A configuration for a non-terminal health check. A support container automatically restarts if it stops
     * functioning or if it fails this health check.
     * </p>
     * 
     * @return A configuration for a non-terminal health check. A support container automatically restarts if it stops
     *         functioning or if it fails this health check.
     */
    public final ContainerHealthCheck healthCheck() {
        return healthCheck;
    }

    /**
     * <p>
     * The URI to the image that Amazon GameLift deploys to a container fleet. For a more specific identifier, see
     * <code>ResolvedImageDigest</code>.
     * </p>
     * 
     * @return The URI to the image that Amazon GameLift deploys to a container fleet. For a more specific identifier,
     *         see <code>ResolvedImageDigest</code>.
     */
    public final String imageUri() {
        return imageUri;
    }

    /**
     * <p>
     * The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set for an
     * individual container, the container shares the container group's total memory allocation.
     * </p>
     * <p>
     * <b>Related data type: </b> <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
     * >ContainerGroupDefinition TotalMemoryLimitMebibytes</a>
     * </p>
     * 
     * @return The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set
     *         for an individual container, the container shares the container group's total memory allocation.</p>
     *         <p>
     *         <b>Related data type: </b> <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
     *         >ContainerGroupDefinition TotalMemoryLimitMebibytes</a>
     */
    public final Integer memoryHardLimitMebibytes() {
        return memoryHardLimitMebibytes;
    }

    /**
     * <p>
     * A set of ports that allow access to the container from external users. Processes running in the container can
     * bind to a one of these ports. Container ports aren't directly accessed by inbound traffic. Amazon GameLift maps
     * these container ports to externally accessible connection ports, which are assigned as needed from the container
     * fleet's <code>ConnectionPortRange</code>.
     * </p>
     * 
     * @return A set of ports that allow access to the container from external users. Processes running in the container
     *         can bind to a one of these ports. Container ports aren't directly accessed by inbound traffic. Amazon
     *         GameLift maps these container ports to externally accessible connection ports, which are assigned as
     *         needed from the container fleet's <code>ConnectionPortRange</code>.
     */
    public final ContainerPortConfiguration portConfiguration() {
        return portConfiguration;
    }

    /**
     * <p>
     * A unique and immutable identifier for the container image. The digest is a SHA 256 hash of the container image
     * manifest.
     * </p>
     * 
     * @return A unique and immutable identifier for the container image. The digest is a SHA 256 hash of the container
     *         image manifest.
     */
    public final String resolvedImageDigest() {
        return resolvedImageDigest;
    }

    /**
     * <p>
     * The number of vCPU units that are reserved for the container. If no resources are reserved, the container shares
     * the total vCPU limit for the container group.
     * </p>
     * <p>
     * <b>Related data type: </b> <a
     * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
     * >ContainerGroupDefinition TotalVcpuLimit</a>
     * </p>
     * 
     * @return The number of vCPU units that are reserved for the container. If no resources are reserved, the container
     *         shares the total vCPU limit for the container group.</p>
     *         <p>
     *         <b>Related data type: </b> <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
     *         >ContainerGroupDefinition TotalVcpuLimit</a>
     */
    public final Double vcpu() {
        return vcpu;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentOverride() ? environmentOverride() : null);
        hashCode = 31 * hashCode + Objects.hashCode(essential());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheck());
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(memoryHardLimitMebibytes());
        hashCode = 31 * hashCode + Objects.hashCode(portConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedImageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(vcpu());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SupportContainerDefinition)) {
            return false;
        }
        SupportContainerDefinition other = (SupportContainerDefinition) obj;
        return Objects.equals(containerName(), other.containerName()) && hasDependsOn() == other.hasDependsOn()
                && Objects.equals(dependsOn(), other.dependsOn()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints())
                && hasEnvironmentOverride() == other.hasEnvironmentOverride()
                && Objects.equals(environmentOverride(), other.environmentOverride())
                && Objects.equals(essential(), other.essential()) && Objects.equals(healthCheck(), other.healthCheck())
                && Objects.equals(imageUri(), other.imageUri())
                && Objects.equals(memoryHardLimitMebibytes(), other.memoryHardLimitMebibytes())
                && Objects.equals(portConfiguration(), other.portConfiguration())
                && Objects.equals(resolvedImageDigest(), other.resolvedImageDigest()) && Objects.equals(vcpu(), other.vcpu());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SupportContainerDefinition").add("ContainerName", containerName())
                .add("DependsOn", hasDependsOn() ? dependsOn() : null)
                .add("MountPoints", hasMountPoints() ? mountPoints() : null)
                .add("EnvironmentOverride", hasEnvironmentOverride() ? environmentOverride() : null)
                .add("Essential", essential()).add("HealthCheck", healthCheck()).add("ImageUri", imageUri())
                .add("MemoryHardLimitMebibytes", memoryHardLimitMebibytes()).add("PortConfiguration", portConfiguration())
                .add("ResolvedImageDigest", resolvedImageDigest()).add("Vcpu", vcpu()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerName":
            return Optional.ofNullable(clazz.cast(containerName()));
        case "DependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        case "MountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "EnvironmentOverride":
            return Optional.ofNullable(clazz.cast(environmentOverride()));
        case "Essential":
            return Optional.ofNullable(clazz.cast(essential()));
        case "HealthCheck":
            return Optional.ofNullable(clazz.cast(healthCheck()));
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "MemoryHardLimitMebibytes":
            return Optional.ofNullable(clazz.cast(memoryHardLimitMebibytes()));
        case "PortConfiguration":
            return Optional.ofNullable(clazz.cast(portConfiguration()));
        case "ResolvedImageDigest":
            return Optional.ofNullable(clazz.cast(resolvedImageDigest()));
        case "Vcpu":
            return Optional.ofNullable(clazz.cast(vcpu()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ContainerName", CONTAINER_NAME_FIELD);
        map.put("DependsOn", DEPENDS_ON_FIELD);
        map.put("MountPoints", MOUNT_POINTS_FIELD);
        map.put("EnvironmentOverride", ENVIRONMENT_OVERRIDE_FIELD);
        map.put("Essential", ESSENTIAL_FIELD);
        map.put("HealthCheck", HEALTH_CHECK_FIELD);
        map.put("ImageUri", IMAGE_URI_FIELD);
        map.put("MemoryHardLimitMebibytes", MEMORY_HARD_LIMIT_MEBIBYTES_FIELD);
        map.put("PortConfiguration", PORT_CONFIGURATION_FIELD);
        map.put("ResolvedImageDigest", RESOLVED_IMAGE_DIGEST_FIELD);
        map.put("Vcpu", VCPU_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SupportContainerDefinition, T> g) {
        return obj -> g.apply((SupportContainerDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SupportContainerDefinition> {
        /**
         * <p>
         * The container definition identifier. Container names are unique within a container group definition.
         * </p>
         * 
         * @param containerName
         *        The container definition identifier. Container names are unique within a container group definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerName(String containerName);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * 
         * @param dependsOn
         *        Indicates that the container relies on the status of other containers in the same container group
         *        during its startup and shutdown sequences. A container might have dependencies on multiple containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<ContainerDependency> dependsOn);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * 
         * @param dependsOn
         *        Indicates that the container relies on the status of other containers in the same container group
         *        during its startup and shutdown sequences. A container might have dependencies on multiple containers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(ContainerDependency... dependsOn);

        /**
         * <p>
         * Indicates that the container relies on the status of other containers in the same container group during its
         * startup and shutdown sequences. A container might have dependencies on multiple containers.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder#build()} is called
         * immediately and its result is passed to {@link #dependsOn(List<ContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<ContainerDependency>)
         */
        Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * 
         * @param mountPoints
         *        A mount point that binds a path inside the container to a file or directory on the host system and
         *        lets it access the file or directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<ContainerMountPoint> mountPoints);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * 
         * @param mountPoints
         *        A mount point that binds a path inside the container to a file or directory on the host system and
         *        lets it access the file or directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(ContainerMountPoint... mountPoints);

        /**
         * <p>
         * A mount point that binds a path inside the container to a file or directory on the host system and lets it
         * access the file or directory.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder#build()} is called
         * immediately and its result is passed to {@link #mountPoints(List<ContainerMountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerMountPoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(java.util.Collection<ContainerMountPoint>)
         */
        Builder mountPoints(Consumer<ContainerMountPoint.Builder>... mountPoints);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environmentOverride
         *        A set of environment variables that's passed to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentOverride(Collection<ContainerEnvironment> environmentOverride);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * 
         * @param environmentOverride
         *        A set of environment variables that's passed to the container on startup. See the <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         *        >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API
         *        Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentOverride(ContainerEnvironment... environmentOverride);

        /**
         * <p>
         * A set of environment variables that's passed to the container on startup. See the <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html#ECS-Type-ContainerDefinition-environment"
         * >ContainerDefinition::environment</a> parameter in the <i>Amazon Elastic Container Service API Reference</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder#build()} is called
         * immediately and its result is passed to {@link #environmentOverride(List<ContainerEnvironment>)}.
         * 
         * @param environmentOverride
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.gamelift.model.ContainerEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environmentOverride(java.util.Collection<ContainerEnvironment>)
         */
        Builder environmentOverride(Consumer<ContainerEnvironment.Builder>... environmentOverride);

        /**
         * <p>
         * Indicates whether the container is vital to the container group. If an essential container fails, the entire
         * container group restarts.
         * </p>
         * 
         * @param essential
         *        Indicates whether the container is vital to the container group. If an essential container fails, the
         *        entire container group restarts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder essential(Boolean essential);

        /**
         * <p>
         * A configuration for a non-terminal health check. A support container automatically restarts if it stops
         * functioning or if it fails this health check.
         * </p>
         * 
         * @param healthCheck
         *        A configuration for a non-terminal health check. A support container automatically restarts if it
         *        stops functioning or if it fails this health check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheck(ContainerHealthCheck healthCheck);

        /**
         * <p>
         * A configuration for a non-terminal health check. A support container automatically restarts if it stops
         * functioning or if it fails this health check.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerHealthCheck.Builder} avoiding
         * the need to create one manually via {@link ContainerHealthCheck#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerHealthCheck.Builder#build()} is called immediately and
         * its result is passed to {@link #healthCheck(ContainerHealthCheck)}.
         * 
         * @param healthCheck
         *        a consumer that will call methods on {@link ContainerHealthCheck.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #healthCheck(ContainerHealthCheck)
         */
        default Builder healthCheck(Consumer<ContainerHealthCheck.Builder> healthCheck) {
            return healthCheck(ContainerHealthCheck.builder().applyMutation(healthCheck).build());
        }

        /**
         * <p>
         * The URI to the image that Amazon GameLift deploys to a container fleet. For a more specific identifier, see
         * <code>ResolvedImageDigest</code>.
         * </p>
         * 
         * @param imageUri
         *        The URI to the image that Amazon GameLift deploys to a container fleet. For a more specific
         *        identifier, see <code>ResolvedImageDigest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't set for
         * an individual container, the container shares the container group's total memory allocation.
         * </p>
         * <p>
         * <b>Related data type: </b> <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
         * >ContainerGroupDefinition TotalMemoryLimitMebibytes</a>
         * </p>
         * 
         * @param memoryHardLimitMebibytes
         *        The amount of memory that Amazon GameLift makes available to the container. If memory limits aren't
         *        set for an individual container, the container shares the container group's total memory
         *        allocation.</p>
         *        <p>
         *        <b>Related data type: </b> <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
         *        >ContainerGroupDefinition TotalMemoryLimitMebibytes</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryHardLimitMebibytes(Integer memoryHardLimitMebibytes);

        /**
         * <p>
         * A set of ports that allow access to the container from external users. Processes running in the container can
         * bind to a one of these ports. Container ports aren't directly accessed by inbound traffic. Amazon GameLift
         * maps these container ports to externally accessible connection ports, which are assigned as needed from the
         * container fleet's <code>ConnectionPortRange</code>.
         * </p>
         * 
         * @param portConfiguration
         *        A set of ports that allow access to the container from external users. Processes running in the
         *        container can bind to a one of these ports. Container ports aren't directly accessed by inbound
         *        traffic. Amazon GameLift maps these container ports to externally accessible connection ports, which
         *        are assigned as needed from the container fleet's <code>ConnectionPortRange</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portConfiguration(ContainerPortConfiguration portConfiguration);

        /**
         * <p>
         * A set of ports that allow access to the container from external users. Processes running in the container can
         * bind to a one of these ports. Container ports aren't directly accessed by inbound traffic. Amazon GameLift
         * maps these container ports to externally accessible connection ports, which are assigned as needed from the
         * container fleet's <code>ConnectionPortRange</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerPortConfiguration.Builder}
         * avoiding the need to create one manually via {@link ContainerPortConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerPortConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #portConfiguration(ContainerPortConfiguration)}.
         * 
         * @param portConfiguration
         *        a consumer that will call methods on {@link ContainerPortConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portConfiguration(ContainerPortConfiguration)
         */
        default Builder portConfiguration(Consumer<ContainerPortConfiguration.Builder> portConfiguration) {
            return portConfiguration(ContainerPortConfiguration.builder().applyMutation(portConfiguration).build());
        }

        /**
         * <p>
         * A unique and immutable identifier for the container image. The digest is a SHA 256 hash of the container
         * image manifest.
         * </p>
         * 
         * @param resolvedImageDigest
         *        A unique and immutable identifier for the container image. The digest is a SHA 256 hash of the
         *        container image manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedImageDigest(String resolvedImageDigest);

        /**
         * <p>
         * The number of vCPU units that are reserved for the container. If no resources are reserved, the container
         * shares the total vCPU limit for the container group.
         * </p>
         * <p>
         * <b>Related data type: </b> <a
         * href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
         * >ContainerGroupDefinition TotalVcpuLimit</a>
         * </p>
         * 
         * @param vcpu
         *        The number of vCPU units that are reserved for the container. If no resources are reserved, the
         *        container shares the total vCPU limit for the container group.</p>
         *        <p>
         *        <b>Related data type: </b> <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/apireference/API_ContainerGroupDefinition.html"
         *        >ContainerGroupDefinition TotalVcpuLimit</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpu(Double vcpu);
    }

    static final class BuilderImpl implements Builder {
        private String containerName;

        private List<ContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerMountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private List<ContainerEnvironment> environmentOverride = DefaultSdkAutoConstructList.getInstance();

        private Boolean essential;

        private ContainerHealthCheck healthCheck;

        private String imageUri;

        private Integer memoryHardLimitMebibytes;

        private ContainerPortConfiguration portConfiguration;

        private String resolvedImageDigest;

        private Double vcpu;

        private BuilderImpl() {
        }

        private BuilderImpl(SupportContainerDefinition model) {
            containerName(model.containerName);
            dependsOn(model.dependsOn);
            mountPoints(model.mountPoints);
            environmentOverride(model.environmentOverride);
            essential(model.essential);
            healthCheck(model.healthCheck);
            imageUri(model.imageUri);
            memoryHardLimitMebibytes(model.memoryHardLimitMebibytes);
            portConfiguration(model.portConfiguration);
            resolvedImageDigest(model.resolvedImageDigest);
            vcpu(model.vcpu);
        }

        public final String getContainerName() {
            return containerName;
        }

        public final void setContainerName(String containerName) {
            this.containerName = containerName;
        }

        @Override
        public final Builder containerName(String containerName) {
            this.containerName = containerName;
            return this;
        }

        public final List<ContainerDependency.Builder> getDependsOn() {
            List<ContainerDependency.Builder> result = ContainerDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<ContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<ContainerDependency> dependsOn) {
            this.dependsOn = ContainerDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(ContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<ContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> ContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerMountPoint.Builder> getMountPoints() {
            List<ContainerMountPoint.Builder> result = ContainerMountPointListCopier.copyToBuilder(this.mountPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMountPoints(Collection<ContainerMountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = ContainerMountPointListCopier.copyFromBuilder(mountPoints);
        }

        @Override
        public final Builder mountPoints(Collection<ContainerMountPoint> mountPoints) {
            this.mountPoints = ContainerMountPointListCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(ContainerMountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<ContainerMountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> ContainerMountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ContainerEnvironment.Builder> getEnvironmentOverride() {
            List<ContainerEnvironment.Builder> result = ContainerEnvironmentListCopier.copyToBuilder(this.environmentOverride);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironmentOverride(Collection<ContainerEnvironment.BuilderImpl> environmentOverride) {
            this.environmentOverride = ContainerEnvironmentListCopier.copyFromBuilder(environmentOverride);
        }

        @Override
        public final Builder environmentOverride(Collection<ContainerEnvironment> environmentOverride) {
            this.environmentOverride = ContainerEnvironmentListCopier.copy(environmentOverride);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentOverride(ContainerEnvironment... environmentOverride) {
            environmentOverride(Arrays.asList(environmentOverride));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environmentOverride(Consumer<ContainerEnvironment.Builder>... environmentOverride) {
            environmentOverride(Stream.of(environmentOverride).map(c -> ContainerEnvironment.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEssential() {
            return essential;
        }

        public final void setEssential(Boolean essential) {
            this.essential = essential;
        }

        @Override
        public final Builder essential(Boolean essential) {
            this.essential = essential;
            return this;
        }

        public final ContainerHealthCheck.Builder getHealthCheck() {
            return healthCheck != null ? healthCheck.toBuilder() : null;
        }

        public final void setHealthCheck(ContainerHealthCheck.BuilderImpl healthCheck) {
            this.healthCheck = healthCheck != null ? healthCheck.build() : null;
        }

        @Override
        public final Builder healthCheck(ContainerHealthCheck healthCheck) {
            this.healthCheck = healthCheck;
            return this;
        }

        public final String getImageUri() {
            return imageUri;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final Integer getMemoryHardLimitMebibytes() {
            return memoryHardLimitMebibytes;
        }

        public final void setMemoryHardLimitMebibytes(Integer memoryHardLimitMebibytes) {
            this.memoryHardLimitMebibytes = memoryHardLimitMebibytes;
        }

        @Override
        public final Builder memoryHardLimitMebibytes(Integer memoryHardLimitMebibytes) {
            this.memoryHardLimitMebibytes = memoryHardLimitMebibytes;
            return this;
        }

        public final ContainerPortConfiguration.Builder getPortConfiguration() {
            return portConfiguration != null ? portConfiguration.toBuilder() : null;
        }

        public final void setPortConfiguration(ContainerPortConfiguration.BuilderImpl portConfiguration) {
            this.portConfiguration = portConfiguration != null ? portConfiguration.build() : null;
        }

        @Override
        public final Builder portConfiguration(ContainerPortConfiguration portConfiguration) {
            this.portConfiguration = portConfiguration;
            return this;
        }

        public final String getResolvedImageDigest() {
            return resolvedImageDigest;
        }

        public final void setResolvedImageDigest(String resolvedImageDigest) {
            this.resolvedImageDigest = resolvedImageDigest;
        }

        @Override
        public final Builder resolvedImageDigest(String resolvedImageDigest) {
            this.resolvedImageDigest = resolvedImageDigest;
            return this;
        }

        public final Double getVcpu() {
            return vcpu;
        }

        public final void setVcpu(Double vcpu) {
            this.vcpu = vcpu;
        }

        @Override
        public final Builder vcpu(Double vcpu) {
            this.vcpu = vcpu;
            return this;
        }

        @Override
        public SupportContainerDefinition build() {
            return new SupportContainerDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
