/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateScriptRequest extends GameLiftRequest implements
        ToCopyableBuilder<CreateScriptRequest.Builder, CreateScriptRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateScriptRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateScriptRequest::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<S3Location> STORAGE_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateScriptRequest::storageLocation)).setter(setter(Builder::storageLocation))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageLocation").build()).build();

    private static final SdkField<SdkBytes> ZIP_FILE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .getter(getter(CreateScriptRequest::zipFile)).setter(setter(Builder::zipFile))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZipFile").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VERSION_FIELD,
            STORAGE_LOCATION_FIELD, ZIP_FILE_FIELD));

    private final String name;

    private final String version;

    private final S3Location storageLocation;

    private final SdkBytes zipFile;

    private CreateScriptRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.version = builder.version;
        this.storageLocation = builder.storageLocation;
        this.zipFile = builder.zipFile;
    }

    /**
     * <p>
     * Descriptive label that is associated with a script. Script names do not need to be unique. You can use
     * <a>UpdateScript</a> to change this value later.
     * </p>
     * 
     * @return Descriptive label that is associated with a script. Script names do not need to be unique. You can use
     *         <a>UpdateScript</a> to change this value later.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * Version that is associated with a build or script. Version strings do not need to be unique. You can use
     * <a>UpdateScript</a> to change this value later.
     * </p>
     * 
     * @return Version that is associated with a build or script. Version strings do not need to be unique. You can use
     *         <a>UpdateScript</a> to change this value later.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * Location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The storage
     * location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that allows Amazon
     * GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same region where you want to
     * create a new script. By default, Amazon GameLift uploads the latest version of the zip file; if you have S3
     * object versioning turned on, you can use the <code>ObjectVersion</code> parameter to specify an earlier version.
     * </p>
     * 
     * @return Location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The
     *         storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN
     *         that allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same
     *         region where you want to create a new script. By default, Amazon GameLift uploads the latest version of
     *         the zip file; if you have S3 object versioning turned on, you can use the <code>ObjectVersion</code>
     *         parameter to specify an earlier version.
     */
    public S3Location storageLocation() {
        return storageLocation;
    }

    /**
     * <p>
     * Data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one or
     * multiple files. Maximum size of a zip file is 5 MB.
     * </p>
     * <p>
     * When using the AWS CLI tool to create a script, this parameter is set to the zip file name. It must be prepended
     * with the string "fileb://" to indicate that the file data is a binary object. For example:
     * <code>--zip-file fileb://myRealtimeScript.zip</code>.
     * </p>
     * 
     * @return Data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one or
     *         multiple files. Maximum size of a zip file is 5 MB.</p>
     *         <p>
     *         When using the AWS CLI tool to create a script, this parameter is set to the zip file name. It must be
     *         prepended with the string "fileb://" to indicate that the file data is a binary object. For example:
     *         <code>--zip-file fileb://myRealtimeScript.zip</code>.
     */
    public SdkBytes zipFile() {
        return zipFile;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(storageLocation());
        hashCode = 31 * hashCode + Objects.hashCode(zipFile());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateScriptRequest)) {
            return false;
        }
        CreateScriptRequest other = (CreateScriptRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(storageLocation(), other.storageLocation()) && Objects.equals(zipFile(), other.zipFile());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateScriptRequest").add("Name", name()).add("Version", version())
                .add("StorageLocation", storageLocation()).add("ZipFile", zipFile()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "StorageLocation":
            return Optional.ofNullable(clazz.cast(storageLocation()));
        case "ZipFile":
            return Optional.ofNullable(clazz.cast(zipFile()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateScriptRequest, T> g) {
        return obj -> g.apply((CreateScriptRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GameLiftRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateScriptRequest> {
        /**
         * <p>
         * Descriptive label that is associated with a script. Script names do not need to be unique. You can use
         * <a>UpdateScript</a> to change this value later.
         * </p>
         * 
         * @param name
         *        Descriptive label that is associated with a script. Script names do not need to be unique. You can use
         *        <a>UpdateScript</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Version that is associated with a build or script. Version strings do not need to be unique. You can use
         * <a>UpdateScript</a> to change this value later.
         * </p>
         * 
         * @param version
         *        Version that is associated with a build or script. Version strings do not need to be unique. You can
         *        use <a>UpdateScript</a> to change this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The storage
         * location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that allows
         * Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same region where you
         * want to create a new script. By default, Amazon GameLift uploads the latest version of the zip file; if you
         * have S3 object versioning turned on, you can use the <code>ObjectVersion</code> parameter to specify an
         * earlier version.
         * </p>
         * 
         * @param storageLocation
         *        Location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The
         *        storage location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN
         *        that allows Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the
         *        same region where you want to create a new script. By default, Amazon GameLift uploads the latest
         *        version of the zip file; if you have S3 object versioning turned on, you can use the
         *        <code>ObjectVersion</code> parameter to specify an earlier version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageLocation(S3Location storageLocation);

        /**
         * <p>
         * Location of the Amazon S3 bucket where a zipped file containing your Realtime scripts is stored. The storage
         * location must specify the Amazon S3 bucket name, the zip file name (the "key"), and a role ARN that allows
         * Amazon GameLift to access the Amazon S3 storage location. The S3 bucket must be in the same region where you
         * want to create a new script. By default, Amazon GameLift uploads the latest version of the zip file; if you
         * have S3 object versioning turned on, you can use the <code>ObjectVersion</code> parameter to specify an
         * earlier version.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Location.Builder} avoiding the need to create
         * one manually via {@link S3Location#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #storageLocation(S3Location)}.
         * 
         * @param storageLocation
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storageLocation(S3Location)
         */
        default Builder storageLocation(Consumer<S3Location.Builder> storageLocation) {
            return storageLocation(S3Location.builder().applyMutation(storageLocation).build());
        }

        /**
         * <p>
         * Data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one or
         * multiple files. Maximum size of a zip file is 5 MB.
         * </p>
         * <p>
         * When using the AWS CLI tool to create a script, this parameter is set to the zip file name. It must be
         * prepended with the string "fileb://" to indicate that the file data is a binary object. For example:
         * <code>--zip-file fileb://myRealtimeScript.zip</code>.
         * </p>
         * 
         * @param zipFile
         *        Data object containing your Realtime scripts and dependencies as a zip file. The zip file can have one
         *        or multiple files. Maximum size of a zip file is 5 MB.</p>
         *        <p>
         *        When using the AWS CLI tool to create a script, this parameter is set to the zip file name. It must be
         *        prepended with the string "fileb://" to indicate that the file data is a binary object. For example:
         *        <code>--zip-file fileb://myRealtimeScript.zip</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zipFile(SdkBytes zipFile);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GameLiftRequest.BuilderImpl implements Builder {
        private String name;

        private String version;

        private S3Location storageLocation;

        private SdkBytes zipFile;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateScriptRequest model) {
            super(model);
            name(model.name);
            version(model.version);
            storageLocation(model.storageLocation);
            zipFile(model.zipFile);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final S3Location.Builder getStorageLocation() {
            return storageLocation != null ? storageLocation.toBuilder() : null;
        }

        @Override
        public final Builder storageLocation(S3Location storageLocation) {
            this.storageLocation = storageLocation;
            return this;
        }

        public final void setStorageLocation(S3Location.BuilderImpl storageLocation) {
            this.storageLocation = storageLocation != null ? storageLocation.build() : null;
        }

        public final ByteBuffer getZipFile() {
            return zipFile == null ? null : zipFile.asByteBuffer();
        }

        @Override
        public final Builder zipFile(SdkBytes zipFile) {
            this.zipFile = StandardMemberCopier.copy(zipFile);
            return this;
        }

        public final void setZipFile(ByteBuffer zipFile) {
            zipFile(zipFile == null ? null : SdkBytes.fromByteBuffer(zipFile));
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateScriptRequest build() {
            return new CreateScriptRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
