/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object that describes a <a>StartGameSessionPlacement</a> request. This object includes the full details of the
 * original request plus the current status and start/end time stamps.
 * </p>
 * <p>
 * Game session placement-related operations include:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>StartGameSessionPlacement</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeGameSessionPlacement</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>StopGameSessionPlacement</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameSessionPlacement implements SdkPojo, Serializable,
        ToCopyableBuilder<GameSessionPlacement.Builder, GameSessionPlacement> {
    private static final SdkField<String> PLACEMENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::placementId)).setter(setter(Builder::placementId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementId").build()).build();

    private static final SdkField<String> GAME_SESSION_QUEUE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionQueueName)).setter(setter(Builder::gameSessionQueueName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionQueueName").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<GameProperty>> GAME_PROPERTIES_FIELD = SdkField
            .<List<GameProperty>> builder(MarshallingType.LIST)
            .getter(getter(GameSessionPlacement::gameProperties))
            .setter(setter(Builder::gameProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameProperties").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GameProperty> builder(MarshallingType.SDK_POJO)
                                            .constructor(GameProperty::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAXIMUM_PLAYER_SESSION_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).getter(getter(GameSessionPlacement::maximumPlayerSessionCount))
            .setter(setter(Builder::maximumPlayerSessionCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumPlayerSessionCount").build())
            .build();

    private static final SdkField<String> GAME_SESSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionName)).setter(setter(Builder::gameSessionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionName").build()).build();

    private static final SdkField<String> GAME_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionId)).setter(setter(Builder::gameSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionId").build()).build();

    private static final SdkField<String> GAME_SESSION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionArn)).setter(setter(Builder::gameSessionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionArn").build()).build();

    private static final SdkField<String> GAME_SESSION_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionRegion)).setter(setter(Builder::gameSessionRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionRegion").build()).build();

    private static final SdkField<List<PlayerLatency>> PLAYER_LATENCIES_FIELD = SdkField
            .<List<PlayerLatency>> builder(MarshallingType.LIST)
            .getter(getter(GameSessionPlacement::playerLatencies))
            .setter(setter(Builder::playerLatencies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerLatencies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlayerLatency> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlayerLatency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameSessionPlacement::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameSessionPlacement::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(GameSessionPlacement::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<List<PlacedPlayerSession>> PLACED_PLAYER_SESSIONS_FIELD = SdkField
            .<List<PlacedPlayerSession>> builder(MarshallingType.LIST)
            .getter(getter(GameSessionPlacement::placedPlayerSessions))
            .setter(setter(Builder::placedPlayerSessions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacedPlayerSessions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PlacedPlayerSession> builder(MarshallingType.SDK_POJO)
                                            .constructor(PlacedPlayerSession::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GAME_SESSION_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::gameSessionData)).setter(setter(Builder::gameSessionData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionData").build()).build();

    private static final SdkField<String> MATCHMAKER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameSessionPlacement::matchmakerData)).setter(setter(Builder::matchmakerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchmakerData").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PLACEMENT_ID_FIELD,
            GAME_SESSION_QUEUE_NAME_FIELD, STATUS_FIELD, GAME_PROPERTIES_FIELD, MAXIMUM_PLAYER_SESSION_COUNT_FIELD,
            GAME_SESSION_NAME_FIELD, GAME_SESSION_ID_FIELD, GAME_SESSION_ARN_FIELD, GAME_SESSION_REGION_FIELD,
            PLAYER_LATENCIES_FIELD, START_TIME_FIELD, END_TIME_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, PORT_FIELD,
            PLACED_PLAYER_SESSIONS_FIELD, GAME_SESSION_DATA_FIELD, MATCHMAKER_DATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String placementId;

    private final String gameSessionQueueName;

    private final String status;

    private final List<GameProperty> gameProperties;

    private final Integer maximumPlayerSessionCount;

    private final String gameSessionName;

    private final String gameSessionId;

    private final String gameSessionArn;

    private final String gameSessionRegion;

    private final List<PlayerLatency> playerLatencies;

    private final Instant startTime;

    private final Instant endTime;

    private final String ipAddress;

    private final String dnsName;

    private final Integer port;

    private final List<PlacedPlayerSession> placedPlayerSessions;

    private final String gameSessionData;

    private final String matchmakerData;

    private GameSessionPlacement(BuilderImpl builder) {
        this.placementId = builder.placementId;
        this.gameSessionQueueName = builder.gameSessionQueueName;
        this.status = builder.status;
        this.gameProperties = builder.gameProperties;
        this.maximumPlayerSessionCount = builder.maximumPlayerSessionCount;
        this.gameSessionName = builder.gameSessionName;
        this.gameSessionId = builder.gameSessionId;
        this.gameSessionArn = builder.gameSessionArn;
        this.gameSessionRegion = builder.gameSessionRegion;
        this.playerLatencies = builder.playerLatencies;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.port = builder.port;
        this.placedPlayerSessions = builder.placedPlayerSessions;
        this.gameSessionData = builder.gameSessionData;
        this.matchmakerData = builder.matchmakerData;
    }

    /**
     * <p>
     * Unique identifier for a game session placement.
     * </p>
     * 
     * @return Unique identifier for a game session placement.
     */
    public String placementId() {
        return placementId;
    }

    /**
     * <p>
     * Descriptive label that is associated with game session queue. Queue names must be unique within each region.
     * </p>
     * 
     * @return Descriptive label that is associated with game session queue. Queue names must be unique within each
     *         region.
     */
    public String gameSessionQueueName() {
        return gameSessionQueueName;
    }

    /**
     * <p>
     * Current status of the game session placement request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created. Values
     * for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
     * resubmit the placement request as needed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionPlacementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session placement request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created.
     *         Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
     *         can resubmit the placement request as needed.
     *         </p>
     *         </li>
     * @see GameSessionPlacementState
     */
    public GameSessionPlacementState status() {
        return GameSessionPlacementState.fromValue(status);
    }

    /**
     * <p>
     * Current status of the game session placement request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created. Values
     * for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
     * resubmit the placement request as needed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameSessionPlacementState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the game session placement request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created.
     *         Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
     *         can resubmit the placement request as needed.
     *         </p>
     *         </li>
     * @see GameSessionPlacementState
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a game
     * server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Set of custom properties for a game session, formatted as key:value pairs. These properties are passed to
     *         a game server process in the <a>GameSession</a> object with a request to start a new game session (see <a
     *         href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>).
     */
    public List<GameProperty> gameProperties() {
        return gameProperties;
    }

    /**
     * <p>
     * Maximum number of players that can be connected simultaneously to the game session.
     * </p>
     * 
     * @return Maximum number of players that can be connected simultaneously to the game session.
     */
    public Integer maximumPlayerSessionCount() {
        return maximumPlayerSessionCount;
    }

    /**
     * <p>
     * Descriptive label that is associated with a game session. Session names do not need to be unique.
     * </p>
     * 
     * @return Descriptive label that is associated with a game session. Session names do not need to be unique.
     */
    public String gameSessionName() {
        return gameSessionName;
    }

    /**
     * <p>
     * Unique identifier for the game session. This value is set once the new game session is placed (placement status
     * is <code>FULFILLED</code>).
     * </p>
     * 
     * @return Unique identifier for the game session. This value is set once the new game session is placed (placement
     *         status is <code>FULFILLED</code>).
     */
    public String gameSessionId() {
        return gameSessionId;
    }

    /**
     * <p>
     * Identifier for the game session created by this placement request. This value is set once the new game session is
     * placed (placement status is <code>FULFILLED</code>). This identifier is unique across all regions. You can use
     * this value as a <code>GameSessionId</code> value as needed.
     * </p>
     * 
     * @return Identifier for the game session created by this placement request. This value is set once the new game
     *         session is placed (placement status is <code>FULFILLED</code>). This identifier is unique across all
     *         regions. You can use this value as a <code>GameSessionId</code> value as needed.
     */
    public String gameSessionArn() {
        return gameSessionArn;
    }

    /**
     * <p>
     * Name of the region where the game session created by this placement request is running. This value is set once
     * the new game session is placed (placement status is <code>FULFILLED</code>).
     * </p>
     * 
     * @return Name of the region where the game session created by this placement request is running. This value is set
     *         once the new game session is placed (placement status is <code>FULFILLED</code>).
     */
    public String gameSessionRegion() {
        return gameSessionRegion;
    }

    /**
     * <p>
     * Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences when
     * connected to AWS regions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences when
     *         connected to AWS regions.
     */
    public List<PlayerLatency> playerLatencies() {
        return playerLatencies;
    }

    /**
     * <p>
     * Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix
     *         time as milliseconds (for example "1469498468.057").
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * Time stamp indicating when this request was completed, canceled, or timed out.
     * </p>
     * 
     * @return Time stamp indicating when this request was completed, canceled, or timed out.
     */
    public Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP address and
     * port number. This value is set once the new game session is placed (placement status is <code>FULFILLED</code>).
     * </p>
     * 
     * @return IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP
     *         address and port number. This value is set once the new game session is placed (placement status is
     *         <code>FULFILLED</code>).
     */
    public String ipAddress() {
        return ipAddress;
    }

    /**
     * Returns the value of the DnsName property for this object.
     * 
     * @return The value of the DnsName property for this object.
     */
    public String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Port number for the game session. To connect to a Amazon GameLift game server, an app needs both the IP address
     * and port number. This value is set once the new game session is placed (placement status is
     * <code>FULFILLED</code>).
     * </p>
     * 
     * @return Port number for the game session. To connect to a Amazon GameLift game server, an app needs both the IP
     *         address and port number. This value is set once the new game session is placed (placement status is
     *         <code>FULFILLED</code>).
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * Collection of information on player sessions created in response to the game session placement request. These
     * player sessions are created only once a new game session is successfully placed (placement status is
     * <code>FULFILLED</code>). This information includes the player ID (as provided in the placement request) and the
     * corresponding player session ID. Retrieve full player sessions by calling <a>DescribePlayerSessions</a> with the
     * player session ID.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Collection of information on player sessions created in response to the game session placement request.
     *         These player sessions are created only once a new game session is successfully placed (placement status
     *         is <code>FULFILLED</code>). This information includes the player ID (as provided in the placement
     *         request) and the corresponding player session ID. Retrieve full player sessions by calling
     *         <a>DescribePlayerSessions</a> with the player session ID.
     */
    public List<PlacedPlayerSession> placedPlayerSessions() {
        return placedPlayerSessions;
    }

    /**
     * <p>
     * Set of custom game session properties, formatted as a single string value. This data is passed to a game server
     * process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     * >Start a Game Session</a>).
     * </p>
     * 
     * @return Set of custom game session properties, formatted as a single string value. This data is passed to a game
     *         server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
     *         "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
     *         >Start a Game Session</a>).
     */
    public String gameSessionData() {
        return gameSessionData;
    }

    /**
     * <p>
     * Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
     * identifies the matchmaking configuration used to create the match, and contains data on all players assigned to
     * the match, including player attributes and team assignments. For more details on matchmaker data, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-server.html#match-server-data">Match
     * Data</a>.
     * </p>
     * 
     * @return Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
     *         identifies the matchmaking configuration used to create the match, and contains data on all players
     *         assigned to the match, including player attributes and team assignments. For more details on matchmaker
     *         data, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-server.html#match-server-data"
     *         >Match Data</a>.
     */
    public String matchmakerData() {
        return matchmakerData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(placementId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionQueueName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gameProperties());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPlayerSessionCount());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionName());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionArn());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionRegion());
        hashCode = 31 * hashCode + Objects.hashCode(playerLatencies());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(placedPlayerSessions());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionData());
        hashCode = 31 * hashCode + Objects.hashCode(matchmakerData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameSessionPlacement)) {
            return false;
        }
        GameSessionPlacement other = (GameSessionPlacement) obj;
        return Objects.equals(placementId(), other.placementId())
                && Objects.equals(gameSessionQueueName(), other.gameSessionQueueName())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(gameProperties(), other.gameProperties())
                && Objects.equals(maximumPlayerSessionCount(), other.maximumPlayerSessionCount())
                && Objects.equals(gameSessionName(), other.gameSessionName())
                && Objects.equals(gameSessionId(), other.gameSessionId())
                && Objects.equals(gameSessionArn(), other.gameSessionArn())
                && Objects.equals(gameSessionRegion(), other.gameSessionRegion())
                && Objects.equals(playerLatencies(), other.playerLatencies()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(port(), other.port())
                && Objects.equals(placedPlayerSessions(), other.placedPlayerSessions())
                && Objects.equals(gameSessionData(), other.gameSessionData())
                && Objects.equals(matchmakerData(), other.matchmakerData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GameSessionPlacement").add("PlacementId", placementId())
                .add("GameSessionQueueName", gameSessionQueueName()).add("Status", statusAsString())
                .add("GameProperties", gameProperties()).add("MaximumPlayerSessionCount", maximumPlayerSessionCount())
                .add("GameSessionName", gameSessionName()).add("GameSessionId", gameSessionId())
                .add("GameSessionArn", gameSessionArn()).add("GameSessionRegion", gameSessionRegion())
                .add("PlayerLatencies", playerLatencies()).add("StartTime", startTime()).add("EndTime", endTime())
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("Port", port())
                .add("PlacedPlayerSessions", placedPlayerSessions()).add("GameSessionData", gameSessionData())
                .add("MatchmakerData", matchmakerData()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlacementId":
            return Optional.ofNullable(clazz.cast(placementId()));
        case "GameSessionQueueName":
            return Optional.ofNullable(clazz.cast(gameSessionQueueName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "GameProperties":
            return Optional.ofNullable(clazz.cast(gameProperties()));
        case "MaximumPlayerSessionCount":
            return Optional.ofNullable(clazz.cast(maximumPlayerSessionCount()));
        case "GameSessionName":
            return Optional.ofNullable(clazz.cast(gameSessionName()));
        case "GameSessionId":
            return Optional.ofNullable(clazz.cast(gameSessionId()));
        case "GameSessionArn":
            return Optional.ofNullable(clazz.cast(gameSessionArn()));
        case "GameSessionRegion":
            return Optional.ofNullable(clazz.cast(gameSessionRegion()));
        case "PlayerLatencies":
            return Optional.ofNullable(clazz.cast(playerLatencies()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "PlacedPlayerSessions":
            return Optional.ofNullable(clazz.cast(placedPlayerSessions()));
        case "GameSessionData":
            return Optional.ofNullable(clazz.cast(gameSessionData()));
        case "MatchmakerData":
            return Optional.ofNullable(clazz.cast(matchmakerData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameSessionPlacement, T> g) {
        return obj -> g.apply((GameSessionPlacement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameSessionPlacement> {
        /**
         * <p>
         * Unique identifier for a game session placement.
         * </p>
         * 
         * @param placementId
         *        Unique identifier for a game session placement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementId(String placementId);

        /**
         * <p>
         * Descriptive label that is associated with game session queue. Queue names must be unique within each region.
         * </p>
         * 
         * @param gameSessionQueueName
         *        Descriptive label that is associated with game session queue. Queue names must be unique within each
         *        region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionQueueName(String gameSessionQueueName);

        /**
         * <p>
         * Current status of the game session placement request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created.
         * Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
         * resubmit the placement request as needed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the game session placement request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully
         *        created. Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
         *        can resubmit the placement request as needed.
         *        </p>
         *        </li>
         * @see GameSessionPlacementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionPlacementState
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the game session placement request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully created.
         * Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You can
         * resubmit the placement request as needed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the game session placement request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>PENDING</b> -- The placement request is currently in the queue waiting to be processed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>FULFILLED</b> -- A new game session and player sessions (if requested) have been successfully
         *        created. Values for <i>GameSessionArn</i> and <i>GameSessionRegion</i> are available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CANCELLED</b> -- The placement request was canceled with a call to <a>StopGameSessionPlacement</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMED_OUT</b> -- A new game session was not successfully created before the time limit expired. You
         *        can resubmit the placement request as needed.
         *        </p>
         *        </li>
         * @see GameSessionPlacementState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameSessionPlacementState
         */
        Builder status(GameSessionPlacementState status);

        /**
         * <p>
         * Set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>).
         * </p>
         * 
         * @param gameProperties
         *        Set of custom properties for a game session, formatted as key:value pairs. These properties are passed
         *        to a game server process in the <a>GameSession</a> object with a request to start a new game session
         *        (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(Collection<GameProperty> gameProperties);

        /**
         * <p>
         * Set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>).
         * </p>
         * 
         * @param gameProperties
         *        Set of custom properties for a game session, formatted as key:value pairs. These properties are passed
         *        to a game server process in the <a>GameSession</a> object with a request to start a new game session
         *        (see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameProperties(GameProperty... gameProperties);

        /**
         * <p>
         * Set of custom properties for a game session, formatted as key:value pairs. These properties are passed to a
         * game server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>).
         * </p>
         * This is a convenience that creates an instance of the {@link List<GameProperty>.Builder} avoiding the need to
         * create one manually via {@link List<GameProperty>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GameProperty>.Builder#build()} is called immediately and its
         * result is passed to {@link #gameProperties(List<GameProperty>)}.
         * 
         * @param gameProperties
         *        a consumer that will call methods on {@link List<GameProperty>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #gameProperties(List<GameProperty>)
         */
        Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties);

        /**
         * <p>
         * Maximum number of players that can be connected simultaneously to the game session.
         * </p>
         * 
         * @param maximumPlayerSessionCount
         *        Maximum number of players that can be connected simultaneously to the game session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount);

        /**
         * <p>
         * Descriptive label that is associated with a game session. Session names do not need to be unique.
         * </p>
         * 
         * @param gameSessionName
         *        Descriptive label that is associated with a game session. Session names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionName(String gameSessionName);

        /**
         * <p>
         * Unique identifier for the game session. This value is set once the new game session is placed (placement
         * status is <code>FULFILLED</code>).
         * </p>
         * 
         * @param gameSessionId
         *        Unique identifier for the game session. This value is set once the new game session is placed
         *        (placement status is <code>FULFILLED</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionId(String gameSessionId);

        /**
         * <p>
         * Identifier for the game session created by this placement request. This value is set once the new game
         * session is placed (placement status is <code>FULFILLED</code>). This identifier is unique across all regions.
         * You can use this value as a <code>GameSessionId</code> value as needed.
         * </p>
         * 
         * @param gameSessionArn
         *        Identifier for the game session created by this placement request. This value is set once the new game
         *        session is placed (placement status is <code>FULFILLED</code>). This identifier is unique across all
         *        regions. You can use this value as a <code>GameSessionId</code> value as needed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionArn(String gameSessionArn);

        /**
         * <p>
         * Name of the region where the game session created by this placement request is running. This value is set
         * once the new game session is placed (placement status is <code>FULFILLED</code>).
         * </p>
         * 
         * @param gameSessionRegion
         *        Name of the region where the game session created by this placement request is running. This value is
         *        set once the new game session is placed (placement status is <code>FULFILLED</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionRegion(String gameSessionRegion);

        /**
         * <p>
         * Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences when
         * connected to AWS regions.
         * </p>
         * 
         * @param playerLatencies
         *        Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences
         *        when connected to AWS regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencies(Collection<PlayerLatency> playerLatencies);

        /**
         * <p>
         * Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences when
         * connected to AWS regions.
         * </p>
         * 
         * @param playerLatencies
         *        Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences
         *        when connected to AWS regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerLatencies(PlayerLatency... playerLatencies);

        /**
         * <p>
         * Set of values, expressed in milliseconds, indicating the amount of latency that a player experiences when
         * connected to AWS regions.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PlayerLatency>.Builder} avoiding the need
         * to create one manually via {@link List<PlayerLatency>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PlayerLatency>.Builder#build()} is called immediately and
         * its result is passed to {@link #playerLatencies(List<PlayerLatency>)}.
         * 
         * @param playerLatencies
         *        a consumer that will call methods on {@link List<PlayerLatency>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #playerLatencies(List<PlayerLatency>)
         */
        Builder playerLatencies(Consumer<PlayerLatency.Builder>... playerLatencies);

        /**
         * <p>
         * Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param startTime
         *        Time stamp indicating when this request was placed in the queue. Format is a number expressed in Unix
         *        time as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * Time stamp indicating when this request was completed, canceled, or timed out.
         * </p>
         * 
         * @param endTime
         *        Time stamp indicating when this request was completed, canceled, or timed out.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP address
         * and port number. This value is set once the new game session is placed (placement status is
         * <code>FULFILLED</code>).
         * </p>
         * 
         * @param ipAddress
         *        IP address of the game session. To connect to a Amazon GameLift game server, an app needs both the IP
         *        address and port number. This value is set once the new game session is placed (placement status is
         *        <code>FULFILLED</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * Sets the value of the DnsName property for this object.
         *
         * @param dnsName
         *        The new value for the DnsName property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Port number for the game session. To connect to a Amazon GameLift game server, an app needs both the IP
         * address and port number. This value is set once the new game session is placed (placement status is
         * <code>FULFILLED</code>).
         * </p>
         * 
         * @param port
         *        Port number for the game session. To connect to a Amazon GameLift game server, an app needs both the
         *        IP address and port number. This value is set once the new game session is placed (placement status is
         *        <code>FULFILLED</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Collection of information on player sessions created in response to the game session placement request. These
         * player sessions are created only once a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID (as provided in the placement request) and
         * the corresponding player session ID. Retrieve full player sessions by calling <a>DescribePlayerSessions</a>
         * with the player session ID.
         * </p>
         * 
         * @param placedPlayerSessions
         *        Collection of information on player sessions created in response to the game session placement
         *        request. These player sessions are created only once a new game session is successfully placed
         *        (placement status is <code>FULFILLED</code>). This information includes the player ID (as provided in
         *        the placement request) and the corresponding player session ID. Retrieve full player sessions by
         *        calling <a>DescribePlayerSessions</a> with the player session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placedPlayerSessions(Collection<PlacedPlayerSession> placedPlayerSessions);

        /**
         * <p>
         * Collection of information on player sessions created in response to the game session placement request. These
         * player sessions are created only once a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID (as provided in the placement request) and
         * the corresponding player session ID. Retrieve full player sessions by calling <a>DescribePlayerSessions</a>
         * with the player session ID.
         * </p>
         * 
         * @param placedPlayerSessions
         *        Collection of information on player sessions created in response to the game session placement
         *        request. These player sessions are created only once a new game session is successfully placed
         *        (placement status is <code>FULFILLED</code>). This information includes the player ID (as provided in
         *        the placement request) and the corresponding player session ID. Retrieve full player sessions by
         *        calling <a>DescribePlayerSessions</a> with the player session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placedPlayerSessions(PlacedPlayerSession... placedPlayerSessions);

        /**
         * <p>
         * Collection of information on player sessions created in response to the game session placement request. These
         * player sessions are created only once a new game session is successfully placed (placement status is
         * <code>FULFILLED</code>). This information includes the player ID (as provided in the placement request) and
         * the corresponding player session ID. Retrieve full player sessions by calling <a>DescribePlayerSessions</a>
         * with the player session ID.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PlacedPlayerSession>.Builder} avoiding the
         * need to create one manually via {@link List<PlacedPlayerSession>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PlacedPlayerSession>.Builder#build()} is called immediately
         * and its result is passed to {@link #placedPlayerSessions(List<PlacedPlayerSession>)}.
         * 
         * @param placedPlayerSessions
         *        a consumer that will call methods on {@link List<PlacedPlayerSession>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placedPlayerSessions(List<PlacedPlayerSession>)
         */
        Builder placedPlayerSessions(Consumer<PlacedPlayerSession.Builder>... placedPlayerSessions);

        /**
         * <p>
         * Set of custom game session properties, formatted as a single string value. This data is passed to a game
         * server process in the <a>GameSession</a> object with a request to start a new game session (see <a href=
         * "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         * >Start a Game Session</a>).
         * </p>
         * 
         * @param gameSessionData
         *        Set of custom game session properties, formatted as a single string value. This data is passed to a
         *        game server process in the <a>GameSession</a> object with a request to start a new game session (see
         *        <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession"
         *        >Start a Game Session</a>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionData(String gameSessionData);

        /**
         * <p>
         * Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string. It
         * identifies the matchmaking configuration used to create the match, and contains data on all players assigned
         * to the match, including player attributes and team assignments. For more details on matchmaker data, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/match-server.html#match-server-data">Match
         * Data</a>.
         * </p>
         * 
         * @param matchmakerData
         *        Information on the matchmaking process for this game. Data is in JSON syntax, formatted as a string.
         *        It identifies the matchmaking configuration used to create the match, and contains data on all players
         *        assigned to the match, including player attributes and team assignments. For more details on
         *        matchmaker data, see <a href=
         *        "https://docs.aws.amazon.com/gamelift/latest/developerguide/match-server.html#match-server-data">Match
         *        Data</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchmakerData(String matchmakerData);
    }

    static final class BuilderImpl implements Builder {
        private String placementId;

        private String gameSessionQueueName;

        private String status;

        private List<GameProperty> gameProperties = DefaultSdkAutoConstructList.getInstance();

        private Integer maximumPlayerSessionCount;

        private String gameSessionName;

        private String gameSessionId;

        private String gameSessionArn;

        private String gameSessionRegion;

        private List<PlayerLatency> playerLatencies = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private String ipAddress;

        private String dnsName;

        private Integer port;

        private List<PlacedPlayerSession> placedPlayerSessions = DefaultSdkAutoConstructList.getInstance();

        private String gameSessionData;

        private String matchmakerData;

        private BuilderImpl() {
        }

        private BuilderImpl(GameSessionPlacement model) {
            placementId(model.placementId);
            gameSessionQueueName(model.gameSessionQueueName);
            status(model.status);
            gameProperties(model.gameProperties);
            maximumPlayerSessionCount(model.maximumPlayerSessionCount);
            gameSessionName(model.gameSessionName);
            gameSessionId(model.gameSessionId);
            gameSessionArn(model.gameSessionArn);
            gameSessionRegion(model.gameSessionRegion);
            playerLatencies(model.playerLatencies);
            startTime(model.startTime);
            endTime(model.endTime);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            port(model.port);
            placedPlayerSessions(model.placedPlayerSessions);
            gameSessionData(model.gameSessionData);
            matchmakerData(model.matchmakerData);
        }

        public final String getPlacementId() {
            return placementId;
        }

        @Override
        public final Builder placementId(String placementId) {
            this.placementId = placementId;
            return this;
        }

        public final void setPlacementId(String placementId) {
            this.placementId = placementId;
        }

        public final String getGameSessionQueueName() {
            return gameSessionQueueName;
        }

        @Override
        public final Builder gameSessionQueueName(String gameSessionQueueName) {
            this.gameSessionQueueName = gameSessionQueueName;
            return this;
        }

        public final void setGameSessionQueueName(String gameSessionQueueName) {
            this.gameSessionQueueName = gameSessionQueueName;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GameSessionPlacementState status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Collection<GameProperty.Builder> getGameProperties() {
            return gameProperties != null ? gameProperties.stream().map(GameProperty::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder gameProperties(Collection<GameProperty> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copy(gameProperties);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(GameProperty... gameProperties) {
            gameProperties(Arrays.asList(gameProperties));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder gameProperties(Consumer<GameProperty.Builder>... gameProperties) {
            gameProperties(Stream.of(gameProperties).map(c -> GameProperty.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setGameProperties(Collection<GameProperty.BuilderImpl> gameProperties) {
            this.gameProperties = GamePropertyListCopier.copyFromBuilder(gameProperties);
        }

        public final Integer getMaximumPlayerSessionCount() {
            return maximumPlayerSessionCount;
        }

        @Override
        public final Builder maximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
            return this;
        }

        public final void setMaximumPlayerSessionCount(Integer maximumPlayerSessionCount) {
            this.maximumPlayerSessionCount = maximumPlayerSessionCount;
        }

        public final String getGameSessionName() {
            return gameSessionName;
        }

        @Override
        public final Builder gameSessionName(String gameSessionName) {
            this.gameSessionName = gameSessionName;
            return this;
        }

        public final void setGameSessionName(String gameSessionName) {
            this.gameSessionName = gameSessionName;
        }

        public final String getGameSessionId() {
            return gameSessionId;
        }

        @Override
        public final Builder gameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
            return this;
        }

        public final void setGameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
        }

        public final String getGameSessionArn() {
            return gameSessionArn;
        }

        @Override
        public final Builder gameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
            return this;
        }

        public final void setGameSessionArn(String gameSessionArn) {
            this.gameSessionArn = gameSessionArn;
        }

        public final String getGameSessionRegion() {
            return gameSessionRegion;
        }

        @Override
        public final Builder gameSessionRegion(String gameSessionRegion) {
            this.gameSessionRegion = gameSessionRegion;
            return this;
        }

        public final void setGameSessionRegion(String gameSessionRegion) {
            this.gameSessionRegion = gameSessionRegion;
        }

        public final Collection<PlayerLatency.Builder> getPlayerLatencies() {
            return playerLatencies != null ? playerLatencies.stream().map(PlayerLatency::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder playerLatencies(Collection<PlayerLatency> playerLatencies) {
            this.playerLatencies = PlayerLatencyListCopier.copy(playerLatencies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencies(PlayerLatency... playerLatencies) {
            playerLatencies(Arrays.asList(playerLatencies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder playerLatencies(Consumer<PlayerLatency.Builder>... playerLatencies) {
            playerLatencies(Stream.of(playerLatencies).map(c -> PlayerLatency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPlayerLatencies(Collection<PlayerLatency.BuilderImpl> playerLatencies) {
            this.playerLatencies = PlayerLatencyListCopier.copyFromBuilder(playerLatencies);
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final Collection<PlacedPlayerSession.Builder> getPlacedPlayerSessions() {
            return placedPlayerSessions != null ? placedPlayerSessions.stream().map(PlacedPlayerSession::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder placedPlayerSessions(Collection<PlacedPlayerSession> placedPlayerSessions) {
            this.placedPlayerSessions = PlacedPlayerSessionListCopier.copy(placedPlayerSessions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placedPlayerSessions(PlacedPlayerSession... placedPlayerSessions) {
            placedPlayerSessions(Arrays.asList(placedPlayerSessions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder placedPlayerSessions(Consumer<PlacedPlayerSession.Builder>... placedPlayerSessions) {
            placedPlayerSessions(Stream.of(placedPlayerSessions).map(c -> PlacedPlayerSession.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPlacedPlayerSessions(Collection<PlacedPlayerSession.BuilderImpl> placedPlayerSessions) {
            this.placedPlayerSessions = PlacedPlayerSessionListCopier.copyFromBuilder(placedPlayerSessions);
        }

        public final String getGameSessionData() {
            return gameSessionData;
        }

        @Override
        public final Builder gameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
            return this;
        }

        public final void setGameSessionData(String gameSessionData) {
            this.gameSessionData = gameSessionData;
        }

        public final String getMatchmakerData() {
            return matchmakerData;
        }

        @Override
        public final Builder matchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
            return this;
        }

        public final void setMatchmakerData(String matchmakerData) {
            this.matchmakerData = matchmakerData;
        }

        @Override
        public GameSessionPlacement build() {
            return new GameSessionPlacement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
