/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gameliftstreams;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.gameliftstreams.internal.GameLiftStreamsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.gameliftstreams.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.gameliftstreams.model.AccessDeniedException;
import software.amazon.awssdk.services.gameliftstreams.model.AddStreamGroupLocationsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.AddStreamGroupLocationsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.AssociateApplicationsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.AssociateApplicationsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ConflictException;
import software.amazon.awssdk.services.gameliftstreams.model.CreateApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.CreateApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.CreateStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.CreateStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.CreateStreamSessionConnectionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.CreateStreamSessionConnectionResponse;
import software.amazon.awssdk.services.gameliftstreams.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.DeleteStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.DeleteStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.DisassociateApplicationsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.DisassociateApplicationsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ExportStreamSessionFilesRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ExportStreamSessionFilesResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GameLiftStreamsException;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionResponse;
import software.amazon.awssdk.services.gameliftstreams.model.InternalServerException;
import software.amazon.awssdk.services.gameliftstreams.model.ListApplicationsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ListApplicationsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamGroupsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamGroupsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamSessionsByAccountRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamSessionsByAccountResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamSessionsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ListStreamSessionsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.gameliftstreams.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.gameliftstreams.model.RemoveStreamGroupLocationsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.RemoveStreamGroupLocationsResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ResourceNotFoundException;
import software.amazon.awssdk.services.gameliftstreams.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.gameliftstreams.model.StartStreamSessionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.StartStreamSessionResponse;
import software.amazon.awssdk.services.gameliftstreams.model.TagResourceRequest;
import software.amazon.awssdk.services.gameliftstreams.model.TagResourceResponse;
import software.amazon.awssdk.services.gameliftstreams.model.TerminateStreamSessionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.TerminateStreamSessionResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ThrottlingException;
import software.amazon.awssdk.services.gameliftstreams.model.UntagResourceRequest;
import software.amazon.awssdk.services.gameliftstreams.model.UntagResourceResponse;
import software.amazon.awssdk.services.gameliftstreams.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.UpdateStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.UpdateStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.ValidationException;
import software.amazon.awssdk.services.gameliftstreams.transform.AddStreamGroupLocationsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.AssociateApplicationsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.CreateStreamGroupRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.CreateStreamSessionConnectionRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.DeleteStreamGroupRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.DisassociateApplicationsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ExportStreamSessionFilesRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.GetStreamGroupRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.GetStreamSessionRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ListStreamGroupsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ListStreamSessionsByAccountRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ListStreamSessionsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.RemoveStreamGroupLocationsRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.StartStreamSessionRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.TerminateStreamSessionRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.transform.UpdateStreamGroupRequestMarshaller;
import software.amazon.awssdk.services.gameliftstreams.waiters.GameLiftStreamsAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GameLiftStreamsAsyncClient}.
 *
 * @see GameLiftStreamsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGameLiftStreamsAsyncClient implements GameLiftStreamsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGameLiftStreamsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultGameLiftStreamsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "GameLiftStreams" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Add locations that can host stream sessions. You configure locations and their corresponding capacity for each
     * stream group. Creating a stream group in a location that's nearest to your end users can help minimize latency
     * and improve quality.
     * </p>
     * <p>
     * This operation provisions stream capacity at the specified locations. By default, all locations have 1 or 2
     * capacity, depending on the stream class option: 2 for 'High' and 1 for 'Ultra' and 'Win2022'. This operation also
     * copies the content files of all associated applications to an internal S3 bucket at each location. This allows
     * Amazon GameLift Streams to host performant stream sessions.
     * </p>
     *
     * @param addStreamGroupLocationsRequest
     * @return A Java Future containing the result of the AddStreamGroupLocations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause the resource to exceed an allowed service
     *         quota. Resolve the issue before you try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.AddStreamGroupLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/AddStreamGroupLocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddStreamGroupLocationsResponse> addStreamGroupLocations(
            AddStreamGroupLocationsRequest addStreamGroupLocationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addStreamGroupLocationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addStreamGroupLocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddStreamGroupLocations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddStreamGroupLocationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddStreamGroupLocationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AddStreamGroupLocationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddStreamGroupLocationsRequest, AddStreamGroupLocationsResponse>()
                            .withOperationName("AddStreamGroupLocations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AddStreamGroupLocationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addStreamGroupLocationsRequest));
            CompletableFuture<AddStreamGroupLocationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you associate, or link, an application with a stream group, then Amazon GameLift Streams can launch the
     * application using the stream group's allocated compute resources. The stream group must be in <code>ACTIVE</code>
     * status. You can reverse this action by using <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DisassociateApplications.html"
     * >DisassociateApplications</a>.
     * </p>
     * <p>
     * If a stream group does not already have a linked application, Amazon GameLift Streams will automatically assign
     * the first application provided in <code>ApplicationIdentifiers</code> as the default.
     * </p>
     *
     * @param associateApplicationsRequest
     * @return A Java Future containing the result of the AssociateApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause the resource to exceed an allowed service
     *         quota. Resolve the issue before you try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.AssociateApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/AssociateApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateApplicationsResponse> associateApplications(
            AssociateApplicationsRequest associateApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateApplicationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateApplicationsRequest, AssociateApplicationsResponse>()
                            .withOperationName("AssociateApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateApplicationsRequest));
            CompletableFuture<AssociateApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an application resource in Amazon GameLift Streams, which specifies the application content you want to
     * stream, such as a game build or other software, and configures the settings to run it.
     * </p>
     * <p>
     * Before you create an application, upload your application content files to an Amazon Simple Storage Service
     * (Amazon S3) bucket. For more information, see <b>Getting Started</b> in the Amazon GameLift Streams Developer
     * Guide.
     * </p>
     * <important>
     * <p>
     * Make sure that your files in the Amazon S3 bucket are the correct version you want to use. If you change the
     * files at a later time, you will need to create a new Amazon GameLift Streams application.
     * </p>
     * </important>
     * <p>
     * If the request is successful, Amazon GameLift Streams begins to create an application and sets the status to
     * <code>INITIALIZED</code>. When an application reaches <code>READY</code> status, you can use the application to
     * set up stream groups and start streams. To track application status, call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetApplication.html"
     * >GetApplication</a>.
     * </p>
     *
     * @param createApplicationRequest
     * @return A Java Future containing the result of the CreateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause the resource to exceed an allowed service
     *         quota. Resolve the issue before you try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/CreateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateApplicationResponse> createApplication(CreateApplicationRequest createApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                            .withOperationName("CreateApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createApplicationRequest));
            CompletableFuture<CreateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Manage how Amazon GameLift Streams streams your applications by using a stream group. A stream group is a
     * collection of resources that Amazon GameLift Streams uses to stream your application to end-users. When you
     * create a stream group, you specify an application to stream by default and the type of hardware to use, such as
     * the graphical processing unit (GPU). You can also link additional applications, which allows you to stream those
     * applications using this stream group. Depending on your expected users, you also scale the number of concurrent
     * streams you want to support at one time, and in what locations.
     * </p>
     * <p>
     * Stream capacity represents the number of concurrent streams that can be active at a time. You set stream capacity
     * per location, per stream group. There are two types of capacity, always-on and on-demand:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Always-on</b>: The streaming capacity that is allocated and ready to handle stream requests without delay. You
     * pay for this capacity whether it's in use or not. Best for quickest time from streaming request to streaming
     * session. Default is 1 when creating a stream group or adding a location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>On-demand</b>: The streaming capacity that Amazon GameLift Streams can allocate in response to stream
     * requests, and then de-allocate when the session has terminated. This offers a cost control measure at the expense
     * of a greater startup time (typically under 5 minutes). Default is 0 when creating a stream group or adding a
     * location.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To adjust the capacity of any <code>ACTIVE</code> stream group, call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UpdateStreamGroup.html"
     * >UpdateStreamGroup</a>.
     * </p>
     * <p>
     * If the request is successful, Amazon GameLift Streams begins creating the stream group. Amazon GameLift Streams
     * assigns a unique ID to the stream group resource and sets the status to <code>ACTIVATING</code>. When the stream
     * group reaches <code>ACTIVE</code> status, you can start stream sessions by using <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_StartStreamSession.html"
     * >StartStreamSession</a>. To check the stream group's status, call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamGroup.html"
     * >GetStreamGroup</a>.
     * </p>
     *
     * @param createStreamGroupRequest
     * @return A Java Future containing the result of the CreateStreamGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause the resource to exceed an allowed service
     *         quota. Resolve the issue before you try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.CreateStreamGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/CreateStreamGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamGroupResponse> createStreamGroup(CreateStreamGroupRequest createStreamGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStreamGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStreamGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStreamGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStreamGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStreamGroupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateStreamGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamGroupRequest, CreateStreamGroupResponse>()
                            .withOperationName("CreateStreamGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStreamGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStreamGroupRequest));
            CompletableFuture<CreateStreamGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables clients to reconnect to a stream session while preserving all session state and data in the disconnected
     * session. This reconnection process can be initiated when a stream session is in either
     * <code>PENDING_CLIENT_RECONNECTION</code> or <code>ACTIVE</code> status. The process works as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Initial disconnect:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When a client disconnects or loses connection, the stream session transitions from <code>CONNECTED</code> to
     * <code>PENDING_CLIENT_RECONNECTION</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Reconnection time window:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Clients have <code>ConnectionTimeoutSeconds</code> (defined in <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_StartStreamSession.html"
     * >StartStreamSession</a>) to reconnect before session termination
     * </p>
     * </li>
     * <li>
     * <p>
     * Your backend server must call <b>CreateStreamSessionConnection</b> to initiate reconnection
     * </p>
     * </li>
     * <li>
     * <p>
     * Session transitions to <code>RECONNECTING</code> status
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Reconnection completion:
     * </p>
     * <ul>
     * <li>
     * <p>
     * On successful <b>CreateStreamSessionConnection</b>, session status changes to <code>ACTIVE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Provide the new connection information to the requesting client
     * </p>
     * </li>
     * <li>
     * <p>
     * Client must establish connection within <code>ConnectionTimeoutSeconds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Session terminates automatically if client fails to connect in time
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <p>
     * For more information about the stream session lifecycle, see <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/stream-sessions.html">Stream sessions</a>
     * in the <i>Amazon GameLift Streams Developer Guide</i>.
     * </p>
     * <p>
     * To begin re-connecting to an existing stream session, specify the stream group ID and stream session ID that you
     * want to reconnect to, and the signal request to use with the stream.
     * </p>
     *
     * @param createStreamSessionConnectionRequest
     * @return A Java Future containing the result of the CreateStreamSessionConnection operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.CreateStreamSessionConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/CreateStreamSessionConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStreamSessionConnectionResponse> createStreamSessionConnection(
            CreateStreamSessionConnectionRequest createStreamSessionConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStreamSessionConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createStreamSessionConnectionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStreamSessionConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStreamSessionConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStreamSessionConnectionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateStreamSessionConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStreamSessionConnectionRequest, CreateStreamSessionConnectionResponse>()
                            .withOperationName("CreateStreamSessionConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStreamSessionConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStreamSessionConnectionRequest));
            CompletableFuture<CreateStreamSessionConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes an Amazon GameLift Streams application resource. This also deletes the application content
     * files stored with Amazon GameLift Streams. However, this does not delete the original files that you uploaded to
     * your Amazon S3 bucket; you can delete these any time after Amazon GameLift Streams creates an application, which
     * is the only time Amazon GameLift Streams accesses your Amazon S3 bucket.
     * </p>
     * <p>
     * You can only delete an application that meets the following conditions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The application is in <code>READY</code> or <code>ERROR</code> status. You cannot delete an application that's in
     * <code>PROCESSING</code> or <code>INITIALIZED</code> status.
     * </p>
     * </li>
     * <li>
     * <p>
     * The application is not the default application of any stream groups. You must first delete the stream group by
     * using <a href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DeleteStreamGroup.html">
     * DeleteStreamGroup</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The application is not linked to any stream groups. You must first unlink the stream group by using <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_DisassociateApplications.html"
     * >DisassociateApplications</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * An application is not streaming in any ongoing stream session. You must wait until the client ends the stream
     * session or call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TerminateStreamSession.html"
     * >TerminateStreamSession</a> to end the stream.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If any active stream groups exist for this application, this request returns a <code>ValidationException</code>.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return A Java Future containing the result of the DeleteApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/DeleteApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteApplicationResponse> deleteApplication(DeleteApplicationRequest deleteApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                            .withOperationName("DeleteApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteApplicationRequest));
            CompletableFuture<DeleteApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently deletes all compute resources and information related to a stream group. To delete a stream group,
     * specify the unique stream group identifier. During the deletion process, the stream group's status is
     * <code>DELETING</code>. This operation stops streams in progress and prevents new streams from starting. As a best
     * practice, before deleting the stream group, call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListStreamSessions.html"
     * >ListStreamSessions</a> to check for streams in progress and take action to stop them. When you delete a stream
     * group, any application associations referring to that stream group are automatically removed.
     * </p>
     *
     * @param deleteStreamGroupRequest
     * @return A Java Future containing the result of the DeleteStreamGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.DeleteStreamGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/DeleteStreamGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStreamGroupResponse> deleteStreamGroup(DeleteStreamGroupRequest deleteStreamGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStreamGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStreamGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStreamGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStreamGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStreamGroupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteStreamGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStreamGroupRequest, DeleteStreamGroupResponse>()
                            .withOperationName("DeleteStreamGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStreamGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStreamGroupRequest));
            CompletableFuture<DeleteStreamGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * When you disassociate, or unlink, an application from a stream group, you can no longer stream this application
     * by using that stream group's allocated compute resources. Any streams in process will continue until they
     * terminate, which helps avoid interrupting an end-user's stream. Amazon GameLift Streams will not initiate new
     * streams in the stream group using the disassociated application. The disassociate action does not affect the
     * stream capacity of a stream group.
     * </p>
     * <p>
     * If you disassociate the default application, Amazon GameLift Streams will automatically choose a new default
     * application from the remaining associated applications. To change which application is the default application,
     * call <a href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UpdateStreamGroup.html">
     * UpdateStreamGroup</a> and specify a new <code>DefaultApplicationIdentifier</code>.
     * </p>
     *
     * @param disassociateApplicationsRequest
     * @return A Java Future containing the result of the DisassociateApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.DisassociateApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/DisassociateApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateApplicationsResponse> disassociateApplications(
            DisassociateApplicationsRequest disassociateApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateApplicationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateApplicationsRequest, DisassociateApplicationsResponse>()
                            .withOperationName("DisassociateApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateApplicationsRequest));
            CompletableFuture<DisassociateApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Export the files that your application modifies or generates in a stream session, which can help you debug or
     * verify your application. When your application runs, it generates output files such as logs, diagnostic
     * information, crash dumps, save files, user data, screenshots, and so on. The files can be defined by the engine
     * or frameworks that your application uses, or information that you've programmed your application to output.
     * </p>
     * <p>
     * You can only call this action on a stream session that is in progress, specifically in one of the following
     * statuses <code>ACTIVE</code>, <code>CONNECTED</code>, <code>PENDING_CLIENT_RECONNECTION</code>, and
     * <code>RECONNECTING</code>. You must provide an Amazon Simple Storage Service (Amazon S3) bucket to store the
     * files in. When the session ends, Amazon GameLift Streams produces a compressed folder that contains all of the
     * files and directories that were modified or created by the application during the stream session. AWS uses your
     * security credentials to authenticate and authorize access to your Amazon S3 bucket.
     * </p>
     * <p>
     * Amazon GameLift Streams collects the following generated and modified files. Find them in the corresponding
     * folders in the <code>.zip</code> archive.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application/</code>: The folder where your application or game is stored.
     * </p>
     * </li>
     * </ul>
     * <ul>
     * <li>
     * <p>
     * <code>profile/</code>: The user profile folder.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>temp/</code>: The system temp folder.
     * </p>
     * </li>
     * </ul>
     * <p/>
     * <p>
     * To verify the status of the exported files, use GetStreamSession.
     * </p>
     * <p>
     * To delete the files, delete the object in the S3 bucket.
     * </p>
     *
     * @param exportStreamSessionFilesRequest
     * @return A Java Future containing the result of the ExportStreamSessionFiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ExportStreamSessionFiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ExportStreamSessionFiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportStreamSessionFilesResponse> exportStreamSessionFiles(
            ExportStreamSessionFilesRequest exportStreamSessionFilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportStreamSessionFilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportStreamSessionFilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportStreamSessionFiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportStreamSessionFilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportStreamSessionFilesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ExportStreamSessionFilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportStreamSessionFilesRequest, ExportStreamSessionFilesResponse>()
                            .withOperationName("ExportStreamSessionFiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ExportStreamSessionFilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportStreamSessionFilesRequest));
            CompletableFuture<ExportStreamSessionFilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for an Amazon GameLift Streams application resource. Specify the ID of the application that
     * you want to retrieve. If the operation is successful, it returns properties for the requested application.
     * </p>
     *
     * @param getApplicationRequest
     * @return A Java Future containing the result of the GetApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/GetApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetApplicationResponse> getApplication(GetApplicationRequest getApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getApplicationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                            .withOperationName("GetApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getApplicationRequest));
            CompletableFuture<GetApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for a Amazon GameLift Streams stream group resource. Specify the ID of the stream group that
     * you want to retrieve. If the operation is successful, it returns properties for the requested stream group.
     * </p>
     *
     * @param getStreamGroupRequest
     * @return A Java Future containing the result of the GetStreamGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.GetStreamGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/GetStreamGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStreamGroupResponse> getStreamGroup(GetStreamGroupRequest getStreamGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamGroupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStreamGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStreamGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStreamGroupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetStreamGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStreamGroupRequest, GetStreamGroupResponse>()
                            .withOperationName("GetStreamGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStreamGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStreamGroupRequest));
            CompletableFuture<GetStreamGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves properties for a Amazon GameLift Streams stream session resource. Specify the Amazon Resource Name
     * (ARN) of the stream session that you want to retrieve and its stream group ARN. If the operation is successful,
     * it returns properties for the requested resource.
     * </p>
     *
     * @param getStreamSessionRequest
     * @return A Java Future containing the result of the GetStreamSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.GetStreamSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/GetStreamSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStreamSessionResponse> getStreamSession(GetStreamSessionRequest getStreamSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStreamSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStreamSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStreamSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStreamSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStreamSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetStreamSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStreamSessionRequest, GetStreamSessionResponse>()
                            .withOperationName("GetStreamSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStreamSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStreamSessionRequest));
            CompletableFuture<GetStreamSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all Amazon GameLift Streams applications that are associated with the Amazon Web Services
     * account in use. This operation returns applications in all statuses, in no particular order. You can paginate the
     * results as needed.
     * </p>
     *
     * @param listApplicationsRequest
     * @return A Java Future containing the result of the ListApplications operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ListApplications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListApplicationsResponse> listApplications(ListApplicationsRequest listApplicationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListApplicationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListApplicationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                            .withOperationName("ListApplications").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listApplicationsRequest));
            CompletableFuture<ListApplicationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all Amazon GameLift Streams stream groups that are associated with the Amazon Web Services
     * account in use. This operation returns stream groups in all statuses, in no particular order. You can paginate
     * the results as needed.
     * </p>
     *
     * @param listStreamGroupsRequest
     * @return A Java Future containing the result of the ListStreamGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ListStreamGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ListStreamGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamGroupsResponse> listStreamGroups(ListStreamGroupsRequest listStreamGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStreamGroupsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListStreamGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamGroupsRequest, ListStreamGroupsResponse>()
                            .withOperationName("ListStreamGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStreamGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStreamGroupsRequest));
            CompletableFuture<ListStreamGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of Amazon GameLift Streams stream sessions that a stream group is hosting.
     * </p>
     * <p>
     * To retrieve stream sessions, specify the stream group, and optionally filter by stream session status. You can
     * paginate the results as needed.
     * </p>
     * <p>
     * This operation returns the requested stream sessions in no particular order.
     * </p>
     *
     * @param listStreamSessionsRequest
     * @return A Java Future containing the result of the ListStreamSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ListStreamSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ListStreamSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamSessionsResponse> listStreamSessions(ListStreamSessionsRequest listStreamSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStreamSessionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListStreamSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamSessionsRequest, ListStreamSessionsResponse>()
                            .withOperationName("ListStreamSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStreamSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStreamSessionsRequest));
            CompletableFuture<ListStreamSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of Amazon GameLift Streams stream sessions that this user account has access to.
     * </p>
     * <p>
     * In the returned list of stream sessions, the <code>ExportFilesMetadata</code> property only shows the
     * <code>Status</code> value. To get the <code>OutpurUri</code> and <code>StatusReason</code> values, use <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html"
     * >GetStreamSession</a>.
     * </p>
     * <p>
     * We don't recommend using this operation to regularly check stream session statuses because it's costly. Instead,
     * to check status updates for a specific stream session, use <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html"
     * >GetStreamSession</a>.
     * </p>
     *
     * @param listStreamSessionsByAccountRequest
     * @return A Java Future containing the result of the ListStreamSessionsByAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ListStreamSessionsByAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ListStreamSessionsByAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStreamSessionsByAccountResponse> listStreamSessionsByAccount(
            ListStreamSessionsByAccountRequest listStreamSessionsByAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamSessionsByAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamSessionsByAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreamSessionsByAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStreamSessionsByAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStreamSessionsByAccountResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListStreamSessionsByAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStreamSessionsByAccountRequest, ListStreamSessionsByAccountResponse>()
                            .withOperationName("ListStreamSessionsByAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStreamSessionsByAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStreamSessionsByAccountRequest));
            CompletableFuture<ListStreamSessionsByAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all tags assigned to a Amazon GameLift Streams resource. To list tags for a resource, specify the ARN
     * value for the resource.
     * </p>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
     * Resources</a> in the <i>Amazon Web Services General Reference</i>
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/answers/account-management/aws-tagging-strategies/"> Amazon Web Services Tagging
     * Strategies</a>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a set of remote locations from this stream group. Amazon GameLift Streams works to release allocated
     * compute resources in these location. Thus, stream sessions can no longer start from these locations by using this
     * stream group. Amazon GameLift Streams also deletes the content files of all associated applications that were in
     * Amazon GameLift Streams's internal S3 bucket at this location.
     * </p>
     * <p>
     * You cannot remove the region where you initially created this stream group, known as the primary location.
     * However, you can set the stream capacity to zero.
     * </p>
     *
     * @param removeStreamGroupLocationsRequest
     * @return A Java Future containing the result of the RemoveStreamGroupLocations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.RemoveStreamGroupLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/RemoveStreamGroupLocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveStreamGroupLocationsResponse> removeStreamGroupLocations(
            RemoveStreamGroupLocationsRequest removeStreamGroupLocationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(removeStreamGroupLocationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeStreamGroupLocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveStreamGroupLocations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveStreamGroupLocationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveStreamGroupLocationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<RemoveStreamGroupLocationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveStreamGroupLocationsRequest, RemoveStreamGroupLocationsResponse>()
                            .withOperationName("RemoveStreamGroupLocations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RemoveStreamGroupLocationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(removeStreamGroupLocationsRequest));
            CompletableFuture<RemoveStreamGroupLocationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This action initiates a new stream session and outputs connection information that clients can use to access the
     * stream. A stream session refers to an instance of a stream that Amazon GameLift Streams transmits from the server
     * to the end-user. A stream session runs on a compute resource that a stream group has allocated. The start stream
     * session process works as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Prerequisites:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must have a stream group in <code>ACTIVE</code> state
     * </p>
     * </li>
     * <li>
     * <p>
     * You must have idle or on-demand capacity in a stream group in the location you want to stream from
     * </p>
     * </li>
     * <li>
     * <p>
     * You must have at least one application associated to the stream group (use <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_AssociateApplications.html"
     * >AssociateApplications</a> if needed)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Start stream request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Your backend server calls <b>StartStreamSession</b> to initiate connection
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon GameLift Streams creates the stream session resource, assigns an Amazon Resource Name (ARN) value, and
     * begins searching for available stream capacity to run the stream
     * </p>
     * </li>
     * <li>
     * <p>
     * Session transitions to <code>ACTIVATING</code> status
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Placement completion:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If Amazon GameLift Streams is successful in finding capacity for the stream, the stream session status changes to
     * <code>ACTIVE</code> status and <b>StartStreamSession</b> returns stream connection information
     * </p>
     * </li>
     * <li>
     * <p>
     * If Amazon GameLift Streams was not successful in finding capacity within the placement timeout period (defined
     * according to the capacity type and platform type), the stream session status changes to <code>ERROR</code> status
     * and <b>StartStreamSession</b> returns a <code>StatusReason</code> of <code>placementTimeout</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Connection completion:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Provide the new connection information to the requesting client
     * </p>
     * </li>
     * <li>
     * <p>
     * Client must establish connection within <code>ConnectionTimeoutSeconds</code> (specified in
     * <b>StartStreamSession</b> parameters)
     * </p>
     * </li>
     * <li>
     * <p>
     * Session terminates automatically if client fails to connect in time
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ol>
     * <p>
     * For more information about the stream session lifecycle, see <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/stream-sessions.html">Stream sessions</a>
     * in the <i>Amazon GameLift Streams Developer Guide</i>.
     * </p>
     * <p>
     * Timeouts to be aware of that affect a stream session:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Placement timeout</b>: The amount of time that Amazon GameLift Streams has to find capacity for a stream
     * request. Placement timeout varies based on the capacity type used to fulfill your stream request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Always-on capacity</b>: 75 seconds
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>On-demand capacity</b>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Linux/Proton runtimes: 90 seconds
     * </p>
     * </li>
     * <li>
     * <p>
     * Windows runtime: 10 minutes
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <b>Connection timeout</b>: The amount of time that Amazon GameLift Streams waits for a client to connect to a
     * stream session in <code>ACTIVE</code> status, or reconnect to a stream session in
     * <code>PENDING_CLIENT_RECONNECTION</code> status, the latter of which occurs when a client disconnects or loses
     * connection from a stream session. If no client connects before the timeout, Amazon GameLift Streams terminates
     * the stream session. This value is specified by <code>ConnectionTimeoutSeconds</code> in the
     * <code>StartStreamSession</code> parameters.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Idle timeout</b>: A stream session will be terminated if no user input has been received for 60 minutes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Maximum session length</b>: A stream session will be terminated after this amount of time has elapsed since it
     * started, regardless of any existing client connections. This value is specified by
     * <code>SessionLengthSeconds</code> in the <code>StartStreamSession</code> parameters.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To start a new stream session, specify a stream group ID and application ID, along with the transport protocol
     * and signal request to use with the stream session.
     * </p>
     * <p>
     * For stream groups that have multiple locations, provide a set of locations ordered by priority using a
     * <code>Locations</code> parameter. Amazon GameLift Streams will start a single stream session in the next
     * available location. An application must be finished replicating to a remote location before the remote location
     * can host a stream.
     * </p>
     * <p>
     * To reconnect to a stream session after a client disconnects or loses connection, use <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_CreateStreamSessionConnection.html"
     * >CreateStreamSessionConnection</a>.
     * </p>
     *
     * @param startStreamSessionRequest
     * @return A Java Future containing the result of the StartStreamSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.StartStreamSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/StartStreamSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartStreamSessionResponse> startStreamSession(StartStreamSessionRequest startStreamSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startStreamSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStreamSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStreamSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartStreamSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartStreamSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<StartStreamSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartStreamSessionRequest, StartStreamSessionResponse>()
                            .withOperationName("StartStreamSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartStreamSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startStreamSessionRequest));
            CompletableFuture<StartStreamSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags to a Amazon GameLift Streams resource. Use tags to organize Amazon Web Services
     * resources for a range of purposes. You can assign tags to the following Amazon GameLift Streams resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Application
     * </p>
     * </li>
     * <li>
     * <p>
     * StreamGroup
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Learn more</b>
     * </p>
     * <p>
     * <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
     * Resources</a> in the <i>Amazon Web Services General Reference</i>
     * </p>
     * <p>
     * <a href="http://aws.amazon.com/answers/account-management/aws-tagging-strategies/"> Amazon Web Services Tagging
     * Strategies</a>
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently terminates an active stream session. When called, the stream session status changes to
     * <code>TERMINATING</code>. You can terminate a stream session in any status except <code>ACTIVATING</code>. If the
     * stream session is in <code>ACTIVATING</code> status, an exception is thrown.
     * </p>
     *
     * @param terminateStreamSessionRequest
     * @return A Java Future containing the result of the TerminateStreamSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.TerminateStreamSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/TerminateStreamSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateStreamSessionResponse> terminateStreamSession(
            TerminateStreamSessionRequest terminateStreamSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(terminateStreamSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateStreamSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateStreamSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateStreamSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TerminateStreamSessionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TerminateStreamSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateStreamSessionRequest, TerminateStreamSessionResponse>()
                            .withOperationName("TerminateStreamSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TerminateStreamSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(terminateStreamSessionRequest));
            CompletableFuture<TerminateStreamSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from a Amazon GameLift Streams resource. To remove tags, specify the Amazon GameLift
     * Streams resource and a list of one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the mutable configuration settings for a Amazon GameLift Streams application resource. You can change the
     * <code>Description</code>, <code>ApplicationLogOutputUri</code>, and <code>ApplicationLogPaths</code>.
     * </p>
     * <p>
     * To update application settings, specify the application ID and provide the new values. If the operation is
     * successful, it returns the complete updated set of settings for the application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return A Java Future containing the result of the UpdateApplication operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/UpdateApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationResponse> updateApplication(UpdateApplicationRequest updateApplicationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateApplicationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateApplicationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                            .withOperationName("UpdateApplication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApplicationRequest));
            CompletableFuture<UpdateApplicationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration settings for an Amazon GameLift Streams stream group resource. You can change the
     * description, the set of locations, and the requested capacity of a stream group per location. If you want to
     * change the stream class, create a new stream group.
     * </p>
     * <p>
     * Stream capacity represents the number of concurrent streams that can be active at a time. You set stream capacity
     * per location, per stream group. There are two types of capacity, always-on and on-demand:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Always-on</b>: The streaming capacity that is allocated and ready to handle stream requests without delay. You
     * pay for this capacity whether it's in use or not. Best for quickest time from streaming request to streaming
     * session. Default is 1 when creating a stream group or adding a location.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>On-demand</b>: The streaming capacity that Amazon GameLift Streams can allocate in response to stream
     * requests, and then de-allocate when the session has terminated. This offers a cost control measure at the expense
     * of a greater startup time (typically under 5 minutes). Default is 0 when creating a stream group or adding a
     * location.
     * </p>
     * </li>
     * </ul>
     * <p>
     * To update a stream group, specify the stream group's Amazon Resource Name (ARN) and provide the new values. If
     * the request is successful, Amazon GameLift Streams returns the complete updated metadata for the stream group.
     * </p>
     *
     * @param updateStreamGroupRequest
     * @return A Java Future containing the result of the UpdateStreamGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You don't have the required permissions to access this Amazon GameLift Streams
     *         resource. Correct the permissions before you try again.</li>
     *         <li>ResourceNotFoundException The resource specified in the request was not found. Correct the request
     *         before you try again.</li>
     *         <li>ThrottlingException The request was denied due to request throttling. Retry the request after the
     *         suggested wait time.</li>
     *         <li>InternalServerException The service encountered an internal error and is unable to complete the
     *         request.</li>
     *         <li>ConflictException The requested operation would cause a conflict with the current state of a service
     *         resource associated with the request. Resolve the conflict before retrying this request.</li>
     *         <li>ValidationException One or more parameter values in the request fail to satisfy the specified
     *         constraints. Correct the invalid parameter values before retrying the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause the resource to exceed an allowed service
     *         quota. Resolve the issue before you try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameLiftStreamsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample GameLiftStreamsAsyncClient.UpdateStreamGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gameliftstreams-2018-05-10/UpdateStreamGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStreamGroupResponse> updateStreamGroup(UpdateStreamGroupRequest updateStreamGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStreamGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStreamGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameLiftStreams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStreamGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStreamGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateStreamGroupResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "ThrottlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateStreamGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStreamGroupRequest, UpdateStreamGroupResponse>()
                            .withOperationName("UpdateStreamGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateStreamGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateStreamGroupRequest));
            CompletableFuture<UpdateStreamGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public GameLiftStreamsAsyncWaiter waiter() {
        return GameLiftStreamsAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final GameLiftStreamsServiceClientConfiguration serviceClientConfiguration() {
        return new GameLiftStreamsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GameLiftStreamsException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        GameLiftStreamsServiceClientConfigurationBuilder serviceConfigBuilder = new GameLiftStreamsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
