/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gameliftstreams.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateApplicationRequest extends GameLiftStreamsRequest implements
        ToCopyableBuilder<CreateApplicationRequest.Builder, CreateApplicationRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateApplicationRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<RuntimeEnvironment> RUNTIME_ENVIRONMENT_FIELD = SdkField
            .<RuntimeEnvironment> builder(MarshallingType.SDK_POJO).memberName("RuntimeEnvironment")
            .getter(getter(CreateApplicationRequest::runtimeEnvironment)).setter(setter(Builder::runtimeEnvironment))
            .constructor(RuntimeEnvironment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuntimeEnvironment").build())
            .build();

    private static final SdkField<String> EXECUTABLE_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutablePath").getter(getter(CreateApplicationRequest::executablePath))
            .setter(setter(Builder::executablePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutablePath").build()).build();

    private static final SdkField<String> APPLICATION_SOURCE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationSourceUri").getter(getter(CreateApplicationRequest::applicationSourceUri))
            .setter(setter(Builder::applicationSourceUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationSourceUri").build())
            .build();

    private static final SdkField<List<String>> APPLICATION_LOG_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ApplicationLogPaths")
            .getter(getter(CreateApplicationRequest::applicationLogPaths))
            .setter(setter(Builder::applicationLogPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationLogPaths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> APPLICATION_LOG_OUTPUT_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationLogOutputUri").getter(getter(CreateApplicationRequest::applicationLogOutputUri))
            .setter(setter(Builder::applicationLogOutputUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationLogOutputUri").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateApplicationRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateApplicationRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            RUNTIME_ENVIRONMENT_FIELD, EXECUTABLE_PATH_FIELD, APPLICATION_SOURCE_URI_FIELD, APPLICATION_LOG_PATHS_FIELD,
            APPLICATION_LOG_OUTPUT_URI_FIELD, TAGS_FIELD, CLIENT_TOKEN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String description;

    private final RuntimeEnvironment runtimeEnvironment;

    private final String executablePath;

    private final String applicationSourceUri;

    private final List<String> applicationLogPaths;

    private final String applicationLogOutputUri;

    private final Map<String, String> tags;

    private final String clientToken;

    private CreateApplicationRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.runtimeEnvironment = builder.runtimeEnvironment;
        this.executablePath = builder.executablePath;
        this.applicationSourceUri = builder.applicationSourceUri;
        this.applicationLogPaths = builder.applicationLogPaths;
        this.applicationLogOutputUri = builder.applicationLogOutputUri;
        this.tags = builder.tags;
        this.clientToken = builder.clientToken;
    }

    /**
     * <p>
     * A human-readable label for the application. You can update this value later.
     * </p>
     * 
     * @return A human-readable label for the application. You can update this value later.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Configuration settings that identify the operating system for an application resource. This can also include a
     * compatibility layer and other drivers.
     * </p>
     * <p>
     * A runtime environment can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Linux applications
     * </p>
     * <ul>
     * <li>
     * <p>
     * Ubuntu 22.04 LTS (<code>Type=UBUNTU, Version=22_04_LTS</code>)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * For Windows applications
     * </p>
     * <ul>
     * <li>
     * <p>
     * Microsoft Windows Server 2022 Base (<code>Type=WINDOWS, Version=2022</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * Proton 9.0-2 (<code>Type=PROTON, Version=20250516</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * Proton 8.0-5 (<code>Type=PROTON, Version=20241007</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * Proton 8.0-2c (<code>Type=PROTON, Version=20230704</code>)
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * 
     * @return Configuration settings that identify the operating system for an application resource. This can also
     *         include a compatibility layer and other drivers.</p>
     *         <p>
     *         A runtime environment can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For Linux applications
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Ubuntu 22.04 LTS (<code>Type=UBUNTU, Version=22_04_LTS</code>)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         For Windows applications
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Microsoft Windows Server 2022 Base (<code>Type=WINDOWS, Version=2022</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Proton 9.0-2 (<code>Type=PROTON, Version=20250516</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Proton 8.0-5 (<code>Type=PROTON, Version=20241007</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Proton 8.0-2c (<code>Type=PROTON, Version=20230704</code>)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final RuntimeEnvironment runtimeEnvironment() {
        return runtimeEnvironment;
    }

    /**
     * <p>
     * The path and file name of the executable file that launches the content for streaming. Enter a path value that is
     * relative to the location set in <code>ApplicationSourceUri</code>.
     * </p>
     * 
     * @return The path and file name of the executable file that launches the content for streaming. Enter a path value
     *         that is relative to the location set in <code>ApplicationSourceUri</code>.
     */
    public final String executablePath() {
        return executablePath;
    }

    /**
     * <p>
     * The location of the content that you want to stream. Enter an Amazon S3 URI to a bucket that contains your game
     * or other application. The location can have a multi-level prefix structure, but it must include all the files
     * needed to run the content. Amazon GameLift Streams copies everything under the specified location.
     * </p>
     * <p>
     * This value is immutable. To designate a different content location, create a new application.
     * </p>
     * <note>
     * <p>
     * The Amazon S3 bucket and the Amazon GameLift Streams application must be in the same Amazon Web Services Region.
     * </p>
     * </note>
     * 
     * @return The location of the content that you want to stream. Enter an Amazon S3 URI to a bucket that contains
     *         your game or other application. The location can have a multi-level prefix structure, but it must include
     *         all the files needed to run the content. Amazon GameLift Streams copies everything under the specified
     *         location.</p>
     *         <p>
     *         This value is immutable. To designate a different content location, create a new application.
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon S3 bucket and the Amazon GameLift Streams application must be in the same Amazon Web Services
     *         Region.
     *         </p>
     */
    public final String applicationSourceUri() {
        return applicationSourceUri;
    }

    /**
     * For responses, this returns true if the service returned a value for the ApplicationLogPaths property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasApplicationLogPaths() {
        return applicationLogPaths != null && !(applicationLogPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Locations of log files that your content generates during a stream session. Enter path values that are relative
     * to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon GameLift Streams
     * uploads designated log files to the Amazon S3 bucket that you specify in <code>ApplicationLogOutputUri</code> at
     * the end of a stream session. To retrieve stored log files, call <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html"
     * >GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApplicationLogPaths} method.
     * </p>
     * 
     * @return Locations of log files that your content generates during a stream session. Enter path values that are
     *         relative to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon
     *         GameLift Streams uploads designated log files to the Amazon S3 bucket that you specify in
     *         <code>ApplicationLogOutputUri</code> at the end of a stream session. To retrieve stored log files, call
     *         <a href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html">
     *         GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
     */
    public final List<String> applicationLogPaths() {
        return applicationLogPaths;
    }

    /**
     * <p>
     * An Amazon S3 URI to a bucket where you would like Amazon GameLift Streams to save application logs. Required if
     * you specify one or more <code>ApplicationLogPaths</code>.
     * </p>
     * <note>
     * <p>
     * The log bucket must have permissions that give Amazon GameLift Streams access to write the log files. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/applications.html#application-bucket-permission-template"
     * >Application log bucket permission policy</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return An Amazon S3 URI to a bucket where you would like Amazon GameLift Streams to save application logs.
     *         Required if you specify one or more <code>ApplicationLogPaths</code>.</p> <note>
     *         <p>
     *         The log bucket must have permissions that give Amazon GameLift Streams access to write the log files. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/applications.html#application-bucket-permission-template"
     *         >Application log bucket permission policy</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
     *         </p>
     */
    public final String applicationLogOutputUri() {
        return applicationLogOutputUri;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of labels to assign to the new application resource. Tags are developer-defined key-value pairs. Tagging
     * Amazon Web Services resources is useful for resource management, access management and cost allocation. See <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web Services Resources</a>
     * in the <i>Amazon Web Services General Reference</i>. You can use <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TagResource.html">TagResource</a> to
     * add tags, <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UntagResource.html">UntagResource</a>
     * to remove tags, and <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListTagsForResource.html"
     * >ListTagsForResource</a> to view tags on existing resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of labels to assign to the new application resource. Tags are developer-defined key-value pairs.
     *         Tagging Amazon Web Services resources is useful for resource management, access management and cost
     *         allocation. See <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon
     *         Web Services Resources</a> in the <i>Amazon Web Services General Reference</i>. You can use <a
     *         href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TagResource.html"
     *         >TagResource</a> to add tags, <a
     *         href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UntagResource.html"
     *         >UntagResource</a> to remove tags, and <a
     *         href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListTagsForResource.html"
     *         >ListTagsForResource</a> to view tags on existing resources.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * A unique identifier that represents a client request. The request is idempotent, which ensures that an API
     * request completes only once. When users send a request, Amazon GameLift Streams automatically populates this
     * field.
     * </p>
     * 
     * @return A unique identifier that represents a client request. The request is idempotent, which ensures that an
     *         API request completes only once. When users send a request, Amazon GameLift Streams automatically
     *         populates this field.
     */
    public final String clientToken() {
        return clientToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(runtimeEnvironment());
        hashCode = 31 * hashCode + Objects.hashCode(executablePath());
        hashCode = 31 * hashCode + Objects.hashCode(applicationSourceUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasApplicationLogPaths() ? applicationLogPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(applicationLogOutputUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateApplicationRequest)) {
            return false;
        }
        CreateApplicationRequest other = (CreateApplicationRequest) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(runtimeEnvironment(), other.runtimeEnvironment())
                && Objects.equals(executablePath(), other.executablePath())
                && Objects.equals(applicationSourceUri(), other.applicationSourceUri())
                && hasApplicationLogPaths() == other.hasApplicationLogPaths()
                && Objects.equals(applicationLogPaths(), other.applicationLogPaths())
                && Objects.equals(applicationLogOutputUri(), other.applicationLogOutputUri()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(clientToken(), other.clientToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateApplicationRequest").add("Description", description())
                .add("RuntimeEnvironment", runtimeEnvironment()).add("ExecutablePath", executablePath())
                .add("ApplicationSourceUri", applicationSourceUri())
                .add("ApplicationLogPaths", hasApplicationLogPaths() ? applicationLogPaths() : null)
                .add("ApplicationLogOutputUri", applicationLogOutputUri()).add("Tags", hasTags() ? tags() : null)
                .add("ClientToken", clientToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RuntimeEnvironment":
            return Optional.ofNullable(clazz.cast(runtimeEnvironment()));
        case "ExecutablePath":
            return Optional.ofNullable(clazz.cast(executablePath()));
        case "ApplicationSourceUri":
            return Optional.ofNullable(clazz.cast(applicationSourceUri()));
        case "ApplicationLogPaths":
            return Optional.ofNullable(clazz.cast(applicationLogPaths()));
        case "ApplicationLogOutputUri":
            return Optional.ofNullable(clazz.cast(applicationLogOutputUri()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("RuntimeEnvironment", RUNTIME_ENVIRONMENT_FIELD);
        map.put("ExecutablePath", EXECUTABLE_PATH_FIELD);
        map.put("ApplicationSourceUri", APPLICATION_SOURCE_URI_FIELD);
        map.put("ApplicationLogPaths", APPLICATION_LOG_PATHS_FIELD);
        map.put("ApplicationLogOutputUri", APPLICATION_LOG_OUTPUT_URI_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("ClientToken", CLIENT_TOKEN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateApplicationRequest, T> g) {
        return obj -> g.apply((CreateApplicationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GameLiftStreamsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateApplicationRequest> {
        /**
         * <p>
         * A human-readable label for the application. You can update this value later.
         * </p>
         * 
         * @param description
         *        A human-readable label for the application. You can update this value later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Configuration settings that identify the operating system for an application resource. This can also include
         * a compatibility layer and other drivers.
         * </p>
         * <p>
         * A runtime environment can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Linux applications
         * </p>
         * <ul>
         * <li>
         * <p>
         * Ubuntu 22.04 LTS (<code>Type=UBUNTU, Version=22_04_LTS</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For Windows applications
         * </p>
         * <ul>
         * <li>
         * <p>
         * Microsoft Windows Server 2022 Base (<code>Type=WINDOWS, Version=2022</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 9.0-2 (<code>Type=PROTON, Version=20250516</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 8.0-5 (<code>Type=PROTON, Version=20241007</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 8.0-2c (<code>Type=PROTON, Version=20230704</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param runtimeEnvironment
         *        Configuration settings that identify the operating system for an application resource. This can also
         *        include a compatibility layer and other drivers.</p>
         *        <p>
         *        A runtime environment can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For Linux applications
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Ubuntu 22.04 LTS (<code>Type=UBUNTU, Version=22_04_LTS</code>)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        For Windows applications
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Microsoft Windows Server 2022 Base (<code>Type=WINDOWS, Version=2022</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Proton 9.0-2 (<code>Type=PROTON, Version=20250516</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Proton 8.0-5 (<code>Type=PROTON, Version=20241007</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Proton 8.0-2c (<code>Type=PROTON, Version=20230704</code>)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder runtimeEnvironment(RuntimeEnvironment runtimeEnvironment);

        /**
         * <p>
         * Configuration settings that identify the operating system for an application resource. This can also include
         * a compatibility layer and other drivers.
         * </p>
         * <p>
         * A runtime environment can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For Linux applications
         * </p>
         * <ul>
         * <li>
         * <p>
         * Ubuntu 22.04 LTS (<code>Type=UBUNTU, Version=22_04_LTS</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * For Windows applications
         * </p>
         * <ul>
         * <li>
         * <p>
         * Microsoft Windows Server 2022 Base (<code>Type=WINDOWS, Version=2022</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 9.0-2 (<code>Type=PROTON, Version=20250516</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 8.0-5 (<code>Type=PROTON, Version=20241007</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * Proton 8.0-2c (<code>Type=PROTON, Version=20230704</code>)
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link RuntimeEnvironment.Builder} avoiding the
         * need to create one manually via {@link RuntimeEnvironment#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RuntimeEnvironment.Builder#build()} is called immediately and its
         * result is passed to {@link #runtimeEnvironment(RuntimeEnvironment)}.
         * 
         * @param runtimeEnvironment
         *        a consumer that will call methods on {@link RuntimeEnvironment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #runtimeEnvironment(RuntimeEnvironment)
         */
        default Builder runtimeEnvironment(Consumer<RuntimeEnvironment.Builder> runtimeEnvironment) {
            return runtimeEnvironment(RuntimeEnvironment.builder().applyMutation(runtimeEnvironment).build());
        }

        /**
         * <p>
         * The path and file name of the executable file that launches the content for streaming. Enter a path value
         * that is relative to the location set in <code>ApplicationSourceUri</code>.
         * </p>
         * 
         * @param executablePath
         *        The path and file name of the executable file that launches the content for streaming. Enter a path
         *        value that is relative to the location set in <code>ApplicationSourceUri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executablePath(String executablePath);

        /**
         * <p>
         * The location of the content that you want to stream. Enter an Amazon S3 URI to a bucket that contains your
         * game or other application. The location can have a multi-level prefix structure, but it must include all the
         * files needed to run the content. Amazon GameLift Streams copies everything under the specified location.
         * </p>
         * <p>
         * This value is immutable. To designate a different content location, create a new application.
         * </p>
         * <note>
         * <p>
         * The Amazon S3 bucket and the Amazon GameLift Streams application must be in the same Amazon Web Services
         * Region.
         * </p>
         * </note>
         * 
         * @param applicationSourceUri
         *        The location of the content that you want to stream. Enter an Amazon S3 URI to a bucket that contains
         *        your game or other application. The location can have a multi-level prefix structure, but it must
         *        include all the files needed to run the content. Amazon GameLift Streams copies everything under the
         *        specified location.</p>
         *        <p>
         *        This value is immutable. To designate a different content location, create a new application.
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon S3 bucket and the Amazon GameLift Streams application must be in the same Amazon Web
         *        Services Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationSourceUri(String applicationSourceUri);

        /**
         * <p>
         * Locations of log files that your content generates during a stream session. Enter path values that are
         * relative to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon
         * GameLift Streams uploads designated log files to the Amazon S3 bucket that you specify in
         * <code>ApplicationLogOutputUri</code> at the end of a stream session. To retrieve stored log files, call <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html">
         * GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
         * </p>
         * 
         * @param applicationLogPaths
         *        Locations of log files that your content generates during a stream session. Enter path values that are
         *        relative to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon
         *        GameLift Streams uploads designated log files to the Amazon S3 bucket that you specify in
         *        <code>ApplicationLogOutputUri</code> at the end of a stream session. To retrieve stored log files,
         *        call <a
         *        href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html"
         *        >GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationLogPaths(Collection<String> applicationLogPaths);

        /**
         * <p>
         * Locations of log files that your content generates during a stream session. Enter path values that are
         * relative to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon
         * GameLift Streams uploads designated log files to the Amazon S3 bucket that you specify in
         * <code>ApplicationLogOutputUri</code> at the end of a stream session. To retrieve stored log files, call <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html">
         * GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
         * </p>
         * 
         * @param applicationLogPaths
         *        Locations of log files that your content generates during a stream session. Enter path values that are
         *        relative to the <code>ApplicationSourceUri</code> location. You can specify up to 10 log paths. Amazon
         *        GameLift Streams uploads designated log files to the Amazon S3 bucket that you specify in
         *        <code>ApplicationLogOutputUri</code> at the end of a stream session. To retrieve stored log files,
         *        call <a
         *        href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html"
         *        >GetStreamSession</a> and get the <code>LogFileLocationUri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationLogPaths(String... applicationLogPaths);

        /**
         * <p>
         * An Amazon S3 URI to a bucket where you would like Amazon GameLift Streams to save application logs. Required
         * if you specify one or more <code>ApplicationLogPaths</code>.
         * </p>
         * <note>
         * <p>
         * The log bucket must have permissions that give Amazon GameLift Streams access to write the log files. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/applications.html#application-bucket-permission-template"
         * >Application log bucket permission policy</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param applicationLogOutputUri
         *        An Amazon S3 URI to a bucket where you would like Amazon GameLift Streams to save application logs.
         *        Required if you specify one or more <code>ApplicationLogPaths</code>.</p> <note>
         *        <p>
         *        The log bucket must have permissions that give Amazon GameLift Streams access to write the log files.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/applications.html#application-bucket-permission-template"
         *        >Application log bucket permission policy</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationLogOutputUri(String applicationLogOutputUri);

        /**
         * <p>
         * A list of labels to assign to the new application resource. Tags are developer-defined key-value pairs.
         * Tagging Amazon Web Services resources is useful for resource management, access management and cost
         * allocation. See <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging Amazon Web
         * Services Resources</a> in the <i>Amazon Web Services General Reference</i>. You can use <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TagResource.html">TagResource</a>
         * to add tags, <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UntagResource.html">
         * UntagResource</a> to remove tags, and <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListTagsForResource.html"
         * >ListTagsForResource</a> to view tags on existing resources.
         * </p>
         * 
         * @param tags
         *        A list of labels to assign to the new application resource. Tags are developer-defined key-value
         *        pairs. Tagging Amazon Web Services resources is useful for resource management, access management and
         *        cost allocation. See <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html"> Tagging
         *        Amazon Web Services Resources</a> in the <i>Amazon Web Services General Reference</i>. You can use <a
         *        href
         *        ="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_TagResource.html">TagResource
         *        </a> to add tags, <a
         *        href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_UntagResource.html"
         *        >UntagResource</a> to remove tags, and <a
         *        href="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_ListTagsForResource.html"
         *        >ListTagsForResource</a> to view tags on existing resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A unique identifier that represents a client request. The request is idempotent, which ensures that an API
         * request completes only once. When users send a request, Amazon GameLift Streams automatically populates this
         * field.
         * </p>
         * 
         * @param clientToken
         *        A unique identifier that represents a client request. The request is idempotent, which ensures that an
         *        API request completes only once. When users send a request, Amazon GameLift Streams automatically
         *        populates this field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GameLiftStreamsRequest.BuilderImpl implements Builder {
        private String description;

        private RuntimeEnvironment runtimeEnvironment;

        private String executablePath;

        private String applicationSourceUri;

        private List<String> applicationLogPaths = DefaultSdkAutoConstructList.getInstance();

        private String applicationLogOutputUri;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String clientToken;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateApplicationRequest model) {
            super(model);
            description(model.description);
            runtimeEnvironment(model.runtimeEnvironment);
            executablePath(model.executablePath);
            applicationSourceUri(model.applicationSourceUri);
            applicationLogPaths(model.applicationLogPaths);
            applicationLogOutputUri(model.applicationLogOutputUri);
            tags(model.tags);
            clientToken(model.clientToken);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final RuntimeEnvironment.Builder getRuntimeEnvironment() {
            return runtimeEnvironment != null ? runtimeEnvironment.toBuilder() : null;
        }

        public final void setRuntimeEnvironment(RuntimeEnvironment.BuilderImpl runtimeEnvironment) {
            this.runtimeEnvironment = runtimeEnvironment != null ? runtimeEnvironment.build() : null;
        }

        @Override
        public final Builder runtimeEnvironment(RuntimeEnvironment runtimeEnvironment) {
            this.runtimeEnvironment = runtimeEnvironment;
            return this;
        }

        public final String getExecutablePath() {
            return executablePath;
        }

        public final void setExecutablePath(String executablePath) {
            this.executablePath = executablePath;
        }

        @Override
        public final Builder executablePath(String executablePath) {
            this.executablePath = executablePath;
            return this;
        }

        public final String getApplicationSourceUri() {
            return applicationSourceUri;
        }

        public final void setApplicationSourceUri(String applicationSourceUri) {
            this.applicationSourceUri = applicationSourceUri;
        }

        @Override
        public final Builder applicationSourceUri(String applicationSourceUri) {
            this.applicationSourceUri = applicationSourceUri;
            return this;
        }

        public final Collection<String> getApplicationLogPaths() {
            if (applicationLogPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return applicationLogPaths;
        }

        public final void setApplicationLogPaths(Collection<String> applicationLogPaths) {
            this.applicationLogPaths = FilePathsCopier.copy(applicationLogPaths);
        }

        @Override
        public final Builder applicationLogPaths(Collection<String> applicationLogPaths) {
            this.applicationLogPaths = FilePathsCopier.copy(applicationLogPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applicationLogPaths(String... applicationLogPaths) {
            applicationLogPaths(Arrays.asList(applicationLogPaths));
            return this;
        }

        public final String getApplicationLogOutputUri() {
            return applicationLogOutputUri;
        }

        public final void setApplicationLogOutputUri(String applicationLogOutputUri) {
            this.applicationLogOutputUri = applicationLogOutputUri;
        }

        @Override
        public final Builder applicationLogOutputUri(String applicationLogOutputUri) {
            this.applicationLogOutputUri = applicationLogOutputUri;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateApplicationRequest build() {
            return new CreateApplicationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
