/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gameliftstreams.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.gameliftstreams.GameLiftStreamsAsyncClient;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface GameLiftStreamsAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition
     * {@code ApplicationDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getApplicationRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            GetApplicationRequest getApplicationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition
     * {@code ApplicationDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetApplicationRequest#builder()}
     *
     * @param getApplicationRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            Consumer<GetApplicationRequest.Builder> getApplicationRequest) {
        return waitUntilApplicationDeleted(GetApplicationRequest.builder().applyMutation(getApplicationRequest).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition
     * {@code ApplicationDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getApplicationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            GetApplicationRequest getApplicationRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition
     * {@code ApplicationDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getApplicationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            Consumer<GetApplicationRequest.Builder> getApplicationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilApplicationDeleted(GetApplicationRequest.builder().applyMutation(getApplicationRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition {@code ApplicationReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getApplicationRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            GetApplicationRequest getApplicationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition {@code ApplicationReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetApplicationRequest#builder()}
     *
     * @param getApplicationRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            Consumer<GetApplicationRequest.Builder> getApplicationRequest) {
        return waitUntilApplicationReady(GetApplicationRequest.builder().applyMutation(getApplicationRequest).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition {@code ApplicationReady}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getApplicationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            GetApplicationRequest getApplicationRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getApplication} API until the desired condition {@code ApplicationReady}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getApplicationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            Consumer<GetApplicationRequest.Builder> getApplicationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilApplicationReady(GetApplicationRequest.builder().applyMutation(getApplicationRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition {@code StreamGroupActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamGroupRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            GetStreamGroupRequest getStreamGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition {@code StreamGroupActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamGroupRequest#builder()}
     *
     * @param getStreamGroupRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            Consumer<GetStreamGroupRequest.Builder> getStreamGroupRequest) {
        return waitUntilStreamGroupActive(GetStreamGroupRequest.builder().applyMutation(getStreamGroupRequest).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition {@code StreamGroupActive}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param getStreamGroupRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            GetStreamGroupRequest getStreamGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition {@code StreamGroupActive}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamGroupRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            Consumer<GetStreamGroupRequest.Builder> getStreamGroupRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamGroupActive(GetStreamGroupRequest.builder().applyMutation(getStreamGroupRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition
     * {@code StreamGroupDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getStreamGroupRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            GetStreamGroupRequest getStreamGroupRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition
     * {@code StreamGroupDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamGroupRequest#builder()}
     *
     * @param getStreamGroupRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            Consumer<GetStreamGroupRequest.Builder> getStreamGroupRequest) {
        return waitUntilStreamGroupDeleted(GetStreamGroupRequest.builder().applyMutation(getStreamGroupRequest).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition
     * {@code StreamGroupDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getStreamGroupRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            GetStreamGroupRequest getStreamGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamGroup} API until the desired condition
     * {@code StreamGroupDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamGroupRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            Consumer<GetStreamGroupRequest.Builder> getStreamGroupRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamGroupDeleted(GetStreamGroupRequest.builder().applyMutation(getStreamGroupRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamSession} API until the desired condition
     * {@code StreamSessionActive} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getStreamSessionRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            GetStreamSessionRequest getStreamSessionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamSession} API until the desired condition
     * {@code StreamSessionActive} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link GetStreamSessionRequest#builder()}
     *
     * @param getStreamSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            Consumer<GetStreamSessionRequest.Builder> getStreamSessionRequest) {
        return waitUntilStreamSessionActive(GetStreamSessionRequest.builder().applyMutation(getStreamSessionRequest).build());
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamSession} API until the desired condition
     * {@code StreamSessionActive} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param getStreamSessionRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            GetStreamSessionRequest getStreamSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link GameLiftStreamsAsyncClient#getStreamSession} API until the desired condition
     * {@code StreamSessionActive} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param getStreamSessionRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            Consumer<GetStreamSessionRequest.Builder> getStreamSessionRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilStreamSessionActive(GetStreamSessionRequest.builder().applyMutation(getStreamSessionRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link GameLiftStreamsAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultGameLiftStreamsAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link GameLiftStreamsAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link GameLiftStreamsAsyncClient} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link GameLiftStreamsAsyncWaiter}
     */
    static GameLiftStreamsAsyncWaiter create() {
        return DefaultGameLiftStreamsAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link GameLiftStreamsAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(GameLiftStreamsAsyncClient client);

        /**
         * Builds an instance of {@link GameLiftStreamsAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link GameLiftStreamsAsyncWaiter}
         */
        GameLiftStreamsAsyncWaiter build();
    }
}
