/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gameliftstreams.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon GameLift Streams stream session. To retrieve additional details for the stream session, call <a
 * href
 * ="https://docs.aws.amazon.com/gameliftstreams/latest/apireference/API_GetStreamSession.html">GetStreamSession</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamSessionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<StreamSessionSummary.Builder, StreamSessionSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(StreamSessionSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> USER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("UserId")
            .getter(getter(StreamSessionSummary::userId)).setter(setter(Builder::userId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(StreamSessionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Protocol").getter(getter(StreamSessionSummary::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt").getter(getter(StreamSessionSummary::lastUpdatedAt))
            .setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(StreamSessionSummary::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<String> APPLICATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ApplicationArn").getter(getter(StreamSessionSummary::applicationArn))
            .setter(setter(Builder::applicationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationArn").build()).build();

    private static final SdkField<ExportFilesMetadata> EXPORT_FILES_METADATA_FIELD = SdkField
            .<ExportFilesMetadata> builder(MarshallingType.SDK_POJO).memberName("ExportFilesMetadata")
            .getter(getter(StreamSessionSummary::exportFilesMetadata)).setter(setter(Builder::exportFilesMetadata))
            .constructor(ExportFilesMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportFilesMetadata").build())
            .build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(StreamSessionSummary::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, USER_ID_FIELD,
            STATUS_FIELD, PROTOCOL_FIELD, LAST_UPDATED_AT_FIELD, CREATED_AT_FIELD, APPLICATION_ARN_FIELD,
            EXPORT_FILES_METADATA_FIELD, LOCATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String userId;

    private final String status;

    private final String protocol;

    private final Instant lastUpdatedAt;

    private final Instant createdAt;

    private final String applicationArn;

    private final ExportFilesMetadata exportFilesMetadata;

    private final String location;

    private StreamSessionSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.userId = builder.userId;
        this.status = builder.status;
        this.protocol = builder.protocol;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.createdAt = builder.createdAt;
        this.applicationArn = builder.applicationArn;
        this.exportFilesMetadata = builder.exportFilesMetadata;
        this.location = builder.location;
    }

    /**
     * <p>
     * An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name (ARN)</a>
     * that uniquely identifies the stream session resource. Example ARN:
     * <code>arn:aws:gameliftstreams:us-west-2:111122223333:streamsession/sg-1AB2C3De4/ABC123def4567</code>.
     * </p>
     * 
     * @return An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
     *         (ARN)</a> that uniquely identifies the stream session resource. Example ARN:
     *         <code>arn:aws:gameliftstreams:us-west-2:111122223333:streamsession/sg-1AB2C3De4/ABC123def4567</code>.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * An opaque, unique identifier for an end-user, defined by the developer.
     * </p>
     * 
     * @return An opaque, unique identifier for an end-user, defined by the developer.
     */
    public final String userId() {
        return userId;
    }

    /**
     * <p>
     * The current status of the stream session resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
     * <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the session
     * reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this timeframe, the
     * session automatically terminates.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate if
     * there is no user input for 60 minutes, or if the maximum length of a session specified by
     * <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
     * <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is waiting
     * for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
     * <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code> state to
     * re-establish a connection. If no client connects within this timeframe, the session automatically terminates.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
     * <code>PENDING_CLIENT_RECONNECTION</code> state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATING</code>: The stream session is ending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATED</code>: The stream session has ended.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StreamSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the stream session resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
     *         <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the
     *         session reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this
     *         timeframe, the session automatically terminates.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate
     *         if there is no user input for 60 minutes, or if the maximum length of a session specified by
     *         <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
     *         <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
     *         waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
     *         <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code>
     *         state to re-establish a connection. If no client connects within this timeframe, the session
     *         automatically terminates.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
     *         <code>PENDING_CLIENT_RECONNECTION</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATING</code>: The stream session is ending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATED</code>: The stream session has ended.
     *         </p>
     *         </li>
     * @see StreamSessionStatus
     */
    public final StreamSessionStatus status() {
        return StreamSessionStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the stream session resource.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
     * <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the session
     * reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this timeframe, the
     * session automatically terminates.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate if
     * there is no user input for 60 minutes, or if the maximum length of a session specified by
     * <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
     * <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is waiting
     * for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
     * <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code> state to
     * re-establish a connection. If no client connects within this timeframe, the session automatically terminates.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
     * <code>PENDING_CLIENT_RECONNECTION</code> state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATING</code>: The stream session is ending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TERMINATED</code>: The stream session has ended.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StreamSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the stream session resource.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
     *         <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the
     *         session reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this
     *         timeframe, the session automatically terminates.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate
     *         if there is no user input for 60 minutes, or if the maximum length of a session specified by
     *         <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
     *         <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
     *         waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
     *         <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code>
     *         state to re-establish a connection. If no client connects within this timeframe, the session
     *         automatically terminates.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
     *         <code>PENDING_CLIENT_RECONNECTION</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATING</code>: The stream session is ending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TERMINATED</code>: The stream session has ended.
     *         </p>
     *         </li>
     * @see StreamSessionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The data transfer protocol in use with the stream session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link Protocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The data transfer protocol in use with the stream session.
     * @see Protocol
     */
    public final Protocol protocol() {
        return Protocol.fromValue(protocol);
    }

    /**
     * <p>
     * The data transfer protocol in use with the stream session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link Protocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The data transfer protocol in use with the stream session.
     * @see Protocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * A timestamp that indicates when this resource was last updated. Timestamps are expressed using in ISO8601 format,
     * such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
     * </p>
     * 
     * @return A timestamp that indicates when this resource was last updated. Timestamps are expressed using in ISO8601
     *         format, such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * A timestamp that indicates when this resource was created. Timestamps are expressed using in ISO8601 format, such
     * as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
     * </p>
     * 
     * @return A timestamp that indicates when this resource was created. Timestamps are expressed using in ISO8601
     *         format, such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name (ARN)</a>
     * that uniquely identifies the application resource. Example ARN:
     * <code>arn:aws:gameliftstreams:us-west-2:111122223333:application/a-9ZY8X7Wv6</code>.
     * </p>
     * 
     * @return An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
     *         (ARN)</a> that uniquely identifies the application resource. Example ARN:
     *         <code>arn:aws:gameliftstreams:us-west-2:111122223333:application/a-9ZY8X7Wv6</code>.
     */
    public final String applicationArn() {
        return applicationArn;
    }

    /**
     * <p>
     * Provides details about the stream session's exported files.
     * </p>
     * 
     * @return Provides details about the stream session's exported files.
     */
    public final ExportFilesMetadata exportFilesMetadata() {
        return exportFilesMetadata;
    }

    /**
     * <p>
     * The location where Amazon GameLift Streams hosts and streams your application. For example,
     * <code>us-east-1</code>. For a complete list of locations that Amazon GameLift Streams supports, refer to <a
     * href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/regions-quotas.html">Regions, quotas, and
     * limitations</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
     * </p>
     * 
     * @return The location where Amazon GameLift Streams hosts and streams your application. For example,
     *         <code>us-east-1</code>. For a complete list of locations that Amazon GameLift Streams supports, refer to
     *         <a href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/regions-quotas.html">Regions,
     *         quotas, and limitations</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
     */
    public final String location() {
        return location;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(userId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(applicationArn());
        hashCode = 31 * hashCode + Objects.hashCode(exportFilesMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamSessionSummary)) {
            return false;
        }
        StreamSessionSummary other = (StreamSessionSummary) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(userId(), other.userId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(protocolAsString(), other.protocolAsString())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(applicationArn(), other.applicationArn())
                && Objects.equals(exportFilesMetadata(), other.exportFilesMetadata())
                && Objects.equals(location(), other.location());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StreamSessionSummary").add("Arn", arn()).add("UserId", userId()).add("Status", statusAsString())
                .add("Protocol", protocolAsString()).add("LastUpdatedAt", lastUpdatedAt()).add("CreatedAt", createdAt())
                .add("ApplicationArn", applicationArn()).add("ExportFilesMetadata", exportFilesMetadata())
                .add("Location", location()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "UserId":
            return Optional.ofNullable(clazz.cast(userId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "ApplicationArn":
            return Optional.ofNullable(clazz.cast(applicationArn()));
        case "ExportFilesMetadata":
            return Optional.ofNullable(clazz.cast(exportFilesMetadata()));
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Arn", ARN_FIELD);
        map.put("UserId", USER_ID_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Protocol", PROTOCOL_FIELD);
        map.put("LastUpdatedAt", LAST_UPDATED_AT_FIELD);
        map.put("CreatedAt", CREATED_AT_FIELD);
        map.put("ApplicationArn", APPLICATION_ARN_FIELD);
        map.put("ExportFilesMetadata", EXPORT_FILES_METADATA_FIELD);
        map.put("Location", LOCATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StreamSessionSummary, T> g) {
        return obj -> g.apply((StreamSessionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StreamSessionSummary> {
        /**
         * <p>
         * An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
         * (ARN)</a> that uniquely identifies the stream session resource. Example ARN:
         * <code>arn:aws:gameliftstreams:us-west-2:111122223333:streamsession/sg-1AB2C3De4/ABC123def4567</code>.
         * </p>
         * 
         * @param arn
         *        An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
         *        (ARN)</a> that uniquely identifies the stream session resource. Example ARN:
         *        <code>arn:aws:gameliftstreams:us-west-2:111122223333:streamsession/sg-1AB2C3De4/ABC123def4567</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * An opaque, unique identifier for an end-user, defined by the developer.
         * </p>
         * 
         * @param userId
         *        An opaque, unique identifier for an end-user, defined by the developer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userId(String userId);

        /**
         * <p>
         * The current status of the stream session resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
         * <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the session
         * reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this timeframe, the
         * session automatically terminates.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate if
         * there is no user input for 60 minutes, or if the maximum length of a session specified by
         * <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
         * <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
         * waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
         * <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code> state
         * to re-establish a connection. If no client connects within this timeframe, the session automatically
         * terminates.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
         * <code>PENDING_CLIENT_RECONNECTION</code> state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATING</code>: The stream session is ending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATED</code>: The stream session has ended.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the stream session resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
         *        <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the
         *        session reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this
         *        timeframe, the session automatically terminates.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONNECTED</code>: The stream session has a connected client. A session will automatically
         *        terminate if there is no user input for 60 minutes, or if the maximum length of a session specified by
         *        <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
         *        <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
         *        waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
         *        <code>StartStreamSession</code>) from when the session reaches
         *        <code>PENDING_CLIENT_RECONNECTION</code> state to re-establish a connection. If no client connects
         *        within this timeframe, the session automatically terminates.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
         *        <code>PENDING_CLIENT_RECONNECTION</code> state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATING</code>: The stream session is ending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATED</code>: The stream session has ended.
         *        </p>
         *        </li>
         * @see StreamSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamSessionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the stream session resource.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
         * <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the session
         * reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this timeframe, the
         * session automatically terminates.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CONNECTED</code>: The stream session has a connected client. A session will automatically terminate if
         * there is no user input for 60 minutes, or if the maximum length of a session specified by
         * <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
         * <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
         * waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
         * <code>StartStreamSession</code>) from when the session reaches <code>PENDING_CLIENT_RECONNECTION</code> state
         * to re-establish a connection. If no client connects within this timeframe, the session automatically
         * terminates.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
         * <code>PENDING_CLIENT_RECONNECTION</code> state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATING</code>: The stream session is ending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TERMINATED</code>: The stream session has ended.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the stream session resource.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACTIVATING</code>: The stream session is starting and preparing to stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code>: The stream session is ready and waiting for a client connection. A client has
         *        <code>ConnectionTimeoutSeconds</code> (specified in <code>StartStreamSession</code>) from when the
         *        session reaches <code>ACTIVE</code> state to establish a connection. If no client connects within this
         *        timeframe, the session automatically terminates.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CONNECTED</code>: The stream session has a connected client. A session will automatically
         *        terminate if there is no user input for 60 minutes, or if the maximum length of a session specified by
         *        <code>SessionLengthSeconds</code> in <code>StartStreamSession</code> is exceeded.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ERROR</code>: The stream session failed to activate. See <code>StatusReason</code> (returned by
         *        <code>GetStreamSession</code> and <code>StartStreamSession</code>) for more information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING_CLIENT_RECONNECTION</code>: A client has recently disconnected and the stream session is
         *        waiting for the client to reconnect. A client has <code>ConnectionTimeoutSeconds</code> (specified in
         *        <code>StartStreamSession</code>) from when the session reaches
         *        <code>PENDING_CLIENT_RECONNECTION</code> state to re-establish a connection. If no client connects
         *        within this timeframe, the session automatically terminates.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RECONNECTING</code>: A client has initiated a reconnect to a session that was in
         *        <code>PENDING_CLIENT_RECONNECTION</code> state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATING</code>: The stream session is ending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TERMINATED</code>: The stream session has ended.
         *        </p>
         *        </li>
         * @see StreamSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamSessionStatus
         */
        Builder status(StreamSessionStatus status);

        /**
         * <p>
         * The data transfer protocol in use with the stream session.
         * </p>
         * 
         * @param protocol
         *        The data transfer protocol in use with the stream session.
         * @see Protocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Protocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The data transfer protocol in use with the stream session.
         * </p>
         * 
         * @param protocol
         *        The data transfer protocol in use with the stream session.
         * @see Protocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Protocol
         */
        Builder protocol(Protocol protocol);

        /**
         * <p>
         * A timestamp that indicates when this resource was last updated. Timestamps are expressed using in ISO8601
         * format, such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
         * </p>
         * 
         * @param lastUpdatedAt
         *        A timestamp that indicates when this resource was last updated. Timestamps are expressed using in
         *        ISO8601 format, such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * A timestamp that indicates when this resource was created. Timestamps are expressed using in ISO8601 format,
         * such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
         * </p>
         * 
         * @param createdAt
         *        A timestamp that indicates when this resource was created. Timestamps are expressed using in ISO8601
         *        format, such as: <code>2022-12-27T22:29:40+00:00</code> (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
         * (ARN)</a> that uniquely identifies the application resource. Example ARN:
         * <code>arn:aws:gameliftstreams:us-west-2:111122223333:application/a-9ZY8X7Wv6</code>.
         * </p>
         * 
         * @param applicationArn
         *        An <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon Resource Name
         *        (ARN)</a> that uniquely identifies the application resource. Example ARN:
         *        <code>arn:aws:gameliftstreams:us-west-2:111122223333:application/a-9ZY8X7Wv6</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationArn(String applicationArn);

        /**
         * <p>
         * Provides details about the stream session's exported files.
         * </p>
         * 
         * @param exportFilesMetadata
         *        Provides details about the stream session's exported files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportFilesMetadata(ExportFilesMetadata exportFilesMetadata);

        /**
         * <p>
         * Provides details about the stream session's exported files.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExportFilesMetadata.Builder} avoiding the
         * need to create one manually via {@link ExportFilesMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExportFilesMetadata.Builder#build()} is called immediately and
         * its result is passed to {@link #exportFilesMetadata(ExportFilesMetadata)}.
         * 
         * @param exportFilesMetadata
         *        a consumer that will call methods on {@link ExportFilesMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exportFilesMetadata(ExportFilesMetadata)
         */
        default Builder exportFilesMetadata(Consumer<ExportFilesMetadata.Builder> exportFilesMetadata) {
            return exportFilesMetadata(ExportFilesMetadata.builder().applyMutation(exportFilesMetadata).build());
        }

        /**
         * <p>
         * The location where Amazon GameLift Streams hosts and streams your application. For example,
         * <code>us-east-1</code>. For a complete list of locations that Amazon GameLift Streams supports, refer to <a
         * href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/regions-quotas.html">Regions, quotas,
         * and limitations</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
         * </p>
         * 
         * @param location
         *        The location where Amazon GameLift Streams hosts and streams your application. For example,
         *        <code>us-east-1</code>. For a complete list of locations that Amazon GameLift Streams supports, refer
         *        to <a
         *        href="https://docs.aws.amazon.com/gameliftstreams/latest/developerguide/regions-quotas.html">Regions,
         *        quotas, and limitations</a> in the <i>Amazon GameLift Streams Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String userId;

        private String status;

        private String protocol;

        private Instant lastUpdatedAt;

        private Instant createdAt;

        private String applicationArn;

        private ExportFilesMetadata exportFilesMetadata;

        private String location;

        private BuilderImpl() {
        }

        private BuilderImpl(StreamSessionSummary model) {
            arn(model.arn);
            userId(model.userId);
            status(model.status);
            protocol(model.protocol);
            lastUpdatedAt(model.lastUpdatedAt);
            createdAt(model.createdAt);
            applicationArn(model.applicationArn);
            exportFilesMetadata(model.exportFilesMetadata);
            location(model.location);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getUserId() {
            return userId;
        }

        public final void setUserId(String userId) {
            this.userId = userId;
        }

        @Override
        public final Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StreamSessionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(Protocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getApplicationArn() {
            return applicationArn;
        }

        public final void setApplicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
        }

        @Override
        public final Builder applicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
            return this;
        }

        public final ExportFilesMetadata.Builder getExportFilesMetadata() {
            return exportFilesMetadata != null ? exportFilesMetadata.toBuilder() : null;
        }

        public final void setExportFilesMetadata(ExportFilesMetadata.BuilderImpl exportFilesMetadata) {
            this.exportFilesMetadata = exportFilesMetadata != null ? exportFilesMetadata.build() : null;
        }

        @Override
        public final Builder exportFilesMetadata(ExportFilesMetadata exportFilesMetadata) {
            this.exportFilesMetadata = exportFilesMetadata;
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        @Override
        public StreamSessionSummary build() {
            return new StreamSessionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
