/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gameliftstreams.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.gameliftstreams.GameLiftStreamsAsyncClient;
import software.amazon.awssdk.services.gameliftstreams.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.gameliftstreams.model.GameLiftStreamsRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetApplicationResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamGroupResponse;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionRequest;
import software.amazon.awssdk.services.gameliftstreams.model.GetStreamSessionResponse;
import software.amazon.awssdk.services.gameliftstreams.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultGameLiftStreamsAsyncWaiter implements GameLiftStreamsAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final GameLiftStreamsAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetApplicationResponse> applicationDeletedWaiter;

    private final AsyncWaiter<GetApplicationResponse> applicationReadyWaiter;

    private final AsyncWaiter<GetStreamGroupResponse> streamGroupActiveWaiter;

    private final AsyncWaiter<GetStreamGroupResponse> streamGroupDeletedWaiter;

    private final AsyncWaiter<GetStreamSessionResponse> streamSessionActiveWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultGameLiftStreamsAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = GameLiftStreamsAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.applicationDeletedWaiter = AsyncWaiter.builder(GetApplicationResponse.class)
                .acceptors(applicationDeletedWaiterAcceptors())
                .overrideConfiguration(applicationDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.applicationReadyWaiter = AsyncWaiter.builder(GetApplicationResponse.class)
                .acceptors(applicationReadyWaiterAcceptors())
                .overrideConfiguration(applicationReadyWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.streamGroupActiveWaiter = AsyncWaiter.builder(GetStreamGroupResponse.class)
                .acceptors(streamGroupActiveWaiterAcceptors())
                .overrideConfiguration(streamGroupActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.streamGroupDeletedWaiter = AsyncWaiter.builder(GetStreamGroupResponse.class)
                .acceptors(streamGroupDeletedWaiterAcceptors())
                .overrideConfiguration(streamGroupDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.streamSessionActiveWaiter = AsyncWaiter.builder(GetStreamSessionResponse.class)
                .acceptors(streamSessionActiveWaiterAcceptors())
                .overrideConfiguration(streamSessionActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            GetApplicationRequest getApplicationRequest) {
        return applicationDeletedWaiter.runAsync(() -> client.getApplication(applyWaitersUserAgent(getApplicationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationDeleted(
            GetApplicationRequest getApplicationRequest, WaiterOverrideConfiguration overrideConfig) {
        return applicationDeletedWaiter.runAsync(() -> client.getApplication(applyWaitersUserAgent(getApplicationRequest)),
                applicationDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            GetApplicationRequest getApplicationRequest) {
        return applicationReadyWaiter.runAsync(() -> client.getApplication(applyWaitersUserAgent(getApplicationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetApplicationResponse>> waitUntilApplicationReady(
            GetApplicationRequest getApplicationRequest, WaiterOverrideConfiguration overrideConfig) {
        return applicationReadyWaiter.runAsync(() -> client.getApplication(applyWaitersUserAgent(getApplicationRequest)),
                applicationReadyWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            GetStreamGroupRequest getStreamGroupRequest) {
        return streamGroupActiveWaiter.runAsync(() -> client.getStreamGroup(applyWaitersUserAgent(getStreamGroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupActive(
            GetStreamGroupRequest getStreamGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamGroupActiveWaiter.runAsync(() -> client.getStreamGroup(applyWaitersUserAgent(getStreamGroupRequest)),
                streamGroupActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            GetStreamGroupRequest getStreamGroupRequest) {
        return streamGroupDeletedWaiter.runAsync(() -> client.getStreamGroup(applyWaitersUserAgent(getStreamGroupRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamGroupResponse>> waitUntilStreamGroupDeleted(
            GetStreamGroupRequest getStreamGroupRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamGroupDeletedWaiter.runAsync(() -> client.getStreamGroup(applyWaitersUserAgent(getStreamGroupRequest)),
                streamGroupDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            GetStreamSessionRequest getStreamSessionRequest) {
        return streamSessionActiveWaiter.runAsync(() -> client.getStreamSession(applyWaitersUserAgent(getStreamSessionRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetStreamSessionResponse>> waitUntilStreamSessionActive(
            GetStreamSessionRequest getStreamSessionRequest, WaiterOverrideConfiguration overrideConfig) {
        return streamSessionActiveWaiter.runAsync(() -> client.getStreamSession(applyWaitersUserAgent(getStreamSessionRequest)),
                streamSessionActiveWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetApplicationResponse>> applicationDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetApplicationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetApplicationResponse>> applicationReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super GetApplicationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "READY");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamGroupResponse>> streamGroupActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=ERROR) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "ACTIVE_WITH_ERRORS");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=ACTIVE_WITH_ERRORS) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("Status").value(), "DELETING");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=DELETING) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamGroupResponse>> streamGroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamGroupResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetStreamSessionResponse>> streamSessionActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super GetStreamSessionResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "ERROR");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=ERROR) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration applicationDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration applicationReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamGroupActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamGroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration streamSessionActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static GameLiftStreamsAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends GameLiftStreamsRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements GameLiftStreamsAsyncWaiter.Builder {
        private GameLiftStreamsAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public GameLiftStreamsAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public GameLiftStreamsAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public GameLiftStreamsAsyncWaiter.Builder client(GameLiftStreamsAsyncClient client) {
            this.client = client;
            return this;
        }

        public GameLiftStreamsAsyncWaiter build() {
            return new DefaultGameLiftStreamsAsyncWaiter(this);
        }
    }
}
