/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamesparks;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.gamesparks.model.AccessDeniedException;
import software.amazon.awssdk.services.gamesparks.model.ConflictException;
import software.amazon.awssdk.services.gamesparks.model.CreateGameRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateGameResponse;
import software.amazon.awssdk.services.gamesparks.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.CreateStageRequest;
import software.amazon.awssdk.services.gamesparks.model.CreateStageResponse;
import software.amazon.awssdk.services.gamesparks.model.DeleteGameRequest;
import software.amazon.awssdk.services.gamesparks.model.DeleteGameResponse;
import software.amazon.awssdk.services.gamesparks.model.DeleteStageRequest;
import software.amazon.awssdk.services.gamesparks.model.DeleteStageResponse;
import software.amazon.awssdk.services.gamesparks.model.DisconnectPlayerRequest;
import software.amazon.awssdk.services.gamesparks.model.DisconnectPlayerResponse;
import software.amazon.awssdk.services.gamesparks.model.ExportSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.ExportSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.GameSparksException;
import software.amazon.awssdk.services.gamesparks.model.GameSparksRequest;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionRequest;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionResponse;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionVersionRequest;
import software.amazon.awssdk.services.gamesparks.model.GetExtensionVersionResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGameRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGameResponse;
import software.amazon.awssdk.services.gamesparks.model.GetGeneratedCodeJobRequest;
import software.amazon.awssdk.services.gamesparks.model.GetGeneratedCodeJobResponse;
import software.amazon.awssdk.services.gamesparks.model.GetPlayerConnectionStatusRequest;
import software.amazon.awssdk.services.gamesparks.model.GetPlayerConnectionStatusResponse;
import software.amazon.awssdk.services.gamesparks.model.GetSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.GetSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.GetStageDeploymentRequest;
import software.amazon.awssdk.services.gamesparks.model.GetStageDeploymentResponse;
import software.amazon.awssdk.services.gamesparks.model.GetStageRequest;
import software.amazon.awssdk.services.gamesparks.model.GetStageResponse;
import software.amazon.awssdk.services.gamesparks.model.ImportGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.ImportGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.InternalServerException;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListExtensionsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListGamesRequest;
import software.amazon.awssdk.services.gamesparks.model.ListGamesResponse;
import software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListSnapshotsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListSnapshotsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsRequest;
import software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsResponse;
import software.amazon.awssdk.services.gamesparks.model.ListStagesRequest;
import software.amazon.awssdk.services.gamesparks.model.ListStagesResponse;
import software.amazon.awssdk.services.gamesparks.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.ResourceNotFoundException;
import software.amazon.awssdk.services.gamesparks.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.gamesparks.model.StartGeneratedCodeJobRequest;
import software.amazon.awssdk.services.gamesparks.model.StartGeneratedCodeJobResponse;
import software.amazon.awssdk.services.gamesparks.model.StartStageDeploymentRequest;
import software.amazon.awssdk.services.gamesparks.model.StartStageDeploymentResponse;
import software.amazon.awssdk.services.gamesparks.model.TagResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.TagResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.ThrottlingException;
import software.amazon.awssdk.services.gamesparks.model.UntagResourceRequest;
import software.amazon.awssdk.services.gamesparks.model.UntagResourceResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameConfigurationRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameConfigurationResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateGameResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateSnapshotRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateSnapshotResponse;
import software.amazon.awssdk.services.gamesparks.model.UpdateStageRequest;
import software.amazon.awssdk.services.gamesparks.model.UpdateStageResponse;
import software.amazon.awssdk.services.gamesparks.model.ValidationException;
import software.amazon.awssdk.services.gamesparks.paginators.ListExtensionVersionsPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListExtensionsPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListGamesPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListGeneratedCodeJobsPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListSnapshotsPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListStageDeploymentsPublisher;
import software.amazon.awssdk.services.gamesparks.paginators.ListStagesPublisher;
import software.amazon.awssdk.services.gamesparks.transform.CreateGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.CreateStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DeleteGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DeleteStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.DisconnectPlayerRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ExportSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetExtensionRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetExtensionVersionRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetGeneratedCodeJobRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetPlayerConnectionStatusRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetStageDeploymentRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.GetStageRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ImportGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListExtensionVersionsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListExtensionsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListGamesRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListGeneratedCodeJobsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListStageDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListStagesRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.StartGeneratedCodeJobRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.StartStageDeploymentRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateGameConfigurationRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateGameRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.gamesparks.transform.UpdateStageRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GameSparksAsyncClient}.
 *
 * @see GameSparksAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGameSparksAsyncClient implements GameSparksAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGameSparksAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGameSparksAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new game with an empty configuration. After creating your game, you can update the configuration using
     * <code>UpdateGameConfiguration</code> or <code>ImportGameConfiguration</code>.
     * </p>
     *
     * @param createGameRequest
     * @return A Java Future containing the result of the CreateGame operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>ServiceQuotaExceededException The request would result in exceeding service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.CreateGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGameResponse> createGame(CreateGameRequest createGameRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGame");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGameRequest, CreateGameResponse>().withOperationName("CreateGame")
                            .withMarshaller(new CreateGameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createGameRequest));
            CompletableFuture<CreateGameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a snapshot of the game configuration.
     * </p>
     *
     * @param createSnapshotRequest
     * @return A Java Future containing the result of the CreateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSnapshotResponse> createSnapshot(CreateSnapshotRequest createSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                            .withOperationName("CreateSnapshot")
                            .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSnapshotRequest));
            CompletableFuture<CreateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new stage for stage-by-stage game development and deployment.
     * </p>
     *
     * @param createStageRequest
     * @return A Java Future containing the result of the CreateStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.CreateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/CreateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStageResponse> createStage(CreateStageRequest createStageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStageRequest, CreateStageResponse>()
                            .withOperationName("CreateStage").withMarshaller(new CreateStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createStageRequest));
            CompletableFuture<CreateStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a game.
     * </p>
     *
     * @param deleteGameRequest
     * @return A Java Future containing the result of the DeleteGame operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.DeleteGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DeleteGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGameResponse> deleteGame(DeleteGameRequest deleteGameRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGame");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGameRequest, DeleteGameResponse>().withOperationName("DeleteGame")
                            .withMarshaller(new DeleteGameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteGameRequest));
            CompletableFuture<DeleteGameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a stage from a game, along with the associated game runtime.
     * </p>
     *
     * @param deleteStageRequest
     * @return A Java Future containing the result of the DeleteStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.DeleteStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DeleteStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStageResponse> deleteStage(DeleteStageRequest deleteStageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStageRequest, DeleteStageResponse>()
                            .withOperationName("DeleteStage").withMarshaller(new DeleteStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteStageRequest));
            CompletableFuture<DeleteStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disconnects a player from the game runtime.
     * </p>
     * <p>
     * If a player has multiple connections, this operation attempts to close all of them.
     * </p>
     *
     * @param disconnectPlayerRequest
     * @return A Java Future containing the result of the DisconnectPlayer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.DisconnectPlayer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/DisconnectPlayer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisconnectPlayerResponse> disconnectPlayer(DisconnectPlayerRequest disconnectPlayerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectPlayerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectPlayer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisconnectPlayerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisconnectPlayerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisconnectPlayerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisconnectPlayerRequest, DisconnectPlayerResponse>()
                            .withOperationName("DisconnectPlayer")
                            .withMarshaller(new DisconnectPlayerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disconnectPlayerRequest));
            CompletableFuture<DisconnectPlayerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports a game configuration snapshot.
     * </p>
     *
     * @param exportSnapshotRequest
     * @return A Java Future containing the result of the ExportSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ExportSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ExportSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ExportSnapshotResponse> exportSnapshot(ExportSnapshotRequest exportSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportSnapshotRequest, ExportSnapshotResponse>()
                            .withOperationName("ExportSnapshot")
                            .withMarshaller(new ExportSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportSnapshotRequest));
            CompletableFuture<ExportSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a specified extension.
     * </p>
     *
     * @param getExtensionRequest
     * @return A Java Future containing the result of the GetExtension operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetExtensionResponse> getExtension(GetExtensionRequest getExtensionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtension");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetExtensionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExtensionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExtensionRequest, GetExtensionResponse>()
                            .withOperationName("GetExtension").withMarshaller(new GetExtensionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getExtensionRequest));
            CompletableFuture<GetExtensionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a specified extension version.
     * </p>
     *
     * @param getExtensionVersionRequest
     * @return A Java Future containing the result of the GetExtensionVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetExtensionVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetExtensionVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExtensionVersionResponse> getExtensionVersion(
            GetExtensionVersionRequest getExtensionVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtensionVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExtensionVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExtensionVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExtensionVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExtensionVersionRequest, GetExtensionVersionResponse>()
                            .withOperationName("GetExtensionVersion")
                            .withMarshaller(new GetExtensionVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getExtensionVersionRequest));
            CompletableFuture<GetExtensionVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a game.
     * </p>
     *
     * @param getGameRequest
     * @return A Java Future containing the result of the GetGame operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGameResponse> getGame(GetGameRequest getGameRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGame");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGameRequest, GetGameResponse>().withOperationName("GetGame")
                            .withMarshaller(new GetGameRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getGameRequest));
            CompletableFuture<GetGameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the configuration of the game.
     * </p>
     *
     * @param getGameConfigurationRequest
     * @return A Java Future containing the result of the GetGameConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGameConfigurationResponse> getGameConfiguration(
            GetGameConfigurationRequest getGameConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGameConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGameConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGameConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGameConfigurationRequest, GetGameConfigurationResponse>()
                            .withOperationName("GetGameConfiguration")
                            .withMarshaller(new GetGameConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGameConfigurationRequest));
            CompletableFuture<GetGameConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a job that is generating code for a snapshot.
     * </p>
     *
     * @param getGeneratedCodeJobRequest
     * @return A Java Future containing the result of the GetGeneratedCodeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetGeneratedCodeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetGeneratedCodeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetGeneratedCodeJobResponse> getGeneratedCodeJob(
            GetGeneratedCodeJobRequest getGeneratedCodeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeneratedCodeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeneratedCodeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGeneratedCodeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetGeneratedCodeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGeneratedCodeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGeneratedCodeJobRequest, GetGeneratedCodeJobResponse>()
                            .withOperationName("GetGeneratedCodeJob")
                            .withMarshaller(new GetGeneratedCodeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGeneratedCodeJobRequest));
            CompletableFuture<GetGeneratedCodeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the status of a player's connection to the game runtime.
     * </p>
     * <p>
     * It's possible for a single player to have multiple connections to the game runtime. If a player is not connected,
     * this operation returns an empty list.
     * </p>
     *
     * @param getPlayerConnectionStatusRequest
     * @return A Java Future containing the result of the GetPlayerConnectionStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetPlayerConnectionStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetPlayerConnectionStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlayerConnectionStatusResponse> getPlayerConnectionStatus(
            GetPlayerConnectionStatusRequest getPlayerConnectionStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlayerConnectionStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlayerConnectionStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlayerConnectionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPlayerConnectionStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPlayerConnectionStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlayerConnectionStatusRequest, GetPlayerConnectionStatusResponse>()
                            .withOperationName("GetPlayerConnectionStatus")
                            .withMarshaller(new GetPlayerConnectionStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPlayerConnectionStatusRequest));
            CompletableFuture<GetPlayerConnectionStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a copy of the game configuration in a snapshot.
     * </p>
     *
     * @param getSnapshotRequest
     * @return A Java Future containing the result of the GetSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSnapshotResponse> getSnapshot(GetSnapshotRequest getSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSnapshotRequest, GetSnapshotResponse>()
                            .withOperationName("GetSnapshot").withMarshaller(new GetSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSnapshotRequest));
            CompletableFuture<GetSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a stage.
     * </p>
     *
     * @param getStageRequest
     * @return A Java Future containing the result of the GetStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetStageResponse> getStage(GetStageRequest getStageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStageRequest, GetStageResponse>().withOperationName("GetStage")
                            .withMarshaller(new GetStageRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStageRequest));
            CompletableFuture<GetStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about a stage deployment.
     * </p>
     *
     * @param getStageDeploymentRequest
     * @return A Java Future containing the result of the GetStageDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.GetStageDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/GetStageDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStageDeploymentResponse> getStageDeployment(GetStageDeploymentRequest getStageDeploymentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStageDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStageDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStageDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStageDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStageDeploymentRequest, GetStageDeploymentResponse>()
                            .withOperationName("GetStageDeployment")
                            .withMarshaller(new GetStageDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getStageDeploymentRequest));
            CompletableFuture<GetStageDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports a game configuration.
     * </p>
     * <p>
     * This operation replaces the current configuration of the game with the provided input. This is not a reversible
     * operation. If you want to preserve the previous configuration, use <code>CreateSnapshot</code> to make a new
     * snapshot before importing.
     * </p>
     *
     * @param importGameConfigurationRequest
     * @return A Java Future containing the result of the ImportGameConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>ServiceQuotaExceededException The request would result in exceeding service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ImportGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ImportGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportGameConfigurationResponse> importGameConfiguration(
            ImportGameConfigurationRequest importGameConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportGameConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportGameConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportGameConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportGameConfigurationRequest, ImportGameConfigurationResponse>()
                            .withOperationName("ImportGameConfiguration")
                            .withMarshaller(new ImportGameConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importGameConfigurationRequest));
            CompletableFuture<ImportGameConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of available versions for the extension.
     * </p>
     * <p>
     * Each time an API change is made to an extension, the version is incremented. The list retrieved by this operation
     * shows the versions that are currently available.
     * </p>
     *
     * @param listExtensionVersionsRequest
     * @return A Java Future containing the result of the ListExtensionVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListExtensionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensionVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListExtensionVersionsResponse> listExtensionVersions(
            ListExtensionVersionsRequest listExtensionVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensionVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExtensionVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExtensionVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExtensionVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExtensionVersionsRequest, ListExtensionVersionsResponse>()
                            .withOperationName("ListExtensionVersions")
                            .withMarshaller(new ListExtensionVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExtensionVersionsRequest));
            CompletableFuture<ListExtensionVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of available versions for the extension.
     * </p>
     * <p>
     * Each time an API change is made to an extension, the version is incremented. The list retrieved by this operation
     * shows the versions that are currently available.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExtensionVersions(software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListExtensionVersionsPublisher publisher = client.listExtensionVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListExtensionVersionsPublisher publisher = client.listExtensionVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExtensionVersions(software.amazon.awssdk.services.gamesparks.model.ListExtensionVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listExtensionVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListExtensionVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensionVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListExtensionVersionsPublisher listExtensionVersionsPaginator(ListExtensionVersionsRequest listExtensionVersionsRequest) {
        return new ListExtensionVersionsPublisher(this, applyPaginatorUserAgent(listExtensionVersionsRequest));
    }

    /**
     * <p>
     * Gets a paginated list of available extensions.
     * </p>
     * <p>
     * Extensions provide features that games can use from scripts.
     * </p>
     *
     * @param listExtensionsRequest
     * @return A Java Future containing the result of the ListExtensions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListExtensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExtensionsResponse> listExtensions(ListExtensionsRequest listExtensionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExtensionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListExtensionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExtensionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExtensionsRequest, ListExtensionsResponse>()
                            .withOperationName("ListExtensions")
                            .withMarshaller(new ListExtensionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExtensionsRequest));
            CompletableFuture<ListExtensionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of available extensions.
     * </p>
     * <p>
     * Extensions provide features that games can use from scripts.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listExtensions(software.amazon.awssdk.services.gamesparks.model.ListExtensionsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListExtensionsPublisher publisher = client.listExtensionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListExtensionsPublisher publisher = client.listExtensionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListExtensionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListExtensionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExtensions(software.amazon.awssdk.services.gamesparks.model.ListExtensionsRequest)} operation.</b>
     * </p>
     *
     * @param listExtensionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListExtensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListExtensionsPublisher listExtensionsPaginator(ListExtensionsRequest listExtensionsRequest) {
        return new ListExtensionsPublisher(this, applyPaginatorUserAgent(listExtensionsRequest));
    }

    /**
     * <p>
     * Gets a paginated list of games.
     * </p>
     *
     * @param listGamesRequest
     * @return A Java Future containing the result of the ListGames operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListGames
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGames" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGamesResponse> listGames(ListGamesRequest listGamesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGamesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGames");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGamesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGamesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGamesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGamesRequest, ListGamesResponse>().withOperationName("ListGames")
                            .withMarshaller(new ListGamesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGamesRequest));
            CompletableFuture<ListGamesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of games.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGames(software.amazon.awssdk.services.gamesparks.model.ListGamesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListGamesPublisher publisher = client.listGamesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListGamesPublisher publisher = client.listGamesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListGamesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListGamesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGames(software.amazon.awssdk.services.gamesparks.model.ListGamesRequest)} operation.</b>
     * </p>
     *
     * @param listGamesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListGames
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGames" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGamesPublisher listGamesPaginator(ListGamesRequest listGamesRequest) {
        return new ListGamesPublisher(this, applyPaginatorUserAgent(listGamesRequest));
    }

    /**
     * <p>
     * Gets a paginated list of code generation jobs for a snapshot.
     * </p>
     *
     * @param listGeneratedCodeJobsRequest
     * @return A Java Future containing the result of the ListGeneratedCodeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListGeneratedCodeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGeneratedCodeJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGeneratedCodeJobsResponse> listGeneratedCodeJobs(
            ListGeneratedCodeJobsRequest listGeneratedCodeJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeneratedCodeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeneratedCodeJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGeneratedCodeJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGeneratedCodeJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGeneratedCodeJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGeneratedCodeJobsRequest, ListGeneratedCodeJobsResponse>()
                            .withOperationName("ListGeneratedCodeJobs")
                            .withMarshaller(new ListGeneratedCodeJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listGeneratedCodeJobsRequest));
            CompletableFuture<ListGeneratedCodeJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of code generation jobs for a snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGeneratedCodeJobs(software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListGeneratedCodeJobsPublisher publisher = client.listGeneratedCodeJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListGeneratedCodeJobsPublisher publisher = client.listGeneratedCodeJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGeneratedCodeJobs(software.amazon.awssdk.services.gamesparks.model.ListGeneratedCodeJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGeneratedCodeJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListGeneratedCodeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListGeneratedCodeJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListGeneratedCodeJobsPublisher listGeneratedCodeJobsPaginator(ListGeneratedCodeJobsRequest listGeneratedCodeJobsRequest) {
        return new ListGeneratedCodeJobsPublisher(this, applyPaginatorUserAgent(listGeneratedCodeJobsRequest));
    }

    /**
     * <p>
     * Gets a paginated list of snapshot summaries from the game.
     * </p>
     *
     * @param listSnapshotsRequest
     * @return A Java Future containing the result of the ListSnapshots operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSnapshotsResponse> listSnapshots(ListSnapshotsRequest listSnapshotsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSnapshots");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSnapshotsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSnapshotsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSnapshotsRequest, ListSnapshotsResponse>()
                            .withOperationName("ListSnapshots")
                            .withMarshaller(new ListSnapshotsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSnapshotsRequest));
            CompletableFuture<ListSnapshotsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of snapshot summaries from the game.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshots(software.amazon.awssdk.services.gamesparks.model.ListSnapshotsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListSnapshotsPublisher publisher = client.listSnapshotsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListSnapshotsPublisher publisher = client.listSnapshotsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListSnapshotsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListSnapshotsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshots(software.amazon.awssdk.services.gamesparks.model.ListSnapshotsRequest)} operation.</b>
     * </p>
     *
     * @param listSnapshotsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSnapshotsPublisher listSnapshotsPaginator(ListSnapshotsRequest listSnapshotsRequest) {
        return new ListSnapshotsPublisher(this, applyPaginatorUserAgent(listSnapshotsRequest));
    }

    /**
     * <p>
     * Gets a paginated list of stage deployment summaries from the game.
     * </p>
     *
     * @param listStageDeploymentsRequest
     * @return A Java Future containing the result of the ListStageDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListStageDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStageDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStageDeploymentsResponse> listStageDeployments(
            ListStageDeploymentsRequest listStageDeploymentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStageDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStageDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStageDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStageDeploymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStageDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStageDeploymentsRequest, ListStageDeploymentsResponse>()
                            .withOperationName("ListStageDeployments")
                            .withMarshaller(new ListStageDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listStageDeploymentsRequest));
            CompletableFuture<ListStageDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of stage deployment summaries from the game.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listStageDeployments(software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListStageDeploymentsPublisher publisher = client.listStageDeploymentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListStageDeploymentsPublisher publisher = client.listStageDeploymentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStageDeployments(software.amazon.awssdk.services.gamesparks.model.ListStageDeploymentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listStageDeploymentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListStageDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStageDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    public ListStageDeploymentsPublisher listStageDeploymentsPaginator(ListStageDeploymentsRequest listStageDeploymentsRequest) {
        return new ListStageDeploymentsPublisher(this, applyPaginatorUserAgent(listStageDeploymentsRequest));
    }

    /**
     * <p>
     * Gets a paginated list of stage summaries from the game.
     * </p>
     *
     * @param listStagesRequest
     * @return A Java Future containing the result of the ListStages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStagesResponse> listStages(ListStagesRequest listStagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListStagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStagesRequest, ListStagesResponse>().withOperationName("ListStages")
                            .withMarshaller(new ListStagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listStagesRequest));
            CompletableFuture<ListStagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a paginated list of stage summaries from the game.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStages(software.amazon.awssdk.services.gamesparks.model.ListStagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListStagesPublisher publisher = client.listStagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.gamesparks.paginators.ListStagesPublisher publisher = client.listStagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.gamesparks.model.ListStagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.gamesparks.model.ListStagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStages(software.amazon.awssdk.services.gamesparks.model.ListStagesRequest)} operation.</b>
     * </p>
     *
     * @param listStagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    public ListStagesPublisher listStagesPaginator(ListStagesRequest listStagesRequest) {
        return new ListStagesPublisher(this, applyPaginatorUserAgent(listStagesRequest));
    }

    /**
     * <p>
     * Lists the tags associated with a GameSparks resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an asynchronous process that generates client code for system-defined and custom messages. The resulting
     * code is collected as a .zip file and uploaded to a pre-signed Amazon S3 URL.
     * </p>
     *
     * @param startGeneratedCodeJobRequest
     * @return A Java Future containing the result of the StartGeneratedCodeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.StartGeneratedCodeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/StartGeneratedCodeJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartGeneratedCodeJobResponse> startGeneratedCodeJob(
            StartGeneratedCodeJobRequest startGeneratedCodeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startGeneratedCodeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartGeneratedCodeJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartGeneratedCodeJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartGeneratedCodeJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartGeneratedCodeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartGeneratedCodeJobRequest, StartGeneratedCodeJobResponse>()
                            .withOperationName("StartGeneratedCodeJob")
                            .withMarshaller(new StartGeneratedCodeJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startGeneratedCodeJobRequest));
            CompletableFuture<StartGeneratedCodeJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deploys a snapshot to the stage and creates a new game runtime.
     * </p>
     * <p>
     * After you call this operation, you can check the deployment status by using <code>GetStageDeployment</code>.
     * </p>
     * <p>
     * If there are any players connected to the previous game runtime, then both runtimes persist. Existing connections
     * to the previous runtime are maintained. When players disconnect and reconnect, they connect to the new runtime.
     * After there are no connections to the previous game runtime, it is deleted.
     * </p>
     *
     * @param startStageDeploymentRequest
     * @return A Java Future containing the result of the StartStageDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ConflictException The resource already exists, or another operation is in progress.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.StartStageDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/StartStageDeployment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartStageDeploymentResponse> startStageDeployment(
            StartStageDeploymentRequest startStageDeploymentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startStageDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartStageDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartStageDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartStageDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartStageDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartStageDeploymentRequest, StartStageDeploymentResponse>()
                            .withOperationName("StartStageDeployment")
                            .withMarshaller(new StartStageDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startStageDeploymentRequest));
            CompletableFuture<StartStageDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to a GameSparks resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a GameSparks resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates details of the game.
     * </p>
     *
     * @param updateGameRequest
     * @return A Java Future containing the result of the UpdateGame operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.UpdateGame
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateGame" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameResponse> updateGame(UpdateGameRequest updateGameRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGame");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateGameResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameRequest, UpdateGameResponse>().withOperationName("UpdateGame")
                            .withMarshaller(new UpdateGameRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameRequest));
            CompletableFuture<UpdateGameResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates one or more sections of the game configuration.
     * </p>
     *
     * @param updateGameConfigurationRequest
     * @return A Java Future containing the result of the UpdateGameConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>ServiceQuotaExceededException The request would result in exceeding service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.UpdateGameConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateGameConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGameConfigurationResponse> updateGameConfiguration(
            UpdateGameConfigurationRequest updateGameConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGameConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGameConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGameConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGameConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateGameConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGameConfigurationRequest, UpdateGameConfigurationResponse>()
                            .withOperationName("UpdateGameConfiguration")
                            .withMarshaller(new UpdateGameConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateGameConfigurationRequest));
            CompletableFuture<UpdateGameConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the metadata of a GameSparks snapshot.
     * </p>
     *
     * @param updateSnapshotRequest
     * @return A Java Future containing the result of the UpdateSnapshot operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.UpdateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSnapshotResponse> updateSnapshot(UpdateSnapshotRequest updateSnapshotRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSnapshot");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSnapshotResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSnapshotResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSnapshotRequest, UpdateSnapshotResponse>()
                            .withOperationName("UpdateSnapshot")
                            .withMarshaller(new UpdateSnapshotRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSnapshotRequest));
            CompletableFuture<UpdateSnapshotResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the metadata of a stage.
     * </p>
     *
     * @param updateStageRequest
     * @return A Java Future containing the result of the UpdateStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One of the parameters in the request is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request throughput limit was exceeded.</li>
     *         <li>ResourceNotFoundException The resource specified in the request does not exist.</li>
     *         <li>InternalServerException The service encountered an internal error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GameSparksException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GameSparksAsyncClient.UpdateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/gamesparks-2021-08-17/UpdateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStageResponse> updateStage(UpdateStageRequest updateStageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GameSparks");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStageRequest, UpdateStageResponse>()
                            .withOperationName("UpdateStage").withMarshaller(new UpdateStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateStageRequest));
            CompletableFuture<UpdateStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GameSparksException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends GameSparksRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
