/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.geomaps.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetStyleDescriptorRequest extends GeoMapsRequest implements
        ToCopyableBuilder<GetStyleDescriptorRequest.Builder, GetStyleDescriptorRequest> {
    private static final SdkField<String> STYLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Style")
            .getter(getter(GetStyleDescriptorRequest::styleAsString)).setter(setter(Builder::style))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Style").build()).build();

    private static final SdkField<String> COLOR_SCHEME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ColorScheme").getter(getter(GetStyleDescriptorRequest::colorSchemeAsString))
            .setter(setter(Builder::colorScheme))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("color-scheme").build()).build();

    private static final SdkField<String> POLITICAL_VIEW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PoliticalView").getter(getter(GetStyleDescriptorRequest::politicalView))
            .setter(setter(Builder::politicalView))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("political-view").build())
            .build();

    private static final SdkField<String> TERRAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Terrain")
            .getter(getter(GetStyleDescriptorRequest::terrainAsString)).setter(setter(Builder::terrain))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("terrain").build()).build();

    private static final SdkField<String> CONTOUR_DENSITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContourDensity").getter(getter(GetStyleDescriptorRequest::contourDensityAsString))
            .setter(setter(Builder::contourDensity))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("contour-density").build())
            .build();

    private static final SdkField<String> TRAFFIC_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Traffic")
            .getter(getter(GetStyleDescriptorRequest::trafficAsString)).setter(setter(Builder::traffic))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("traffic").build()).build();

    private static final SdkField<List<String>> TRAVEL_MODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TravelModes")
            .getter(getter(GetStyleDescriptorRequest::travelModesAsStrings))
            .setter(setter(Builder::travelModesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("travel-modes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(GetStyleDescriptorRequest::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("key").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STYLE_FIELD,
            COLOR_SCHEME_FIELD, POLITICAL_VIEW_FIELD, TERRAIN_FIELD, CONTOUR_DENSITY_FIELD, TRAFFIC_FIELD, TRAVEL_MODES_FIELD,
            KEY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String style;

    private final String colorScheme;

    private final String politicalView;

    private final String terrain;

    private final String contourDensity;

    private final String traffic;

    private final List<String> travelModes;

    private final String key;

    private GetStyleDescriptorRequest(BuilderImpl builder) {
        super(builder);
        this.style = builder.style;
        this.colorScheme = builder.colorScheme;
        this.politicalView = builder.politicalView;
        this.terrain = builder.terrain;
        this.contourDensity = builder.contourDensity;
        this.traffic = builder.traffic;
        this.travelModes = builder.travelModes;
        this.key = builder.key;
    }

    /**
     * <p>
     * Style specifies the desired map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #style} will return
     * {@link MapStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #styleAsString}.
     * </p>
     * 
     * @return Style specifies the desired map style.
     * @see MapStyle
     */
    public final MapStyle style() {
        return MapStyle.fromValue(style);
    }

    /**
     * <p>
     * Style specifies the desired map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #style} will return
     * {@link MapStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #styleAsString}.
     * </p>
     * 
     * @return Style specifies the desired map style.
     * @see MapStyle
     */
    public final String styleAsString() {
        return style;
    }

    /**
     * <p>
     * Sets color tone for map such as dark and light for specific map styles. It applies to only vector map styles such
     * as Standard and Monochrome.
     * </p>
     * <p>
     * Example: <code>Light</code>
     * </p>
     * <p>
     * Default value: <code>Light</code>
     * </p>
     * <note>
     * <p>
     * Valid values for ColorScheme are case sensitive.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #colorScheme} will
     * return {@link ColorScheme#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #colorSchemeAsString}.
     * </p>
     * 
     * @return Sets color tone for map such as dark and light for specific map styles. It applies to only vector map
     *         styles such as Standard and Monochrome.</p>
     *         <p>
     *         Example: <code>Light</code>
     *         </p>
     *         <p>
     *         Default value: <code>Light</code>
     *         </p>
     *         <note>
     *         <p>
     *         Valid values for ColorScheme are case sensitive.
     *         </p>
     * @see ColorScheme
     */
    public final ColorScheme colorScheme() {
        return ColorScheme.fromValue(colorScheme);
    }

    /**
     * <p>
     * Sets color tone for map such as dark and light for specific map styles. It applies to only vector map styles such
     * as Standard and Monochrome.
     * </p>
     * <p>
     * Example: <code>Light</code>
     * </p>
     * <p>
     * Default value: <code>Light</code>
     * </p>
     * <note>
     * <p>
     * Valid values for ColorScheme are case sensitive.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #colorScheme} will
     * return {@link ColorScheme#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #colorSchemeAsString}.
     * </p>
     * 
     * @return Sets color tone for map such as dark and light for specific map styles. It applies to only vector map
     *         styles such as Standard and Monochrome.</p>
     *         <p>
     *         Example: <code>Light</code>
     *         </p>
     *         <p>
     *         Default value: <code>Light</code>
     *         </p>
     *         <note>
     *         <p>
     *         Valid values for ColorScheme are case sensitive.
     *         </p>
     * @see ColorScheme
     */
    public final String colorSchemeAsString() {
        return colorScheme;
    }

    /**
     * <p>
     * Specifies the political view using ISO 3166-2 or ISO 3166-3 country code format.
     * </p>
     * <p>
     * The following political views are currently supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ARG</code>: Argentina's view on the Southern Patagonian Ice Field and Tierra Del Fuego, including the
     * Falkland Islands, South Georgia, and South Sandwich Islands
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EGY</code>: Egypt's view on Bir Tawil
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IND</code>: India's view on Gilgit-Baltistan
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KEN</code>: Kenya's view on the Ilemi Triangle
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MAR</code>: Morocco's view on Western Sahara
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUS</code>: Russia's view on Crimea
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SDN</code>: Sudan's view on the Halaib Triangle
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SRB</code>: Serbia's view on Kosovo, Vukovar, and Sarengrad Islands
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUR</code>: Suriname's view on the Courantyne Headwaters and Lawa Headwaters
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SYR</code>: Syria's view on the Golan Heights
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TUR</code>: Turkey's view on Cyprus and Northern Cyprus
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TZA</code>: Tanzania's view on Lake Malawi
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>URY</code>: Uruguay's view on Rincon de Artigas
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VNM</code>: Vietnam's view on the Paracel Islands and Spratly Islands
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the political view using ISO 3166-2 or ISO 3166-3 country code format.</p>
     *         <p>
     *         The following political views are currently supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ARG</code>: Argentina's view on the Southern Patagonian Ice Field and Tierra Del Fuego, including
     *         the Falkland Islands, South Georgia, and South Sandwich Islands
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EGY</code>: Egypt's view on Bir Tawil
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IND</code>: India's view on Gilgit-Baltistan
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KEN</code>: Kenya's view on the Ilemi Triangle
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MAR</code>: Morocco's view on Western Sahara
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUS</code>: Russia's view on Crimea
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SDN</code>: Sudan's view on the Halaib Triangle
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SRB</code>: Serbia's view on Kosovo, Vukovar, and Sarengrad Islands
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUR</code>: Suriname's view on the Courantyne Headwaters and Lawa Headwaters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SYR</code>: Syria's view on the Golan Heights
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TUR</code>: Turkey's view on Cyprus and Northern Cyprus
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TZA</code>: Tanzania's view on Lake Malawi
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>URY</code>: Uruguay's view on Rincon de Artigas
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VNM</code>: Vietnam's view on the Paracel Islands and Spratly Islands
     *         </p>
     *         </li>
     */
    public final String politicalView() {
        return politicalView;
    }

    /**
     * <p>
     * Adjusts how physical terrain details are rendered on the map.
     * </p>
     * <p>
     * The following terrain styles are currently supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of elevation
     * change and geographic features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #terrain} will
     * return {@link Terrain#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #terrainAsString}.
     * </p>
     * 
     * @return Adjusts how physical terrain details are rendered on the map.</p>
     *         <p>
     *         The following terrain styles are currently supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of
     *         elevation change and geographic features.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see Terrain
     */
    public final Terrain terrain() {
        return Terrain.fromValue(terrain);
    }

    /**
     * <p>
     * Adjusts how physical terrain details are rendered on the map.
     * </p>
     * <p>
     * The following terrain styles are currently supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of elevation
     * change and geographic features.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #terrain} will
     * return {@link Terrain#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #terrainAsString}.
     * </p>
     * 
     * @return Adjusts how physical terrain details are rendered on the map.</p>
     *         <p>
     *         The following terrain styles are currently supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of
     *         elevation change and geographic features.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see Terrain
     */
    public final String terrainAsString() {
        return terrain;
    }

    /**
     * <p>
     * Displays the shape and steepness of terrain features using elevation lines. The density value controls how
     * densely the available contour line information is rendered on the map.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contourDensity}
     * will return {@link ContourDensity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #contourDensityAsString}.
     * </p>
     * 
     * @return Displays the shape and steepness of terrain features using elevation lines. The density value controls
     *         how densely the available contour line information is rendered on the map.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see ContourDensity
     */
    public final ContourDensity contourDensity() {
        return ContourDensity.fromValue(contourDensity);
    }

    /**
     * <p>
     * Displays the shape and steepness of terrain features using elevation lines. The density value controls how
     * densely the available contour line information is rendered on the map.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contourDensity}
     * will return {@link ContourDensity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #contourDensityAsString}.
     * </p>
     * 
     * @return Displays the shape and steepness of terrain features using elevation lines. The density value controls
     *         how densely the available contour line information is rendered on the map.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see ContourDensity
     */
    public final String contourDensityAsString() {
        return contourDensity;
    }

    /**
     * <p>
     * Displays real-time traffic information overlay on map, such as incident events and flow events.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #traffic} will
     * return {@link Traffic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficAsString}.
     * </p>
     * 
     * @return Displays real-time traffic information overlay on map, such as incident events and flow events.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see Traffic
     */
    public final Traffic traffic() {
        return Traffic.fromValue(traffic);
    }

    /**
     * <p>
     * Displays real-time traffic information overlay on map, such as incident events and flow events.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #traffic} will
     * return {@link Traffic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficAsString}.
     * </p>
     * 
     * @return Displays real-time traffic information overlay on map, such as incident events and flow events.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     * @see Traffic
     */
    public final String trafficAsString() {
        return traffic;
    }

    /**
     * <p>
     * Renders additional map information relevant to selected travel modes. Information for multiple travel modes can
     * be displayed simultaneously, although this increases the overall information density rendered on the map.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTravelModes} method.
     * </p>
     * 
     * @return Renders additional map information relevant to selected travel modes. Information for multiple travel
     *         modes can be displayed simultaneously, although this increases the overall information density rendered
     *         on the map.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     */
    public final List<TravelMode> travelModes() {
        return TravelModeListCopier.copyStringToEnum(travelModes);
    }

    /**
     * For responses, this returns true if the service returned a value for the TravelModes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTravelModes() {
        return travelModes != null && !(travelModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Renders additional map information relevant to selected travel modes. Information for multiple travel modes can
     * be displayed simultaneously, although this increases the overall information density rendered on the map.
     * </p>
     * <p>
     * This parameter is valid only for the <code>Standard</code> map style.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTravelModes} method.
     * </p>
     * 
     * @return Renders additional map information relevant to selected travel modes. Information for multiple travel
     *         modes can be displayed simultaneously, although this increases the overall information density rendered
     *         on the map.</p>
     *         <p>
     *         This parameter is valid only for the <code>Standard</code> map style.
     */
    public final List<String> travelModesAsStrings() {
        return travelModes;
    }

    /**
     * <p>
     * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be provided
     * when making a request.
     * </p>
     * 
     * @return Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
     *         provided when making a request.
     */
    public final String key() {
        return key;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(styleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(colorSchemeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(politicalView());
        hashCode = 31 * hashCode + Objects.hashCode(terrainAsString());
        hashCode = 31 * hashCode + Objects.hashCode(contourDensityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(trafficAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTravelModes() ? travelModesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(key());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetStyleDescriptorRequest)) {
            return false;
        }
        GetStyleDescriptorRequest other = (GetStyleDescriptorRequest) obj;
        return Objects.equals(styleAsString(), other.styleAsString())
                && Objects.equals(colorSchemeAsString(), other.colorSchemeAsString())
                && Objects.equals(politicalView(), other.politicalView())
                && Objects.equals(terrainAsString(), other.terrainAsString())
                && Objects.equals(contourDensityAsString(), other.contourDensityAsString())
                && Objects.equals(trafficAsString(), other.trafficAsString()) && hasTravelModes() == other.hasTravelModes()
                && Objects.equals(travelModesAsStrings(), other.travelModesAsStrings()) && Objects.equals(key(), other.key());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetStyleDescriptorRequest").add("Style", styleAsString())
                .add("ColorScheme", colorSchemeAsString())
                .add("PoliticalView", politicalView() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Terrain", terrainAsString()).add("ContourDensity", contourDensityAsString())
                .add("Traffic", trafficAsString()).add("TravelModes", hasTravelModes() ? travelModesAsStrings() : null)
                .add("Key", key() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Style":
            return Optional.ofNullable(clazz.cast(styleAsString()));
        case "ColorScheme":
            return Optional.ofNullable(clazz.cast(colorSchemeAsString()));
        case "PoliticalView":
            return Optional.ofNullable(clazz.cast(politicalView()));
        case "Terrain":
            return Optional.ofNullable(clazz.cast(terrainAsString()));
        case "ContourDensity":
            return Optional.ofNullable(clazz.cast(contourDensityAsString()));
        case "Traffic":
            return Optional.ofNullable(clazz.cast(trafficAsString()));
        case "TravelModes":
            return Optional.ofNullable(clazz.cast(travelModesAsStrings()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Style", STYLE_FIELD);
        map.put("color-scheme", COLOR_SCHEME_FIELD);
        map.put("political-view", POLITICAL_VIEW_FIELD);
        map.put("terrain", TERRAIN_FIELD);
        map.put("contour-density", CONTOUR_DENSITY_FIELD);
        map.put("traffic", TRAFFIC_FIELD);
        map.put("travel-modes", TRAVEL_MODES_FIELD);
        map.put("key", KEY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GetStyleDescriptorRequest, T> g) {
        return obj -> g.apply((GetStyleDescriptorRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GeoMapsRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetStyleDescriptorRequest> {
        /**
         * <p>
         * Style specifies the desired map style.
         * </p>
         * 
         * @param style
         *        Style specifies the desired map style.
         * @see MapStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MapStyle
         */
        Builder style(String style);

        /**
         * <p>
         * Style specifies the desired map style.
         * </p>
         * 
         * @param style
         *        Style specifies the desired map style.
         * @see MapStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MapStyle
         */
        Builder style(MapStyle style);

        /**
         * <p>
         * Sets color tone for map such as dark and light for specific map styles. It applies to only vector map styles
         * such as Standard and Monochrome.
         * </p>
         * <p>
         * Example: <code>Light</code>
         * </p>
         * <p>
         * Default value: <code>Light</code>
         * </p>
         * <note>
         * <p>
         * Valid values for ColorScheme are case sensitive.
         * </p>
         * </note>
         * 
         * @param colorScheme
         *        Sets color tone for map such as dark and light for specific map styles. It applies to only vector map
         *        styles such as Standard and Monochrome.</p>
         *        <p>
         *        Example: <code>Light</code>
         *        </p>
         *        <p>
         *        Default value: <code>Light</code>
         *        </p>
         *        <note>
         *        <p>
         *        Valid values for ColorScheme are case sensitive.
         *        </p>
         * @see ColorScheme
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ColorScheme
         */
        Builder colorScheme(String colorScheme);

        /**
         * <p>
         * Sets color tone for map such as dark and light for specific map styles. It applies to only vector map styles
         * such as Standard and Monochrome.
         * </p>
         * <p>
         * Example: <code>Light</code>
         * </p>
         * <p>
         * Default value: <code>Light</code>
         * </p>
         * <note>
         * <p>
         * Valid values for ColorScheme are case sensitive.
         * </p>
         * </note>
         * 
         * @param colorScheme
         *        Sets color tone for map such as dark and light for specific map styles. It applies to only vector map
         *        styles such as Standard and Monochrome.</p>
         *        <p>
         *        Example: <code>Light</code>
         *        </p>
         *        <p>
         *        Default value: <code>Light</code>
         *        </p>
         *        <note>
         *        <p>
         *        Valid values for ColorScheme are case sensitive.
         *        </p>
         * @see ColorScheme
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ColorScheme
         */
        Builder colorScheme(ColorScheme colorScheme);

        /**
         * <p>
         * Specifies the political view using ISO 3166-2 or ISO 3166-3 country code format.
         * </p>
         * <p>
         * The following political views are currently supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ARG</code>: Argentina's view on the Southern Patagonian Ice Field and Tierra Del Fuego, including the
         * Falkland Islands, South Georgia, and South Sandwich Islands
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EGY</code>: Egypt's view on Bir Tawil
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IND</code>: India's view on Gilgit-Baltistan
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KEN</code>: Kenya's view on the Ilemi Triangle
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MAR</code>: Morocco's view on Western Sahara
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUS</code>: Russia's view on Crimea
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SDN</code>: Sudan's view on the Halaib Triangle
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SRB</code>: Serbia's view on Kosovo, Vukovar, and Sarengrad Islands
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUR</code>: Suriname's view on the Courantyne Headwaters and Lawa Headwaters
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SYR</code>: Syria's view on the Golan Heights
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TUR</code>: Turkey's view on Cyprus and Northern Cyprus
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TZA</code>: Tanzania's view on Lake Malawi
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>URY</code>: Uruguay's view on Rincon de Artigas
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VNM</code>: Vietnam's view on the Paracel Islands and Spratly Islands
         * </p>
         * </li>
         * </ul>
         * 
         * @param politicalView
         *        Specifies the political view using ISO 3166-2 or ISO 3166-3 country code format.</p>
         *        <p>
         *        The following political views are currently supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ARG</code>: Argentina's view on the Southern Patagonian Ice Field and Tierra Del Fuego,
         *        including the Falkland Islands, South Georgia, and South Sandwich Islands
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EGY</code>: Egypt's view on Bir Tawil
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IND</code>: India's view on Gilgit-Baltistan
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KEN</code>: Kenya's view on the Ilemi Triangle
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MAR</code>: Morocco's view on Western Sahara
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUS</code>: Russia's view on Crimea
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SDN</code>: Sudan's view on the Halaib Triangle
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SRB</code>: Serbia's view on Kosovo, Vukovar, and Sarengrad Islands
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUR</code>: Suriname's view on the Courantyne Headwaters and Lawa Headwaters
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SYR</code>: Syria's view on the Golan Heights
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TUR</code>: Turkey's view on Cyprus and Northern Cyprus
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TZA</code>: Tanzania's view on Lake Malawi
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>URY</code>: Uruguay's view on Rincon de Artigas
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VNM</code>: Vietnam's view on the Paracel Islands and Spratly Islands
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder politicalView(String politicalView);

        /**
         * <p>
         * Adjusts how physical terrain details are rendered on the map.
         * </p>
         * <p>
         * The following terrain styles are currently supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of elevation
         * change and geographic features.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param terrain
         *        Adjusts how physical terrain details are rendered on the map.</p>
         *        <p>
         *        The following terrain styles are currently supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of
         *        elevation change and geographic features.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see Terrain
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Terrain
         */
        Builder terrain(String terrain);

        /**
         * <p>
         * Adjusts how physical terrain details are rendered on the map.
         * </p>
         * <p>
         * The following terrain styles are currently supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of elevation
         * change and geographic features.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param terrain
         *        Adjusts how physical terrain details are rendered on the map.</p>
         *        <p>
         *        The following terrain styles are currently supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Hillshade</code>: Displays the physical terrain details through shading and highlighting of
         *        elevation change and geographic features.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see Terrain
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Terrain
         */
        Builder terrain(Terrain terrain);

        /**
         * <p>
         * Displays the shape and steepness of terrain features using elevation lines. The density value controls how
         * densely the available contour line information is rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param contourDensity
         *        Displays the shape and steepness of terrain features using elevation lines. The density value controls
         *        how densely the available contour line information is rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see ContourDensity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContourDensity
         */
        Builder contourDensity(String contourDensity);

        /**
         * <p>
         * Displays the shape and steepness of terrain features using elevation lines. The density value controls how
         * densely the available contour line information is rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param contourDensity
         *        Displays the shape and steepness of terrain features using elevation lines. The density value controls
         *        how densely the available contour line information is rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see ContourDensity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContourDensity
         */
        Builder contourDensity(ContourDensity contourDensity);

        /**
         * <p>
         * Displays real-time traffic information overlay on map, such as incident events and flow events.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param traffic
         *        Displays real-time traffic information overlay on map, such as incident events and flow events.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see Traffic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Traffic
         */
        Builder traffic(String traffic);

        /**
         * <p>
         * Displays real-time traffic information overlay on map, such as incident events and flow events.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param traffic
         *        Displays real-time traffic information overlay on map, such as incident events and flow events.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @see Traffic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Traffic
         */
        Builder traffic(Traffic traffic);

        /**
         * <p>
         * Renders additional map information relevant to selected travel modes. Information for multiple travel modes
         * can be displayed simultaneously, although this increases the overall information density rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param travelModes
         *        Renders additional map information relevant to selected travel modes. Information for multiple travel
         *        modes can be displayed simultaneously, although this increases the overall information density
         *        rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModesWithStrings(Collection<String> travelModes);

        /**
         * <p>
         * Renders additional map information relevant to selected travel modes. Information for multiple travel modes
         * can be displayed simultaneously, although this increases the overall information density rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param travelModes
         *        Renders additional map information relevant to selected travel modes. Information for multiple travel
         *        modes can be displayed simultaneously, although this increases the overall information density
         *        rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModesWithStrings(String... travelModes);

        /**
         * <p>
         * Renders additional map information relevant to selected travel modes. Information for multiple travel modes
         * can be displayed simultaneously, although this increases the overall information density rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param travelModes
         *        Renders additional map information relevant to selected travel modes. Information for multiple travel
         *        modes can be displayed simultaneously, although this increases the overall information density
         *        rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModes(Collection<TravelMode> travelModes);

        /**
         * <p>
         * Renders additional map information relevant to selected travel modes. Information for multiple travel modes
         * can be displayed simultaneously, although this increases the overall information density rendered on the map.
         * </p>
         * <p>
         * This parameter is valid only for the <code>Standard</code> map style.
         * </p>
         * 
         * @param travelModes
         *        Renders additional map information relevant to selected travel modes. Information for multiple travel
         *        modes can be displayed simultaneously, although this increases the overall information density
         *        rendered on the map.</p>
         *        <p>
         *        This parameter is valid only for the <code>Standard</code> map style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModes(TravelMode... travelModes);

        /**
         * <p>
         * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         * provided when making a request.
         * </p>
         * 
         * @param key
         *        Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         *        provided when making a request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GeoMapsRequest.BuilderImpl implements Builder {
        private String style;

        private String colorScheme;

        private String politicalView;

        private String terrain;

        private String contourDensity;

        private String traffic;

        private List<String> travelModes = DefaultSdkAutoConstructList.getInstance();

        private String key;

        private BuilderImpl() {
        }

        private BuilderImpl(GetStyleDescriptorRequest model) {
            super(model);
            style(model.style);
            colorScheme(model.colorScheme);
            politicalView(model.politicalView);
            terrain(model.terrain);
            contourDensity(model.contourDensity);
            traffic(model.traffic);
            travelModesWithStrings(model.travelModes);
            key(model.key);
        }

        public final String getStyle() {
            return style;
        }

        public final void setStyle(String style) {
            this.style = style;
        }

        @Override
        public final Builder style(String style) {
            this.style = style;
            return this;
        }

        @Override
        public final Builder style(MapStyle style) {
            this.style(style == null ? null : style.toString());
            return this;
        }

        public final String getColorScheme() {
            return colorScheme;
        }

        public final void setColorScheme(String colorScheme) {
            this.colorScheme = colorScheme;
        }

        @Override
        public final Builder colorScheme(String colorScheme) {
            this.colorScheme = colorScheme;
            return this;
        }

        @Override
        public final Builder colorScheme(ColorScheme colorScheme) {
            this.colorScheme(colorScheme == null ? null : colorScheme.toString());
            return this;
        }

        public final String getPoliticalView() {
            return politicalView;
        }

        public final void setPoliticalView(String politicalView) {
            this.politicalView = politicalView;
        }

        @Override
        public final Builder politicalView(String politicalView) {
            this.politicalView = politicalView;
            return this;
        }

        public final String getTerrain() {
            return terrain;
        }

        public final void setTerrain(String terrain) {
            this.terrain = terrain;
        }

        @Override
        public final Builder terrain(String terrain) {
            this.terrain = terrain;
            return this;
        }

        @Override
        public final Builder terrain(Terrain terrain) {
            this.terrain(terrain == null ? null : terrain.toString());
            return this;
        }

        public final String getContourDensity() {
            return contourDensity;
        }

        public final void setContourDensity(String contourDensity) {
            this.contourDensity = contourDensity;
        }

        @Override
        public final Builder contourDensity(String contourDensity) {
            this.contourDensity = contourDensity;
            return this;
        }

        @Override
        public final Builder contourDensity(ContourDensity contourDensity) {
            this.contourDensity(contourDensity == null ? null : contourDensity.toString());
            return this;
        }

        public final String getTraffic() {
            return traffic;
        }

        public final void setTraffic(String traffic) {
            this.traffic = traffic;
        }

        @Override
        public final Builder traffic(String traffic) {
            this.traffic = traffic;
            return this;
        }

        @Override
        public final Builder traffic(Traffic traffic) {
            this.traffic(traffic == null ? null : traffic.toString());
            return this;
        }

        public final Collection<String> getTravelModes() {
            if (travelModes instanceof SdkAutoConstructList) {
                return null;
            }
            return travelModes;
        }

        public final void setTravelModes(Collection<String> travelModes) {
            this.travelModes = TravelModeListCopier.copy(travelModes);
        }

        @Override
        public final Builder travelModesWithStrings(Collection<String> travelModes) {
            this.travelModes = TravelModeListCopier.copy(travelModes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder travelModesWithStrings(String... travelModes) {
            travelModesWithStrings(Arrays.asList(travelModes));
            return this;
        }

        @Override
        public final Builder travelModes(Collection<TravelMode> travelModes) {
            this.travelModes = TravelModeListCopier.copyEnumToString(travelModes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder travelModes(TravelMode... travelModes) {
            travelModes(Arrays.asList(travelModes));
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetStyleDescriptorRequest build() {
            return new GetStyleDescriptorRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
