/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.geoplaces;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.geoplaces.internal.GeoPlacesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.geoplaces.model.AccessDeniedException;
import software.amazon.awssdk.services.geoplaces.model.AutocompleteRequest;
import software.amazon.awssdk.services.geoplaces.model.AutocompleteResponse;
import software.amazon.awssdk.services.geoplaces.model.GeoPlacesException;
import software.amazon.awssdk.services.geoplaces.model.GeocodeRequest;
import software.amazon.awssdk.services.geoplaces.model.GeocodeResponse;
import software.amazon.awssdk.services.geoplaces.model.GetPlaceRequest;
import software.amazon.awssdk.services.geoplaces.model.GetPlaceResponse;
import software.amazon.awssdk.services.geoplaces.model.InternalServerException;
import software.amazon.awssdk.services.geoplaces.model.ReverseGeocodeRequest;
import software.amazon.awssdk.services.geoplaces.model.ReverseGeocodeResponse;
import software.amazon.awssdk.services.geoplaces.model.SearchNearbyRequest;
import software.amazon.awssdk.services.geoplaces.model.SearchNearbyResponse;
import software.amazon.awssdk.services.geoplaces.model.SearchTextRequest;
import software.amazon.awssdk.services.geoplaces.model.SearchTextResponse;
import software.amazon.awssdk.services.geoplaces.model.SuggestRequest;
import software.amazon.awssdk.services.geoplaces.model.SuggestResponse;
import software.amazon.awssdk.services.geoplaces.model.ThrottlingException;
import software.amazon.awssdk.services.geoplaces.model.ValidationException;
import software.amazon.awssdk.services.geoplaces.transform.AutocompleteRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.GeocodeRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.GetPlaceRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.ReverseGeocodeRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SearchNearbyRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SearchTextRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SuggestRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GeoPlacesAsyncClient}.
 *
 * @see GeoPlacesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGeoPlacesAsyncClient implements GeoPlacesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGeoPlacesAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGeoPlacesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * The autocomplete operation speeds up and increases the accuracy of entering addresses by providing a list of
     * address candidates matching a partially entered address. Results are sorted from most to least matching.
     * Filtering and biasing can be used to increase the relevance of the results if additional search context is known
     * </p>
     *
     * @param autocompleteRequest
     * @return A Java Future containing the result of the Autocomplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.Autocomplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Autocomplete" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AutocompleteResponse> autocomplete(AutocompleteRequest autocompleteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(autocompleteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, autocompleteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Autocomplete");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AutocompleteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AutocompleteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AutocompleteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AutocompleteRequest, AutocompleteResponse>()
                            .withOperationName("Autocomplete").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AutocompleteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(autocompleteRequest));
            CompletableFuture<AutocompleteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Geocode</code> action allows you to obtain coordinates, addresses, and other information about places.
     * </p>
     *
     * @param geocodeRequest
     * @return A Java Future containing the result of the Geocode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.Geocode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Geocode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GeocodeResponse> geocode(GeocodeRequest geocodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(geocodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, geocodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Geocode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GeocodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GeocodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GeocodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GeocodeRequest, GeocodeResponse>().withOperationName("Geocode")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GeocodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(geocodeRequest));
            CompletableFuture<GeocodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Finds a place by its unique ID. A <code>PlaceId</code> is returned by other place operations.
     * </p>
     *
     * @param getPlaceRequest
     * @return A Java Future containing the result of the GetPlace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.GetPlace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/GetPlace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlaceResponse> getPlace(GetPlaceRequest getPlaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPlaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPlaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlaceRequest, GetPlaceResponse>().withOperationName("GetPlace")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPlaceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPlaceRequest));
            CompletableFuture<GetPlaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ReverseGeocode</code> operation allows you to retrieve addresses and place information from
     * coordinates.
     * </p>
     *
     * @param reverseGeocodeRequest
     * @return A Java Future containing the result of the ReverseGeocode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.ReverseGeocode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/ReverseGeocode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ReverseGeocodeResponse> reverseGeocode(ReverseGeocodeRequest reverseGeocodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(reverseGeocodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reverseGeocodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReverseGeocode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReverseGeocodeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReverseGeocodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReverseGeocodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReverseGeocodeRequest, ReverseGeocodeResponse>()
                            .withOperationName("ReverseGeocode").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ReverseGeocodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(reverseGeocodeRequest));
            CompletableFuture<ReverseGeocodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Search nearby a specified location.
     * </p>
     *
     * @param searchNearbyRequest
     * @return A Java Future containing the result of the SearchNearby operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.SearchNearby
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/SearchNearby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchNearbyResponse> searchNearby(SearchNearbyRequest searchNearbyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchNearbyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchNearbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchNearby");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchNearbyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchNearbyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchNearbyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchNearbyRequest, SearchNearbyResponse>()
                            .withOperationName("SearchNearby").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchNearbyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchNearbyRequest));
            CompletableFuture<SearchNearbyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use the <code>SearchText</code> operation to search for geocode and place information. You can then complete a
     * follow-up query suggested from the <code>Suggest</code> API via a query id.
     * </p>
     *
     * @param searchTextRequest
     * @return A Java Future containing the result of the SearchText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.SearchText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/SearchText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchTextResponse> searchText(SearchTextRequest searchTextRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchTextRequest, SearchTextResponse>().withOperationName("SearchText")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchTextRequest));
            CompletableFuture<SearchTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Suggest</code> operation finds addresses or place candidates based on incomplete or misspelled queries.
     * You then select the best query to submit based on the returned results.
     * </p>
     *
     * @param suggestRequest
     * @return A Java Future containing the result of the Suggest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You don't have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GeoPlacesException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GeoPlacesAsyncClient.Suggest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Suggest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SuggestResponse> suggest(SuggestRequest suggestRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(suggestRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suggestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Suggest");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SuggestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SuggestResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SuggestResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SuggestRequest, SuggestResponse>().withOperationName("Suggest")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new SuggestRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(suggestRequest));
            CompletableFuture<SuggestResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final GeoPlacesServiceClientConfiguration serviceClientConfiguration() {
        return new GeoPlacesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GeoPlacesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        GeoPlacesServiceClientConfigurationBuilder serviceConfigBuilder = new GeoPlacesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
