/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.geoplaces.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The place address.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Address implements SdkPojo, Serializable, ToCopyableBuilder<Address.Builder, Address> {
    private static final SdkField<String> LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Label")
            .getter(getter(Address::label)).setter(setter(Builder::label))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Label").build()).build();

    private static final SdkField<Country> COUNTRY_FIELD = SdkField.<Country> builder(MarshallingType.SDK_POJO)
            .memberName("Country").getter(getter(Address::country)).setter(setter(Builder::country))
            .constructor(Country::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Country").build()).build();

    private static final SdkField<Region> REGION_FIELD = SdkField.<Region> builder(MarshallingType.SDK_POJO).memberName("Region")
            .getter(getter(Address::region)).setter(setter(Builder::region)).constructor(Region::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<SubRegion> SUB_REGION_FIELD = SdkField.<SubRegion> builder(MarshallingType.SDK_POJO)
            .memberName("SubRegion").getter(getter(Address::subRegion)).setter(setter(Builder::subRegion))
            .constructor(SubRegion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubRegion").build()).build();

    private static final SdkField<String> LOCALITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Locality").getter(getter(Address::locality)).setter(setter(Builder::locality))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Locality").build()).build();

    private static final SdkField<String> DISTRICT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("District").getter(getter(Address::district)).setter(setter(Builder::district))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("District").build()).build();

    private static final SdkField<String> SUB_DISTRICT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubDistrict").getter(getter(Address::subDistrict)).setter(setter(Builder::subDistrict))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubDistrict").build()).build();

    private static final SdkField<String> POSTAL_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PostalCode").getter(getter(Address::postalCode)).setter(setter(Builder::postalCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PostalCode").build()).build();

    private static final SdkField<String> BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Block")
            .getter(getter(Address::block)).setter(setter(Builder::block))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Block").build()).build();

    private static final SdkField<String> SUB_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubBlock").getter(getter(Address::subBlock)).setter(setter(Builder::subBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubBlock").build()).build();

    private static final SdkField<List<String>> INTERSECTION_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Intersection")
            .getter(getter(Address::intersection))
            .setter(setter(Builder::intersection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Intersection").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STREET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Street")
            .getter(getter(Address::street)).setter(setter(Builder::street))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Street").build()).build();

    private static final SdkField<List<StreetComponents>> STREET_COMPONENTS_FIELD = SdkField
            .<List<StreetComponents>> builder(MarshallingType.LIST)
            .memberName("StreetComponents")
            .getter(getter(Address::streetComponents))
            .setter(setter(Builder::streetComponents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreetComponents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StreetComponents> builder(MarshallingType.SDK_POJO)
                                            .constructor(StreetComponents::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ADDRESS_NUMBER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AddressNumber").getter(getter(Address::addressNumber)).setter(setter(Builder::addressNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressNumber").build()).build();

    private static final SdkField<String> BUILDING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Building").getter(getter(Address::building)).setter(setter(Builder::building))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Building").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LABEL_FIELD, COUNTRY_FIELD,
            REGION_FIELD, SUB_REGION_FIELD, LOCALITY_FIELD, DISTRICT_FIELD, SUB_DISTRICT_FIELD, POSTAL_CODE_FIELD, BLOCK_FIELD,
            SUB_BLOCK_FIELD, INTERSECTION_FIELD, STREET_FIELD, STREET_COMPONENTS_FIELD, ADDRESS_NUMBER_FIELD, BUILDING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String label;

    private final Country country;

    private final Region region;

    private final SubRegion subRegion;

    private final String locality;

    private final String district;

    private final String subDistrict;

    private final String postalCode;

    private final String block;

    private final String subBlock;

    private final List<String> intersection;

    private final String street;

    private final List<StreetComponents> streetComponents;

    private final String addressNumber;

    private final String building;

    private Address(BuilderImpl builder) {
        this.label = builder.label;
        this.country = builder.country;
        this.region = builder.region;
        this.subRegion = builder.subRegion;
        this.locality = builder.locality;
        this.district = builder.district;
        this.subDistrict = builder.subDistrict;
        this.postalCode = builder.postalCode;
        this.block = builder.block;
        this.subBlock = builder.subBlock;
        this.intersection = builder.intersection;
        this.street = builder.street;
        this.streetComponents = builder.streetComponents;
        this.addressNumber = builder.addressNumber;
        this.building = builder.building;
    }

    /**
     * <p>
     * Assembled address value built out of the address components, according to the regional postal rules. This is the
     * correctly formatted address.
     * </p>
     * 
     * @return Assembled address value built out of the address components, according to the regional postal rules. This
     *         is the correctly formatted address.
     */
    public final String label() {
        return label;
    }

    /**
     * <p>
     * The country component of the address.
     * </p>
     * 
     * @return The country component of the address.
     */
    public final Country country() {
        return country;
    }

    /**
     * <p>
     * The region or state results should be present in.
     * </p>
     * <p>
     * Example: <code>North Rhine-Westphalia</code>.
     * </p>
     * 
     * @return The region or state results should be present in. </p>
     *         <p>
     *         Example: <code>North Rhine-Westphalia</code>.
     */
    public final Region region() {
        return region;
    }

    /**
     * <p>
     * The sub-region or county for which results should be present in.
     * </p>
     * 
     * @return The sub-region or county for which results should be present in.
     */
    public final SubRegion subRegion() {
        return subRegion;
    }

    /**
     * <p>
     * The locality or city of the address.
     * </p>
     * <p>
     * Example: <code>Vancouver</code>.
     * </p>
     * 
     * @return The locality or city of the address.</p>
     *         <p>
     *         Example: <code>Vancouver</code>.
     */
    public final String locality() {
        return locality;
    }

    /**
     * <p>
     * The district or division of a locality associated with this address.
     * </p>
     * 
     * @return The district or division of a locality associated with this address.
     */
    public final String district() {
        return district;
    }

    /**
     * <p>
     * A subdivision of a district.
     * </p>
     * <p>
     * Example: <code>Minden-Lübbecke</code>.
     * </p>
     * 
     * @return A subdivision of a district. </p>
     *         <p>
     *         Example: <code>Minden-Lübbecke</code>.
     */
    public final String subDistrict() {
        return subDistrict;
    }

    /**
     * <p>
     * An alphanumeric string included in a postal address to facilitate mail sorting, such as post code, postcode, or
     * ZIP code, for which the result should posses.
     * </p>
     * 
     * @return An alphanumeric string included in a postal address to facilitate mail sorting, such as post code,
     *         postcode, or ZIP code, for which the result should posses.
     */
    public final String postalCode() {
        return postalCode;
    }

    /**
     * <p>
     * Name of the block.
     * </p>
     * <p>
     * Example: <code>Sunny Mansion 203 block: 2 Chome</code>
     * </p>
     * 
     * @return Name of the block. </p>
     *         <p>
     *         Example: <code>Sunny Mansion 203 block: 2 Chome</code>
     */
    public final String block() {
        return block;
    }

    /**
     * <p>
     * Name of sub-block.
     * </p>
     * <p>
     * Example: <code>Sunny Mansion 203 sub-block: 4</code>
     * </p>
     * 
     * @return Name of sub-block. </p>
     *         <p>
     *         Example: <code>Sunny Mansion 203 sub-block: 4</code>
     */
    public final String subBlock() {
        return subBlock;
    }

    /**
     * For responses, this returns true if the service returned a value for the Intersection property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIntersection() {
        return intersection != null && !(intersection instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Name of the streets in the intersection.
     * </p>
     * <p>
     * Example: <code>["Friedrichstraße","Unter den Linden"]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIntersection} method.
     * </p>
     * 
     * @return Name of the streets in the intersection. </p>
     *         <p>
     *         Example: <code>["Friedrichstraße","Unter den Linden"]</code>
     */
    public final List<String> intersection() {
        return intersection;
    }

    /**
     * <p>
     * The name of the street results should be present in.
     * </p>
     * 
     * @return The name of the street results should be present in.
     */
    public final String street() {
        return street;
    }

    /**
     * For responses, this returns true if the service returned a value for the StreetComponents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStreetComponents() {
        return streetComponents != null && !(streetComponents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Components of the street.
     * </p>
     * <p>
     * Example: Younge from the "Younge street".
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStreetComponents} method.
     * </p>
     * 
     * @return Components of the street. </p>
     *         <p>
     *         Example: Younge from the "Younge street".
     */
    public final List<StreetComponents> streetComponents() {
        return streetComponents;
    }

    /**
     * <p>
     * The number that identifies an address within a street.
     * </p>
     * 
     * @return The number that identifies an address within a street.
     */
    public final String addressNumber() {
        return addressNumber;
    }

    /**
     * <p>
     * The name of the building at the address.
     * </p>
     * 
     * @return The name of the building at the address.
     */
    public final String building() {
        return building;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(label());
        hashCode = 31 * hashCode + Objects.hashCode(country());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(subRegion());
        hashCode = 31 * hashCode + Objects.hashCode(locality());
        hashCode = 31 * hashCode + Objects.hashCode(district());
        hashCode = 31 * hashCode + Objects.hashCode(subDistrict());
        hashCode = 31 * hashCode + Objects.hashCode(postalCode());
        hashCode = 31 * hashCode + Objects.hashCode(block());
        hashCode = 31 * hashCode + Objects.hashCode(subBlock());
        hashCode = 31 * hashCode + Objects.hashCode(hasIntersection() ? intersection() : null);
        hashCode = 31 * hashCode + Objects.hashCode(street());
        hashCode = 31 * hashCode + Objects.hashCode(hasStreetComponents() ? streetComponents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(addressNumber());
        hashCode = 31 * hashCode + Objects.hashCode(building());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Address)) {
            return false;
        }
        Address other = (Address) obj;
        return Objects.equals(label(), other.label()) && Objects.equals(country(), other.country())
                && Objects.equals(region(), other.region()) && Objects.equals(subRegion(), other.subRegion())
                && Objects.equals(locality(), other.locality()) && Objects.equals(district(), other.district())
                && Objects.equals(subDistrict(), other.subDistrict()) && Objects.equals(postalCode(), other.postalCode())
                && Objects.equals(block(), other.block()) && Objects.equals(subBlock(), other.subBlock())
                && hasIntersection() == other.hasIntersection() && Objects.equals(intersection(), other.intersection())
                && Objects.equals(street(), other.street()) && hasStreetComponents() == other.hasStreetComponents()
                && Objects.equals(streetComponents(), other.streetComponents())
                && Objects.equals(addressNumber(), other.addressNumber()) && Objects.equals(building(), other.building());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Address").add("Label", label()).add("Country", country()).add("Region", region())
                .add("SubRegion", subRegion()).add("Locality", locality()).add("District", district())
                .add("SubDistrict", subDistrict()).add("PostalCode", postalCode()).add("Block", block())
                .add("SubBlock", subBlock()).add("Intersection", hasIntersection() ? intersection() : null)
                .add("Street", street()).add("StreetComponents", hasStreetComponents() ? streetComponents() : null)
                .add("AddressNumber", addressNumber()).add("Building", building()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Label":
            return Optional.ofNullable(clazz.cast(label()));
        case "Country":
            return Optional.ofNullable(clazz.cast(country()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "SubRegion":
            return Optional.ofNullable(clazz.cast(subRegion()));
        case "Locality":
            return Optional.ofNullable(clazz.cast(locality()));
        case "District":
            return Optional.ofNullable(clazz.cast(district()));
        case "SubDistrict":
            return Optional.ofNullable(clazz.cast(subDistrict()));
        case "PostalCode":
            return Optional.ofNullable(clazz.cast(postalCode()));
        case "Block":
            return Optional.ofNullable(clazz.cast(block()));
        case "SubBlock":
            return Optional.ofNullable(clazz.cast(subBlock()));
        case "Intersection":
            return Optional.ofNullable(clazz.cast(intersection()));
        case "Street":
            return Optional.ofNullable(clazz.cast(street()));
        case "StreetComponents":
            return Optional.ofNullable(clazz.cast(streetComponents()));
        case "AddressNumber":
            return Optional.ofNullable(clazz.cast(addressNumber()));
        case "Building":
            return Optional.ofNullable(clazz.cast(building()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Label", LABEL_FIELD);
        map.put("Country", COUNTRY_FIELD);
        map.put("Region", REGION_FIELD);
        map.put("SubRegion", SUB_REGION_FIELD);
        map.put("Locality", LOCALITY_FIELD);
        map.put("District", DISTRICT_FIELD);
        map.put("SubDistrict", SUB_DISTRICT_FIELD);
        map.put("PostalCode", POSTAL_CODE_FIELD);
        map.put("Block", BLOCK_FIELD);
        map.put("SubBlock", SUB_BLOCK_FIELD);
        map.put("Intersection", INTERSECTION_FIELD);
        map.put("Street", STREET_FIELD);
        map.put("StreetComponents", STREET_COMPONENTS_FIELD);
        map.put("AddressNumber", ADDRESS_NUMBER_FIELD);
        map.put("Building", BUILDING_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Address, T> g) {
        return obj -> g.apply((Address) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Address> {
        /**
         * <p>
         * Assembled address value built out of the address components, according to the regional postal rules. This is
         * the correctly formatted address.
         * </p>
         * 
         * @param label
         *        Assembled address value built out of the address components, according to the regional postal rules.
         *        This is the correctly formatted address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder label(String label);

        /**
         * <p>
         * The country component of the address.
         * </p>
         * 
         * @param country
         *        The country component of the address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder country(Country country);

        /**
         * <p>
         * The country component of the address.
         * </p>
         * This is a convenience method that creates an instance of the {@link Country.Builder} avoiding the need to
         * create one manually via {@link Country#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Country.Builder#build()} is called immediately and its result is
         * passed to {@link #country(Country)}.
         * 
         * @param country
         *        a consumer that will call methods on {@link Country.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #country(Country)
         */
        default Builder country(Consumer<Country.Builder> country) {
            return country(Country.builder().applyMutation(country).build());
        }

        /**
         * <p>
         * The region or state results should be present in.
         * </p>
         * <p>
         * Example: <code>North Rhine-Westphalia</code>.
         * </p>
         * 
         * @param region
         *        The region or state results should be present in. </p>
         *        <p>
         *        Example: <code>North Rhine-Westphalia</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(Region region);

        /**
         * <p>
         * The region or state results should be present in.
         * </p>
         * <p>
         * Example: <code>North Rhine-Westphalia</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Region.Builder} avoiding the need to
         * create one manually via {@link Region#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Region.Builder#build()} is called immediately and its result is
         * passed to {@link #region(Region)}.
         * 
         * @param region
         *        a consumer that will call methods on {@link Region.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #region(Region)
         */
        default Builder region(Consumer<Region.Builder> region) {
            return region(Region.builder().applyMutation(region).build());
        }

        /**
         * <p>
         * The sub-region or county for which results should be present in.
         * </p>
         * 
         * @param subRegion
         *        The sub-region or county for which results should be present in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subRegion(SubRegion subRegion);

        /**
         * <p>
         * The sub-region or county for which results should be present in.
         * </p>
         * This is a convenience method that creates an instance of the {@link SubRegion.Builder} avoiding the need to
         * create one manually via {@link SubRegion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SubRegion.Builder#build()} is called immediately and its result
         * is passed to {@link #subRegion(SubRegion)}.
         * 
         * @param subRegion
         *        a consumer that will call methods on {@link SubRegion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subRegion(SubRegion)
         */
        default Builder subRegion(Consumer<SubRegion.Builder> subRegion) {
            return subRegion(SubRegion.builder().applyMutation(subRegion).build());
        }

        /**
         * <p>
         * The locality or city of the address.
         * </p>
         * <p>
         * Example: <code>Vancouver</code>.
         * </p>
         * 
         * @param locality
         *        The locality or city of the address.</p>
         *        <p>
         *        Example: <code>Vancouver</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locality(String locality);

        /**
         * <p>
         * The district or division of a locality associated with this address.
         * </p>
         * 
         * @param district
         *        The district or division of a locality associated with this address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder district(String district);

        /**
         * <p>
         * A subdivision of a district.
         * </p>
         * <p>
         * Example: <code>Minden-Lübbecke</code>.
         * </p>
         * 
         * @param subDistrict
         *        A subdivision of a district. </p>
         *        <p>
         *        Example: <code>Minden-Lübbecke</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subDistrict(String subDistrict);

        /**
         * <p>
         * An alphanumeric string included in a postal address to facilitate mail sorting, such as post code, postcode,
         * or ZIP code, for which the result should posses.
         * </p>
         * 
         * @param postalCode
         *        An alphanumeric string included in a postal address to facilitate mail sorting, such as post code,
         *        postcode, or ZIP code, for which the result should posses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postalCode(String postalCode);

        /**
         * <p>
         * Name of the block.
         * </p>
         * <p>
         * Example: <code>Sunny Mansion 203 block: 2 Chome</code>
         * </p>
         * 
         * @param block
         *        Name of the block. </p>
         *        <p>
         *        Example: <code>Sunny Mansion 203 block: 2 Chome</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder block(String block);

        /**
         * <p>
         * Name of sub-block.
         * </p>
         * <p>
         * Example: <code>Sunny Mansion 203 sub-block: 4</code>
         * </p>
         * 
         * @param subBlock
         *        Name of sub-block. </p>
         *        <p>
         *        Example: <code>Sunny Mansion 203 sub-block: 4</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subBlock(String subBlock);

        /**
         * <p>
         * Name of the streets in the intersection.
         * </p>
         * <p>
         * Example: <code>["Friedrichstraße","Unter den Linden"]</code>
         * </p>
         * 
         * @param intersection
         *        Name of the streets in the intersection. </p>
         *        <p>
         *        Example: <code>["Friedrichstraße","Unter den Linden"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intersection(Collection<String> intersection);

        /**
         * <p>
         * Name of the streets in the intersection.
         * </p>
         * <p>
         * Example: <code>["Friedrichstraße","Unter den Linden"]</code>
         * </p>
         * 
         * @param intersection
         *        Name of the streets in the intersection. </p>
         *        <p>
         *        Example: <code>["Friedrichstraße","Unter den Linden"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intersection(String... intersection);

        /**
         * <p>
         * The name of the street results should be present in.
         * </p>
         * 
         * @param street
         *        The name of the street results should be present in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder street(String street);

        /**
         * <p>
         * Components of the street.
         * </p>
         * <p>
         * Example: Younge from the "Younge street".
         * </p>
         * 
         * @param streetComponents
         *        Components of the street. </p>
         *        <p>
         *        Example: Younge from the "Younge street".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streetComponents(Collection<StreetComponents> streetComponents);

        /**
         * <p>
         * Components of the street.
         * </p>
         * <p>
         * Example: Younge from the "Younge street".
         * </p>
         * 
         * @param streetComponents
         *        Components of the street. </p>
         *        <p>
         *        Example: Younge from the "Younge street".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streetComponents(StreetComponents... streetComponents);

        /**
         * <p>
         * Components of the street.
         * </p>
         * <p>
         * Example: Younge from the "Younge street".
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.geoplaces.model.StreetComponents.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.geoplaces.model.StreetComponents#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.geoplaces.model.StreetComponents.Builder#build()} is called
         * immediately and its result is passed to {@link #streetComponents(List<StreetComponents>)}.
         * 
         * @param streetComponents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.geoplaces.model.StreetComponents.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #streetComponents(java.util.Collection<StreetComponents>)
         */
        Builder streetComponents(Consumer<StreetComponents.Builder>... streetComponents);

        /**
         * <p>
         * The number that identifies an address within a street.
         * </p>
         * 
         * @param addressNumber
         *        The number that identifies an address within a street.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addressNumber(String addressNumber);

        /**
         * <p>
         * The name of the building at the address.
         * </p>
         * 
         * @param building
         *        The name of the building at the address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder building(String building);
    }

    static final class BuilderImpl implements Builder {
        private String label;

        private Country country;

        private Region region;

        private SubRegion subRegion;

        private String locality;

        private String district;

        private String subDistrict;

        private String postalCode;

        private String block;

        private String subBlock;

        private List<String> intersection = DefaultSdkAutoConstructList.getInstance();

        private String street;

        private List<StreetComponents> streetComponents = DefaultSdkAutoConstructList.getInstance();

        private String addressNumber;

        private String building;

        private BuilderImpl() {
        }

        private BuilderImpl(Address model) {
            label(model.label);
            country(model.country);
            region(model.region);
            subRegion(model.subRegion);
            locality(model.locality);
            district(model.district);
            subDistrict(model.subDistrict);
            postalCode(model.postalCode);
            block(model.block);
            subBlock(model.subBlock);
            intersection(model.intersection);
            street(model.street);
            streetComponents(model.streetComponents);
            addressNumber(model.addressNumber);
            building(model.building);
        }

        public final String getLabel() {
            return label;
        }

        public final void setLabel(String label) {
            this.label = label;
        }

        @Override
        public final Builder label(String label) {
            this.label = label;
            return this;
        }

        public final Country.Builder getCountry() {
            return country != null ? country.toBuilder() : null;
        }

        public final void setCountry(Country.BuilderImpl country) {
            this.country = country != null ? country.build() : null;
        }

        @Override
        public final Builder country(Country country) {
            this.country = country;
            return this;
        }

        public final Region.Builder getRegion() {
            return region != null ? region.toBuilder() : null;
        }

        public final void setRegion(Region.BuilderImpl region) {
            this.region = region != null ? region.build() : null;
        }

        @Override
        public final Builder region(Region region) {
            this.region = region;
            return this;
        }

        public final SubRegion.Builder getSubRegion() {
            return subRegion != null ? subRegion.toBuilder() : null;
        }

        public final void setSubRegion(SubRegion.BuilderImpl subRegion) {
            this.subRegion = subRegion != null ? subRegion.build() : null;
        }

        @Override
        public final Builder subRegion(SubRegion subRegion) {
            this.subRegion = subRegion;
            return this;
        }

        public final String getLocality() {
            return locality;
        }

        public final void setLocality(String locality) {
            this.locality = locality;
        }

        @Override
        public final Builder locality(String locality) {
            this.locality = locality;
            return this;
        }

        public final String getDistrict() {
            return district;
        }

        public final void setDistrict(String district) {
            this.district = district;
        }

        @Override
        public final Builder district(String district) {
            this.district = district;
            return this;
        }

        public final String getSubDistrict() {
            return subDistrict;
        }

        public final void setSubDistrict(String subDistrict) {
            this.subDistrict = subDistrict;
        }

        @Override
        public final Builder subDistrict(String subDistrict) {
            this.subDistrict = subDistrict;
            return this;
        }

        public final String getPostalCode() {
            return postalCode;
        }

        public final void setPostalCode(String postalCode) {
            this.postalCode = postalCode;
        }

        @Override
        public final Builder postalCode(String postalCode) {
            this.postalCode = postalCode;
            return this;
        }

        public final String getBlock() {
            return block;
        }

        public final void setBlock(String block) {
            this.block = block;
        }

        @Override
        public final Builder block(String block) {
            this.block = block;
            return this;
        }

        public final String getSubBlock() {
            return subBlock;
        }

        public final void setSubBlock(String subBlock) {
            this.subBlock = subBlock;
        }

        @Override
        public final Builder subBlock(String subBlock) {
            this.subBlock = subBlock;
            return this;
        }

        public final Collection<String> getIntersection() {
            if (intersection instanceof SdkAutoConstructList) {
                return null;
            }
            return intersection;
        }

        public final void setIntersection(Collection<String> intersection) {
            this.intersection = IntersectionListCopier.copy(intersection);
        }

        @Override
        public final Builder intersection(Collection<String> intersection) {
            this.intersection = IntersectionListCopier.copy(intersection);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder intersection(String... intersection) {
            intersection(Arrays.asList(intersection));
            return this;
        }

        public final String getStreet() {
            return street;
        }

        public final void setStreet(String street) {
            this.street = street;
        }

        @Override
        public final Builder street(String street) {
            this.street = street;
            return this;
        }

        public final List<StreetComponents.Builder> getStreetComponents() {
            List<StreetComponents.Builder> result = StreetComponentsListCopier.copyToBuilder(this.streetComponents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStreetComponents(Collection<StreetComponents.BuilderImpl> streetComponents) {
            this.streetComponents = StreetComponentsListCopier.copyFromBuilder(streetComponents);
        }

        @Override
        public final Builder streetComponents(Collection<StreetComponents> streetComponents) {
            this.streetComponents = StreetComponentsListCopier.copy(streetComponents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder streetComponents(StreetComponents... streetComponents) {
            streetComponents(Arrays.asList(streetComponents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder streetComponents(Consumer<StreetComponents.Builder>... streetComponents) {
            streetComponents(Stream.of(streetComponents).map(c -> StreetComponents.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getAddressNumber() {
            return addressNumber;
        }

        public final void setAddressNumber(String addressNumber) {
            this.addressNumber = addressNumber;
        }

        @Override
        public final Builder addressNumber(String addressNumber) {
            this.addressNumber = addressNumber;
            return this;
        }

        public final String getBuilding() {
            return building;
        }

        public final void setBuilding(String building) {
            this.building = building;
        }

        @Override
        public final Builder building(String building) {
            this.building = building;
            return this;
        }

        @Override
        public Address build() {
            return new Address(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
