/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.geoplaces;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.geoplaces.internal.GeoPlacesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.geoplaces.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.geoplaces.model.AccessDeniedException;
import software.amazon.awssdk.services.geoplaces.model.AutocompleteRequest;
import software.amazon.awssdk.services.geoplaces.model.AutocompleteResponse;
import software.amazon.awssdk.services.geoplaces.model.GeoPlacesException;
import software.amazon.awssdk.services.geoplaces.model.GeocodeRequest;
import software.amazon.awssdk.services.geoplaces.model.GeocodeResponse;
import software.amazon.awssdk.services.geoplaces.model.GetPlaceRequest;
import software.amazon.awssdk.services.geoplaces.model.GetPlaceResponse;
import software.amazon.awssdk.services.geoplaces.model.InternalServerException;
import software.amazon.awssdk.services.geoplaces.model.ReverseGeocodeRequest;
import software.amazon.awssdk.services.geoplaces.model.ReverseGeocodeResponse;
import software.amazon.awssdk.services.geoplaces.model.SearchNearbyRequest;
import software.amazon.awssdk.services.geoplaces.model.SearchNearbyResponse;
import software.amazon.awssdk.services.geoplaces.model.SearchTextRequest;
import software.amazon.awssdk.services.geoplaces.model.SearchTextResponse;
import software.amazon.awssdk.services.geoplaces.model.SuggestRequest;
import software.amazon.awssdk.services.geoplaces.model.SuggestResponse;
import software.amazon.awssdk.services.geoplaces.model.ThrottlingException;
import software.amazon.awssdk.services.geoplaces.model.ValidationException;
import software.amazon.awssdk.services.geoplaces.transform.AutocompleteRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.GeocodeRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.GetPlaceRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.ReverseGeocodeRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SearchNearbyRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SearchTextRequestMarshaller;
import software.amazon.awssdk.services.geoplaces.transform.SuggestRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GeoPlacesClient}.
 *
 * @see GeoPlacesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGeoPlacesClient implements GeoPlacesClient {
    private static final Logger log = Logger.loggerFor(DefaultGeoPlacesClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGeoPlacesClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Geo_Places" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * <code>Autocomplete</code> completes potential places and addresses as the user types, based on the partial input.
     * The API enhances the efficiency and accuracy of address by completing query based on a few entered keystrokes. It
     * helps you by completing partial queries with valid address completion. Also, the API supports the filtering of
     * results based on geographic location, country, or specific place types, and can be tailored using optional
     * parameters like language and political views.
     * </p>
     *
     * @param autocompleteRequest
     * @return Result of the Autocomplete operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.Autocomplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Autocomplete" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AutocompleteResponse autocomplete(AutocompleteRequest autocompleteRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AutocompleteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AutocompleteResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(autocompleteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, autocompleteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Autocomplete");

            return clientHandler.execute(new ClientExecutionParams<AutocompleteRequest, AutocompleteResponse>()
                    .withOperationName("Autocomplete").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(autocompleteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AutocompleteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>Geocode</code> converts a textual address or place into geographic coordinates. You can obtain geographic
     * coordinates, address component, and other related information. It supports flexible queries, including free-form
     * text or structured queries with components like street names, postal codes, and regions. The Geocode API can also
     * provide additional features such as time zone information and the inclusion of political views.
     * </p>
     *
     * @param geocodeRequest
     * @return Result of the Geocode operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.Geocode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Geocode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GeocodeResponse geocode(GeocodeRequest geocodeRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GeocodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GeocodeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(geocodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, geocodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Geocode");

            return clientHandler.execute(new ClientExecutionParams<GeocodeRequest, GeocodeResponse>()
                    .withOperationName("Geocode").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(geocodeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GeocodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>GetPlace</code> finds a place by its unique ID. A <code>PlaceId</code> is returned by other place
     * operations.
     * </p>
     *
     * @param getPlaceRequest
     * @return Result of the GetPlace operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.GetPlace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/GetPlace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPlaceResponse getPlace(GetPlaceRequest getPlaceRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPlaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPlaceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPlaceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlace");

            return clientHandler.execute(new ClientExecutionParams<GetPlaceRequest, GetPlaceResponse>()
                    .withOperationName("GetPlace").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getPlaceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPlaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>ReverseGeocode</code> converts geographic coordinates into a human-readable address or place. You can
     * obtain address component, and other related information such as place type, category, street information. The
     * Reverse Geocode API supports filtering to on place type so that you can refine result based on your need. Also,
     * The Reverse Geocode API can also provide additional features such as time zone information and the inclusion of
     * political views.
     * </p>
     *
     * @param reverseGeocodeRequest
     * @return Result of the ReverseGeocode operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.ReverseGeocode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/ReverseGeocode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ReverseGeocodeResponse reverseGeocode(ReverseGeocodeRequest reverseGeocodeRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReverseGeocodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReverseGeocodeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(reverseGeocodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reverseGeocodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReverseGeocode");

            return clientHandler.execute(new ClientExecutionParams<ReverseGeocodeRequest, ReverseGeocodeResponse>()
                    .withOperationName("ReverseGeocode").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(reverseGeocodeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReverseGeocodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>SearchNearby</code> queries for points of interest within a radius from a central coordinates, returning
     * place results with optional filters such as categories, business chains, food types and more. The API returns
     * details such as a place name, address, phone, category, food type, contact, opening hours. Also, the API can
     * return phonemes, time zones and more based on requested parameters.
     * </p>
     *
     * @param searchNearbyRequest
     * @return Result of the SearchNearby operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.SearchNearby
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/SearchNearby" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchNearbyResponse searchNearby(SearchNearbyRequest searchNearbyRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchNearbyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchNearbyResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchNearbyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchNearbyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchNearby");

            return clientHandler.execute(new ClientExecutionParams<SearchNearbyRequest, SearchNearbyResponse>()
                    .withOperationName("SearchNearby").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(searchNearbyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchNearbyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>SearchText</code> searches for geocode and place information. You can then complete a follow-up query
     * suggested from the <code>Suggest</code> API via a query id.
     * </p>
     *
     * @param searchTextRequest
     * @return Result of the SearchText operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.SearchText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/SearchText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SearchTextResponse searchText(SearchTextRequest searchTextRequest) throws InternalServerException,
            AccessDeniedException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SearchTextResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchText");

            return clientHandler.execute(new ClientExecutionParams<SearchTextRequest, SearchTextResponse>()
                    .withOperationName("SearchText").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(searchTextRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SearchTextRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>Suggest</code> provides intelligent predictions or recommendations based on the user's input or context,
     * such as relevant places, points of interest, query terms or search category. It is designed to help users find
     * places or point of interests candidates or identify a follow on query based on incomplete or misspelled queries.
     * It returns a list of possible matches or refinements that can be used to formulate a more accurate query. Users
     * can select the most appropriate suggestion and use it for further searching. The API provides options for
     * filtering results by location and other attributes, and allows for additional features like phonemes and
     * timezones. The response includes refined query terms and detailed place information.
     * </p>
     *
     * @param suggestRequest
     * @return Result of the Suggest operation returned by the service.
     * @throws InternalServerException
     *         The request processing has failed because of an unknown error, exception or failure.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GeoPlacesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GeoPlacesClient.Suggest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/geo-places-2020-11-19/Suggest" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SuggestResponse suggest(SuggestRequest suggestRequest) throws InternalServerException, AccessDeniedException,
            ValidationException, ThrottlingException, AwsServiceException, SdkClientException, GeoPlacesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SuggestResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SuggestResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(suggestRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suggestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Geo Places");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Suggest");

            return clientHandler.execute(new ClientExecutionParams<SuggestRequest, SuggestResponse>()
                    .withOperationName("Suggest").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(suggestRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SuggestRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        GeoPlacesServiceClientConfigurationBuilder serviceConfigBuilder = new GeoPlacesServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(GeoPlacesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final GeoPlacesServiceClientConfiguration serviceClientConfiguration() {
        return new GeoPlacesServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
