/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculateRouteMatrixRequest extends GeoRoutesRequest implements
        ToCopyableBuilder<CalculateRouteMatrixRequest.Builder, CalculateRouteMatrixRequest> {
    private static final SdkField<RouteMatrixAllowOptions> ALLOW_FIELD = SdkField
            .<RouteMatrixAllowOptions> builder(MarshallingType.SDK_POJO).memberName("Allow")
            .getter(getter(CalculateRouteMatrixRequest::allow)).setter(setter(Builder::allow))
            .constructor(RouteMatrixAllowOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Allow").build()).build();

    private static final SdkField<RouteMatrixAvoidanceOptions> AVOID_FIELD = SdkField
            .<RouteMatrixAvoidanceOptions> builder(MarshallingType.SDK_POJO).memberName("Avoid")
            .getter(getter(CalculateRouteMatrixRequest::avoid)).setter(setter(Builder::avoid))
            .constructor(RouteMatrixAvoidanceOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Avoid").build()).build();

    private static final SdkField<Boolean> DEPART_NOW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DepartNow").getter(getter(CalculateRouteMatrixRequest::departNow)).setter(setter(Builder::departNow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartNow").build()).build();

    private static final SdkField<String> DEPARTURE_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DepartureTime").getter(getter(CalculateRouteMatrixRequest::departureTime))
            .setter(setter(Builder::departureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartureTime").build()).build();

    private static final SdkField<List<RouteMatrixDestination>> DESTINATIONS_FIELD = SdkField
            .<List<RouteMatrixDestination>> builder(MarshallingType.LIST)
            .memberName("Destinations")
            .getter(getter(CalculateRouteMatrixRequest::destinations))
            .setter(setter(Builder::destinations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RouteMatrixDestination> builder(MarshallingType.SDK_POJO)
                                            .constructor(RouteMatrixDestination::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RouteMatrixExclusionOptions> EXCLUDE_FIELD = SdkField
            .<RouteMatrixExclusionOptions> builder(MarshallingType.SDK_POJO).memberName("Exclude")
            .getter(getter(CalculateRouteMatrixRequest::exclude)).setter(setter(Builder::exclude))
            .constructor(RouteMatrixExclusionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exclude").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(CalculateRouteMatrixRequest::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("key").build()).build();

    private static final SdkField<String> OPTIMIZE_ROUTING_FOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptimizeRoutingFor").getter(getter(CalculateRouteMatrixRequest::optimizeRoutingForAsString))
            .setter(setter(Builder::optimizeRoutingFor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptimizeRoutingFor").build())
            .build();

    private static final SdkField<List<RouteMatrixOrigin>> ORIGINS_FIELD = SdkField
            .<List<RouteMatrixOrigin>> builder(MarshallingType.LIST)
            .memberName("Origins")
            .getter(getter(CalculateRouteMatrixRequest::origins))
            .setter(setter(Builder::origins))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Origins").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RouteMatrixOrigin> builder(MarshallingType.SDK_POJO)
                                            .constructor(RouteMatrixOrigin::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RouteMatrixBoundary> ROUTING_BOUNDARY_FIELD = SdkField
            .<RouteMatrixBoundary> builder(MarshallingType.SDK_POJO).memberName("RoutingBoundary")
            .getter(getter(CalculateRouteMatrixRequest::routingBoundary)).setter(setter(Builder::routingBoundary))
            .constructor(RouteMatrixBoundary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoutingBoundary").build()).build();

    private static final SdkField<RouteMatrixTrafficOptions> TRAFFIC_FIELD = SdkField
            .<RouteMatrixTrafficOptions> builder(MarshallingType.SDK_POJO).memberName("Traffic")
            .getter(getter(CalculateRouteMatrixRequest::traffic)).setter(setter(Builder::traffic))
            .constructor(RouteMatrixTrafficOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Traffic").build()).build();

    private static final SdkField<String> TRAVEL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TravelMode").getter(getter(CalculateRouteMatrixRequest::travelModeAsString))
            .setter(setter(Builder::travelMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelMode").build()).build();

    private static final SdkField<RouteMatrixTravelModeOptions> TRAVEL_MODE_OPTIONS_FIELD = SdkField
            .<RouteMatrixTravelModeOptions> builder(MarshallingType.SDK_POJO).memberName("TravelModeOptions")
            .getter(getter(CalculateRouteMatrixRequest::travelModeOptions)).setter(setter(Builder::travelModeOptions))
            .constructor(RouteMatrixTravelModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelModeOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOW_FIELD, AVOID_FIELD,
            DEPART_NOW_FIELD, DEPARTURE_TIME_FIELD, DESTINATIONS_FIELD, EXCLUDE_FIELD, KEY_FIELD, OPTIMIZE_ROUTING_FOR_FIELD,
            ORIGINS_FIELD, ROUTING_BOUNDARY_FIELD, TRAFFIC_FIELD, TRAVEL_MODE_FIELD, TRAVEL_MODE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final RouteMatrixAllowOptions allow;

    private final RouteMatrixAvoidanceOptions avoid;

    private final Boolean departNow;

    private final String departureTime;

    private final List<RouteMatrixDestination> destinations;

    private final RouteMatrixExclusionOptions exclude;

    private final String key;

    private final String optimizeRoutingFor;

    private final List<RouteMatrixOrigin> origins;

    private final RouteMatrixBoundary routingBoundary;

    private final RouteMatrixTrafficOptions traffic;

    private final String travelMode;

    private final RouteMatrixTravelModeOptions travelModeOptions;

    private CalculateRouteMatrixRequest(BuilderImpl builder) {
        super(builder);
        this.allow = builder.allow;
        this.avoid = builder.avoid;
        this.departNow = builder.departNow;
        this.departureTime = builder.departureTime;
        this.destinations = builder.destinations;
        this.exclude = builder.exclude;
        this.key = builder.key;
        this.optimizeRoutingFor = builder.optimizeRoutingFor;
        this.origins = builder.origins;
        this.routingBoundary = builder.routingBoundary;
        this.traffic = builder.traffic;
        this.travelMode = builder.travelMode;
        this.travelModeOptions = builder.travelModeOptions;
    }

    /**
     * <p>
     * Features that are allowed while calculating a route.
     * </p>
     * 
     * @return Features that are allowed while calculating a route.
     */
    public final RouteMatrixAllowOptions allow() {
        return allow;
    }

    /**
     * <p>
     * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't be
     * satisfied for a particular case, it violates the avoidance and the returned response produces a notice for the
     * violation.
     * </p>
     * 
     * @return Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance
     *         can't be satisfied for a particular case, it violates the avoidance and the returned response produces a
     *         notice for the violation.
     */
    public final RouteMatrixAvoidanceOptions avoid() {
        return avoid;
    }

    /**
     * <p>
     * Uses the current time as the time of departure.
     * </p>
     * 
     * @return Uses the current time as the time of departure.
     */
    public final Boolean departNow() {
        return departNow;
    }

    /**
     * <p>
     * Time of departure from thr origin.
     * </p>
     * <p>
     * Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     * </p>
     * <p>
     * Examples:
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24Z</code>
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24+02:00</code>
     * </p>
     * 
     * @return Time of departure from thr origin.</p>
     *         <p>
     *         Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     *         </p>
     *         <p>
     *         Examples:
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24Z</code>
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24+02:00</code>
     */
    public final String departureTime() {
        return departureTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Destinations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinations() {
        return destinations != null && !(destinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of destinations for the route.
     * </p>
     * <note>
     * <p>
     * Route calculations are billed for each origin and destination pair. If you use a large matrix of origins and
     * destinations, your costs will increase accordingly. See <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's pricing
     * page</a> for more information.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinations} method.
     * </p>
     * 
     * @return List of destinations for the route.</p> <note>
     *         <p>
     *         Route calculations are billed for each origin and destination pair. If you use a large matrix of origins
     *         and destinations, your costs will increase accordingly. See <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
     *         pricing page</a> for more information.
     *         </p>
     */
    public final List<RouteMatrixDestination> destinations() {
        return destinations;
    }

    /**
     * <p>
     * Features to be strictly excluded while calculating the route.
     * </p>
     * 
     * @return Features to be strictly excluded while calculating the route.
     */
    public final RouteMatrixExclusionOptions exclude() {
        return exclude;
    }

    /**
     * <p>
     * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be provided
     * when making a request.
     * </p>
     * 
     * @return Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
     *         provided when making a request.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Specifies the optimization criteria for calculating a route.
     * </p>
     * <p>
     * Default Value: <code>FastestRoute</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeRoutingFor} will return {@link RoutingObjective#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizeRoutingForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for calculating a route.</p>
     *         <p>
     *         Default Value: <code>FastestRoute</code>
     * @see RoutingObjective
     */
    public final RoutingObjective optimizeRoutingFor() {
        return RoutingObjective.fromValue(optimizeRoutingFor);
    }

    /**
     * <p>
     * Specifies the optimization criteria for calculating a route.
     * </p>
     * <p>
     * Default Value: <code>FastestRoute</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeRoutingFor} will return {@link RoutingObjective#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizeRoutingForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for calculating a route.</p>
     *         <p>
     *         Default Value: <code>FastestRoute</code>
     * @see RoutingObjective
     */
    public final String optimizeRoutingForAsString() {
        return optimizeRoutingFor;
    }

    /**
     * For responses, this returns true if the service returned a value for the Origins property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOrigins() {
        return origins != null && !(origins instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The position in longitude and latitude for the origin.
     * </p>
     * <note>
     * <p>
     * Route calculations are billed for each origin and destination pair. Using a large amount of Origins in a request
     * can lead you to incur unexpected charges. See <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's pricing
     * page</a> for more information.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrigins} method.
     * </p>
     * 
     * @return The position in longitude and latitude for the origin.</p> <note>
     *         <p>
     *         Route calculations are billed for each origin and destination pair. Using a large amount of Origins in a
     *         request can lead you to incur unexpected charges. See <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
     *         pricing page</a> for more information.
     *         </p>
     */
    public final List<RouteMatrixOrigin> origins() {
        return origins;
    }

    /**
     * <p>
     * Boundary within which the matrix is to be calculated. All data, origins and destinations outside the boundary are
     * considered invalid.
     * </p>
     * <note>
     * <p>
     * When request routing boundary was set as AutoCircle, the response routing boundary will return Circle derived
     * from the AutoCircle settings.
     * </p>
     * </note>
     * 
     * @return Boundary within which the matrix is to be calculated. All data, origins and destinations outside the
     *         boundary are considered invalid.</p> <note>
     *         <p>
     *         When request routing boundary was set as AutoCircle, the response routing boundary will return Circle
     *         derived from the AutoCircle settings.
     *         </p>
     */
    public final RouteMatrixBoundary routingBoundary() {
        return routingBoundary;
    }

    /**
     * <p>
     * Traffic related options.
     * </p>
     * 
     * @return Traffic related options.
     */
    public final RouteMatrixTrafficOptions traffic() {
        return traffic;
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link RouteMatrixTravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see RouteMatrixTravelMode
     */
    public final RouteMatrixTravelMode travelMode() {
        return RouteMatrixTravelMode.fromValue(travelMode);
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link RouteMatrixTravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see RouteMatrixTravelMode
     */
    public final String travelModeAsString() {
        return travelMode;
    }

    /**
     * <p>
     * Travel mode related options for the provided travel mode.
     * </p>
     * 
     * @return Travel mode related options for the provided travel mode.
     */
    public final RouteMatrixTravelModeOptions travelModeOptions() {
        return travelModeOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(allow());
        hashCode = 31 * hashCode + Objects.hashCode(avoid());
        hashCode = 31 * hashCode + Objects.hashCode(departNow());
        hashCode = 31 * hashCode + Objects.hashCode(departureTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinations() ? destinations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(exclude());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(optimizeRoutingForAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOrigins() ? origins() : null);
        hashCode = 31 * hashCode + Objects.hashCode(routingBoundary());
        hashCode = 31 * hashCode + Objects.hashCode(traffic());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculateRouteMatrixRequest)) {
            return false;
        }
        CalculateRouteMatrixRequest other = (CalculateRouteMatrixRequest) obj;
        return Objects.equals(allow(), other.allow()) && Objects.equals(avoid(), other.avoid())
                && Objects.equals(departNow(), other.departNow()) && Objects.equals(departureTime(), other.departureTime())
                && hasDestinations() == other.hasDestinations() && Objects.equals(destinations(), other.destinations())
                && Objects.equals(exclude(), other.exclude()) && Objects.equals(key(), other.key())
                && Objects.equals(optimizeRoutingForAsString(), other.optimizeRoutingForAsString())
                && hasOrigins() == other.hasOrigins() && Objects.equals(origins(), other.origins())
                && Objects.equals(routingBoundary(), other.routingBoundary()) && Objects.equals(traffic(), other.traffic())
                && Objects.equals(travelModeAsString(), other.travelModeAsString())
                && Objects.equals(travelModeOptions(), other.travelModeOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculateRouteMatrixRequest")
                .add("Allow", allow() == null ? null : "*** Sensitive Data Redacted ***").add("Avoid", avoid())
                .add("DepartNow", departNow() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DepartureTime", departureTime() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Destinations", hasDestinations() ? destinations() : null).add("Exclude", exclude())
                .add("Key", key() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OptimizeRoutingFor", optimizeRoutingForAsString()).add("Origins", hasOrigins() ? origins() : null)
                .add("RoutingBoundary", routingBoundary()).add("Traffic", traffic()).add("TravelMode", travelModeAsString())
                .add("TravelModeOptions", travelModeOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Allow":
            return Optional.ofNullable(clazz.cast(allow()));
        case "Avoid":
            return Optional.ofNullable(clazz.cast(avoid()));
        case "DepartNow":
            return Optional.ofNullable(clazz.cast(departNow()));
        case "DepartureTime":
            return Optional.ofNullable(clazz.cast(departureTime()));
        case "Destinations":
            return Optional.ofNullable(clazz.cast(destinations()));
        case "Exclude":
            return Optional.ofNullable(clazz.cast(exclude()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "OptimizeRoutingFor":
            return Optional.ofNullable(clazz.cast(optimizeRoutingForAsString()));
        case "Origins":
            return Optional.ofNullable(clazz.cast(origins()));
        case "RoutingBoundary":
            return Optional.ofNullable(clazz.cast(routingBoundary()));
        case "Traffic":
            return Optional.ofNullable(clazz.cast(traffic()));
        case "TravelMode":
            return Optional.ofNullable(clazz.cast(travelModeAsString()));
        case "TravelModeOptions":
            return Optional.ofNullable(clazz.cast(travelModeOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Allow", ALLOW_FIELD);
        map.put("Avoid", AVOID_FIELD);
        map.put("DepartNow", DEPART_NOW_FIELD);
        map.put("DepartureTime", DEPARTURE_TIME_FIELD);
        map.put("Destinations", DESTINATIONS_FIELD);
        map.put("Exclude", EXCLUDE_FIELD);
        map.put("key", KEY_FIELD);
        map.put("OptimizeRoutingFor", OPTIMIZE_ROUTING_FOR_FIELD);
        map.put("Origins", ORIGINS_FIELD);
        map.put("RoutingBoundary", ROUTING_BOUNDARY_FIELD);
        map.put("Traffic", TRAFFIC_FIELD);
        map.put("TravelMode", TRAVEL_MODE_FIELD);
        map.put("TravelModeOptions", TRAVEL_MODE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CalculateRouteMatrixRequest, T> g) {
        return obj -> g.apply((CalculateRouteMatrixRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GeoRoutesRequest.Builder, SdkPojo, CopyableBuilder<Builder, CalculateRouteMatrixRequest> {
        /**
         * <p>
         * Features that are allowed while calculating a route.
         * </p>
         * 
         * @param allow
         *        Features that are allowed while calculating a route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allow(RouteMatrixAllowOptions allow);

        /**
         * <p>
         * Features that are allowed while calculating a route.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixAllowOptions.Builder} avoiding
         * the need to create one manually via {@link RouteMatrixAllowOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixAllowOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #allow(RouteMatrixAllowOptions)}.
         * 
         * @param allow
         *        a consumer that will call methods on {@link RouteMatrixAllowOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allow(RouteMatrixAllowOptions)
         */
        default Builder allow(Consumer<RouteMatrixAllowOptions.Builder> allow) {
            return allow(RouteMatrixAllowOptions.builder().applyMutation(allow).build());
        }

        /**
         * <p>
         * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't
         * be satisfied for a particular case, it violates the avoidance and the returned response produces a notice for
         * the violation.
         * </p>
         * 
         * @param avoid
         *        Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an
         *        avoidance can't be satisfied for a particular case, it violates the avoidance and the returned
         *        response produces a notice for the violation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoid(RouteMatrixAvoidanceOptions avoid);

        /**
         * <p>
         * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't
         * be satisfied for a particular case, it violates the avoidance and the returned response produces a notice for
         * the violation.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixAvoidanceOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixAvoidanceOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixAvoidanceOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #avoid(RouteMatrixAvoidanceOptions)}.
         * 
         * @param avoid
         *        a consumer that will call methods on {@link RouteMatrixAvoidanceOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #avoid(RouteMatrixAvoidanceOptions)
         */
        default Builder avoid(Consumer<RouteMatrixAvoidanceOptions.Builder> avoid) {
            return avoid(RouteMatrixAvoidanceOptions.builder().applyMutation(avoid).build());
        }

        /**
         * <p>
         * Uses the current time as the time of departure.
         * </p>
         * 
         * @param departNow
         *        Uses the current time as the time of departure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departNow(Boolean departNow);

        /**
         * <p>
         * Time of departure from thr origin.
         * </p>
         * <p>
         * Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         * </p>
         * <p>
         * Examples:
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24Z</code>
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24+02:00</code>
         * </p>
         * 
         * @param departureTime
         *        Time of departure from thr origin.</p>
         *        <p>
         *        Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         *        </p>
         *        <p>
         *        Examples:
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24Z</code>
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24+02:00</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departureTime(String departureTime);

        /**
         * <p>
         * List of destinations for the route.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. If you use a large matrix of origins and
         * destinations, your costs will increase accordingly. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note>
         * 
         * @param destinations
         *        List of destinations for the route.</p> <note>
         *        <p>
         *        Route calculations are billed for each origin and destination pair. If you use a large matrix of
         *        origins and destinations, your costs will increase accordingly. See <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon
         *        Location's pricing page</a> for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(Collection<RouteMatrixDestination> destinations);

        /**
         * <p>
         * List of destinations for the route.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. If you use a large matrix of origins and
         * destinations, your costs will increase accordingly. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note>
         * 
         * @param destinations
         *        List of destinations for the route.</p> <note>
         *        <p>
         *        Route calculations are billed for each origin and destination pair. If you use a large matrix of
         *        origins and destinations, your costs will increase accordingly. See <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon
         *        Location's pricing page</a> for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinations(RouteMatrixDestination... destinations);

        /**
         * <p>
         * List of destinations for the route.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. If you use a large matrix of origins and
         * destinations, your costs will increase accordingly. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixDestination.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixDestination#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixDestination.Builder#build()} is called
         * immediately and its result is passed to {@link #destinations(List<RouteMatrixDestination>)}.
         * 
         * @param destinations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixDestination.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinations(java.util.Collection<RouteMatrixDestination>)
         */
        Builder destinations(Consumer<RouteMatrixDestination.Builder>... destinations);

        /**
         * <p>
         * Features to be strictly excluded while calculating the route.
         * </p>
         * 
         * @param exclude
         *        Features to be strictly excluded while calculating the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclude(RouteMatrixExclusionOptions exclude);

        /**
         * <p>
         * Features to be strictly excluded while calculating the route.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixExclusionOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixExclusionOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixExclusionOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #exclude(RouteMatrixExclusionOptions)}.
         * 
         * @param exclude
         *        a consumer that will call methods on {@link RouteMatrixExclusionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #exclude(RouteMatrixExclusionOptions)
         */
        default Builder exclude(Consumer<RouteMatrixExclusionOptions.Builder> exclude) {
            return exclude(RouteMatrixExclusionOptions.builder().applyMutation(exclude).build());
        }

        /**
         * <p>
         * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         * provided when making a request.
         * </p>
         * 
         * @param key
         *        Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         *        provided when making a request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Specifies the optimization criteria for calculating a route.
         * </p>
         * <p>
         * Default Value: <code>FastestRoute</code>
         * </p>
         * 
         * @param optimizeRoutingFor
         *        Specifies the optimization criteria for calculating a route.</p>
         *        <p>
         *        Default Value: <code>FastestRoute</code>
         * @see RoutingObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingObjective
         */
        Builder optimizeRoutingFor(String optimizeRoutingFor);

        /**
         * <p>
         * Specifies the optimization criteria for calculating a route.
         * </p>
         * <p>
         * Default Value: <code>FastestRoute</code>
         * </p>
         * 
         * @param optimizeRoutingFor
         *        Specifies the optimization criteria for calculating a route.</p>
         *        <p>
         *        Default Value: <code>FastestRoute</code>
         * @see RoutingObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingObjective
         */
        Builder optimizeRoutingFor(RoutingObjective optimizeRoutingFor);

        /**
         * <p>
         * The position in longitude and latitude for the origin.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. Using a large amount of Origins in a
         * request can lead you to incur unexpected charges. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note>
         * 
         * @param origins
         *        The position in longitude and latitude for the origin.</p> <note>
         *        <p>
         *        Route calculations are billed for each origin and destination pair. Using a large amount of Origins in
         *        a request can lead you to incur unexpected charges. See <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon
         *        Location's pricing page</a> for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origins(Collection<RouteMatrixOrigin> origins);

        /**
         * <p>
         * The position in longitude and latitude for the origin.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. Using a large amount of Origins in a
         * request can lead you to incur unexpected charges. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note>
         * 
         * @param origins
         *        The position in longitude and latitude for the origin.</p> <note>
         *        <p>
         *        Route calculations are billed for each origin and destination pair. Using a large amount of Origins in
         *        a request can lead you to incur unexpected charges. See <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon
         *        Location's pricing page</a> for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origins(RouteMatrixOrigin... origins);

        /**
         * <p>
         * The position in longitude and latitude for the origin.
         * </p>
         * <note>
         * <p>
         * Route calculations are billed for each origin and destination pair. Using a large amount of Origins in a
         * request can lead you to incur unexpected charges. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixOrigin.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixOrigin#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixOrigin.Builder#build()} is called
         * immediately and its result is passed to {@link #origins(List<RouteMatrixOrigin>)}.
         * 
         * @param origins
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.georoutes.model.RouteMatrixOrigin.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #origins(java.util.Collection<RouteMatrixOrigin>)
         */
        Builder origins(Consumer<RouteMatrixOrigin.Builder>... origins);

        /**
         * <p>
         * Boundary within which the matrix is to be calculated. All data, origins and destinations outside the boundary
         * are considered invalid.
         * </p>
         * <note>
         * <p>
         * When request routing boundary was set as AutoCircle, the response routing boundary will return Circle derived
         * from the AutoCircle settings.
         * </p>
         * </note>
         * 
         * @param routingBoundary
         *        Boundary within which the matrix is to be calculated. All data, origins and destinations outside the
         *        boundary are considered invalid.</p> <note>
         *        <p>
         *        When request routing boundary was set as AutoCircle, the response routing boundary will return Circle
         *        derived from the AutoCircle settings.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routingBoundary(RouteMatrixBoundary routingBoundary);

        /**
         * <p>
         * Boundary within which the matrix is to be calculated. All data, origins and destinations outside the boundary
         * are considered invalid.
         * </p>
         * <note>
         * <p>
         * When request routing boundary was set as AutoCircle, the response routing boundary will return Circle derived
         * from the AutoCircle settings.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link RouteMatrixBoundary.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixBoundary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixBoundary.Builder#build()} is called immediately and
         * its result is passed to {@link #routingBoundary(RouteMatrixBoundary)}.
         * 
         * @param routingBoundary
         *        a consumer that will call methods on {@link RouteMatrixBoundary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routingBoundary(RouteMatrixBoundary)
         */
        default Builder routingBoundary(Consumer<RouteMatrixBoundary.Builder> routingBoundary) {
            return routingBoundary(RouteMatrixBoundary.builder().applyMutation(routingBoundary).build());
        }

        /**
         * <p>
         * Traffic related options.
         * </p>
         * 
         * @param traffic
         *        Traffic related options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traffic(RouteMatrixTrafficOptions traffic);

        /**
         * <p>
         * Traffic related options.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixTrafficOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixTrafficOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixTrafficOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #traffic(RouteMatrixTrafficOptions)}.
         * 
         * @param traffic
         *        a consumer that will call methods on {@link RouteMatrixTrafficOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #traffic(RouteMatrixTrafficOptions)
         */
        default Builder traffic(Consumer<RouteMatrixTrafficOptions.Builder> traffic) {
            return traffic(RouteMatrixTrafficOptions.builder().applyMutation(traffic).build());
        }

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see RouteMatrixTravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixTravelMode
         */
        Builder travelMode(String travelMode);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see RouteMatrixTravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixTravelMode
         */
        Builder travelMode(RouteMatrixTravelMode travelMode);

        /**
         * <p>
         * Travel mode related options for the provided travel mode.
         * </p>
         * 
         * @param travelModeOptions
         *        Travel mode related options for the provided travel mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModeOptions(RouteMatrixTravelModeOptions travelModeOptions);

        /**
         * <p>
         * Travel mode related options for the provided travel mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixTravelModeOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixTravelModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixTravelModeOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #travelModeOptions(RouteMatrixTravelModeOptions)}.
         * 
         * @param travelModeOptions
         *        a consumer that will call methods on {@link RouteMatrixTravelModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #travelModeOptions(RouteMatrixTravelModeOptions)
         */
        default Builder travelModeOptions(Consumer<RouteMatrixTravelModeOptions.Builder> travelModeOptions) {
            return travelModeOptions(RouteMatrixTravelModeOptions.builder().applyMutation(travelModeOptions).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GeoRoutesRequest.BuilderImpl implements Builder {
        private RouteMatrixAllowOptions allow;

        private RouteMatrixAvoidanceOptions avoid;

        private Boolean departNow;

        private String departureTime;

        private List<RouteMatrixDestination> destinations = DefaultSdkAutoConstructList.getInstance();

        private RouteMatrixExclusionOptions exclude;

        private String key;

        private String optimizeRoutingFor;

        private List<RouteMatrixOrigin> origins = DefaultSdkAutoConstructList.getInstance();

        private RouteMatrixBoundary routingBoundary;

        private RouteMatrixTrafficOptions traffic;

        private String travelMode;

        private RouteMatrixTravelModeOptions travelModeOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(CalculateRouteMatrixRequest model) {
            super(model);
            allow(model.allow);
            avoid(model.avoid);
            departNow(model.departNow);
            departureTime(model.departureTime);
            destinations(model.destinations);
            exclude(model.exclude);
            key(model.key);
            optimizeRoutingFor(model.optimizeRoutingFor);
            origins(model.origins);
            routingBoundary(model.routingBoundary);
            traffic(model.traffic);
            travelMode(model.travelMode);
            travelModeOptions(model.travelModeOptions);
        }

        public final RouteMatrixAllowOptions.Builder getAllow() {
            return allow != null ? allow.toBuilder() : null;
        }

        public final void setAllow(RouteMatrixAllowOptions.BuilderImpl allow) {
            this.allow = allow != null ? allow.build() : null;
        }

        @Override
        public final Builder allow(RouteMatrixAllowOptions allow) {
            this.allow = allow;
            return this;
        }

        public final RouteMatrixAvoidanceOptions.Builder getAvoid() {
            return avoid != null ? avoid.toBuilder() : null;
        }

        public final void setAvoid(RouteMatrixAvoidanceOptions.BuilderImpl avoid) {
            this.avoid = avoid != null ? avoid.build() : null;
        }

        @Override
        public final Builder avoid(RouteMatrixAvoidanceOptions avoid) {
            this.avoid = avoid;
            return this;
        }

        public final Boolean getDepartNow() {
            return departNow;
        }

        public final void setDepartNow(Boolean departNow) {
            this.departNow = departNow;
        }

        @Override
        public final Builder departNow(Boolean departNow) {
            this.departNow = departNow;
            return this;
        }

        public final String getDepartureTime() {
            return departureTime;
        }

        public final void setDepartureTime(String departureTime) {
            this.departureTime = departureTime;
        }

        @Override
        public final Builder departureTime(String departureTime) {
            this.departureTime = departureTime;
            return this;
        }

        public final List<RouteMatrixDestination.Builder> getDestinations() {
            List<RouteMatrixDestination.Builder> result = CalculateRouteMatrixRequestDestinationsListCopier
                    .copyToBuilder(this.destinations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDestinations(Collection<RouteMatrixDestination.BuilderImpl> destinations) {
            this.destinations = CalculateRouteMatrixRequestDestinationsListCopier.copyFromBuilder(destinations);
        }

        @Override
        public final Builder destinations(Collection<RouteMatrixDestination> destinations) {
            this.destinations = CalculateRouteMatrixRequestDestinationsListCopier.copy(destinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(RouteMatrixDestination... destinations) {
            destinations(Arrays.asList(destinations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinations(Consumer<RouteMatrixDestination.Builder>... destinations) {
            destinations(Stream.of(destinations).map(c -> RouteMatrixDestination.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final RouteMatrixExclusionOptions.Builder getExclude() {
            return exclude != null ? exclude.toBuilder() : null;
        }

        public final void setExclude(RouteMatrixExclusionOptions.BuilderImpl exclude) {
            this.exclude = exclude != null ? exclude.build() : null;
        }

        @Override
        public final Builder exclude(RouteMatrixExclusionOptions exclude) {
            this.exclude = exclude;
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final String getOptimizeRoutingFor() {
            return optimizeRoutingFor;
        }

        public final void setOptimizeRoutingFor(String optimizeRoutingFor) {
            this.optimizeRoutingFor = optimizeRoutingFor;
        }

        @Override
        public final Builder optimizeRoutingFor(String optimizeRoutingFor) {
            this.optimizeRoutingFor = optimizeRoutingFor;
            return this;
        }

        @Override
        public final Builder optimizeRoutingFor(RoutingObjective optimizeRoutingFor) {
            this.optimizeRoutingFor(optimizeRoutingFor == null ? null : optimizeRoutingFor.toString());
            return this;
        }

        public final List<RouteMatrixOrigin.Builder> getOrigins() {
            List<RouteMatrixOrigin.Builder> result = CalculateRouteMatrixRequestOriginsListCopier.copyToBuilder(this.origins);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOrigins(Collection<RouteMatrixOrigin.BuilderImpl> origins) {
            this.origins = CalculateRouteMatrixRequestOriginsListCopier.copyFromBuilder(origins);
        }

        @Override
        public final Builder origins(Collection<RouteMatrixOrigin> origins) {
            this.origins = CalculateRouteMatrixRequestOriginsListCopier.copy(origins);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder origins(RouteMatrixOrigin... origins) {
            origins(Arrays.asList(origins));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder origins(Consumer<RouteMatrixOrigin.Builder>... origins) {
            origins(Stream.of(origins).map(c -> RouteMatrixOrigin.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final RouteMatrixBoundary.Builder getRoutingBoundary() {
            return routingBoundary != null ? routingBoundary.toBuilder() : null;
        }

        public final void setRoutingBoundary(RouteMatrixBoundary.BuilderImpl routingBoundary) {
            this.routingBoundary = routingBoundary != null ? routingBoundary.build() : null;
        }

        @Override
        public final Builder routingBoundary(RouteMatrixBoundary routingBoundary) {
            this.routingBoundary = routingBoundary;
            return this;
        }

        public final RouteMatrixTrafficOptions.Builder getTraffic() {
            return traffic != null ? traffic.toBuilder() : null;
        }

        public final void setTraffic(RouteMatrixTrafficOptions.BuilderImpl traffic) {
            this.traffic = traffic != null ? traffic.build() : null;
        }

        @Override
        public final Builder traffic(RouteMatrixTrafficOptions traffic) {
            this.traffic = traffic;
            return this;
        }

        public final String getTravelMode() {
            return travelMode;
        }

        public final void setTravelMode(String travelMode) {
            this.travelMode = travelMode;
        }

        @Override
        public final Builder travelMode(String travelMode) {
            this.travelMode = travelMode;
            return this;
        }

        @Override
        public final Builder travelMode(RouteMatrixTravelMode travelMode) {
            this.travelMode(travelMode == null ? null : travelMode.toString());
            return this;
        }

        public final RouteMatrixTravelModeOptions.Builder getTravelModeOptions() {
            return travelModeOptions != null ? travelModeOptions.toBuilder() : null;
        }

        public final void setTravelModeOptions(RouteMatrixTravelModeOptions.BuilderImpl travelModeOptions) {
            this.travelModeOptions = travelModeOptions != null ? travelModeOptions.build() : null;
        }

        @Override
        public final Builder travelModeOptions(RouteMatrixTravelModeOptions travelModeOptions) {
            this.travelModeOptions = travelModeOptions;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CalculateRouteMatrixRequest build() {
            return new CalculateRouteMatrixRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
