/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Travel mode options when the provided travel mode is "Truck"
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteMatrixTruckOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteMatrixTruckOptions.Builder, RouteMatrixTruckOptions> {
    private static final SdkField<Integer> AXLE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AxleCount").getter(getter(RouteMatrixTruckOptions::axleCount)).setter(setter(Builder::axleCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AxleCount").build()).build();

    private static final SdkField<Long> GROSS_WEIGHT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("GrossWeight").getter(getter(RouteMatrixTruckOptions::grossWeight)).setter(setter(Builder::grossWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GrossWeight").build()).build();

    private static final SdkField<List<String>> HAZARDOUS_CARGOS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("HazardousCargos")
            .getter(getter(RouteMatrixTruckOptions::hazardousCargosAsStrings))
            .setter(setter(Builder::hazardousCargosWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HazardousCargos").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> HEIGHT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Height")
            .getter(getter(RouteMatrixTruckOptions::height)).setter(setter(Builder::height))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Height").build()).build();

    private static final SdkField<Long> KPRA_LENGTH_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("KpraLength").getter(getter(RouteMatrixTruckOptions::kpraLength)).setter(setter(Builder::kpraLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KpraLength").build()).build();

    private static final SdkField<Long> LENGTH_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Length")
            .getter(getter(RouteMatrixTruckOptions::length)).setter(setter(Builder::length))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Length").build()).build();

    private static final SdkField<RouteMatrixVehicleLicensePlate> LICENSE_PLATE_FIELD = SdkField
            .<RouteMatrixVehicleLicensePlate> builder(MarshallingType.SDK_POJO).memberName("LicensePlate")
            .getter(getter(RouteMatrixTruckOptions::licensePlate)).setter(setter(Builder::licensePlate))
            .constructor(RouteMatrixVehicleLicensePlate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LicensePlate").build()).build();

    private static final SdkField<Double> MAX_SPEED_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxSpeed").getter(getter(RouteMatrixTruckOptions::maxSpeed)).setter(setter(Builder::maxSpeed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxSpeed").build()).build();

    private static final SdkField<Integer> OCCUPANCY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Occupancy").getter(getter(RouteMatrixTruckOptions::occupancy)).setter(setter(Builder::occupancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Occupancy").build()).build();

    private static final SdkField<Long> PAYLOAD_CAPACITY_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("PayloadCapacity").getter(getter(RouteMatrixTruckOptions::payloadCapacity))
            .setter(setter(Builder::payloadCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PayloadCapacity").build()).build();

    private static final SdkField<RouteMatrixTrailerOptions> TRAILER_FIELD = SdkField
            .<RouteMatrixTrailerOptions> builder(MarshallingType.SDK_POJO).memberName("Trailer")
            .getter(getter(RouteMatrixTruckOptions::trailer)).setter(setter(Builder::trailer))
            .constructor(RouteMatrixTrailerOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Trailer").build()).build();

    private static final SdkField<String> TRUCK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TruckType").getter(getter(RouteMatrixTruckOptions::truckTypeAsString))
            .setter(setter(Builder::truckType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TruckType").build()).build();

    private static final SdkField<String> TUNNEL_RESTRICTION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TunnelRestrictionCode").getter(getter(RouteMatrixTruckOptions::tunnelRestrictionCode))
            .setter(setter(Builder::tunnelRestrictionCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TunnelRestrictionCode").build())
            .build();

    private static final SdkField<Long> WEIGHT_PER_AXLE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("WeightPerAxle").getter(getter(RouteMatrixTruckOptions::weightPerAxle))
            .setter(setter(Builder::weightPerAxle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightPerAxle").build()).build();

    private static final SdkField<WeightPerAxleGroup> WEIGHT_PER_AXLE_GROUP_FIELD = SdkField
            .<WeightPerAxleGroup> builder(MarshallingType.SDK_POJO).memberName("WeightPerAxleGroup")
            .getter(getter(RouteMatrixTruckOptions::weightPerAxleGroup)).setter(setter(Builder::weightPerAxleGroup))
            .constructor(WeightPerAxleGroup::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightPerAxleGroup").build())
            .build();

    private static final SdkField<Long> WIDTH_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Width")
            .getter(getter(RouteMatrixTruckOptions::width)).setter(setter(Builder::width))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Width").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AXLE_COUNT_FIELD,
            GROSS_WEIGHT_FIELD, HAZARDOUS_CARGOS_FIELD, HEIGHT_FIELD, KPRA_LENGTH_FIELD, LENGTH_FIELD, LICENSE_PLATE_FIELD,
            MAX_SPEED_FIELD, OCCUPANCY_FIELD, PAYLOAD_CAPACITY_FIELD, TRAILER_FIELD, TRUCK_TYPE_FIELD,
            TUNNEL_RESTRICTION_CODE_FIELD, WEIGHT_PER_AXLE_FIELD, WEIGHT_PER_AXLE_GROUP_FIELD, WIDTH_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer axleCount;

    private final Long grossWeight;

    private final List<String> hazardousCargos;

    private final Long height;

    private final Long kpraLength;

    private final Long length;

    private final RouteMatrixVehicleLicensePlate licensePlate;

    private final Double maxSpeed;

    private final Integer occupancy;

    private final Long payloadCapacity;

    private final RouteMatrixTrailerOptions trailer;

    private final String truckType;

    private final String tunnelRestrictionCode;

    private final Long weightPerAxle;

    private final WeightPerAxleGroup weightPerAxleGroup;

    private final Long width;

    private RouteMatrixTruckOptions(BuilderImpl builder) {
        this.axleCount = builder.axleCount;
        this.grossWeight = builder.grossWeight;
        this.hazardousCargos = builder.hazardousCargos;
        this.height = builder.height;
        this.kpraLength = builder.kpraLength;
        this.length = builder.length;
        this.licensePlate = builder.licensePlate;
        this.maxSpeed = builder.maxSpeed;
        this.occupancy = builder.occupancy;
        this.payloadCapacity = builder.payloadCapacity;
        this.trailer = builder.trailer;
        this.truckType = builder.truckType;
        this.tunnelRestrictionCode = builder.tunnelRestrictionCode;
        this.weightPerAxle = builder.weightPerAxle;
        this.weightPerAxleGroup = builder.weightPerAxleGroup;
        this.width = builder.width;
    }

    /**
     * <p>
     * Total number of axles of the vehicle.
     * </p>
     * 
     * @return Total number of axles of the vehicle.
     */
    public final Integer axleCount() {
        return axleCount;
    }

    /**
     * <p>
     * Gross weight of the vehicle including trailers, and goods at capacity.
     * </p>
     * <p>
     * <b>Unit</b>: <code>Kilograms</code>
     * </p>
     * 
     * @return Gross weight of the vehicle including trailers, and goods at capacity.</p>
     *         <p>
     *         <b>Unit</b>: <code>Kilograms</code>
     */
    public final Long grossWeight() {
        return grossWeight;
    }

    /**
     * <p>
     * List of Hazardous cargo contained in the vehicle.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHazardousCargos} method.
     * </p>
     * 
     * @return List of Hazardous cargo contained in the vehicle.
     */
    public final List<RouteMatrixHazardousCargoType> hazardousCargos() {
        return RouteMatrixHazardousCargoTypeListCopier.copyStringToEnum(hazardousCargos);
    }

    /**
     * For responses, this returns true if the service returned a value for the HazardousCargos property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHazardousCargos() {
        return hazardousCargos != null && !(hazardousCargos instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of Hazardous cargo contained in the vehicle.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHazardousCargos} method.
     * </p>
     * 
     * @return List of Hazardous cargo contained in the vehicle.
     */
    public final List<String> hazardousCargosAsStrings() {
        return hazardousCargos;
    }

    /**
     * <p>
     * Height of the vehicle.
     * </p>
     * <p>
     * <b>Unit</b>: <code>centimeters</code>
     * </p>
     * 
     * @return Height of the vehicle.</p>
     *         <p>
     *         <b>Unit</b>: <code>centimeters</code>
     */
    public final Long height() {
        return height;
    }

    /**
     * <p>
     * Kingpin to rear axle length of the vehicle
     * </p>
     * <p>
     * <b>Unit</b>: <code>centimeters</code>
     * </p>
     * 
     * @return Kingpin to rear axle length of the vehicle</p>
     *         <p>
     *         <b>Unit</b>: <code>centimeters</code>
     */
    public final Long kpraLength() {
        return kpraLength;
    }

    /**
     * <p>
     * Length of the vehicle.
     * </p>
     * <p>
     * <b>Unit</b>: <code>centimeters</code>
     * </p>
     * 
     * @return Length of the vehicle.</p>
     *         <p>
     *         <b>Unit</b>: <code>centimeters</code>
     */
    public final Long length() {
        return length;
    }

    /**
     * <p>
     * The vehicle License Plate.
     * </p>
     * 
     * @return The vehicle License Plate.
     */
    public final RouteMatrixVehicleLicensePlate licensePlate() {
        return licensePlate;
    }

    /**
     * <p>
     * Maximum speed
     * </p>
     * <p>
     * <b>Unit</b>: <code>KilometersPerHour</code>
     * </p>
     * 
     * @return Maximum speed</p>
     *         <p>
     *         <b>Unit</b>: <code>KilometersPerHour</code>
     */
    public final Double maxSpeed() {
        return maxSpeed;
    }

    /**
     * <p>
     * The number of occupants in the vehicle.
     * </p>
     * <p>
     * Default Value: <code>1</code>
     * </p>
     * 
     * @return The number of occupants in the vehicle.</p>
     *         <p>
     *         Default Value: <code>1</code>
     */
    public final Integer occupancy() {
        return occupancy;
    }

    /**
     * <p>
     * Payload capacity of the vehicle and trailers attached.
     * </p>
     * <p>
     * <b>Unit</b>: <code>kilograms</code>
     * </p>
     * 
     * @return Payload capacity of the vehicle and trailers attached.</p>
     *         <p>
     *         <b>Unit</b>: <code>kilograms</code>
     */
    public final Long payloadCapacity() {
        return payloadCapacity;
    }

    /**
     * <p>
     * Trailer options corresponding to the vehicle.
     * </p>
     * 
     * @return Trailer options corresponding to the vehicle.
     */
    public final RouteMatrixTrailerOptions trailer() {
        return trailer;
    }

    /**
     * <p>
     * Type of the truck.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #truckType} will
     * return {@link RouteMatrixTruckType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #truckTypeAsString}.
     * </p>
     * 
     * @return Type of the truck.
     * @see RouteMatrixTruckType
     */
    public final RouteMatrixTruckType truckType() {
        return RouteMatrixTruckType.fromValue(truckType);
    }

    /**
     * <p>
     * Type of the truck.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #truckType} will
     * return {@link RouteMatrixTruckType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #truckTypeAsString}.
     * </p>
     * 
     * @return Type of the truck.
     * @see RouteMatrixTruckType
     */
    public final String truckTypeAsString() {
        return truckType;
    }

    /**
     * <p>
     * The tunnel restriction code.
     * </p>
     * <p>
     * Tunnel categories in this list indicate the restrictions which apply to certain tunnels in Great Britain. They
     * relate to the types of dangerous goods that can be transported through them.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Tunnel Category B</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Risk Level</i>: Limited risk
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Restrictions</i>: Few restrictions
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <i>Tunnel Category C</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Risk Level</i>: Medium risk
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Restrictions</i>: Some restrictions
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <i>Tunnel Category D</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Risk Level</i>: High risk
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Restrictions</i>: Many restrictions occur
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <i>Tunnel Category E</i>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>Risk Level</i>: Very high risk
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Restrictions</i>: Restricted tunnel
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * 
     * @return The tunnel restriction code.</p>
     *         <p>
     *         Tunnel categories in this list indicate the restrictions which apply to certain tunnels in Great Britain.
     *         They relate to the types of dangerous goods that can be transported through them.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>Tunnel Category B</i>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>Risk Level</i>: Limited risk
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Restrictions</i>: Few restrictions
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Tunnel Category C</i>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>Risk Level</i>: Medium risk
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Restrictions</i>: Some restrictions
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Tunnel Category D</i>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>Risk Level</i>: High risk
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Restrictions</i>: Many restrictions occur
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Tunnel Category E</i>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>Risk Level</i>: Very high risk
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Restrictions</i>: Restricted tunnel
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final String tunnelRestrictionCode() {
        return tunnelRestrictionCode;
    }

    /**
     * <p>
     * Heaviest weight per axle irrespective of the axle type or the axle group. Meant for usage in countries where the
     * differences in axle types or axle groups are not distinguished.
     * </p>
     * <p>
     * <b>Unit</b>: <code>Kilograms</code>
     * </p>
     * 
     * @return Heaviest weight per axle irrespective of the axle type or the axle group. Meant for usage in countries
     *         where the differences in axle types or axle groups are not distinguished.</p>
     *         <p>
     *         <b>Unit</b>: <code>Kilograms</code>
     */
    public final Long weightPerAxle() {
        return weightPerAxle;
    }

    /**
     * <p>
     * Specifies the total weight for the specified axle group. Meant for usage in countries that have different
     * regulations based on the axle group type.
     * </p>
     * 
     * @return Specifies the total weight for the specified axle group. Meant for usage in countries that have different
     *         regulations based on the axle group type.
     */
    public final WeightPerAxleGroup weightPerAxleGroup() {
        return weightPerAxleGroup;
    }

    /**
     * <p>
     * Width of the vehicle.
     * </p>
     * <p>
     * <b>Unit</b>: <code>centimeters</code>
     * </p>
     * 
     * @return Width of the vehicle.</p>
     *         <p>
     *         <b>Unit</b>: <code>centimeters</code>
     */
    public final Long width() {
        return width;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(axleCount());
        hashCode = 31 * hashCode + Objects.hashCode(grossWeight());
        hashCode = 31 * hashCode + Objects.hashCode(hasHazardousCargos() ? hazardousCargosAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(height());
        hashCode = 31 * hashCode + Objects.hashCode(kpraLength());
        hashCode = 31 * hashCode + Objects.hashCode(length());
        hashCode = 31 * hashCode + Objects.hashCode(licensePlate());
        hashCode = 31 * hashCode + Objects.hashCode(maxSpeed());
        hashCode = 31 * hashCode + Objects.hashCode(occupancy());
        hashCode = 31 * hashCode + Objects.hashCode(payloadCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(trailer());
        hashCode = 31 * hashCode + Objects.hashCode(truckTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tunnelRestrictionCode());
        hashCode = 31 * hashCode + Objects.hashCode(weightPerAxle());
        hashCode = 31 * hashCode + Objects.hashCode(weightPerAxleGroup());
        hashCode = 31 * hashCode + Objects.hashCode(width());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteMatrixTruckOptions)) {
            return false;
        }
        RouteMatrixTruckOptions other = (RouteMatrixTruckOptions) obj;
        return Objects.equals(axleCount(), other.axleCount()) && Objects.equals(grossWeight(), other.grossWeight())
                && hasHazardousCargos() == other.hasHazardousCargos()
                && Objects.equals(hazardousCargosAsStrings(), other.hazardousCargosAsStrings())
                && Objects.equals(height(), other.height()) && Objects.equals(kpraLength(), other.kpraLength())
                && Objects.equals(length(), other.length()) && Objects.equals(licensePlate(), other.licensePlate())
                && Objects.equals(maxSpeed(), other.maxSpeed()) && Objects.equals(occupancy(), other.occupancy())
                && Objects.equals(payloadCapacity(), other.payloadCapacity()) && Objects.equals(trailer(), other.trailer())
                && Objects.equals(truckTypeAsString(), other.truckTypeAsString())
                && Objects.equals(tunnelRestrictionCode(), other.tunnelRestrictionCode())
                && Objects.equals(weightPerAxle(), other.weightPerAxle())
                && Objects.equals(weightPerAxleGroup(), other.weightPerAxleGroup()) && Objects.equals(width(), other.width());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteMatrixTruckOptions")
                .add("AxleCount", axleCount() == null ? null : "*** Sensitive Data Redacted ***")
                .add("GrossWeight", grossWeight() == null ? null : "*** Sensitive Data Redacted ***")
                .add("HazardousCargos", hazardousCargosAsStrings() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Height", height() == null ? null : "*** Sensitive Data Redacted ***")
                .add("KpraLength", kpraLength() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Length", length() == null ? null : "*** Sensitive Data Redacted ***")
                .add("LicensePlate", licensePlate() == null ? null : "*** Sensitive Data Redacted ***")
                .add("MaxSpeed", maxSpeed() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Occupancy", occupancy() == null ? null : "*** Sensitive Data Redacted ***")
                .add("PayloadCapacity", payloadCapacity() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Trailer", trailer())
                .add("TruckType", truckTypeAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TunnelRestrictionCode", tunnelRestrictionCode() == null ? null : "*** Sensitive Data Redacted ***")
                .add("WeightPerAxle", weightPerAxle() == null ? null : "*** Sensitive Data Redacted ***")
                .add("WeightPerAxleGroup", weightPerAxleGroup())
                .add("Width", width() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AxleCount":
            return Optional.ofNullable(clazz.cast(axleCount()));
        case "GrossWeight":
            return Optional.ofNullable(clazz.cast(grossWeight()));
        case "HazardousCargos":
            return Optional.ofNullable(clazz.cast(hazardousCargosAsStrings()));
        case "Height":
            return Optional.ofNullable(clazz.cast(height()));
        case "KpraLength":
            return Optional.ofNullable(clazz.cast(kpraLength()));
        case "Length":
            return Optional.ofNullable(clazz.cast(length()));
        case "LicensePlate":
            return Optional.ofNullable(clazz.cast(licensePlate()));
        case "MaxSpeed":
            return Optional.ofNullable(clazz.cast(maxSpeed()));
        case "Occupancy":
            return Optional.ofNullable(clazz.cast(occupancy()));
        case "PayloadCapacity":
            return Optional.ofNullable(clazz.cast(payloadCapacity()));
        case "Trailer":
            return Optional.ofNullable(clazz.cast(trailer()));
        case "TruckType":
            return Optional.ofNullable(clazz.cast(truckTypeAsString()));
        case "TunnelRestrictionCode":
            return Optional.ofNullable(clazz.cast(tunnelRestrictionCode()));
        case "WeightPerAxle":
            return Optional.ofNullable(clazz.cast(weightPerAxle()));
        case "WeightPerAxleGroup":
            return Optional.ofNullable(clazz.cast(weightPerAxleGroup()));
        case "Width":
            return Optional.ofNullable(clazz.cast(width()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AxleCount", AXLE_COUNT_FIELD);
        map.put("GrossWeight", GROSS_WEIGHT_FIELD);
        map.put("HazardousCargos", HAZARDOUS_CARGOS_FIELD);
        map.put("Height", HEIGHT_FIELD);
        map.put("KpraLength", KPRA_LENGTH_FIELD);
        map.put("Length", LENGTH_FIELD);
        map.put("LicensePlate", LICENSE_PLATE_FIELD);
        map.put("MaxSpeed", MAX_SPEED_FIELD);
        map.put("Occupancy", OCCUPANCY_FIELD);
        map.put("PayloadCapacity", PAYLOAD_CAPACITY_FIELD);
        map.put("Trailer", TRAILER_FIELD);
        map.put("TruckType", TRUCK_TYPE_FIELD);
        map.put("TunnelRestrictionCode", TUNNEL_RESTRICTION_CODE_FIELD);
        map.put("WeightPerAxle", WEIGHT_PER_AXLE_FIELD);
        map.put("WeightPerAxleGroup", WEIGHT_PER_AXLE_GROUP_FIELD);
        map.put("Width", WIDTH_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteMatrixTruckOptions, T> g) {
        return obj -> g.apply((RouteMatrixTruckOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteMatrixTruckOptions> {
        /**
         * <p>
         * Total number of axles of the vehicle.
         * </p>
         * 
         * @param axleCount
         *        Total number of axles of the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder axleCount(Integer axleCount);

        /**
         * <p>
         * Gross weight of the vehicle including trailers, and goods at capacity.
         * </p>
         * <p>
         * <b>Unit</b>: <code>Kilograms</code>
         * </p>
         * 
         * @param grossWeight
         *        Gross weight of the vehicle including trailers, and goods at capacity.</p>
         *        <p>
         *        <b>Unit</b>: <code>Kilograms</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grossWeight(Long grossWeight);

        /**
         * <p>
         * List of Hazardous cargo contained in the vehicle.
         * </p>
         * 
         * @param hazardousCargos
         *        List of Hazardous cargo contained in the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hazardousCargosWithStrings(Collection<String> hazardousCargos);

        /**
         * <p>
         * List of Hazardous cargo contained in the vehicle.
         * </p>
         * 
         * @param hazardousCargos
         *        List of Hazardous cargo contained in the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hazardousCargosWithStrings(String... hazardousCargos);

        /**
         * <p>
         * List of Hazardous cargo contained in the vehicle.
         * </p>
         * 
         * @param hazardousCargos
         *        List of Hazardous cargo contained in the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hazardousCargos(Collection<RouteMatrixHazardousCargoType> hazardousCargos);

        /**
         * <p>
         * List of Hazardous cargo contained in the vehicle.
         * </p>
         * 
         * @param hazardousCargos
         *        List of Hazardous cargo contained in the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hazardousCargos(RouteMatrixHazardousCargoType... hazardousCargos);

        /**
         * <p>
         * Height of the vehicle.
         * </p>
         * <p>
         * <b>Unit</b>: <code>centimeters</code>
         * </p>
         * 
         * @param height
         *        Height of the vehicle.</p>
         *        <p>
         *        <b>Unit</b>: <code>centimeters</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder height(Long height);

        /**
         * <p>
         * Kingpin to rear axle length of the vehicle
         * </p>
         * <p>
         * <b>Unit</b>: <code>centimeters</code>
         * </p>
         * 
         * @param kpraLength
         *        Kingpin to rear axle length of the vehicle</p>
         *        <p>
         *        <b>Unit</b>: <code>centimeters</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kpraLength(Long kpraLength);

        /**
         * <p>
         * Length of the vehicle.
         * </p>
         * <p>
         * <b>Unit</b>: <code>centimeters</code>
         * </p>
         * 
         * @param length
         *        Length of the vehicle.</p>
         *        <p>
         *        <b>Unit</b>: <code>centimeters</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder length(Long length);

        /**
         * <p>
         * The vehicle License Plate.
         * </p>
         * 
         * @param licensePlate
         *        The vehicle License Plate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder licensePlate(RouteMatrixVehicleLicensePlate licensePlate);

        /**
         * <p>
         * The vehicle License Plate.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixVehicleLicensePlate.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixVehicleLicensePlate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixVehicleLicensePlate.Builder#build()} is called
         * immediately and its result is passed to {@link #licensePlate(RouteMatrixVehicleLicensePlate)}.
         * 
         * @param licensePlate
         *        a consumer that will call methods on {@link RouteMatrixVehicleLicensePlate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #licensePlate(RouteMatrixVehicleLicensePlate)
         */
        default Builder licensePlate(Consumer<RouteMatrixVehicleLicensePlate.Builder> licensePlate) {
            return licensePlate(RouteMatrixVehicleLicensePlate.builder().applyMutation(licensePlate).build());
        }

        /**
         * <p>
         * Maximum speed
         * </p>
         * <p>
         * <b>Unit</b>: <code>KilometersPerHour</code>
         * </p>
         * 
         * @param maxSpeed
         *        Maximum speed</p>
         *        <p>
         *        <b>Unit</b>: <code>KilometersPerHour</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSpeed(Double maxSpeed);

        /**
         * <p>
         * The number of occupants in the vehicle.
         * </p>
         * <p>
         * Default Value: <code>1</code>
         * </p>
         * 
         * @param occupancy
         *        The number of occupants in the vehicle.</p>
         *        <p>
         *        Default Value: <code>1</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder occupancy(Integer occupancy);

        /**
         * <p>
         * Payload capacity of the vehicle and trailers attached.
         * </p>
         * <p>
         * <b>Unit</b>: <code>kilograms</code>
         * </p>
         * 
         * @param payloadCapacity
         *        Payload capacity of the vehicle and trailers attached.</p>
         *        <p>
         *        <b>Unit</b>: <code>kilograms</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloadCapacity(Long payloadCapacity);

        /**
         * <p>
         * Trailer options corresponding to the vehicle.
         * </p>
         * 
         * @param trailer
         *        Trailer options corresponding to the vehicle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trailer(RouteMatrixTrailerOptions trailer);

        /**
         * <p>
         * Trailer options corresponding to the vehicle.
         * </p>
         * This is a convenience method that creates an instance of the {@link RouteMatrixTrailerOptions.Builder}
         * avoiding the need to create one manually via {@link RouteMatrixTrailerOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RouteMatrixTrailerOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #trailer(RouteMatrixTrailerOptions)}.
         * 
         * @param trailer
         *        a consumer that will call methods on {@link RouteMatrixTrailerOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trailer(RouteMatrixTrailerOptions)
         */
        default Builder trailer(Consumer<RouteMatrixTrailerOptions.Builder> trailer) {
            return trailer(RouteMatrixTrailerOptions.builder().applyMutation(trailer).build());
        }

        /**
         * <p>
         * Type of the truck.
         * </p>
         * 
         * @param truckType
         *        Type of the truck.
         * @see RouteMatrixTruckType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixTruckType
         */
        Builder truckType(String truckType);

        /**
         * <p>
         * Type of the truck.
         * </p>
         * 
         * @param truckType
         *        Type of the truck.
         * @see RouteMatrixTruckType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteMatrixTruckType
         */
        Builder truckType(RouteMatrixTruckType truckType);

        /**
         * <p>
         * The tunnel restriction code.
         * </p>
         * <p>
         * Tunnel categories in this list indicate the restrictions which apply to certain tunnels in Great Britain.
         * They relate to the types of dangerous goods that can be transported through them.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>Tunnel Category B</i>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>Risk Level</i>: Limited risk
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>Restrictions</i>: Few restrictions
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <i>Tunnel Category C</i>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>Risk Level</i>: Medium risk
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>Restrictions</i>: Some restrictions
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <i>Tunnel Category D</i>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>Risk Level</i>: High risk
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>Restrictions</i>: Many restrictions occur
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <i>Tunnel Category E</i>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>Risk Level</i>: Very high risk
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>Restrictions</i>: Restricted tunnel
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param tunnelRestrictionCode
         *        The tunnel restriction code.</p>
         *        <p>
         *        Tunnel categories in this list indicate the restrictions which apply to certain tunnels in Great
         *        Britain. They relate to the types of dangerous goods that can be transported through them.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>Tunnel Category B</i>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>Risk Level</i>: Limited risk
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Restrictions</i>: Few restrictions
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Tunnel Category C</i>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>Risk Level</i>: Medium risk
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Restrictions</i>: Some restrictions
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Tunnel Category D</i>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>Risk Level</i>: High risk
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Restrictions</i>: Many restrictions occur
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Tunnel Category E</i>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>Risk Level</i>: Very high risk
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Restrictions</i>: Restricted tunnel
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tunnelRestrictionCode(String tunnelRestrictionCode);

        /**
         * <p>
         * Heaviest weight per axle irrespective of the axle type or the axle group. Meant for usage in countries where
         * the differences in axle types or axle groups are not distinguished.
         * </p>
         * <p>
         * <b>Unit</b>: <code>Kilograms</code>
         * </p>
         * 
         * @param weightPerAxle
         *        Heaviest weight per axle irrespective of the axle type or the axle group. Meant for usage in countries
         *        where the differences in axle types or axle groups are not distinguished.</p>
         *        <p>
         *        <b>Unit</b>: <code>Kilograms</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightPerAxle(Long weightPerAxle);

        /**
         * <p>
         * Specifies the total weight for the specified axle group. Meant for usage in countries that have different
         * regulations based on the axle group type.
         * </p>
         * 
         * @param weightPerAxleGroup
         *        Specifies the total weight for the specified axle group. Meant for usage in countries that have
         *        different regulations based on the axle group type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightPerAxleGroup(WeightPerAxleGroup weightPerAxleGroup);

        /**
         * <p>
         * Specifies the total weight for the specified axle group. Meant for usage in countries that have different
         * regulations based on the axle group type.
         * </p>
         * This is a convenience method that creates an instance of the {@link WeightPerAxleGroup.Builder} avoiding the
         * need to create one manually via {@link WeightPerAxleGroup#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WeightPerAxleGroup.Builder#build()} is called immediately and its
         * result is passed to {@link #weightPerAxleGroup(WeightPerAxleGroup)}.
         * 
         * @param weightPerAxleGroup
         *        a consumer that will call methods on {@link WeightPerAxleGroup.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #weightPerAxleGroup(WeightPerAxleGroup)
         */
        default Builder weightPerAxleGroup(Consumer<WeightPerAxleGroup.Builder> weightPerAxleGroup) {
            return weightPerAxleGroup(WeightPerAxleGroup.builder().applyMutation(weightPerAxleGroup).build());
        }

        /**
         * <p>
         * Width of the vehicle.
         * </p>
         * <p>
         * <b>Unit</b>: <code>centimeters</code>
         * </p>
         * 
         * @param width
         *        Width of the vehicle.</p>
         *        <p>
         *        <b>Unit</b>: <code>centimeters</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder width(Long width);
    }

    static final class BuilderImpl implements Builder {
        private Integer axleCount;

        private Long grossWeight;

        private List<String> hazardousCargos = DefaultSdkAutoConstructList.getInstance();

        private Long height;

        private Long kpraLength;

        private Long length;

        private RouteMatrixVehicleLicensePlate licensePlate;

        private Double maxSpeed;

        private Integer occupancy;

        private Long payloadCapacity;

        private RouteMatrixTrailerOptions trailer;

        private String truckType;

        private String tunnelRestrictionCode;

        private Long weightPerAxle;

        private WeightPerAxleGroup weightPerAxleGroup;

        private Long width;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteMatrixTruckOptions model) {
            axleCount(model.axleCount);
            grossWeight(model.grossWeight);
            hazardousCargosWithStrings(model.hazardousCargos);
            height(model.height);
            kpraLength(model.kpraLength);
            length(model.length);
            licensePlate(model.licensePlate);
            maxSpeed(model.maxSpeed);
            occupancy(model.occupancy);
            payloadCapacity(model.payloadCapacity);
            trailer(model.trailer);
            truckType(model.truckType);
            tunnelRestrictionCode(model.tunnelRestrictionCode);
            weightPerAxle(model.weightPerAxle);
            weightPerAxleGroup(model.weightPerAxleGroup);
            width(model.width);
        }

        public final Integer getAxleCount() {
            return axleCount;
        }

        public final void setAxleCount(Integer axleCount) {
            this.axleCount = axleCount;
        }

        @Override
        public final Builder axleCount(Integer axleCount) {
            this.axleCount = axleCount;
            return this;
        }

        public final Long getGrossWeight() {
            return grossWeight;
        }

        public final void setGrossWeight(Long grossWeight) {
            this.grossWeight = grossWeight;
        }

        @Override
        public final Builder grossWeight(Long grossWeight) {
            this.grossWeight = grossWeight;
            return this;
        }

        public final Collection<String> getHazardousCargos() {
            if (hazardousCargos instanceof SdkAutoConstructList) {
                return null;
            }
            return hazardousCargos;
        }

        public final void setHazardousCargos(Collection<String> hazardousCargos) {
            this.hazardousCargos = RouteMatrixHazardousCargoTypeListCopier.copy(hazardousCargos);
        }

        @Override
        public final Builder hazardousCargosWithStrings(Collection<String> hazardousCargos) {
            this.hazardousCargos = RouteMatrixHazardousCargoTypeListCopier.copy(hazardousCargos);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hazardousCargosWithStrings(String... hazardousCargos) {
            hazardousCargosWithStrings(Arrays.asList(hazardousCargos));
            return this;
        }

        @Override
        public final Builder hazardousCargos(Collection<RouteMatrixHazardousCargoType> hazardousCargos) {
            this.hazardousCargos = RouteMatrixHazardousCargoTypeListCopier.copyEnumToString(hazardousCargos);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder hazardousCargos(RouteMatrixHazardousCargoType... hazardousCargos) {
            hazardousCargos(Arrays.asList(hazardousCargos));
            return this;
        }

        public final Long getHeight() {
            return height;
        }

        public final void setHeight(Long height) {
            this.height = height;
        }

        @Override
        public final Builder height(Long height) {
            this.height = height;
            return this;
        }

        public final Long getKpraLength() {
            return kpraLength;
        }

        public final void setKpraLength(Long kpraLength) {
            this.kpraLength = kpraLength;
        }

        @Override
        public final Builder kpraLength(Long kpraLength) {
            this.kpraLength = kpraLength;
            return this;
        }

        public final Long getLength() {
            return length;
        }

        public final void setLength(Long length) {
            this.length = length;
        }

        @Override
        public final Builder length(Long length) {
            this.length = length;
            return this;
        }

        public final RouteMatrixVehicleLicensePlate.Builder getLicensePlate() {
            return licensePlate != null ? licensePlate.toBuilder() : null;
        }

        public final void setLicensePlate(RouteMatrixVehicleLicensePlate.BuilderImpl licensePlate) {
            this.licensePlate = licensePlate != null ? licensePlate.build() : null;
        }

        @Override
        public final Builder licensePlate(RouteMatrixVehicleLicensePlate licensePlate) {
            this.licensePlate = licensePlate;
            return this;
        }

        public final Double getMaxSpeed() {
            return maxSpeed;
        }

        public final void setMaxSpeed(Double maxSpeed) {
            this.maxSpeed = maxSpeed;
        }

        @Override
        public final Builder maxSpeed(Double maxSpeed) {
            this.maxSpeed = maxSpeed;
            return this;
        }

        public final Integer getOccupancy() {
            return occupancy;
        }

        public final void setOccupancy(Integer occupancy) {
            this.occupancy = occupancy;
        }

        @Override
        public final Builder occupancy(Integer occupancy) {
            this.occupancy = occupancy;
            return this;
        }

        public final Long getPayloadCapacity() {
            return payloadCapacity;
        }

        public final void setPayloadCapacity(Long payloadCapacity) {
            this.payloadCapacity = payloadCapacity;
        }

        @Override
        public final Builder payloadCapacity(Long payloadCapacity) {
            this.payloadCapacity = payloadCapacity;
            return this;
        }

        public final RouteMatrixTrailerOptions.Builder getTrailer() {
            return trailer != null ? trailer.toBuilder() : null;
        }

        public final void setTrailer(RouteMatrixTrailerOptions.BuilderImpl trailer) {
            this.trailer = trailer != null ? trailer.build() : null;
        }

        @Override
        public final Builder trailer(RouteMatrixTrailerOptions trailer) {
            this.trailer = trailer;
            return this;
        }

        public final String getTruckType() {
            return truckType;
        }

        public final void setTruckType(String truckType) {
            this.truckType = truckType;
        }

        @Override
        public final Builder truckType(String truckType) {
            this.truckType = truckType;
            return this;
        }

        @Override
        public final Builder truckType(RouteMatrixTruckType truckType) {
            this.truckType(truckType == null ? null : truckType.toString());
            return this;
        }

        public final String getTunnelRestrictionCode() {
            return tunnelRestrictionCode;
        }

        public final void setTunnelRestrictionCode(String tunnelRestrictionCode) {
            this.tunnelRestrictionCode = tunnelRestrictionCode;
        }

        @Override
        public final Builder tunnelRestrictionCode(String tunnelRestrictionCode) {
            this.tunnelRestrictionCode = tunnelRestrictionCode;
            return this;
        }

        public final Long getWeightPerAxle() {
            return weightPerAxle;
        }

        public final void setWeightPerAxle(Long weightPerAxle) {
            this.weightPerAxle = weightPerAxle;
        }

        @Override
        public final Builder weightPerAxle(Long weightPerAxle) {
            this.weightPerAxle = weightPerAxle;
            return this;
        }

        public final WeightPerAxleGroup.Builder getWeightPerAxleGroup() {
            return weightPerAxleGroup != null ? weightPerAxleGroup.toBuilder() : null;
        }

        public final void setWeightPerAxleGroup(WeightPerAxleGroup.BuilderImpl weightPerAxleGroup) {
            this.weightPerAxleGroup = weightPerAxleGroup != null ? weightPerAxleGroup.build() : null;
        }

        @Override
        public final Builder weightPerAxleGroup(WeightPerAxleGroup weightPerAxleGroup) {
            this.weightPerAxleGroup = weightPerAxleGroup;
            return this;
        }

        public final Long getWidth() {
            return width;
        }

        public final void setWidth(Long width) {
            this.width = width;
        }

        @Override
        public final Builder width(Long width) {
            this.width = width;
            return this;
        }

        @Override
        public RouteMatrixTruckOptions build() {
            return new RouteMatrixTruckOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
