/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculateIsolinesRequest extends GeoRoutesRequest implements
        ToCopyableBuilder<CalculateIsolinesRequest.Builder, CalculateIsolinesRequest> {
    private static final SdkField<IsolineAllowOptions> ALLOW_FIELD = SdkField
            .<IsolineAllowOptions> builder(MarshallingType.SDK_POJO).memberName("Allow")
            .getter(getter(CalculateIsolinesRequest::allow)).setter(setter(Builder::allow))
            .constructor(IsolineAllowOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Allow").build()).build();

    private static final SdkField<String> ARRIVAL_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ArrivalTime").getter(getter(CalculateIsolinesRequest::arrivalTime)).setter(setter(Builder::arrivalTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArrivalTime").build()).build();

    private static final SdkField<IsolineAvoidanceOptions> AVOID_FIELD = SdkField
            .<IsolineAvoidanceOptions> builder(MarshallingType.SDK_POJO).memberName("Avoid")
            .getter(getter(CalculateIsolinesRequest::avoid)).setter(setter(Builder::avoid))
            .constructor(IsolineAvoidanceOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Avoid").build()).build();

    private static final SdkField<Boolean> DEPART_NOW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DepartNow").getter(getter(CalculateIsolinesRequest::departNow)).setter(setter(Builder::departNow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartNow").build()).build();

    private static final SdkField<String> DEPARTURE_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DepartureTime").getter(getter(CalculateIsolinesRequest::departureTime))
            .setter(setter(Builder::departureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartureTime").build()).build();

    private static final SdkField<List<Double>> DESTINATION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Destination")
            .getter(getter(CalculateIsolinesRequest::destination))
            .setter(setter(Builder::destination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destination").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<IsolineDestinationOptions> DESTINATION_OPTIONS_FIELD = SdkField
            .<IsolineDestinationOptions> builder(MarshallingType.SDK_POJO).memberName("DestinationOptions")
            .getter(getter(CalculateIsolinesRequest::destinationOptions)).setter(setter(Builder::destinationOptions))
            .constructor(IsolineDestinationOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationOptions").build())
            .build();

    private static final SdkField<String> ISOLINE_GEOMETRY_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IsolineGeometryFormat").getter(getter(CalculateIsolinesRequest::isolineGeometryFormatAsString))
            .setter(setter(Builder::isolineGeometryFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsolineGeometryFormat").build())
            .build();

    private static final SdkField<IsolineGranularityOptions> ISOLINE_GRANULARITY_FIELD = SdkField
            .<IsolineGranularityOptions> builder(MarshallingType.SDK_POJO).memberName("IsolineGranularity")
            .getter(getter(CalculateIsolinesRequest::isolineGranularity)).setter(setter(Builder::isolineGranularity))
            .constructor(IsolineGranularityOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsolineGranularity").build())
            .build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(CalculateIsolinesRequest::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("key").build()).build();

    private static final SdkField<String> OPTIMIZE_ISOLINE_FOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptimizeIsolineFor").getter(getter(CalculateIsolinesRequest::optimizeIsolineForAsString))
            .setter(setter(Builder::optimizeIsolineFor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptimizeIsolineFor").build())
            .build();

    private static final SdkField<String> OPTIMIZE_ROUTING_FOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptimizeRoutingFor").getter(getter(CalculateIsolinesRequest::optimizeRoutingForAsString))
            .setter(setter(Builder::optimizeRoutingFor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptimizeRoutingFor").build())
            .build();

    private static final SdkField<List<Double>> ORIGIN_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Origin")
            .getter(getter(CalculateIsolinesRequest::origin))
            .setter(setter(Builder::origin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Origin").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<IsolineOriginOptions> ORIGIN_OPTIONS_FIELD = SdkField
            .<IsolineOriginOptions> builder(MarshallingType.SDK_POJO).memberName("OriginOptions")
            .getter(getter(CalculateIsolinesRequest::originOptions)).setter(setter(Builder::originOptions))
            .constructor(IsolineOriginOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginOptions").build()).build();

    private static final SdkField<IsolineThresholds> THRESHOLDS_FIELD = SdkField
            .<IsolineThresholds> builder(MarshallingType.SDK_POJO).memberName("Thresholds")
            .getter(getter(CalculateIsolinesRequest::thresholds)).setter(setter(Builder::thresholds))
            .constructor(IsolineThresholds::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Thresholds").build()).build();

    private static final SdkField<IsolineTrafficOptions> TRAFFIC_FIELD = SdkField
            .<IsolineTrafficOptions> builder(MarshallingType.SDK_POJO).memberName("Traffic")
            .getter(getter(CalculateIsolinesRequest::traffic)).setter(setter(Builder::traffic))
            .constructor(IsolineTrafficOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Traffic").build()).build();

    private static final SdkField<String> TRAVEL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TravelMode").getter(getter(CalculateIsolinesRequest::travelModeAsString))
            .setter(setter(Builder::travelMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelMode").build()).build();

    private static final SdkField<IsolineTravelModeOptions> TRAVEL_MODE_OPTIONS_FIELD = SdkField
            .<IsolineTravelModeOptions> builder(MarshallingType.SDK_POJO).memberName("TravelModeOptions")
            .getter(getter(CalculateIsolinesRequest::travelModeOptions)).setter(setter(Builder::travelModeOptions))
            .constructor(IsolineTravelModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelModeOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOW_FIELD,
            ARRIVAL_TIME_FIELD, AVOID_FIELD, DEPART_NOW_FIELD, DEPARTURE_TIME_FIELD, DESTINATION_FIELD,
            DESTINATION_OPTIONS_FIELD, ISOLINE_GEOMETRY_FORMAT_FIELD, ISOLINE_GRANULARITY_FIELD, KEY_FIELD,
            OPTIMIZE_ISOLINE_FOR_FIELD, OPTIMIZE_ROUTING_FOR_FIELD, ORIGIN_FIELD, ORIGIN_OPTIONS_FIELD, THRESHOLDS_FIELD,
            TRAFFIC_FIELD, TRAVEL_MODE_FIELD, TRAVEL_MODE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final IsolineAllowOptions allow;

    private final String arrivalTime;

    private final IsolineAvoidanceOptions avoid;

    private final Boolean departNow;

    private final String departureTime;

    private final List<Double> destination;

    private final IsolineDestinationOptions destinationOptions;

    private final String isolineGeometryFormat;

    private final IsolineGranularityOptions isolineGranularity;

    private final String key;

    private final String optimizeIsolineFor;

    private final String optimizeRoutingFor;

    private final List<Double> origin;

    private final IsolineOriginOptions originOptions;

    private final IsolineThresholds thresholds;

    private final IsolineTrafficOptions traffic;

    private final String travelMode;

    private final IsolineTravelModeOptions travelModeOptions;

    private CalculateIsolinesRequest(BuilderImpl builder) {
        super(builder);
        this.allow = builder.allow;
        this.arrivalTime = builder.arrivalTime;
        this.avoid = builder.avoid;
        this.departNow = builder.departNow;
        this.departureTime = builder.departureTime;
        this.destination = builder.destination;
        this.destinationOptions = builder.destinationOptions;
        this.isolineGeometryFormat = builder.isolineGeometryFormat;
        this.isolineGranularity = builder.isolineGranularity;
        this.key = builder.key;
        this.optimizeIsolineFor = builder.optimizeIsolineFor;
        this.optimizeRoutingFor = builder.optimizeRoutingFor;
        this.origin = builder.origin;
        this.originOptions = builder.originOptions;
        this.thresholds = builder.thresholds;
        this.traffic = builder.traffic;
        this.travelMode = builder.travelMode;
        this.travelModeOptions = builder.travelModeOptions;
    }

    /**
     * <p>
     * Features that are allowed while calculating an isoline.
     * </p>
     * 
     * @return Features that are allowed while calculating an isoline.
     */
    public final IsolineAllowOptions allow() {
        return allow;
    }

    /**
     * <p>
     * Time of arrival at the destination.
     * </p>
     * <p>
     * Time format: <code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     * </p>
     * <p>
     * Examples:
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24Z</code>
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24+02:00</code>
     * </p>
     * 
     * @return Time of arrival at the destination.</p>
     *         <p>
     *         Time format: <code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     *         </p>
     *         <p>
     *         Examples:
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24Z</code>
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24+02:00</code>
     */
    public final String arrivalTime() {
        return arrivalTime;
    }

    /**
     * <p>
     * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't be
     * satisfied for a particular case, it violates the avoidance and the returned response produces a notice for the
     * violation.
     * </p>
     * 
     * @return Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance
     *         can't be satisfied for a particular case, it violates the avoidance and the returned response produces a
     *         notice for the violation.
     */
    public final IsolineAvoidanceOptions avoid() {
        return avoid;
    }

    /**
     * <p>
     * Uses the current time as the time of departure.
     * </p>
     * 
     * @return Uses the current time as the time of departure.
     */
    public final Boolean departNow() {
        return departNow;
    }

    /**
     * <p>
     * Time of departure from thr origin.
     * </p>
     * <p>
     * Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     * </p>
     * <p>
     * Examples:
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24Z</code>
     * </p>
     * <p>
     * <code>2020-04-22T17:57:24+02:00</code>
     * </p>
     * 
     * @return Time of departure from thr origin.</p>
     *         <p>
     *         Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
     *         </p>
     *         <p>
     *         Examples:
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24Z</code>
     *         </p>
     *         <p>
     *         <code>2020-04-22T17:57:24+02:00</code>
     */
    public final String departureTime() {
        return departureTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Destination property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestination() {
        return destination != null && !(destination instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The final position for the route. In the World Geodetic System (WGS 84) format:
     * <code>[longitude, latitude]</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestination} method.
     * </p>
     * 
     * @return The final position for the route. In the World Geodetic System (WGS 84) format:
     *         <code>[longitude, latitude]</code>.
     */
    public final List<Double> destination() {
        return destination;
    }

    /**
     * <p>
     * Destination related options.
     * </p>
     * 
     * @return Destination related options.
     */
    public final IsolineDestinationOptions destinationOptions() {
        return destinationOptions;
    }

    /**
     * <p>
     * The format of the returned IsolineGeometry.
     * </p>
     * <p>
     * Default Value:<code>FlexiblePolyline</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #isolineGeometryFormat} will return {@link GeometryFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #isolineGeometryFormatAsString}.
     * </p>
     * 
     * @return The format of the returned IsolineGeometry. </p>
     *         <p>
     *         Default Value:<code>FlexiblePolyline</code>
     * @see GeometryFormat
     */
    public final GeometryFormat isolineGeometryFormat() {
        return GeometryFormat.fromValue(isolineGeometryFormat);
    }

    /**
     * <p>
     * The format of the returned IsolineGeometry.
     * </p>
     * <p>
     * Default Value:<code>FlexiblePolyline</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #isolineGeometryFormat} will return {@link GeometryFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #isolineGeometryFormatAsString}.
     * </p>
     * 
     * @return The format of the returned IsolineGeometry. </p>
     *         <p>
     *         Default Value:<code>FlexiblePolyline</code>
     * @see GeometryFormat
     */
    public final String isolineGeometryFormatAsString() {
        return isolineGeometryFormat;
    }

    /**
     * <p>
     * Defines the granularity of the returned Isoline.
     * </p>
     * 
     * @return Defines the granularity of the returned Isoline.
     */
    public final IsolineGranularityOptions isolineGranularity() {
        return isolineGranularity;
    }

    /**
     * <p>
     * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be provided
     * when making a request.
     * </p>
     * 
     * @return Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
     *         provided when making a request.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an isoline of
     * higher granularity that is more precise. FastCalculation generates an isoline faster by reducing the granularity,
     * and in turn the quality of the isoline. BalancedCalculation generates an isoline by balancing between quality and
     * performance.
     * </p>
     * <p>
     * Default Value: <code>BalancedCalculation</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeIsolineFor} will return {@link IsolineOptimizationObjective#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #optimizeIsolineForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an
     *         isoline of higher granularity that is more precise. FastCalculation generates an isoline faster by
     *         reducing the granularity, and in turn the quality of the isoline. BalancedCalculation generates an
     *         isoline by balancing between quality and performance. </p>
     *         <p>
     *         Default Value: <code>BalancedCalculation</code>
     * @see IsolineOptimizationObjective
     */
    public final IsolineOptimizationObjective optimizeIsolineFor() {
        return IsolineOptimizationObjective.fromValue(optimizeIsolineFor);
    }

    /**
     * <p>
     * Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an isoline of
     * higher granularity that is more precise. FastCalculation generates an isoline faster by reducing the granularity,
     * and in turn the quality of the isoline. BalancedCalculation generates an isoline by balancing between quality and
     * performance.
     * </p>
     * <p>
     * Default Value: <code>BalancedCalculation</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeIsolineFor} will return {@link IsolineOptimizationObjective#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #optimizeIsolineForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an
     *         isoline of higher granularity that is more precise. FastCalculation generates an isoline faster by
     *         reducing the granularity, and in turn the quality of the isoline. BalancedCalculation generates an
     *         isoline by balancing between quality and performance. </p>
     *         <p>
     *         Default Value: <code>BalancedCalculation</code>
     * @see IsolineOptimizationObjective
     */
    public final String optimizeIsolineForAsString() {
        return optimizeIsolineFor;
    }

    /**
     * <p>
     * Specifies the optimization criteria for calculating a route.
     * </p>
     * <p>
     * Default Value: <code>FastestRoute</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeRoutingFor} will return {@link RoutingObjective#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizeRoutingForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for calculating a route.</p>
     *         <p>
     *         Default Value: <code>FastestRoute</code>
     * @see RoutingObjective
     */
    public final RoutingObjective optimizeRoutingFor() {
        return RoutingObjective.fromValue(optimizeRoutingFor);
    }

    /**
     * <p>
     * Specifies the optimization criteria for calculating a route.
     * </p>
     * <p>
     * Default Value: <code>FastestRoute</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #optimizeRoutingFor} will return {@link RoutingObjective#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #optimizeRoutingForAsString}.
     * </p>
     * 
     * @return Specifies the optimization criteria for calculating a route.</p>
     *         <p>
     *         Default Value: <code>FastestRoute</code>
     * @see RoutingObjective
     */
    public final String optimizeRoutingForAsString() {
        return optimizeRoutingFor;
    }

    /**
     * For responses, this returns true if the service returned a value for the Origin property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOrigin() {
        return origin != null && !(origin instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The start position for the route.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrigin} method.
     * </p>
     * 
     * @return The start position for the route.
     */
    public final List<Double> origin() {
        return origin;
    }

    /**
     * <p>
     * Origin related options.
     * </p>
     * 
     * @return Origin related options.
     */
    public final IsolineOriginOptions originOptions() {
        return originOptions;
    }

    /**
     * <p>
     * Threshold to be used for the isoline calculation. Up to 3 thresholds per provided type can be requested.
     * </p>
     * <p>
     * You incur a calculation charge for each threshold. Using a large amount of thresholds in a request can lead you
     * to incur unexpected charges. See <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's pricing
     * page</a> for more information.
     * </p>
     * 
     * @return Threshold to be used for the isoline calculation. Up to 3 thresholds per provided type can be
     *         requested.</p>
     *         <p>
     *         You incur a calculation charge for each threshold. Using a large amount of thresholds in a request can
     *         lead you to incur unexpected charges. See <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
     *         pricing page</a> for more information.
     */
    public final IsolineThresholds thresholds() {
        return thresholds;
    }

    /**
     * <p>
     * Traffic related options.
     * </p>
     * 
     * @return Traffic related options.
     */
    public final IsolineTrafficOptions traffic() {
        return traffic;
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <note>
     * <p>
     * The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to calculate
     * options for motorcycles.
     * </p>
     * </note>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link IsolineTravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p> <note>
     *         <p>
     *         The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
     *         calculate options for motorcycles.
     *         </p>
     *         </note>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see IsolineTravelMode
     */
    public final IsolineTravelMode travelMode() {
        return IsolineTravelMode.fromValue(travelMode);
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <note>
     * <p>
     * The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to calculate
     * options for motorcycles.
     * </p>
     * </note>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link IsolineTravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p> <note>
     *         <p>
     *         The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
     *         calculate options for motorcycles.
     *         </p>
     *         </note>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see IsolineTravelMode
     */
    public final String travelModeAsString() {
        return travelMode;
    }

    /**
     * <p>
     * Travel mode related options for the provided travel mode.
     * </p>
     * 
     * @return Travel mode related options for the provided travel mode.
     */
    public final IsolineTravelModeOptions travelModeOptions() {
        return travelModeOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(allow());
        hashCode = 31 * hashCode + Objects.hashCode(arrivalTime());
        hashCode = 31 * hashCode + Objects.hashCode(avoid());
        hashCode = 31 * hashCode + Objects.hashCode(departNow());
        hashCode = 31 * hashCode + Objects.hashCode(departureTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestination() ? destination() : null);
        hashCode = 31 * hashCode + Objects.hashCode(destinationOptions());
        hashCode = 31 * hashCode + Objects.hashCode(isolineGeometryFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(isolineGranularity());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(optimizeIsolineForAsString());
        hashCode = 31 * hashCode + Objects.hashCode(optimizeRoutingForAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOrigin() ? origin() : null);
        hashCode = 31 * hashCode + Objects.hashCode(originOptions());
        hashCode = 31 * hashCode + Objects.hashCode(thresholds());
        hashCode = 31 * hashCode + Objects.hashCode(traffic());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculateIsolinesRequest)) {
            return false;
        }
        CalculateIsolinesRequest other = (CalculateIsolinesRequest) obj;
        return Objects.equals(allow(), other.allow()) && Objects.equals(arrivalTime(), other.arrivalTime())
                && Objects.equals(avoid(), other.avoid()) && Objects.equals(departNow(), other.departNow())
                && Objects.equals(departureTime(), other.departureTime()) && hasDestination() == other.hasDestination()
                && Objects.equals(destination(), other.destination())
                && Objects.equals(destinationOptions(), other.destinationOptions())
                && Objects.equals(isolineGeometryFormatAsString(), other.isolineGeometryFormatAsString())
                && Objects.equals(isolineGranularity(), other.isolineGranularity()) && Objects.equals(key(), other.key())
                && Objects.equals(optimizeIsolineForAsString(), other.optimizeIsolineForAsString())
                && Objects.equals(optimizeRoutingForAsString(), other.optimizeRoutingForAsString())
                && hasOrigin() == other.hasOrigin() && Objects.equals(origin(), other.origin())
                && Objects.equals(originOptions(), other.originOptions()) && Objects.equals(thresholds(), other.thresholds())
                && Objects.equals(traffic(), other.traffic()) && Objects.equals(travelModeAsString(), other.travelModeAsString())
                && Objects.equals(travelModeOptions(), other.travelModeOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculateIsolinesRequest").add("Allow", allow())
                .add("ArrivalTime", arrivalTime() == null ? null : "*** Sensitive Data Redacted ***").add("Avoid", avoid())
                .add("DepartNow", departNow() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DepartureTime", departureTime() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Destination", destination() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DestinationOptions", destinationOptions()).add("IsolineGeometryFormat", isolineGeometryFormatAsString())
                .add("IsolineGranularity", isolineGranularity())
                .add("Key", key() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OptimizeIsolineFor", optimizeIsolineForAsString()).add("OptimizeRoutingFor", optimizeRoutingForAsString())
                .add("Origin", origin() == null ? null : "*** Sensitive Data Redacted ***").add("OriginOptions", originOptions())
                .add("Thresholds", thresholds()).add("Traffic", traffic()).add("TravelMode", travelModeAsString())
                .add("TravelModeOptions", travelModeOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Allow":
            return Optional.ofNullable(clazz.cast(allow()));
        case "ArrivalTime":
            return Optional.ofNullable(clazz.cast(arrivalTime()));
        case "Avoid":
            return Optional.ofNullable(clazz.cast(avoid()));
        case "DepartNow":
            return Optional.ofNullable(clazz.cast(departNow()));
        case "DepartureTime":
            return Optional.ofNullable(clazz.cast(departureTime()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        case "DestinationOptions":
            return Optional.ofNullable(clazz.cast(destinationOptions()));
        case "IsolineGeometryFormat":
            return Optional.ofNullable(clazz.cast(isolineGeometryFormatAsString()));
        case "IsolineGranularity":
            return Optional.ofNullable(clazz.cast(isolineGranularity()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "OptimizeIsolineFor":
            return Optional.ofNullable(clazz.cast(optimizeIsolineForAsString()));
        case "OptimizeRoutingFor":
            return Optional.ofNullable(clazz.cast(optimizeRoutingForAsString()));
        case "Origin":
            return Optional.ofNullable(clazz.cast(origin()));
        case "OriginOptions":
            return Optional.ofNullable(clazz.cast(originOptions()));
        case "Thresholds":
            return Optional.ofNullable(clazz.cast(thresholds()));
        case "Traffic":
            return Optional.ofNullable(clazz.cast(traffic()));
        case "TravelMode":
            return Optional.ofNullable(clazz.cast(travelModeAsString()));
        case "TravelModeOptions":
            return Optional.ofNullable(clazz.cast(travelModeOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Allow", ALLOW_FIELD);
        map.put("ArrivalTime", ARRIVAL_TIME_FIELD);
        map.put("Avoid", AVOID_FIELD);
        map.put("DepartNow", DEPART_NOW_FIELD);
        map.put("DepartureTime", DEPARTURE_TIME_FIELD);
        map.put("Destination", DESTINATION_FIELD);
        map.put("DestinationOptions", DESTINATION_OPTIONS_FIELD);
        map.put("IsolineGeometryFormat", ISOLINE_GEOMETRY_FORMAT_FIELD);
        map.put("IsolineGranularity", ISOLINE_GRANULARITY_FIELD);
        map.put("key", KEY_FIELD);
        map.put("OptimizeIsolineFor", OPTIMIZE_ISOLINE_FOR_FIELD);
        map.put("OptimizeRoutingFor", OPTIMIZE_ROUTING_FOR_FIELD);
        map.put("Origin", ORIGIN_FIELD);
        map.put("OriginOptions", ORIGIN_OPTIONS_FIELD);
        map.put("Thresholds", THRESHOLDS_FIELD);
        map.put("Traffic", TRAFFIC_FIELD);
        map.put("TravelMode", TRAVEL_MODE_FIELD);
        map.put("TravelModeOptions", TRAVEL_MODE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CalculateIsolinesRequest, T> g) {
        return obj -> g.apply((CalculateIsolinesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GeoRoutesRequest.Builder, SdkPojo, CopyableBuilder<Builder, CalculateIsolinesRequest> {
        /**
         * <p>
         * Features that are allowed while calculating an isoline.
         * </p>
         * 
         * @param allow
         *        Features that are allowed while calculating an isoline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allow(IsolineAllowOptions allow);

        /**
         * <p>
         * Features that are allowed while calculating an isoline.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineAllowOptions.Builder} avoiding the
         * need to create one manually via {@link IsolineAllowOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineAllowOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #allow(IsolineAllowOptions)}.
         * 
         * @param allow
         *        a consumer that will call methods on {@link IsolineAllowOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allow(IsolineAllowOptions)
         */
        default Builder allow(Consumer<IsolineAllowOptions.Builder> allow) {
            return allow(IsolineAllowOptions.builder().applyMutation(allow).build());
        }

        /**
         * <p>
         * Time of arrival at the destination.
         * </p>
         * <p>
         * Time format: <code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         * </p>
         * <p>
         * Examples:
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24Z</code>
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24+02:00</code>
         * </p>
         * 
         * @param arrivalTime
         *        Time of arrival at the destination.</p>
         *        <p>
         *        Time format: <code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         *        </p>
         *        <p>
         *        Examples:
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24Z</code>
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24+02:00</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrivalTime(String arrivalTime);

        /**
         * <p>
         * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't
         * be satisfied for a particular case, it violates the avoidance and the returned response produces a notice for
         * the violation.
         * </p>
         * 
         * @param avoid
         *        Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an
         *        avoidance can't be satisfied for a particular case, it violates the avoidance and the returned
         *        response produces a notice for the violation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoid(IsolineAvoidanceOptions avoid);

        /**
         * <p>
         * Features that are avoided while calculating a route. Avoidance is on a best-case basis. If an avoidance can't
         * be satisfied for a particular case, it violates the avoidance and the returned response produces a notice for
         * the violation.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineAvoidanceOptions.Builder} avoiding
         * the need to create one manually via {@link IsolineAvoidanceOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineAvoidanceOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #avoid(IsolineAvoidanceOptions)}.
         * 
         * @param avoid
         *        a consumer that will call methods on {@link IsolineAvoidanceOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #avoid(IsolineAvoidanceOptions)
         */
        default Builder avoid(Consumer<IsolineAvoidanceOptions.Builder> avoid) {
            return avoid(IsolineAvoidanceOptions.builder().applyMutation(avoid).build());
        }

        /**
         * <p>
         * Uses the current time as the time of departure.
         * </p>
         * 
         * @param departNow
         *        Uses the current time as the time of departure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departNow(Boolean departNow);

        /**
         * <p>
         * Time of departure from thr origin.
         * </p>
         * <p>
         * Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         * </p>
         * <p>
         * Examples:
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24Z</code>
         * </p>
         * <p>
         * <code>2020-04-22T17:57:24+02:00</code>
         * </p>
         * 
         * @param departureTime
         *        Time of departure from thr origin.</p>
         *        <p>
         *        Time format:<code>YYYY-MM-DDThh:mm:ss.sssZ | YYYY-MM-DDThh:mm:ss.sss+hh:mm</code>
         *        </p>
         *        <p>
         *        Examples:
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24Z</code>
         *        </p>
         *        <p>
         *        <code>2020-04-22T17:57:24+02:00</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departureTime(String departureTime);

        /**
         * <p>
         * The final position for the route. In the World Geodetic System (WGS 84) format:
         * <code>[longitude, latitude]</code>.
         * </p>
         * 
         * @param destination
         *        The final position for the route. In the World Geodetic System (WGS 84) format:
         *        <code>[longitude, latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Collection<Double> destination);

        /**
         * <p>
         * The final position for the route. In the World Geodetic System (WGS 84) format:
         * <code>[longitude, latitude]</code>.
         * </p>
         * 
         * @param destination
         *        The final position for the route. In the World Geodetic System (WGS 84) format:
         *        <code>[longitude, latitude]</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(Double... destination);

        /**
         * <p>
         * Destination related options.
         * </p>
         * 
         * @param destinationOptions
         *        Destination related options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationOptions(IsolineDestinationOptions destinationOptions);

        /**
         * <p>
         * Destination related options.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineDestinationOptions.Builder}
         * avoiding the need to create one manually via {@link IsolineDestinationOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineDestinationOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #destinationOptions(IsolineDestinationOptions)}.
         * 
         * @param destinationOptions
         *        a consumer that will call methods on {@link IsolineDestinationOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationOptions(IsolineDestinationOptions)
         */
        default Builder destinationOptions(Consumer<IsolineDestinationOptions.Builder> destinationOptions) {
            return destinationOptions(IsolineDestinationOptions.builder().applyMutation(destinationOptions).build());
        }

        /**
         * <p>
         * The format of the returned IsolineGeometry.
         * </p>
         * <p>
         * Default Value:<code>FlexiblePolyline</code>
         * </p>
         * 
         * @param isolineGeometryFormat
         *        The format of the returned IsolineGeometry. </p>
         *        <p>
         *        Default Value:<code>FlexiblePolyline</code>
         * @see GeometryFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeometryFormat
         */
        Builder isolineGeometryFormat(String isolineGeometryFormat);

        /**
         * <p>
         * The format of the returned IsolineGeometry.
         * </p>
         * <p>
         * Default Value:<code>FlexiblePolyline</code>
         * </p>
         * 
         * @param isolineGeometryFormat
         *        The format of the returned IsolineGeometry. </p>
         *        <p>
         *        Default Value:<code>FlexiblePolyline</code>
         * @see GeometryFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeometryFormat
         */
        Builder isolineGeometryFormat(GeometryFormat isolineGeometryFormat);

        /**
         * <p>
         * Defines the granularity of the returned Isoline.
         * </p>
         * 
         * @param isolineGranularity
         *        Defines the granularity of the returned Isoline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isolineGranularity(IsolineGranularityOptions isolineGranularity);

        /**
         * <p>
         * Defines the granularity of the returned Isoline.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineGranularityOptions.Builder}
         * avoiding the need to create one manually via {@link IsolineGranularityOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineGranularityOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #isolineGranularity(IsolineGranularityOptions)}.
         * 
         * @param isolineGranularity
         *        a consumer that will call methods on {@link IsolineGranularityOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #isolineGranularity(IsolineGranularityOptions)
         */
        default Builder isolineGranularity(Consumer<IsolineGranularityOptions.Builder> isolineGranularity) {
            return isolineGranularity(IsolineGranularityOptions.builder().applyMutation(isolineGranularity).build());
        }

        /**
         * <p>
         * Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         * provided when making a request.
         * </p>
         * 
         * @param key
         *        Optional: The API key to be used for authorization. Either an API key or valid SigV4 signature must be
         *        provided when making a request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an isoline
         * of higher granularity that is more precise. FastCalculation generates an isoline faster by reducing the
         * granularity, and in turn the quality of the isoline. BalancedCalculation generates an isoline by balancing
         * between quality and performance.
         * </p>
         * <p>
         * Default Value: <code>BalancedCalculation</code>
         * </p>
         * 
         * @param optimizeIsolineFor
         *        Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an
         *        isoline of higher granularity that is more precise. FastCalculation generates an isoline faster by
         *        reducing the granularity, and in turn the quality of the isoline. BalancedCalculation generates an
         *        isoline by balancing between quality and performance. </p>
         *        <p>
         *        Default Value: <code>BalancedCalculation</code>
         * @see IsolineOptimizationObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IsolineOptimizationObjective
         */
        Builder optimizeIsolineFor(String optimizeIsolineFor);

        /**
         * <p>
         * Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an isoline
         * of higher granularity that is more precise. FastCalculation generates an isoline faster by reducing the
         * granularity, and in turn the quality of the isoline. BalancedCalculation generates an isoline by balancing
         * between quality and performance.
         * </p>
         * <p>
         * Default Value: <code>BalancedCalculation</code>
         * </p>
         * 
         * @param optimizeIsolineFor
         *        Specifies the optimization criteria for when calculating an isoline. AccurateCalculation generates an
         *        isoline of higher granularity that is more precise. FastCalculation generates an isoline faster by
         *        reducing the granularity, and in turn the quality of the isoline. BalancedCalculation generates an
         *        isoline by balancing between quality and performance. </p>
         *        <p>
         *        Default Value: <code>BalancedCalculation</code>
         * @see IsolineOptimizationObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IsolineOptimizationObjective
         */
        Builder optimizeIsolineFor(IsolineOptimizationObjective optimizeIsolineFor);

        /**
         * <p>
         * Specifies the optimization criteria for calculating a route.
         * </p>
         * <p>
         * Default Value: <code>FastestRoute</code>
         * </p>
         * 
         * @param optimizeRoutingFor
         *        Specifies the optimization criteria for calculating a route.</p>
         *        <p>
         *        Default Value: <code>FastestRoute</code>
         * @see RoutingObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingObjective
         */
        Builder optimizeRoutingFor(String optimizeRoutingFor);

        /**
         * <p>
         * Specifies the optimization criteria for calculating a route.
         * </p>
         * <p>
         * Default Value: <code>FastestRoute</code>
         * </p>
         * 
         * @param optimizeRoutingFor
         *        Specifies the optimization criteria for calculating a route.</p>
         *        <p>
         *        Default Value: <code>FastestRoute</code>
         * @see RoutingObjective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RoutingObjective
         */
        Builder optimizeRoutingFor(RoutingObjective optimizeRoutingFor);

        /**
         * <p>
         * The start position for the route.
         * </p>
         * 
         * @param origin
         *        The start position for the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origin(Collection<Double> origin);

        /**
         * <p>
         * The start position for the route.
         * </p>
         * 
         * @param origin
         *        The start position for the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origin(Double... origin);

        /**
         * <p>
         * Origin related options.
         * </p>
         * 
         * @param originOptions
         *        Origin related options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originOptions(IsolineOriginOptions originOptions);

        /**
         * <p>
         * Origin related options.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineOriginOptions.Builder} avoiding
         * the need to create one manually via {@link IsolineOriginOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineOriginOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #originOptions(IsolineOriginOptions)}.
         * 
         * @param originOptions
         *        a consumer that will call methods on {@link IsolineOriginOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #originOptions(IsolineOriginOptions)
         */
        default Builder originOptions(Consumer<IsolineOriginOptions.Builder> originOptions) {
            return originOptions(IsolineOriginOptions.builder().applyMutation(originOptions).build());
        }

        /**
         * <p>
         * Threshold to be used for the isoline calculation. Up to 3 thresholds per provided type can be requested.
         * </p>
         * <p>
         * You incur a calculation charge for each threshold. Using a large amount of thresholds in a request can lead
         * you to incur unexpected charges. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * 
         * @param thresholds
         *        Threshold to be used for the isoline calculation. Up to 3 thresholds per provided type can be
         *        requested.</p>
         *        <p>
         *        You incur a calculation charge for each threshold. Using a large amount of thresholds in a request can
         *        lead you to incur unexpected charges. See <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon
         *        Location's pricing page</a> for more information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder thresholds(IsolineThresholds thresholds);

        /**
         * <p>
         * Threshold to be used for the isoline calculation. Up to 3 thresholds per provided type can be requested.
         * </p>
         * <p>
         * You incur a calculation charge for each threshold. Using a large amount of thresholds in a request can lead
         * you to incur unexpected charges. See <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/routes-pricing.html`"> Amazon Location's
         * pricing page</a> for more information.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineThresholds.Builder} avoiding the
         * need to create one manually via {@link IsolineThresholds#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineThresholds.Builder#build()} is called immediately and its
         * result is passed to {@link #thresholds(IsolineThresholds)}.
         * 
         * @param thresholds
         *        a consumer that will call methods on {@link IsolineThresholds.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #thresholds(IsolineThresholds)
         */
        default Builder thresholds(Consumer<IsolineThresholds.Builder> thresholds) {
            return thresholds(IsolineThresholds.builder().applyMutation(thresholds).build());
        }

        /**
         * <p>
         * Traffic related options.
         * </p>
         * 
         * @param traffic
         *        Traffic related options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder traffic(IsolineTrafficOptions traffic);

        /**
         * <p>
         * Traffic related options.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineTrafficOptions.Builder} avoiding
         * the need to create one manually via {@link IsolineTrafficOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineTrafficOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #traffic(IsolineTrafficOptions)}.
         * 
         * @param traffic
         *        a consumer that will call methods on {@link IsolineTrafficOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #traffic(IsolineTrafficOptions)
         */
        default Builder traffic(Consumer<IsolineTrafficOptions.Builder> traffic) {
            return traffic(IsolineTrafficOptions.builder().applyMutation(traffic).build());
        }

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <note>
         * <p>
         * The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
         * calculate options for motorcycles.
         * </p>
         * </note>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p> <note>
         *        <p>
         *        The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
         *        calculate options for motorcycles.
         *        </p>
         *        </note>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see IsolineTravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IsolineTravelMode
         */
        Builder travelMode(String travelMode);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <note>
         * <p>
         * The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
         * calculate options for motorcycles.
         * </p>
         * </note>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p> <note>
         *        <p>
         *        The mode <code>Scooter</code> also applies to motorcycles, set to <code>Scooter</code> when wanted to
         *        calculate options for motorcycles.
         *        </p>
         *        </note>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see IsolineTravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IsolineTravelMode
         */
        Builder travelMode(IsolineTravelMode travelMode);

        /**
         * <p>
         * Travel mode related options for the provided travel mode.
         * </p>
         * 
         * @param travelModeOptions
         *        Travel mode related options for the provided travel mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder travelModeOptions(IsolineTravelModeOptions travelModeOptions);

        /**
         * <p>
         * Travel mode related options for the provided travel mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link IsolineTravelModeOptions.Builder}
         * avoiding the need to create one manually via {@link IsolineTravelModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IsolineTravelModeOptions.Builder#build()} is called immediately
         * and its result is passed to {@link #travelModeOptions(IsolineTravelModeOptions)}.
         * 
         * @param travelModeOptions
         *        a consumer that will call methods on {@link IsolineTravelModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #travelModeOptions(IsolineTravelModeOptions)
         */
        default Builder travelModeOptions(Consumer<IsolineTravelModeOptions.Builder> travelModeOptions) {
            return travelModeOptions(IsolineTravelModeOptions.builder().applyMutation(travelModeOptions).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GeoRoutesRequest.BuilderImpl implements Builder {
        private IsolineAllowOptions allow;

        private String arrivalTime;

        private IsolineAvoidanceOptions avoid;

        private Boolean departNow;

        private String departureTime;

        private List<Double> destination = DefaultSdkAutoConstructList.getInstance();

        private IsolineDestinationOptions destinationOptions;

        private String isolineGeometryFormat;

        private IsolineGranularityOptions isolineGranularity;

        private String key;

        private String optimizeIsolineFor;

        private String optimizeRoutingFor;

        private List<Double> origin = DefaultSdkAutoConstructList.getInstance();

        private IsolineOriginOptions originOptions;

        private IsolineThresholds thresholds;

        private IsolineTrafficOptions traffic;

        private String travelMode;

        private IsolineTravelModeOptions travelModeOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(CalculateIsolinesRequest model) {
            super(model);
            allow(model.allow);
            arrivalTime(model.arrivalTime);
            avoid(model.avoid);
            departNow(model.departNow);
            departureTime(model.departureTime);
            destination(model.destination);
            destinationOptions(model.destinationOptions);
            isolineGeometryFormat(model.isolineGeometryFormat);
            isolineGranularity(model.isolineGranularity);
            key(model.key);
            optimizeIsolineFor(model.optimizeIsolineFor);
            optimizeRoutingFor(model.optimizeRoutingFor);
            origin(model.origin);
            originOptions(model.originOptions);
            thresholds(model.thresholds);
            traffic(model.traffic);
            travelMode(model.travelMode);
            travelModeOptions(model.travelModeOptions);
        }

        public final IsolineAllowOptions.Builder getAllow() {
            return allow != null ? allow.toBuilder() : null;
        }

        public final void setAllow(IsolineAllowOptions.BuilderImpl allow) {
            this.allow = allow != null ? allow.build() : null;
        }

        @Override
        public final Builder allow(IsolineAllowOptions allow) {
            this.allow = allow;
            return this;
        }

        public final String getArrivalTime() {
            return arrivalTime;
        }

        public final void setArrivalTime(String arrivalTime) {
            this.arrivalTime = arrivalTime;
        }

        @Override
        public final Builder arrivalTime(String arrivalTime) {
            this.arrivalTime = arrivalTime;
            return this;
        }

        public final IsolineAvoidanceOptions.Builder getAvoid() {
            return avoid != null ? avoid.toBuilder() : null;
        }

        public final void setAvoid(IsolineAvoidanceOptions.BuilderImpl avoid) {
            this.avoid = avoid != null ? avoid.build() : null;
        }

        @Override
        public final Builder avoid(IsolineAvoidanceOptions avoid) {
            this.avoid = avoid;
            return this;
        }

        public final Boolean getDepartNow() {
            return departNow;
        }

        public final void setDepartNow(Boolean departNow) {
            this.departNow = departNow;
        }

        @Override
        public final Builder departNow(Boolean departNow) {
            this.departNow = departNow;
            return this;
        }

        public final String getDepartureTime() {
            return departureTime;
        }

        public final void setDepartureTime(String departureTime) {
            this.departureTime = departureTime;
        }

        @Override
        public final Builder departureTime(String departureTime) {
            this.departureTime = departureTime;
            return this;
        }

        public final Collection<Double> getDestination() {
            if (destination instanceof SdkAutoConstructList) {
                return null;
            }
            return destination;
        }

        public final void setDestination(Collection<Double> destination) {
            this.destination = PositionCopier.copy(destination);
        }

        @Override
        public final Builder destination(Collection<Double> destination) {
            this.destination = PositionCopier.copy(destination);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destination(Double... destination) {
            destination(Arrays.asList(destination));
            return this;
        }

        public final IsolineDestinationOptions.Builder getDestinationOptions() {
            return destinationOptions != null ? destinationOptions.toBuilder() : null;
        }

        public final void setDestinationOptions(IsolineDestinationOptions.BuilderImpl destinationOptions) {
            this.destinationOptions = destinationOptions != null ? destinationOptions.build() : null;
        }

        @Override
        public final Builder destinationOptions(IsolineDestinationOptions destinationOptions) {
            this.destinationOptions = destinationOptions;
            return this;
        }

        public final String getIsolineGeometryFormat() {
            return isolineGeometryFormat;
        }

        public final void setIsolineGeometryFormat(String isolineGeometryFormat) {
            this.isolineGeometryFormat = isolineGeometryFormat;
        }

        @Override
        public final Builder isolineGeometryFormat(String isolineGeometryFormat) {
            this.isolineGeometryFormat = isolineGeometryFormat;
            return this;
        }

        @Override
        public final Builder isolineGeometryFormat(GeometryFormat isolineGeometryFormat) {
            this.isolineGeometryFormat(isolineGeometryFormat == null ? null : isolineGeometryFormat.toString());
            return this;
        }

        public final IsolineGranularityOptions.Builder getIsolineGranularity() {
            return isolineGranularity != null ? isolineGranularity.toBuilder() : null;
        }

        public final void setIsolineGranularity(IsolineGranularityOptions.BuilderImpl isolineGranularity) {
            this.isolineGranularity = isolineGranularity != null ? isolineGranularity.build() : null;
        }

        @Override
        public final Builder isolineGranularity(IsolineGranularityOptions isolineGranularity) {
            this.isolineGranularity = isolineGranularity;
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final String getOptimizeIsolineFor() {
            return optimizeIsolineFor;
        }

        public final void setOptimizeIsolineFor(String optimizeIsolineFor) {
            this.optimizeIsolineFor = optimizeIsolineFor;
        }

        @Override
        public final Builder optimizeIsolineFor(String optimizeIsolineFor) {
            this.optimizeIsolineFor = optimizeIsolineFor;
            return this;
        }

        @Override
        public final Builder optimizeIsolineFor(IsolineOptimizationObjective optimizeIsolineFor) {
            this.optimizeIsolineFor(optimizeIsolineFor == null ? null : optimizeIsolineFor.toString());
            return this;
        }

        public final String getOptimizeRoutingFor() {
            return optimizeRoutingFor;
        }

        public final void setOptimizeRoutingFor(String optimizeRoutingFor) {
            this.optimizeRoutingFor = optimizeRoutingFor;
        }

        @Override
        public final Builder optimizeRoutingFor(String optimizeRoutingFor) {
            this.optimizeRoutingFor = optimizeRoutingFor;
            return this;
        }

        @Override
        public final Builder optimizeRoutingFor(RoutingObjective optimizeRoutingFor) {
            this.optimizeRoutingFor(optimizeRoutingFor == null ? null : optimizeRoutingFor.toString());
            return this;
        }

        public final Collection<Double> getOrigin() {
            if (origin instanceof SdkAutoConstructList) {
                return null;
            }
            return origin;
        }

        public final void setOrigin(Collection<Double> origin) {
            this.origin = PositionCopier.copy(origin);
        }

        @Override
        public final Builder origin(Collection<Double> origin) {
            this.origin = PositionCopier.copy(origin);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder origin(Double... origin) {
            origin(Arrays.asList(origin));
            return this;
        }

        public final IsolineOriginOptions.Builder getOriginOptions() {
            return originOptions != null ? originOptions.toBuilder() : null;
        }

        public final void setOriginOptions(IsolineOriginOptions.BuilderImpl originOptions) {
            this.originOptions = originOptions != null ? originOptions.build() : null;
        }

        @Override
        public final Builder originOptions(IsolineOriginOptions originOptions) {
            this.originOptions = originOptions;
            return this;
        }

        public final IsolineThresholds.Builder getThresholds() {
            return thresholds != null ? thresholds.toBuilder() : null;
        }

        public final void setThresholds(IsolineThresholds.BuilderImpl thresholds) {
            this.thresholds = thresholds != null ? thresholds.build() : null;
        }

        @Override
        public final Builder thresholds(IsolineThresholds thresholds) {
            this.thresholds = thresholds;
            return this;
        }

        public final IsolineTrafficOptions.Builder getTraffic() {
            return traffic != null ? traffic.toBuilder() : null;
        }

        public final void setTraffic(IsolineTrafficOptions.BuilderImpl traffic) {
            this.traffic = traffic != null ? traffic.build() : null;
        }

        @Override
        public final Builder traffic(IsolineTrafficOptions traffic) {
            this.traffic = traffic;
            return this;
        }

        public final String getTravelMode() {
            return travelMode;
        }

        public final void setTravelMode(String travelMode) {
            this.travelMode = travelMode;
        }

        @Override
        public final Builder travelMode(String travelMode) {
            this.travelMode = travelMode;
            return this;
        }

        @Override
        public final Builder travelMode(IsolineTravelMode travelMode) {
            this.travelMode(travelMode == null ? null : travelMode.toString());
            return this;
        }

        public final IsolineTravelModeOptions.Builder getTravelModeOptions() {
            return travelModeOptions != null ? travelModeOptions.toBuilder() : null;
        }

        public final void setTravelModeOptions(IsolineTravelModeOptions.BuilderImpl travelModeOptions) {
            this.travelModeOptions = travelModeOptions != null ? travelModeOptions.build() : null;
        }

        @Override
        public final Builder travelModeOptions(IsolineTravelModeOptions travelModeOptions) {
            this.travelModeOptions = travelModeOptions;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CalculateIsolinesRequest build() {
            return new CalculateIsolinesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
