/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.georoutes.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Options for WaypointOptimizationClustering.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WaypointOptimizationClusteringOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<WaypointOptimizationClusteringOptions.Builder, WaypointOptimizationClusteringOptions> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Algorithm").getter(getter(WaypointOptimizationClusteringOptions::algorithmAsString))
            .setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Algorithm").build()).build();

    private static final SdkField<WaypointOptimizationDrivingDistanceOptions> DRIVING_DISTANCE_OPTIONS_FIELD = SdkField
            .<WaypointOptimizationDrivingDistanceOptions> builder(MarshallingType.SDK_POJO).memberName("DrivingDistanceOptions")
            .getter(getter(WaypointOptimizationClusteringOptions::drivingDistanceOptions))
            .setter(setter(Builder::drivingDistanceOptions)).constructor(WaypointOptimizationDrivingDistanceOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DrivingDistanceOptions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            DRIVING_DISTANCE_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final WaypointOptimizationDrivingDistanceOptions drivingDistanceOptions;

    private WaypointOptimizationClusteringOptions(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.drivingDistanceOptions = builder.drivingDistanceOptions;
    }

    /**
     * <p>
     * The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving distance
     * of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that are within the
     * same topology segment into a single cluster. A Topology segment is a linear stretch of road between two
     * junctions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link WaypointOptimizationClusteringAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving
     *         distance of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that
     *         are within the same topology segment into a single cluster. A Topology segment is a linear stretch of
     *         road between two junctions.
     * @see WaypointOptimizationClusteringAlgorithm
     */
    public final WaypointOptimizationClusteringAlgorithm algorithm() {
        return WaypointOptimizationClusteringAlgorithm.fromValue(algorithm);
    }

    /**
     * <p>
     * The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving distance
     * of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that are within the
     * same topology segment into a single cluster. A Topology segment is a linear stretch of road between two
     * junctions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link WaypointOptimizationClusteringAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving
     *         distance of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that
     *         are within the same topology segment into a single cluster. A Topology segment is a linear stretch of
     *         road between two junctions.
     * @see WaypointOptimizationClusteringAlgorithm
     */
    public final String algorithmAsString() {
        return algorithm;
    }

    /**
     * <p>
     * Driving distance options to be used when the clustering algorithm is DrivingDistance.
     * </p>
     * 
     * @return Driving distance options to be used when the clustering algorithm is DrivingDistance.
     */
    public final WaypointOptimizationDrivingDistanceOptions drivingDistanceOptions() {
        return drivingDistanceOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(drivingDistanceOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WaypointOptimizationClusteringOptions)) {
            return false;
        }
        WaypointOptimizationClusteringOptions other = (WaypointOptimizationClusteringOptions) obj;
        return Objects.equals(algorithmAsString(), other.algorithmAsString())
                && Objects.equals(drivingDistanceOptions(), other.drivingDistanceOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WaypointOptimizationClusteringOptions")
                .add("Algorithm", algorithmAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DrivingDistanceOptions", drivingDistanceOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Algorithm":
            return Optional.ofNullable(clazz.cast(algorithmAsString()));
        case "DrivingDistanceOptions":
            return Optional.ofNullable(clazz.cast(drivingDistanceOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Algorithm", ALGORITHM_FIELD);
        map.put("DrivingDistanceOptions", DRIVING_DISTANCE_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<WaypointOptimizationClusteringOptions, T> g) {
        return obj -> g.apply((WaypointOptimizationClusteringOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WaypointOptimizationClusteringOptions> {
        /**
         * <p>
         * The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving
         * distance of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that are
         * within the same topology segment into a single cluster. A Topology segment is a linear stretch of road
         * between two junctions.
         * </p>
         * 
         * @param algorithm
         *        The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within
         *        driving distance of each other into a single cluster. <code>TopologySegment</code> assigns all the
         *        waypoints that are within the same topology segment into a single cluster. A Topology segment is a
         *        linear stretch of road between two junctions.
         * @see WaypointOptimizationClusteringAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WaypointOptimizationClusteringAlgorithm
         */
        Builder algorithm(String algorithm);

        /**
         * <p>
         * The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within driving
         * distance of each other into a single cluster. <code>TopologySegment</code> assigns all the waypoints that are
         * within the same topology segment into a single cluster. A Topology segment is a linear stretch of road
         * between two junctions.
         * </p>
         * 
         * @param algorithm
         *        The algorithm to be used. <code>DrivingDistance</code> assigns all the waypoints that are within
         *        driving distance of each other into a single cluster. <code>TopologySegment</code> assigns all the
         *        waypoints that are within the same topology segment into a single cluster. A Topology segment is a
         *        linear stretch of road between two junctions.
         * @see WaypointOptimizationClusteringAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WaypointOptimizationClusteringAlgorithm
         */
        Builder algorithm(WaypointOptimizationClusteringAlgorithm algorithm);

        /**
         * <p>
         * Driving distance options to be used when the clustering algorithm is DrivingDistance.
         * </p>
         * 
         * @param drivingDistanceOptions
         *        Driving distance options to be used when the clustering algorithm is DrivingDistance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder drivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions drivingDistanceOptions);

        /**
         * <p>
         * Driving distance options to be used when the clustering algorithm is DrivingDistance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link WaypointOptimizationDrivingDistanceOptions.Builder} avoiding the need to create one manually via
         * {@link WaypointOptimizationDrivingDistanceOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WaypointOptimizationDrivingDistanceOptions.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #drivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions)}.
         * 
         * @param drivingDistanceOptions
         *        a consumer that will call methods on {@link WaypointOptimizationDrivingDistanceOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #drivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions)
         */
        default Builder drivingDistanceOptions(Consumer<WaypointOptimizationDrivingDistanceOptions.Builder> drivingDistanceOptions) {
            return drivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions.builder()
                    .applyMutation(drivingDistanceOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private WaypointOptimizationDrivingDistanceOptions drivingDistanceOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(WaypointOptimizationClusteringOptions model) {
            algorithm(model.algorithm);
            drivingDistanceOptions(model.drivingDistanceOptions);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        @Override
        public final Builder algorithm(WaypointOptimizationClusteringAlgorithm algorithm) {
            this.algorithm(algorithm == null ? null : algorithm.toString());
            return this;
        }

        public final WaypointOptimizationDrivingDistanceOptions.Builder getDrivingDistanceOptions() {
            return drivingDistanceOptions != null ? drivingDistanceOptions.toBuilder() : null;
        }

        public final void setDrivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions.BuilderImpl drivingDistanceOptions) {
            this.drivingDistanceOptions = drivingDistanceOptions != null ? drivingDistanceOptions.build() : null;
        }

        @Override
        public final Builder drivingDistanceOptions(WaypointOptimizationDrivingDistanceOptions drivingDistanceOptions) {
            this.drivingDistanceOptions = drivingDistanceOptions;
            return this;
        }

        @Override
        public WaypointOptimizationClusteringOptions build() {
            return new WaypointOptimizationClusteringOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
