/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glacier.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.services.glacier.transform.DefaultAccountIdSupplier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input value for <code>RemoveTagsFromVaultInput</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RemoveTagsFromVaultRequest extends GlacierRequest implements
        ToCopyableBuilder<RemoveTagsFromVaultRequest.Builder, RemoveTagsFromVaultRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("accountId")
            .getter(getter(RemoveTagsFromVaultRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("accountId").build(),
                    DefaultValueTrait.create(DefaultAccountIdSupplier.getInstance())).build();

    private static final SdkField<String> VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("vaultName").getter(getter(RemoveTagsFromVaultRequest::vaultName)).setter(setter(Builder::vaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("vaultName").build()).build();

    private static final SdkField<List<String>> TAG_KEYS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TagKeys")
            .getter(getter(RemoveTagsFromVaultRequest::tagKeys))
            .setter(setter(Builder::tagKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            VAULT_NAME_FIELD, TAG_KEYS_FIELD));

    private final String accountId;

    private final String vaultName;

    private final List<String> tagKeys;

    private RemoveTagsFromVaultRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.vaultName = builder.vaultName;
        this.tagKeys = builder.tagKeys;
    }

    /**
     * <p>
     * The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either specify
     * an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3 Glacier uses the AWS
     * account ID associated with the credentials used to sign the request. If you use an account ID, do not include any
     * hyphens ('-') in the ID.
     * </p>
     * 
     * @return The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either
     *         specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3
     *         Glacier uses the AWS account ID associated with the credentials used to sign the request. If you use an
     *         account ID, do not include any hyphens ('-') in the ID.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name of the vault.
     * </p>
     * 
     * @return The name of the vault.
     */
    public final String vaultName() {
        return vaultName;
    }

    /**
     * Returns true if the TagKeys property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTagKeys() {
        return tagKeys != null && !(tagKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tag keys. Each corresponding tag is removed from the vault.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTagKeys()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of tag keys. Each corresponding tag is removed from the vault.
     */
    public final List<String> tagKeys() {
        return tagKeys;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(vaultName());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagKeys() ? tagKeys() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RemoveTagsFromVaultRequest)) {
            return false;
        }
        RemoveTagsFromVaultRequest other = (RemoveTagsFromVaultRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(vaultName(), other.vaultName())
                && hasTagKeys() == other.hasTagKeys() && Objects.equals(tagKeys(), other.tagKeys());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RemoveTagsFromVaultRequest").add("AccountId", accountId()).add("VaultName", vaultName())
                .add("TagKeys", hasTagKeys() ? tagKeys() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "vaultName":
            return Optional.ofNullable(clazz.cast(vaultName()));
        case "TagKeys":
            return Optional.ofNullable(clazz.cast(tagKeys()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RemoveTagsFromVaultRequest, T> g) {
        return obj -> g.apply((RemoveTagsFromVaultRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GlacierRequest.Builder, SdkPojo, CopyableBuilder<Builder, RemoveTagsFromVaultRequest> {
        /**
         * <p>
         * The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either
         * specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3 Glacier
         * uses the AWS account ID associated with the credentials used to sign the request. If you use an account ID,
         * do not include any hyphens ('-') in the ID.
         * </p>
         * 
         * @param accountId
         *        The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can
         *        either specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case
         *        Amazon S3 Glacier uses the AWS account ID associated with the credentials used to sign the request. If
         *        you use an account ID, do not include any hyphens ('-') in the ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name of the vault.
         * </p>
         * 
         * @param vaultName
         *        The name of the vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vaultName(String vaultName);

        /**
         * <p>
         * A list of tag keys. Each corresponding tag is removed from the vault.
         * </p>
         * 
         * @param tagKeys
         *        A list of tag keys. Each corresponding tag is removed from the vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKeys(Collection<String> tagKeys);

        /**
         * <p>
         * A list of tag keys. Each corresponding tag is removed from the vault.
         * </p>
         * 
         * @param tagKeys
         *        A list of tag keys. Each corresponding tag is removed from the vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKeys(String... tagKeys);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlacierRequest.BuilderImpl implements Builder {
        private String accountId;

        private String vaultName;

        private List<String> tagKeys = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RemoveTagsFromVaultRequest model) {
            super(model);
            accountId(model.accountId);
            vaultName(model.vaultName);
            tagKeys(model.tagKeys);
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getVaultName() {
            return vaultName;
        }

        @Override
        public final Builder vaultName(String vaultName) {
            this.vaultName = vaultName;
            return this;
        }

        public final void setVaultName(String vaultName) {
            this.vaultName = vaultName;
        }

        public final Collection<String> getTagKeys() {
            if (tagKeys instanceof SdkAutoConstructList) {
                return null;
            }
            return tagKeys;
        }

        @Override
        public final Builder tagKeys(Collection<String> tagKeys) {
            this.tagKeys = TagKeyListCopier.copy(tagKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagKeys(String... tagKeys) {
            tagKeys(Arrays.asList(tagKeys));
            return this;
        }

        public final void setTagKeys(Collection<String> tagKeys) {
            this.tagKeys = TagKeyListCopier.copy(tagKeys);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RemoveTagsFromVaultRequest build() {
            return new RemoveTagsFromVaultRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
