/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A cross-account attachment in Global Accelerator. A cross-account attachment specifies the <i>principals</i> who have
 * permission to add to accelerators in their own account the resources in your account that you also list in the
 * attachment.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Attachment implements SdkPojo, Serializable, ToCopyableBuilder<Attachment.Builder, Attachment> {
    private static final SdkField<String> ATTACHMENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AttachmentArn").getter(getter(Attachment::attachmentArn)).setter(setter(Builder::attachmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Attachment::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> PRINCIPALS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Principals")
            .getter(getter(Attachment::principals))
            .setter(setter(Builder::principals))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Principals").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Resource>> RESOURCES_FIELD = SdkField
            .<List<Resource>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(Attachment::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(Attachment::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(Attachment::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTACHMENT_ARN_FIELD,
            NAME_FIELD, PRINCIPALS_FIELD, RESOURCES_FIELD, LAST_MODIFIED_TIME_FIELD, CREATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String attachmentArn;

    private final String name;

    private final List<String> principals;

    private final List<Resource> resources;

    private final Instant lastModifiedTime;

    private final Instant createdTime;

    private Attachment(BuilderImpl builder) {
        this.attachmentArn = builder.attachmentArn;
        this.name = builder.name;
        this.principals = builder.principals;
        this.resources = builder.resources;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.createdTime = builder.createdTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the cross-account attachment.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the cross-account attachment.
     */
    public final String attachmentArn() {
        return attachmentArn;
    }

    /**
     * <p>
     * The name of the cross-account attachment.
     * </p>
     * 
     * @return The name of the cross-account attachment.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Principals property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPrincipals() {
        return principals != null && !(principals instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The principals included in the cross-account attachment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrincipals} method.
     * </p>
     * 
     * @return The principals included in the cross-account attachment.
     */
    public final List<String> principals() {
        return principals;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resources included in the cross-account attachment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return The resources included in the cross-account attachment.
     */
    public final List<Resource> resources() {
        return resources;
    }

    /**
     * <p>
     * The date and time that the cross-account attachment was last modified.
     * </p>
     * 
     * @return The date and time that the cross-account attachment was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The date and time that the cross-account attachment was created.
     * </p>
     * 
     * @return The date and time that the cross-account attachment was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attachmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrincipals() ? principals() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Attachment)) {
            return false;
        }
        Attachment other = (Attachment) obj;
        return Objects.equals(attachmentArn(), other.attachmentArn()) && Objects.equals(name(), other.name())
                && hasPrincipals() == other.hasPrincipals() && Objects.equals(principals(), other.principals())
                && hasResources() == other.hasResources() && Objects.equals(resources(), other.resources())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(createdTime(), other.createdTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Attachment").add("AttachmentArn", attachmentArn()).add("Name", name())
                .add("Principals", hasPrincipals() ? principals() : null).add("Resources", hasResources() ? resources() : null)
                .add("LastModifiedTime", lastModifiedTime()).add("CreatedTime", createdTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttachmentArn":
            return Optional.ofNullable(clazz.cast(attachmentArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Principals":
            return Optional.ofNullable(clazz.cast(principals()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Attachment, T> g) {
        return obj -> g.apply((Attachment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Attachment> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the cross-account attachment.
         * </p>
         * 
         * @param attachmentArn
         *        The Amazon Resource Name (ARN) of the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentArn(String attachmentArn);

        /**
         * <p>
         * The name of the cross-account attachment.
         * </p>
         * 
         * @param name
         *        The name of the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The principals included in the cross-account attachment.
         * </p>
         * 
         * @param principals
         *        The principals included in the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principals(Collection<String> principals);

        /**
         * <p>
         * The principals included in the cross-account attachment.
         * </p>
         * 
         * @param principals
         *        The principals included in the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder principals(String... principals);

        /**
         * <p>
         * The resources included in the cross-account attachment.
         * </p>
         * 
         * @param resources
         *        The resources included in the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<Resource> resources);

        /**
         * <p>
         * The resources included in the cross-account attachment.
         * </p>
         * 
         * @param resources
         *        The resources included in the cross-account attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Resource... resources);

        /**
         * <p>
         * The resources included in the cross-account attachment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.globalaccelerator.model.Resource.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.globalaccelerator.model.Resource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.globalaccelerator.model.Resource.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(List<Resource>)}.
         * 
         * @param resources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.globalaccelerator.model.Resource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(java.util.Collection<Resource>)
         */
        Builder resources(Consumer<Resource.Builder>... resources);

        /**
         * <p>
         * The date and time that the cross-account attachment was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time that the cross-account attachment was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The date and time that the cross-account attachment was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time that the cross-account attachment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);
    }

    static final class BuilderImpl implements Builder {
        private String attachmentArn;

        private String name;

        private List<String> principals = DefaultSdkAutoConstructList.getInstance();

        private List<Resource> resources = DefaultSdkAutoConstructList.getInstance();

        private Instant lastModifiedTime;

        private Instant createdTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Attachment model) {
            attachmentArn(model.attachmentArn);
            name(model.name);
            principals(model.principals);
            resources(model.resources);
            lastModifiedTime(model.lastModifiedTime);
            createdTime(model.createdTime);
        }

        public final String getAttachmentArn() {
            return attachmentArn;
        }

        public final void setAttachmentArn(String attachmentArn) {
            this.attachmentArn = attachmentArn;
        }

        @Override
        public final Builder attachmentArn(String attachmentArn) {
            this.attachmentArn = attachmentArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getPrincipals() {
            if (principals instanceof SdkAutoConstructList) {
                return null;
            }
            return principals;
        }

        public final void setPrincipals(Collection<String> principals) {
            this.principals = PrincipalsCopier.copy(principals);
        }

        @Override
        public final Builder principals(Collection<String> principals) {
            this.principals = PrincipalsCopier.copy(principals);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder principals(String... principals) {
            principals(Arrays.asList(principals));
            return this;
        }

        public final List<Resource.Builder> getResources() {
            List<Resource.Builder> result = ResourcesCopier.copyToBuilder(this.resources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResources(Collection<Resource.BuilderImpl> resources) {
            this.resources = ResourcesCopier.copyFromBuilder(resources);
        }

        @Override
        public final Builder resources(Collection<Resource> resources) {
            this.resources = ResourcesCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Resource... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<Resource.Builder>... resources) {
            resources(Stream.of(resources).map(c -> Resource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        @Override
        public Attachment build() {
            return new Attachment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
