/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies information used to update an existing job definition. The previous job definition is completely
 * overwritten by this information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobUpdate implements SdkPojo, Serializable, ToCopyableBuilder<JobUpdate.Builder, JobUpdate> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(JobUpdate::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LOG_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("LogUri")
            .getter(getter(JobUpdate::logUri)).setter(setter(Builder::logUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogUri").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(JobUpdate::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<ExecutionProperty> EXECUTION_PROPERTY_FIELD = SdkField
            .<ExecutionProperty> builder(MarshallingType.SDK_POJO).memberName("ExecutionProperty")
            .getter(getter(JobUpdate::executionProperty)).setter(setter(Builder::executionProperty))
            .constructor(ExecutionProperty::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionProperty").build()).build();

    private static final SdkField<JobCommand> COMMAND_FIELD = SdkField.<JobCommand> builder(MarshallingType.SDK_POJO)
            .memberName("Command").getter(getter(JobUpdate::command)).setter(setter(Builder::command))
            .constructor(JobCommand::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Command").build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DefaultArguments")
            .getter(getter(JobUpdate::defaultArguments))
            .setter(setter(Builder::defaultArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultArguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> NON_OVERRIDABLE_ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("NonOverridableArguments")
            .getter(getter(JobUpdate::nonOverridableArguments))
            .setter(setter(Builder::nonOverridableArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NonOverridableArguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ConnectionsList> CONNECTIONS_FIELD = SdkField
            .<ConnectionsList> builder(MarshallingType.SDK_POJO).memberName("Connections").getter(getter(JobUpdate::connections))
            .setter(setter(Builder::connections)).constructor(ConnectionsList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Connections").build()).build();

    private static final SdkField<Integer> MAX_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRetries").getter(getter(JobUpdate::maxRetries)).setter(setter(Builder::maxRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRetries").build()).build();

    private static final SdkField<Integer> ALLOCATED_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocatedCapacity").getter(getter(JobUpdate::allocatedCapacity))
            .setter(setter(Builder::allocatedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedCapacity").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(JobUpdate::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Double> MAX_CAPACITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxCapacity").getter(getter(JobUpdate::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<String> WORKER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkerType").getter(getter(JobUpdate::workerTypeAsString)).setter(setter(Builder::workerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkerType").build()).build();

    private static final SdkField<Integer> NUMBER_OF_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfWorkers").getter(getter(JobUpdate::numberOfWorkers)).setter(setter(Builder::numberOfWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfWorkers").build()).build();

    private static final SdkField<String> SECURITY_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityConfiguration").getter(getter(JobUpdate::securityConfiguration))
            .setter(setter(Builder::securityConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityConfiguration").build())
            .build();

    private static final SdkField<NotificationProperty> NOTIFICATION_PROPERTY_FIELD = SdkField
            .<NotificationProperty> builder(MarshallingType.SDK_POJO).memberName("NotificationProperty")
            .getter(getter(JobUpdate::notificationProperty)).setter(setter(Builder::notificationProperty))
            .constructor(NotificationProperty::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationProperty").build())
            .build();

    private static final SdkField<String> GLUE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GlueVersion").getter(getter(JobUpdate::glueVersion)).setter(setter(Builder::glueVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlueVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            LOG_URI_FIELD, ROLE_FIELD, EXECUTION_PROPERTY_FIELD, COMMAND_FIELD, DEFAULT_ARGUMENTS_FIELD,
            NON_OVERRIDABLE_ARGUMENTS_FIELD, CONNECTIONS_FIELD, MAX_RETRIES_FIELD, ALLOCATED_CAPACITY_FIELD, TIMEOUT_FIELD,
            MAX_CAPACITY_FIELD, WORKER_TYPE_FIELD, NUMBER_OF_WORKERS_FIELD, SECURITY_CONFIGURATION_FIELD,
            NOTIFICATION_PROPERTY_FIELD, GLUE_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final String logUri;

    private final String role;

    private final ExecutionProperty executionProperty;

    private final JobCommand command;

    private final Map<String, String> defaultArguments;

    private final Map<String, String> nonOverridableArguments;

    private final ConnectionsList connections;

    private final Integer maxRetries;

    private final Integer allocatedCapacity;

    private final Integer timeout;

    private final Double maxCapacity;

    private final String workerType;

    private final Integer numberOfWorkers;

    private final String securityConfiguration;

    private final NotificationProperty notificationProperty;

    private final String glueVersion;

    private JobUpdate(BuilderImpl builder) {
        this.description = builder.description;
        this.logUri = builder.logUri;
        this.role = builder.role;
        this.executionProperty = builder.executionProperty;
        this.command = builder.command;
        this.defaultArguments = builder.defaultArguments;
        this.nonOverridableArguments = builder.nonOverridableArguments;
        this.connections = builder.connections;
        this.maxRetries = builder.maxRetries;
        this.allocatedCapacity = builder.allocatedCapacity;
        this.timeout = builder.timeout;
        this.maxCapacity = builder.maxCapacity;
        this.workerType = builder.workerType;
        this.numberOfWorkers = builder.numberOfWorkers;
        this.securityConfiguration = builder.securityConfiguration;
        this.notificationProperty = builder.notificationProperty;
        this.glueVersion = builder.glueVersion;
    }

    /**
     * <p>
     * Description of the job being defined.
     * </p>
     * 
     * @return Description of the job being defined.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * This field is reserved for future use.
     * </p>
     * 
     * @return This field is reserved for future use.
     */
    public final String logUri() {
        return logUri;
    }

    /**
     * <p>
     * The name or Amazon Resource Name (ARN) of the IAM role associated with this job (required).
     * </p>
     * 
     * @return The name or Amazon Resource Name (ARN) of the IAM role associated with this job (required).
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * An <code>ExecutionProperty</code> specifying the maximum number of concurrent runs allowed for this job.
     * </p>
     * 
     * @return An <code>ExecutionProperty</code> specifying the maximum number of concurrent runs allowed for this job.
     */
    public final ExecutionProperty executionProperty() {
        return executionProperty;
    }

    /**
     * <p>
     * The <code>JobCommand</code> that executes this job (required).
     * </p>
     * 
     * @return The <code>JobCommand</code> that executes this job (required).
     */
    public final JobCommand command() {
        return command;
    }

    /**
     * Returns true if the DefaultArguments property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasDefaultArguments() {
        return defaultArguments != null && !(defaultArguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The default arguments for this job.
     * </p>
     * <p>
     * You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS Glue
     * itself consumes.
     * </p>
     * <p>
     * For information about how to specify and consume your own Job arguments, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling AWS Glue APIs
     * in Python</a> topic in the developer guide.
     * </p>
     * <p>
     * For information about the key-value pairs that AWS Glue consumes to set up your job, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special Parameters
     * Used by AWS Glue</a> topic in the developer guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDefaultArguments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The default arguments for this job.</p>
     *         <p>
     *         You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS
     *         Glue itself consumes.
     *         </p>
     *         <p>
     *         For information about how to specify and consume your own Job arguments, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling AWS
     *         Glue APIs in Python</a> topic in the developer guide.
     *         </p>
     *         <p>
     *         For information about the key-value pairs that AWS Glue consumes to set up your job, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
     *         Parameters Used by AWS Glue</a> topic in the developer guide.
     */
    public final Map<String, String> defaultArguments() {
        return defaultArguments;
    }

    /**
     * Returns true if the NonOverridableArguments property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasNonOverridableArguments() {
        return nonOverridableArguments != null && !(nonOverridableArguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Non-overridable arguments for this job, specified as name-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNonOverridableArguments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Non-overridable arguments for this job, specified as name-value pairs.
     */
    public final Map<String, String> nonOverridableArguments() {
        return nonOverridableArguments;
    }

    /**
     * <p>
     * The connections used for this job.
     * </p>
     * 
     * @return The connections used for this job.
     */
    public final ConnectionsList connections() {
        return connections;
    }

    /**
     * <p>
     * The maximum number of times to retry this job if it fails.
     * </p>
     * 
     * @return The maximum number of times to retry this job if it fails.
     */
    public final Integer maxRetries() {
        return maxRetries;
    }

    /**
     * <p>
     * This field is deprecated. Use <code>MaxCapacity</code> instead.
     * </p>
     * <p>
     * The number of AWS Glue data processing units (DPUs) to allocate to this job. You can allocate from 2 to 100 DPUs;
     * the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute capacity
     * and 16 GB of memory. For more information, see the <a href="https://aws.amazon.com/glue/pricing/">AWS Glue
     * pricing page</a>.
     * </p>
     * 
     * @return This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
     *         <p>
     *         The number of AWS Glue data processing units (DPUs) to allocate to this job. You can allocate from 2 to
     *         100 DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of
     *         compute capacity and 16 GB of memory. For more information, see the <a
     *         href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
     */
    public final Integer allocatedCapacity() {
        return allocatedCapacity;
    }

    /**
     * <p>
     * The job timeout in minutes. This is the maximum time that a job run can consume resources before it is terminated
     * and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours).
     * </p>
     * 
     * @return The job timeout in minutes. This is the maximum time that a job run can consume resources before it is
     *         terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours).
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a relative
     * measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more
     * information, see the <a href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
     * </p>
     * <p>
     * Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     * </p>
     * <p>
     * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python shell
     * job or an Apache Spark ETL job:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either 0.0625
     * or 1 DPU. The default is 0.0625 DPU.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark streaming ETL
     * job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs. The default is 10 DPUs.
     * This job type cannot have a fractional DPU allocation.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a
     *         relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory.
     *         For more information, see the <a href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing
     *         page</a>.</p>
     *         <p>
     *         Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     *         </p>
     *         <p>
     *         The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
     *         shell job or an Apache Spark ETL job:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
     *         0.0625 or 1 DPU. The default is 0.0625 DPU.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark
     *         streaming ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs.
     *         The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
     *         </p>
     *         </li>
     */
    public final Double maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and
     * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and
     * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
     *         G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk),
     *         and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk),
     *         and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final WorkerType workerType() {
        return WorkerType.fromValue(workerType);
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and
     * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and
     * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
     *         G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk),
     *         and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk),
     *         and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final String workerTypeAsString() {
        return workerType;
    }

    /**
     * <p>
     * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
     * </p>
     * <p>
     * The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
     * </p>
     * 
     * @return The number of workers of a defined <code>workerType</code> that are allocated when a job runs.</p>
     *         <p>
     *         The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
     */
    public final Integer numberOfWorkers() {
        return numberOfWorkers;
    }

    /**
     * <p>
     * The name of the <code>SecurityConfiguration</code> structure to be used with this job.
     * </p>
     * 
     * @return The name of the <code>SecurityConfiguration</code> structure to be used with this job.
     */
    public final String securityConfiguration() {
        return securityConfiguration;
    }

    /**
     * <p>
     * Specifies the configuration properties of a job notification.
     * </p>
     * 
     * @return Specifies the configuration properties of a job notification.
     */
    public final NotificationProperty notificationProperty() {
        return notificationProperty;
    }

    /**
     * <p>
     * Glue version determines the versions of Apache Spark and Python that AWS Glue supports. The Python version
     * indicates the version supported for jobs of type Spark.
     * </p>
     * <p>
     * For more information about the available AWS Glue versions and corresponding Spark and Python versions, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer guide.
     * </p>
     * 
     * @return Glue version determines the versions of Apache Spark and Python that AWS Glue supports. The Python
     *         version indicates the version supported for jobs of type Spark. </p>
     *         <p>
     *         For more information about the available AWS Glue versions and corresponding Spark and Python versions,
     *         see <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer
     *         guide.
     */
    public final String glueVersion() {
        return glueVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(logUri());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(executionProperty());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(hasDefaultArguments() ? defaultArguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNonOverridableArguments() ? nonOverridableArguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connections());
        hashCode = 31 * hashCode + Objects.hashCode(maxRetries());
        hashCode = 31 * hashCode + Objects.hashCode(allocatedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(workerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(securityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(notificationProperty());
        hashCode = 31 * hashCode + Objects.hashCode(glueVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobUpdate)) {
            return false;
        }
        JobUpdate other = (JobUpdate) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(logUri(), other.logUri())
                && Objects.equals(role(), other.role()) && Objects.equals(executionProperty(), other.executionProperty())
                && Objects.equals(command(), other.command()) && hasDefaultArguments() == other.hasDefaultArguments()
                && Objects.equals(defaultArguments(), other.defaultArguments())
                && hasNonOverridableArguments() == other.hasNonOverridableArguments()
                && Objects.equals(nonOverridableArguments(), other.nonOverridableArguments())
                && Objects.equals(connections(), other.connections()) && Objects.equals(maxRetries(), other.maxRetries())
                && Objects.equals(allocatedCapacity(), other.allocatedCapacity()) && Objects.equals(timeout(), other.timeout())
                && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(workerTypeAsString(), other.workerTypeAsString())
                && Objects.equals(numberOfWorkers(), other.numberOfWorkers())
                && Objects.equals(securityConfiguration(), other.securityConfiguration())
                && Objects.equals(notificationProperty(), other.notificationProperty())
                && Objects.equals(glueVersion(), other.glueVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobUpdate").add("Description", description()).add("LogUri", logUri()).add("Role", role())
                .add("ExecutionProperty", executionProperty()).add("Command", command())
                .add("DefaultArguments", hasDefaultArguments() ? defaultArguments() : null)
                .add("NonOverridableArguments", hasNonOverridableArguments() ? nonOverridableArguments() : null)
                .add("Connections", connections()).add("MaxRetries", maxRetries()).add("AllocatedCapacity", allocatedCapacity())
                .add("Timeout", timeout()).add("MaxCapacity", maxCapacity()).add("WorkerType", workerTypeAsString())
                .add("NumberOfWorkers", numberOfWorkers()).add("SecurityConfiguration", securityConfiguration())
                .add("NotificationProperty", notificationProperty()).add("GlueVersion", glueVersion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LogUri":
            return Optional.ofNullable(clazz.cast(logUri()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "ExecutionProperty":
            return Optional.ofNullable(clazz.cast(executionProperty()));
        case "Command":
            return Optional.ofNullable(clazz.cast(command()));
        case "DefaultArguments":
            return Optional.ofNullable(clazz.cast(defaultArguments()));
        case "NonOverridableArguments":
            return Optional.ofNullable(clazz.cast(nonOverridableArguments()));
        case "Connections":
            return Optional.ofNullable(clazz.cast(connections()));
        case "MaxRetries":
            return Optional.ofNullable(clazz.cast(maxRetries()));
        case "AllocatedCapacity":
            return Optional.ofNullable(clazz.cast(allocatedCapacity()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "WorkerType":
            return Optional.ofNullable(clazz.cast(workerTypeAsString()));
        case "NumberOfWorkers":
            return Optional.ofNullable(clazz.cast(numberOfWorkers()));
        case "SecurityConfiguration":
            return Optional.ofNullable(clazz.cast(securityConfiguration()));
        case "NotificationProperty":
            return Optional.ofNullable(clazz.cast(notificationProperty()));
        case "GlueVersion":
            return Optional.ofNullable(clazz.cast(glueVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobUpdate, T> g) {
        return obj -> g.apply((JobUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobUpdate> {
        /**
         * <p>
         * Description of the job being defined.
         * </p>
         * 
         * @param description
         *        Description of the job being defined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * This field is reserved for future use.
         * </p>
         * 
         * @param logUri
         *        This field is reserved for future use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logUri(String logUri);

        /**
         * <p>
         * The name or Amazon Resource Name (ARN) of the IAM role associated with this job (required).
         * </p>
         * 
         * @param role
         *        The name or Amazon Resource Name (ARN) of the IAM role associated with this job (required).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * An <code>ExecutionProperty</code> specifying the maximum number of concurrent runs allowed for this job.
         * </p>
         * 
         * @param executionProperty
         *        An <code>ExecutionProperty</code> specifying the maximum number of concurrent runs allowed for this
         *        job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionProperty(ExecutionProperty executionProperty);

        /**
         * <p>
         * An <code>ExecutionProperty</code> specifying the maximum number of concurrent runs allowed for this job.
         * </p>
         * This is a convenience that creates an instance of the {@link ExecutionProperty.Builder} avoiding the need to
         * create one manually via {@link ExecutionProperty#builder()}.
         *
         * When the {@link Consumer} completes, {@link ExecutionProperty.Builder#build()} is called immediately and its
         * result is passed to {@link #executionProperty(ExecutionProperty)}.
         * 
         * @param executionProperty
         *        a consumer that will call methods on {@link ExecutionProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #executionProperty(ExecutionProperty)
         */
        default Builder executionProperty(Consumer<ExecutionProperty.Builder> executionProperty) {
            return executionProperty(ExecutionProperty.builder().applyMutation(executionProperty).build());
        }

        /**
         * <p>
         * The <code>JobCommand</code> that executes this job (required).
         * </p>
         * 
         * @param command
         *        The <code>JobCommand</code> that executes this job (required).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(JobCommand command);

        /**
         * <p>
         * The <code>JobCommand</code> that executes this job (required).
         * </p>
         * This is a convenience that creates an instance of the {@link JobCommand.Builder} avoiding the need to create
         * one manually via {@link JobCommand#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobCommand.Builder#build()} is called immediately and its result
         * is passed to {@link #command(JobCommand)}.
         * 
         * @param command
         *        a consumer that will call methods on {@link JobCommand.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #command(JobCommand)
         */
        default Builder command(Consumer<JobCommand.Builder> command) {
            return command(JobCommand.builder().applyMutation(command).build());
        }

        /**
         * <p>
         * The default arguments for this job.
         * </p>
         * <p>
         * You can specify arguments here that your own job-execution script consumes, as well as arguments that AWS
         * Glue itself consumes.
         * </p>
         * <p>
         * For information about how to specify and consume your own Job arguments, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling AWS Glue
         * APIs in Python</a> topic in the developer guide.
         * </p>
         * <p>
         * For information about the key-value pairs that AWS Glue consumes to set up your job, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
         * Parameters Used by AWS Glue</a> topic in the developer guide.
         * </p>
         * 
         * @param defaultArguments
         *        The default arguments for this job.</p>
         *        <p>
         *        You can specify arguments here that your own job-execution script consumes, as well as arguments that
         *        AWS Glue itself consumes.
         *        </p>
         *        <p>
         *        For information about how to specify and consume your own Job arguments, see the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling AWS
         *        Glue APIs in Python</a> topic in the developer guide.
         *        </p>
         *        <p>
         *        For information about the key-value pairs that AWS Glue consumes to set up your job, see the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
         *        Parameters Used by AWS Glue</a> topic in the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultArguments(Map<String, String> defaultArguments);

        /**
         * <p>
         * Non-overridable arguments for this job, specified as name-value pairs.
         * </p>
         * 
         * @param nonOverridableArguments
         *        Non-overridable arguments for this job, specified as name-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonOverridableArguments(Map<String, String> nonOverridableArguments);

        /**
         * <p>
         * The connections used for this job.
         * </p>
         * 
         * @param connections
         *        The connections used for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connections(ConnectionsList connections);

        /**
         * <p>
         * The connections used for this job.
         * </p>
         * This is a convenience that creates an instance of the {@link ConnectionsList.Builder} avoiding the need to
         * create one manually via {@link ConnectionsList#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConnectionsList.Builder#build()} is called immediately and its
         * result is passed to {@link #connections(ConnectionsList)}.
         * 
         * @param connections
         *        a consumer that will call methods on {@link ConnectionsList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connections(ConnectionsList)
         */
        default Builder connections(Consumer<ConnectionsList.Builder> connections) {
            return connections(ConnectionsList.builder().applyMutation(connections).build());
        }

        /**
         * <p>
         * The maximum number of times to retry this job if it fails.
         * </p>
         * 
         * @param maxRetries
         *        The maximum number of times to retry this job if it fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetries(Integer maxRetries);

        /**
         * <p>
         * This field is deprecated. Use <code>MaxCapacity</code> instead.
         * </p>
         * <p>
         * The number of AWS Glue data processing units (DPUs) to allocate to this job. You can allocate from 2 to 100
         * DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute
         * capacity and 16 GB of memory. For more information, see the <a
         * href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
         * </p>
         * 
         * @param allocatedCapacity
         *        This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
         *        <p>
         *        The number of AWS Glue data processing units (DPUs) to allocate to this job. You can allocate from 2
         *        to 100 DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4
         *        vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
         *        href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocatedCapacity(Integer allocatedCapacity);

        /**
         * <p>
         * The job timeout in minutes. This is the maximum time that a job run can consume resources before it is
         * terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours).
         * </p>
         * 
         * @param timeout
         *        The job timeout in minutes. This is the maximum time that a job run can consume resources before it is
         *        terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a
         * relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For
         * more information, see the <a href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
         * </p>
         * <p>
         * Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
         * </p>
         * <p>
         * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
         * shell job or an Apache Spark ETL job:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
         * 0.0625 or 1 DPU. The default is 0.0625 DPU.
         * </p>
         * </li>
         * <li>
         * <p>
         * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark streaming
         * ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs. The default is
         * 10 DPUs. This job type cannot have a fractional DPU allocation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param maxCapacity
         *        The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is
         *        a relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of
         *        memory. For more information, see the <a href="https://aws.amazon.com/glue/pricing/">AWS Glue pricing
         *        page</a>.</p>
         *        <p>
         *        Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>
         *        .
         *        </p>
         *        <p>
         *        The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a
         *        Python shell job or an Apache Spark ETL job:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate
         *        either 0.0625 or 1 DPU. The default is 0.0625 DPU.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark
         *        streaming ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs.
         *        The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Double maxCapacity);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and
         * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and
         * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
         *        G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB
         *        disk), and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB
         *        disk), and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(String workerType);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and
         * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and
         * provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
         *        G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB
         *        disk), and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB
         *        disk), and provides 1 executor per worker. We recommend this worker type for memory-intensive jobs.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(WorkerType workerType);

        /**
         * <p>
         * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
         * </p>
         * <p>
         * The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
         * </p>
         * 
         * @param numberOfWorkers
         *        The number of workers of a defined <code>workerType</code> that are allocated when a job runs.</p>
         *        <p>
         *        The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for
         *        <code>G.2X</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfWorkers(Integer numberOfWorkers);

        /**
         * <p>
         * The name of the <code>SecurityConfiguration</code> structure to be used with this job.
         * </p>
         * 
         * @param securityConfiguration
         *        The name of the <code>SecurityConfiguration</code> structure to be used with this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityConfiguration(String securityConfiguration);

        /**
         * <p>
         * Specifies the configuration properties of a job notification.
         * </p>
         * 
         * @param notificationProperty
         *        Specifies the configuration properties of a job notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationProperty(NotificationProperty notificationProperty);

        /**
         * <p>
         * Specifies the configuration properties of a job notification.
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationProperty.Builder} avoiding the need
         * to create one manually via {@link NotificationProperty#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #notificationProperty(NotificationProperty)}.
         * 
         * @param notificationProperty
         *        a consumer that will call methods on {@link NotificationProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationProperty(NotificationProperty)
         */
        default Builder notificationProperty(Consumer<NotificationProperty.Builder> notificationProperty) {
            return notificationProperty(NotificationProperty.builder().applyMutation(notificationProperty).build());
        }

        /**
         * <p>
         * Glue version determines the versions of Apache Spark and Python that AWS Glue supports. The Python version
         * indicates the version supported for jobs of type Spark.
         * </p>
         * <p>
         * For more information about the available AWS Glue versions and corresponding Spark and Python versions, see
         * <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer guide.
         * </p>
         * 
         * @param glueVersion
         *        Glue version determines the versions of Apache Spark and Python that AWS Glue supports. The Python
         *        version indicates the version supported for jobs of type Spark. </p>
         *        <p>
         *        For more information about the available AWS Glue versions and corresponding Spark and Python
         *        versions, see <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in
         *        the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueVersion(String glueVersion);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private String logUri;

        private String role;

        private ExecutionProperty executionProperty;

        private JobCommand command;

        private Map<String, String> defaultArguments = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> nonOverridableArguments = DefaultSdkAutoConstructMap.getInstance();

        private ConnectionsList connections;

        private Integer maxRetries;

        private Integer allocatedCapacity;

        private Integer timeout;

        private Double maxCapacity;

        private String workerType;

        private Integer numberOfWorkers;

        private String securityConfiguration;

        private NotificationProperty notificationProperty;

        private String glueVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(JobUpdate model) {
            description(model.description);
            logUri(model.logUri);
            role(model.role);
            executionProperty(model.executionProperty);
            command(model.command);
            defaultArguments(model.defaultArguments);
            nonOverridableArguments(model.nonOverridableArguments);
            connections(model.connections);
            maxRetries(model.maxRetries);
            allocatedCapacity(model.allocatedCapacity);
            timeout(model.timeout);
            maxCapacity(model.maxCapacity);
            workerType(model.workerType);
            numberOfWorkers(model.numberOfWorkers);
            securityConfiguration(model.securityConfiguration);
            notificationProperty(model.notificationProperty);
            glueVersion(model.glueVersion);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getLogUri() {
            return logUri;
        }

        @Override
        public final Builder logUri(String logUri) {
            this.logUri = logUri;
            return this;
        }

        public final void setLogUri(String logUri) {
            this.logUri = logUri;
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final ExecutionProperty.Builder getExecutionProperty() {
            return executionProperty != null ? executionProperty.toBuilder() : null;
        }

        @Override
        public final Builder executionProperty(ExecutionProperty executionProperty) {
            this.executionProperty = executionProperty;
            return this;
        }

        public final void setExecutionProperty(ExecutionProperty.BuilderImpl executionProperty) {
            this.executionProperty = executionProperty != null ? executionProperty.build() : null;
        }

        public final JobCommand.Builder getCommand() {
            return command != null ? command.toBuilder() : null;
        }

        @Override
        public final Builder command(JobCommand command) {
            this.command = command;
            return this;
        }

        public final void setCommand(JobCommand.BuilderImpl command) {
            this.command = command != null ? command.build() : null;
        }

        public final Map<String, String> getDefaultArguments() {
            if (defaultArguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return defaultArguments;
        }

        @Override
        public final Builder defaultArguments(Map<String, String> defaultArguments) {
            this.defaultArguments = GenericMapCopier.copy(defaultArguments);
            return this;
        }

        public final void setDefaultArguments(Map<String, String> defaultArguments) {
            this.defaultArguments = GenericMapCopier.copy(defaultArguments);
        }

        public final Map<String, String> getNonOverridableArguments() {
            if (nonOverridableArguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return nonOverridableArguments;
        }

        @Override
        public final Builder nonOverridableArguments(Map<String, String> nonOverridableArguments) {
            this.nonOverridableArguments = GenericMapCopier.copy(nonOverridableArguments);
            return this;
        }

        public final void setNonOverridableArguments(Map<String, String> nonOverridableArguments) {
            this.nonOverridableArguments = GenericMapCopier.copy(nonOverridableArguments);
        }

        public final ConnectionsList.Builder getConnections() {
            return connections != null ? connections.toBuilder() : null;
        }

        @Override
        public final Builder connections(ConnectionsList connections) {
            this.connections = connections;
            return this;
        }

        public final void setConnections(ConnectionsList.BuilderImpl connections) {
            this.connections = connections != null ? connections.build() : null;
        }

        public final Integer getMaxRetries() {
            return maxRetries;
        }

        @Override
        public final Builder maxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public final void setMaxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
        }

        public final Integer getAllocatedCapacity() {
            return allocatedCapacity;
        }

        @Override
        public final Builder allocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
            return this;
        }

        public final void setAllocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        public final Double getMaxCapacity() {
            return maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final void setMaxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        public final String getWorkerType() {
            return workerType;
        }

        @Override
        public final Builder workerType(String workerType) {
            this.workerType = workerType;
            return this;
        }

        @Override
        public final Builder workerType(WorkerType workerType) {
            this.workerType(workerType == null ? null : workerType.toString());
            return this;
        }

        public final void setWorkerType(String workerType) {
            this.workerType = workerType;
        }

        public final Integer getNumberOfWorkers() {
            return numberOfWorkers;
        }

        @Override
        public final Builder numberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
            return this;
        }

        public final void setNumberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
        }

        public final String getSecurityConfiguration() {
            return securityConfiguration;
        }

        @Override
        public final Builder securityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return this;
        }

        public final void setSecurityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
        }

        public final NotificationProperty.Builder getNotificationProperty() {
            return notificationProperty != null ? notificationProperty.toBuilder() : null;
        }

        @Override
        public final Builder notificationProperty(NotificationProperty notificationProperty) {
            this.notificationProperty = notificationProperty;
            return this;
        }

        public final void setNotificationProperty(NotificationProperty.BuilderImpl notificationProperty) {
            this.notificationProperty = notificationProperty != null ? notificationProperty.build() : null;
        }

        public final String getGlueVersion() {
            return glueVersion;
        }

        @Override
        public final Builder glueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
            return this;
        }

        public final void setGlueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
        }

        @Override
        public JobUpdate build() {
            return new JobUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
