/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure for a machine learning transform.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MLTransform implements SdkPojo, Serializable, ToCopyableBuilder<MLTransform.Builder, MLTransform> {
    private static final SdkField<String> TRANSFORM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransformId").getter(getter(MLTransform::transformId)).setter(setter(Builder::transformId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(MLTransform::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(MLTransform::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MLTransform::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedOn").getter(getter(MLTransform::createdOn)).setter(setter(Builder::createdOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedOn").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedOn").getter(getter(MLTransform::lastModifiedOn)).setter(setter(Builder::lastModifiedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedOn").build()).build();

    private static final SdkField<List<GlueTable>> INPUT_RECORD_TABLES_FIELD = SdkField
            .<List<GlueTable>> builder(MarshallingType.LIST)
            .memberName("InputRecordTables")
            .getter(getter(MLTransform::inputRecordTables))
            .setter(setter(Builder::inputRecordTables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputRecordTables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GlueTable> builder(MarshallingType.SDK_POJO)
                                            .constructor(GlueTable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TransformParameters> PARAMETERS_FIELD = SdkField
            .<TransformParameters> builder(MarshallingType.SDK_POJO).memberName("Parameters")
            .getter(getter(MLTransform::parameters)).setter(setter(Builder::parameters))
            .constructor(TransformParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build()).build();

    private static final SdkField<EvaluationMetrics> EVALUATION_METRICS_FIELD = SdkField
            .<EvaluationMetrics> builder(MarshallingType.SDK_POJO).memberName("EvaluationMetrics")
            .getter(getter(MLTransform::evaluationMetrics)).setter(setter(Builder::evaluationMetrics))
            .constructor(EvaluationMetrics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationMetrics").build()).build();

    private static final SdkField<Integer> LABEL_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("LabelCount").getter(getter(MLTransform::labelCount)).setter(setter(Builder::labelCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelCount").build()).build();

    private static final SdkField<List<SchemaColumn>> SCHEMA_FIELD = SdkField
            .<List<SchemaColumn>> builder(MarshallingType.LIST)
            .memberName("Schema")
            .getter(getter(MLTransform::schema))
            .setter(setter(Builder::schema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SchemaColumn> builder(MarshallingType.SDK_POJO)
                                            .constructor(SchemaColumn::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(MLTransform::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<String> GLUE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GlueVersion").getter(getter(MLTransform::glueVersion)).setter(setter(Builder::glueVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlueVersion").build()).build();

    private static final SdkField<Double> MAX_CAPACITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxCapacity").getter(getter(MLTransform::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<String> WORKER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkerType").getter(getter(MLTransform::workerTypeAsString)).setter(setter(Builder::workerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkerType").build()).build();

    private static final SdkField<Integer> NUMBER_OF_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfWorkers").getter(getter(MLTransform::numberOfWorkers)).setter(setter(Builder::numberOfWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfWorkers").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(MLTransform::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Integer> MAX_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRetries").getter(getter(MLTransform::maxRetries)).setter(setter(Builder::maxRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRetries").build()).build();

    private static final SdkField<TransformEncryption> TRANSFORM_ENCRYPTION_FIELD = SdkField
            .<TransformEncryption> builder(MarshallingType.SDK_POJO).memberName("TransformEncryption")
            .getter(getter(MLTransform::transformEncryption)).setter(setter(Builder::transformEncryption))
            .constructor(TransformEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformEncryption").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSFORM_ID_FIELD,
            NAME_FIELD, DESCRIPTION_FIELD, STATUS_FIELD, CREATED_ON_FIELD, LAST_MODIFIED_ON_FIELD, INPUT_RECORD_TABLES_FIELD,
            PARAMETERS_FIELD, EVALUATION_METRICS_FIELD, LABEL_COUNT_FIELD, SCHEMA_FIELD, ROLE_FIELD, GLUE_VERSION_FIELD,
            MAX_CAPACITY_FIELD, WORKER_TYPE_FIELD, NUMBER_OF_WORKERS_FIELD, TIMEOUT_FIELD, MAX_RETRIES_FIELD,
            TRANSFORM_ENCRYPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transformId;

    private final String name;

    private final String description;

    private final String status;

    private final Instant createdOn;

    private final Instant lastModifiedOn;

    private final List<GlueTable> inputRecordTables;

    private final TransformParameters parameters;

    private final EvaluationMetrics evaluationMetrics;

    private final Integer labelCount;

    private final List<SchemaColumn> schema;

    private final String role;

    private final String glueVersion;

    private final Double maxCapacity;

    private final String workerType;

    private final Integer numberOfWorkers;

    private final Integer timeout;

    private final Integer maxRetries;

    private final TransformEncryption transformEncryption;

    private MLTransform(BuilderImpl builder) {
        this.transformId = builder.transformId;
        this.name = builder.name;
        this.description = builder.description;
        this.status = builder.status;
        this.createdOn = builder.createdOn;
        this.lastModifiedOn = builder.lastModifiedOn;
        this.inputRecordTables = builder.inputRecordTables;
        this.parameters = builder.parameters;
        this.evaluationMetrics = builder.evaluationMetrics;
        this.labelCount = builder.labelCount;
        this.schema = builder.schema;
        this.role = builder.role;
        this.glueVersion = builder.glueVersion;
        this.maxCapacity = builder.maxCapacity;
        this.workerType = builder.workerType;
        this.numberOfWorkers = builder.numberOfWorkers;
        this.timeout = builder.timeout;
        this.maxRetries = builder.maxRetries;
        this.transformEncryption = builder.transformEncryption;
    }

    /**
     * <p>
     * The unique transform ID that is generated for the machine learning transform. The ID is guaranteed to be unique
     * and does not change.
     * </p>
     * 
     * @return The unique transform ID that is generated for the machine learning transform. The ID is guaranteed to be
     *         unique and does not change.
     */
    public final String transformId() {
        return transformId;
    }

    /**
     * <p>
     * A user-defined name for the machine learning transform. Names are not guaranteed unique and can be changed at any
     * time.
     * </p>
     * 
     * @return A user-defined name for the machine learning transform. Names are not guaranteed unique and can be
     *         changed at any time.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A user-defined, long-form description text for the machine learning transform. Descriptions are not guaranteed to
     * be unique and can be changed at any time.
     * </p>
     * 
     * @return A user-defined, long-form description text for the machine learning transform. Descriptions are not
     *         guaranteed to be unique and can be changed at any time.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The current status of the machine learning transform.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the machine learning transform.
     * @see TransformStatusType
     */
    public final TransformStatusType status() {
        return TransformStatusType.fromValue(status);
    }

    /**
     * <p>
     * The current status of the machine learning transform.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the machine learning transform.
     * @see TransformStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A timestamp. The time and date that this machine learning transform was created.
     * </p>
     * 
     * @return A timestamp. The time and date that this machine learning transform was created.
     */
    public final Instant createdOn() {
        return createdOn;
    }

    /**
     * <p>
     * A timestamp. The last point in time when this machine learning transform was modified.
     * </p>
     * 
     * @return A timestamp. The last point in time when this machine learning transform was modified.
     */
    public final Instant lastModifiedOn() {
        return lastModifiedOn;
    }

    /**
     * Returns true if the InputRecordTables property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasInputRecordTables() {
        return inputRecordTables != null && !(inputRecordTables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of AWS Glue table definitions used by the transform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInputRecordTables()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of AWS Glue table definitions used by the transform.
     */
    public final List<GlueTable> inputRecordTables() {
        return inputRecordTables;
    }

    /**
     * <p>
     * A <code>TransformParameters</code> object. You can use parameters to tune (customize) the behavior of the machine
     * learning transform by specifying what data it learns from and your preference on various tradeoffs (such as
     * precious vs. recall, or accuracy vs. cost).
     * </p>
     * 
     * @return A <code>TransformParameters</code> object. You can use parameters to tune (customize) the behavior of the
     *         machine learning transform by specifying what data it learns from and your preference on various
     *         tradeoffs (such as precious vs. recall, or accuracy vs. cost).
     */
    public final TransformParameters parameters() {
        return parameters;
    }

    /**
     * <p>
     * An <code>EvaluationMetrics</code> object. Evaluation metrics provide an estimate of the quality of your machine
     * learning transform.
     * </p>
     * 
     * @return An <code>EvaluationMetrics</code> object. Evaluation metrics provide an estimate of the quality of your
     *         machine learning transform.
     */
    public final EvaluationMetrics evaluationMetrics() {
        return evaluationMetrics;
    }

    /**
     * <p>
     * A count identifier for the labeling files generated by AWS Glue for this transform. As you create a better
     * transform, you can iteratively download, label, and upload the labeling file.
     * </p>
     * 
     * @return A count identifier for the labeling files generated by AWS Glue for this transform. As you create a
     *         better transform, you can iteratively download, label, and upload the labeling file.
     */
    public final Integer labelCount() {
        return labelCount;
    }

    /**
     * Returns true if the Schema property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSchema() {
        return schema != null && !(schema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A map of key-value pairs representing the columns and data types that this transform can run against. Has an
     * upper bound of 100 columns.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSchema()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of key-value pairs representing the columns and data types that this transform can run against. Has
     *         an upper bound of 100 columns.
     */
    public final List<SchemaColumn> schema() {
        return schema;
    }

    /**
     * <p>
     * The name or Amazon Resource Name (ARN) of the IAM role with the required permissions. The required permissions
     * include both AWS Glue service role permissions to AWS Glue resources, and Amazon S3 permissions required by the
     * transform.
     * </p>
     * <ul>
     * <li>
     * <p>
     * This role needs AWS Glue service role permissions to allow access to resources in AWS Glue. See <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/attach-policy-iam-user.html">Attach a Policy to IAM Users That
     * Access AWS Glue</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * This role needs permission to your Amazon Simple Storage Service (Amazon S3) sources, targets, temporary
     * directory, scripts, and any libraries used by the task run for this transform.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name or Amazon Resource Name (ARN) of the IAM role with the required permissions. The required
     *         permissions include both AWS Glue service role permissions to AWS Glue resources, and Amazon S3
     *         permissions required by the transform. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         This role needs AWS Glue service role permissions to allow access to resources in AWS Glue. See <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/attach-policy-iam-user.html">Attach a Policy to IAM
     *         Users That Access AWS Glue</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         This role needs permission to your Amazon Simple Storage Service (Amazon S3) sources, targets, temporary
     *         directory, scripts, and any libraries used by the task run for this transform.
     *         </p>
     *         </li>
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * This value determines which version of AWS Glue this machine learning transform is compatible with. Glue 1.0 is
     * recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue 0.9. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">AWS Glue Versions</a>
     * in the developer guide.
     * </p>
     * 
     * @return This value determines which version of AWS Glue this machine learning transform is compatible with. Glue
     *         1.0 is recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue
     *         0.9. For more information, see <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">AWS Glue
     *         Versions</a> in the developer guide.
     */
    public final String glueVersion() {
        return glueVersion;
    }

    /**
     * <p>
     * The number of AWS Glue data processing units (DPUs) that are allocated to task runs for this transform. You can
     * allocate from 2 to 100 DPUs; the default is 10. A DPU is a relative measure of processing power that consists of
     * 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
     * href="http://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
     * </p>
     * <p>
     * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     * <code>WorkerType</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code> cannot be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When the <code>WorkerType</code> field is set to a value other than <code>Standard</code>, the
     * <code>MaxCapacity</code> field is set automatically and becomes read-only.
     * </p>
     * 
     * @return The number of AWS Glue data processing units (DPUs) that are allocated to task runs for this transform.
     *         You can allocate from 2 to 100 DPUs; the default is 10. A DPU is a relative measure of processing power
     *         that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
     *         href="http://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>. </p>
     *         <p>
     *         <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     *         <code>WorkerType</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
     *         cannot be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code>
     *         can be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When the <code>WorkerType</code> field is set to a value other than <code>Standard</code>, the
     *         <code>MaxCapacity</code> field is set automatically and becomes read-only.
     */
    public final Double maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of Standard,
     * G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     * <code>WorkerType</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code> cannot be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
     *         Standard, G.1X, or G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     *         <code>WorkerType</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
     *         cannot be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code>
     *         can be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final WorkerType workerType() {
        return WorkerType.fromValue(workerType);
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of Standard,
     * G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     * <code>WorkerType</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code> cannot be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can be
     * set.
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
     *         Standard, G.1X, or G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
     *         <code>WorkerType</code>.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
     *         cannot be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code>
     *         can be set.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final String workerTypeAsString() {
        return workerType;
    }

    /**
     * <p>
     * The number of workers of a defined <code>workerType</code> that are allocated when a task of the transform runs.
     * </p>
     * <p>
     * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     * </p>
     * 
     * @return The number of workers of a defined <code>workerType</code> that are allocated when a task of the
     *         transform runs.</p>
     *         <p>
     *         If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
     */
    public final Integer numberOfWorkers() {
        return numberOfWorkers;
    }

    /**
     * <p>
     * The timeout in minutes of the machine learning transform.
     * </p>
     * 
     * @return The timeout in minutes of the machine learning transform.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * The maximum number of times to retry after an <code>MLTaskRun</code> of the machine learning transform fails.
     * </p>
     * 
     * @return The maximum number of times to retry after an <code>MLTaskRun</code> of the machine learning transform
     *         fails.
     */
    public final Integer maxRetries() {
        return maxRetries;
    }

    /**
     * <p>
     * The encryption-at-rest settings of the transform that apply to accessing user data. Machine learning transforms
     * can access user data encrypted in Amazon S3 using KMS.
     * </p>
     * 
     * @return The encryption-at-rest settings of the transform that apply to accessing user data. Machine learning
     *         transforms can access user data encrypted in Amazon S3 using KMS.
     */
    public final TransformEncryption transformEncryption() {
        return transformEncryption;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transformId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedOn());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputRecordTables() ? inputRecordTables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(labelCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasSchema() ? schema() : null);
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(glueVersion());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(workerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(maxRetries());
        hashCode = 31 * hashCode + Objects.hashCode(transformEncryption());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MLTransform)) {
            return false;
        }
        MLTransform other = (MLTransform) obj;
        return Objects.equals(transformId(), other.transformId()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(createdOn(), other.createdOn()) && Objects.equals(lastModifiedOn(), other.lastModifiedOn())
                && hasInputRecordTables() == other.hasInputRecordTables()
                && Objects.equals(inputRecordTables(), other.inputRecordTables())
                && Objects.equals(parameters(), other.parameters())
                && Objects.equals(evaluationMetrics(), other.evaluationMetrics())
                && Objects.equals(labelCount(), other.labelCount()) && hasSchema() == other.hasSchema()
                && Objects.equals(schema(), other.schema()) && Objects.equals(role(), other.role())
                && Objects.equals(glueVersion(), other.glueVersion()) && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(workerTypeAsString(), other.workerTypeAsString())
                && Objects.equals(numberOfWorkers(), other.numberOfWorkers()) && Objects.equals(timeout(), other.timeout())
                && Objects.equals(maxRetries(), other.maxRetries())
                && Objects.equals(transformEncryption(), other.transformEncryption());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MLTransform").add("TransformId", transformId()).add("Name", name())
                .add("Description", description()).add("Status", statusAsString()).add("CreatedOn", createdOn())
                .add("LastModifiedOn", lastModifiedOn())
                .add("InputRecordTables", hasInputRecordTables() ? inputRecordTables() : null).add("Parameters", parameters())
                .add("EvaluationMetrics", evaluationMetrics()).add("LabelCount", labelCount())
                .add("Schema", hasSchema() ? schema() : null).add("Role", role()).add("GlueVersion", glueVersion())
                .add("MaxCapacity", maxCapacity()).add("WorkerType", workerTypeAsString())
                .add("NumberOfWorkers", numberOfWorkers()).add("Timeout", timeout()).add("MaxRetries", maxRetries())
                .add("TransformEncryption", transformEncryption()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TransformId":
            return Optional.ofNullable(clazz.cast(transformId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedOn":
            return Optional.ofNullable(clazz.cast(createdOn()));
        case "LastModifiedOn":
            return Optional.ofNullable(clazz.cast(lastModifiedOn()));
        case "InputRecordTables":
            return Optional.ofNullable(clazz.cast(inputRecordTables()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "EvaluationMetrics":
            return Optional.ofNullable(clazz.cast(evaluationMetrics()));
        case "LabelCount":
            return Optional.ofNullable(clazz.cast(labelCount()));
        case "Schema":
            return Optional.ofNullable(clazz.cast(schema()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "GlueVersion":
            return Optional.ofNullable(clazz.cast(glueVersion()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "WorkerType":
            return Optional.ofNullable(clazz.cast(workerTypeAsString()));
        case "NumberOfWorkers":
            return Optional.ofNullable(clazz.cast(numberOfWorkers()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "MaxRetries":
            return Optional.ofNullable(clazz.cast(maxRetries()));
        case "TransformEncryption":
            return Optional.ofNullable(clazz.cast(transformEncryption()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MLTransform, T> g) {
        return obj -> g.apply((MLTransform) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MLTransform> {
        /**
         * <p>
         * The unique transform ID that is generated for the machine learning transform. The ID is guaranteed to be
         * unique and does not change.
         * </p>
         * 
         * @param transformId
         *        The unique transform ID that is generated for the machine learning transform. The ID is guaranteed to
         *        be unique and does not change.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformId(String transformId);

        /**
         * <p>
         * A user-defined name for the machine learning transform. Names are not guaranteed unique and can be changed at
         * any time.
         * </p>
         * 
         * @param name
         *        A user-defined name for the machine learning transform. Names are not guaranteed unique and can be
         *        changed at any time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A user-defined, long-form description text for the machine learning transform. Descriptions are not
         * guaranteed to be unique and can be changed at any time.
         * </p>
         * 
         * @param description
         *        A user-defined, long-form description text for the machine learning transform. Descriptions are not
         *        guaranteed to be unique and can be changed at any time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The current status of the machine learning transform.
         * </p>
         * 
         * @param status
         *        The current status of the machine learning transform.
         * @see TransformStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the machine learning transform.
         * </p>
         * 
         * @param status
         *        The current status of the machine learning transform.
         * @see TransformStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformStatusType
         */
        Builder status(TransformStatusType status);

        /**
         * <p>
         * A timestamp. The time and date that this machine learning transform was created.
         * </p>
         * 
         * @param createdOn
         *        A timestamp. The time and date that this machine learning transform was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdOn(Instant createdOn);

        /**
         * <p>
         * A timestamp. The last point in time when this machine learning transform was modified.
         * </p>
         * 
         * @param lastModifiedOn
         *        A timestamp. The last point in time when this machine learning transform was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedOn(Instant lastModifiedOn);

        /**
         * <p>
         * A list of AWS Glue table definitions used by the transform.
         * </p>
         * 
         * @param inputRecordTables
         *        A list of AWS Glue table definitions used by the transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputRecordTables(Collection<GlueTable> inputRecordTables);

        /**
         * <p>
         * A list of AWS Glue table definitions used by the transform.
         * </p>
         * 
         * @param inputRecordTables
         *        A list of AWS Glue table definitions used by the transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputRecordTables(GlueTable... inputRecordTables);

        /**
         * <p>
         * A list of AWS Glue table definitions used by the transform.
         * </p>
         * This is a convenience that creates an instance of the {@link List<GlueTable>.Builder} avoiding the need to
         * create one manually via {@link List<GlueTable>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GlueTable>.Builder#build()} is called immediately and its
         * result is passed to {@link #inputRecordTables(List<GlueTable>)}.
         * 
         * @param inputRecordTables
         *        a consumer that will call methods on {@link List<GlueTable>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputRecordTables(List<GlueTable>)
         */
        Builder inputRecordTables(Consumer<GlueTable.Builder>... inputRecordTables);

        /**
         * <p>
         * A <code>TransformParameters</code> object. You can use parameters to tune (customize) the behavior of the
         * machine learning transform by specifying what data it learns from and your preference on various tradeoffs
         * (such as precious vs. recall, or accuracy vs. cost).
         * </p>
         * 
         * @param parameters
         *        A <code>TransformParameters</code> object. You can use parameters to tune (customize) the behavior of
         *        the machine learning transform by specifying what data it learns from and your preference on various
         *        tradeoffs (such as precious vs. recall, or accuracy vs. cost).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(TransformParameters parameters);

        /**
         * <p>
         * A <code>TransformParameters</code> object. You can use parameters to tune (customize) the behavior of the
         * machine learning transform by specifying what data it learns from and your preference on various tradeoffs
         * (such as precious vs. recall, or accuracy vs. cost).
         * </p>
         * This is a convenience that creates an instance of the {@link TransformParameters.Builder} avoiding the need
         * to create one manually via {@link TransformParameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link TransformParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #parameters(TransformParameters)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link TransformParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(TransformParameters)
         */
        default Builder parameters(Consumer<TransformParameters.Builder> parameters) {
            return parameters(TransformParameters.builder().applyMutation(parameters).build());
        }

        /**
         * <p>
         * An <code>EvaluationMetrics</code> object. Evaluation metrics provide an estimate of the quality of your
         * machine learning transform.
         * </p>
         * 
         * @param evaluationMetrics
         *        An <code>EvaluationMetrics</code> object. Evaluation metrics provide an estimate of the quality of
         *        your machine learning transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationMetrics(EvaluationMetrics evaluationMetrics);

        /**
         * <p>
         * An <code>EvaluationMetrics</code> object. Evaluation metrics provide an estimate of the quality of your
         * machine learning transform.
         * </p>
         * This is a convenience that creates an instance of the {@link EvaluationMetrics.Builder} avoiding the need to
         * create one manually via {@link EvaluationMetrics#builder()}.
         *
         * When the {@link Consumer} completes, {@link EvaluationMetrics.Builder#build()} is called immediately and its
         * result is passed to {@link #evaluationMetrics(EvaluationMetrics)}.
         * 
         * @param evaluationMetrics
         *        a consumer that will call methods on {@link EvaluationMetrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationMetrics(EvaluationMetrics)
         */
        default Builder evaluationMetrics(Consumer<EvaluationMetrics.Builder> evaluationMetrics) {
            return evaluationMetrics(EvaluationMetrics.builder().applyMutation(evaluationMetrics).build());
        }

        /**
         * <p>
         * A count identifier for the labeling files generated by AWS Glue for this transform. As you create a better
         * transform, you can iteratively download, label, and upload the labeling file.
         * </p>
         * 
         * @param labelCount
         *        A count identifier for the labeling files generated by AWS Glue for this transform. As you create a
         *        better transform, you can iteratively download, label, and upload the labeling file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelCount(Integer labelCount);

        /**
         * <p>
         * A map of key-value pairs representing the columns and data types that this transform can run against. Has an
         * upper bound of 100 columns.
         * </p>
         * 
         * @param schema
         *        A map of key-value pairs representing the columns and data types that this transform can run against.
         *        Has an upper bound of 100 columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(Collection<SchemaColumn> schema);

        /**
         * <p>
         * A map of key-value pairs representing the columns and data types that this transform can run against. Has an
         * upper bound of 100 columns.
         * </p>
         * 
         * @param schema
         *        A map of key-value pairs representing the columns and data types that this transform can run against.
         *        Has an upper bound of 100 columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(SchemaColumn... schema);

        /**
         * <p>
         * A map of key-value pairs representing the columns and data types that this transform can run against. Has an
         * upper bound of 100 columns.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SchemaColumn>.Builder} avoiding the need to
         * create one manually via {@link List<SchemaColumn>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SchemaColumn>.Builder#build()} is called immediately and its
         * result is passed to {@link #schema(List<SchemaColumn>)}.
         * 
         * @param schema
         *        a consumer that will call methods on {@link List<SchemaColumn>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schema(List<SchemaColumn>)
         */
        Builder schema(Consumer<SchemaColumn.Builder>... schema);

        /**
         * <p>
         * The name or Amazon Resource Name (ARN) of the IAM role with the required permissions. The required
         * permissions include both AWS Glue service role permissions to AWS Glue resources, and Amazon S3 permissions
         * required by the transform.
         * </p>
         * <ul>
         * <li>
         * <p>
         * This role needs AWS Glue service role permissions to allow access to resources in AWS Glue. See <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/attach-policy-iam-user.html">Attach a Policy to IAM Users
         * That Access AWS Glue</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * This role needs permission to your Amazon Simple Storage Service (Amazon S3) sources, targets, temporary
         * directory, scripts, and any libraries used by the task run for this transform.
         * </p>
         * </li>
         * </ul>
         * 
         * @param role
         *        The name or Amazon Resource Name (ARN) of the IAM role with the required permissions. The required
         *        permissions include both AWS Glue service role permissions to AWS Glue resources, and Amazon S3
         *        permissions required by the transform. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        This role needs AWS Glue service role permissions to allow access to resources in AWS Glue. See <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/attach-policy-iam-user.html">Attach a Policy to IAM
         *        Users That Access AWS Glue</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        This role needs permission to your Amazon Simple Storage Service (Amazon S3) sources, targets,
         *        temporary directory, scripts, and any libraries used by the task run for this transform.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * This value determines which version of AWS Glue this machine learning transform is compatible with. Glue 1.0
         * is recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue 0.9. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">AWS Glue
         * Versions</a> in the developer guide.
         * </p>
         * 
         * @param glueVersion
         *        This value determines which version of AWS Glue this machine learning transform is compatible with.
         *        Glue 1.0 is recommended for most customers. If the value is not set, the Glue compatibility defaults
         *        to Glue 0.9. For more information, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">AWS Glue
         *        Versions</a> in the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueVersion(String glueVersion);

        /**
         * <p>
         * The number of AWS Glue data processing units (DPUs) that are allocated to task runs for this transform. You
         * can allocate from 2 to 100 DPUs; the default is 10. A DPU is a relative measure of processing power that
         * consists of 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
         * href="http://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>.
         * </p>
         * <p>
         * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         * <code>WorkerType</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
         * cannot be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can
         * be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When the <code>WorkerType</code> field is set to a value other than <code>Standard</code>, the
         * <code>MaxCapacity</code> field is set automatically and becomes read-only.
         * </p>
         * 
         * @param maxCapacity
         *        The number of AWS Glue data processing units (DPUs) that are allocated to task runs for this
         *        transform. You can allocate from 2 to 100 DPUs; the default is 10. A DPU is a relative measure of
         *        processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more
         *        information, see the <a href="http://aws.amazon.com/glue/pricing/">AWS Glue pricing page</a>. </p>
         *        <p>
         *        <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         *        <code>WorkerType</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then
         *        <code>MaxCapacity</code> cannot be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or
         *        <code>WorkerType</code> can be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When the <code>WorkerType</code> field is set to a value other than <code>Standard</code>, the
         *        <code>MaxCapacity</code> field is set automatically and becomes read-only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Double maxCapacity);

        /**
         * <p>
         * The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
         * Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         * <code>WorkerType</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
         * cannot be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can
         * be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
         *        Standard, G.1X, or G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         *        <code>WorkerType</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then
         *        <code>MaxCapacity</code> cannot be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or
         *        <code>WorkerType</code> can be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(String workerType);

        /**
         * <p>
         * The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
         * Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         * <code>WorkerType</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then <code>MaxCapacity</code>
         * cannot be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or <code>WorkerType</code> can
         * be set.
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a task of this transform runs. Accepts a value of
         *        Standard, G.1X, or G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>MaxCapacity</code> is a mutually exclusive option with <code>NumberOfWorkers</code> and
         *        <code>WorkerType</code>.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If either <code>NumberOfWorkers</code> or <code>WorkerType</code> is set, then
         *        <code>MaxCapacity</code> cannot be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>MaxCapacity</code> is set then neither <code>NumberOfWorkers</code> or
         *        <code>WorkerType</code> can be set.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MaxCapacity</code> and <code>NumberOfWorkers</code> must both be at least 1.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(WorkerType workerType);

        /**
         * <p>
         * The number of workers of a defined <code>workerType</code> that are allocated when a task of the transform
         * runs.
         * </p>
         * <p>
         * If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         * </p>
         * 
         * @param numberOfWorkers
         *        The number of workers of a defined <code>workerType</code> that are allocated when a task of the
         *        transform runs.</p>
         *        <p>
         *        If <code>WorkerType</code> is set, then <code>NumberOfWorkers</code> is required (and vice versa).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfWorkers(Integer numberOfWorkers);

        /**
         * <p>
         * The timeout in minutes of the machine learning transform.
         * </p>
         * 
         * @param timeout
         *        The timeout in minutes of the machine learning transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * The maximum number of times to retry after an <code>MLTaskRun</code> of the machine learning transform fails.
         * </p>
         * 
         * @param maxRetries
         *        The maximum number of times to retry after an <code>MLTaskRun</code> of the machine learning transform
         *        fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetries(Integer maxRetries);

        /**
         * <p>
         * The encryption-at-rest settings of the transform that apply to accessing user data. Machine learning
         * transforms can access user data encrypted in Amazon S3 using KMS.
         * </p>
         * 
         * @param transformEncryption
         *        The encryption-at-rest settings of the transform that apply to accessing user data. Machine learning
         *        transforms can access user data encrypted in Amazon S3 using KMS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformEncryption(TransformEncryption transformEncryption);

        /**
         * <p>
         * The encryption-at-rest settings of the transform that apply to accessing user data. Machine learning
         * transforms can access user data encrypted in Amazon S3 using KMS.
         * </p>
         * This is a convenience that creates an instance of the {@link TransformEncryption.Builder} avoiding the need
         * to create one manually via {@link TransformEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link TransformEncryption.Builder#build()} is called immediately and
         * its result is passed to {@link #transformEncryption(TransformEncryption)}.
         * 
         * @param transformEncryption
         *        a consumer that will call methods on {@link TransformEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformEncryption(TransformEncryption)
         */
        default Builder transformEncryption(Consumer<TransformEncryption.Builder> transformEncryption) {
            return transformEncryption(TransformEncryption.builder().applyMutation(transformEncryption).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String transformId;

        private String name;

        private String description;

        private String status;

        private Instant createdOn;

        private Instant lastModifiedOn;

        private List<GlueTable> inputRecordTables = DefaultSdkAutoConstructList.getInstance();

        private TransformParameters parameters;

        private EvaluationMetrics evaluationMetrics;

        private Integer labelCount;

        private List<SchemaColumn> schema = DefaultSdkAutoConstructList.getInstance();

        private String role;

        private String glueVersion;

        private Double maxCapacity;

        private String workerType;

        private Integer numberOfWorkers;

        private Integer timeout;

        private Integer maxRetries;

        private TransformEncryption transformEncryption;

        private BuilderImpl() {
        }

        private BuilderImpl(MLTransform model) {
            transformId(model.transformId);
            name(model.name);
            description(model.description);
            status(model.status);
            createdOn(model.createdOn);
            lastModifiedOn(model.lastModifiedOn);
            inputRecordTables(model.inputRecordTables);
            parameters(model.parameters);
            evaluationMetrics(model.evaluationMetrics);
            labelCount(model.labelCount);
            schema(model.schema);
            role(model.role);
            glueVersion(model.glueVersion);
            maxCapacity(model.maxCapacity);
            workerType(model.workerType);
            numberOfWorkers(model.numberOfWorkers);
            timeout(model.timeout);
            maxRetries(model.maxRetries);
            transformEncryption(model.transformEncryption);
        }

        public final String getTransformId() {
            return transformId;
        }

        @Override
        public final Builder transformId(String transformId) {
            this.transformId = transformId;
            return this;
        }

        public final void setTransformId(String transformId) {
            this.transformId = transformId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TransformStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getCreatedOn() {
            return createdOn;
        }

        @Override
        public final Builder createdOn(Instant createdOn) {
            this.createdOn = createdOn;
            return this;
        }

        public final void setCreatedOn(Instant createdOn) {
            this.createdOn = createdOn;
        }

        public final Instant getLastModifiedOn() {
            return lastModifiedOn;
        }

        @Override
        public final Builder lastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
            return this;
        }

        public final void setLastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
        }

        public final List<GlueTable.Builder> getInputRecordTables() {
            List<GlueTable.Builder> result = GlueTablesCopier.copyToBuilder(this.inputRecordTables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder inputRecordTables(Collection<GlueTable> inputRecordTables) {
            this.inputRecordTables = GlueTablesCopier.copy(inputRecordTables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputRecordTables(GlueTable... inputRecordTables) {
            inputRecordTables(Arrays.asList(inputRecordTables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputRecordTables(Consumer<GlueTable.Builder>... inputRecordTables) {
            inputRecordTables(Stream.of(inputRecordTables).map(c -> GlueTable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInputRecordTables(Collection<GlueTable.BuilderImpl> inputRecordTables) {
            this.inputRecordTables = GlueTablesCopier.copyFromBuilder(inputRecordTables);
        }

        public final TransformParameters.Builder getParameters() {
            return parameters != null ? parameters.toBuilder() : null;
        }

        @Override
        public final Builder parameters(TransformParameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public final void setParameters(TransformParameters.BuilderImpl parameters) {
            this.parameters = parameters != null ? parameters.build() : null;
        }

        public final EvaluationMetrics.Builder getEvaluationMetrics() {
            return evaluationMetrics != null ? evaluationMetrics.toBuilder() : null;
        }

        @Override
        public final Builder evaluationMetrics(EvaluationMetrics evaluationMetrics) {
            this.evaluationMetrics = evaluationMetrics;
            return this;
        }

        public final void setEvaluationMetrics(EvaluationMetrics.BuilderImpl evaluationMetrics) {
            this.evaluationMetrics = evaluationMetrics != null ? evaluationMetrics.build() : null;
        }

        public final Integer getLabelCount() {
            return labelCount;
        }

        @Override
        public final Builder labelCount(Integer labelCount) {
            this.labelCount = labelCount;
            return this;
        }

        public final void setLabelCount(Integer labelCount) {
            this.labelCount = labelCount;
        }

        public final List<SchemaColumn.Builder> getSchema() {
            List<SchemaColumn.Builder> result = TransformSchemaCopier.copyToBuilder(this.schema);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder schema(Collection<SchemaColumn> schema) {
            this.schema = TransformSchemaCopier.copy(schema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(SchemaColumn... schema) {
            schema(Arrays.asList(schema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(Consumer<SchemaColumn.Builder>... schema) {
            schema(Stream.of(schema).map(c -> SchemaColumn.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSchema(Collection<SchemaColumn.BuilderImpl> schema) {
            this.schema = TransformSchemaCopier.copyFromBuilder(schema);
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final String getGlueVersion() {
            return glueVersion;
        }

        @Override
        public final Builder glueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
            return this;
        }

        public final void setGlueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
        }

        public final Double getMaxCapacity() {
            return maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final void setMaxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        public final String getWorkerType() {
            return workerType;
        }

        @Override
        public final Builder workerType(String workerType) {
            this.workerType = workerType;
            return this;
        }

        @Override
        public final Builder workerType(WorkerType workerType) {
            this.workerType(workerType == null ? null : workerType.toString());
            return this;
        }

        public final void setWorkerType(String workerType) {
            this.workerType = workerType;
        }

        public final Integer getNumberOfWorkers() {
            return numberOfWorkers;
        }

        @Override
        public final Builder numberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
            return this;
        }

        public final void setNumberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        public final Integer getMaxRetries() {
            return maxRetries;
        }

        @Override
        public final Builder maxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public final void setMaxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
        }

        public final TransformEncryption.Builder getTransformEncryption() {
            return transformEncryption != null ? transformEncryption.toBuilder() : null;
        }

        @Override
        public final Builder transformEncryption(TransformEncryption transformEncryption) {
            this.transformEncryption = transformEncryption;
            return this;
        }

        public final void setTransformEncryption(TransformEncryption.BuilderImpl transformEncryption) {
            this.transformEncryption = transformEncryption != null ? transformEncryption.build() : null;
        }

        @Override
        public MLTransform build() {
            return new MLTransform(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
