/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The criteria used to filter the machine learning transforms.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TransformFilterCriteria implements SdkPojo, Serializable,
        ToCopyableBuilder<TransformFilterCriteria.Builder, TransformFilterCriteria> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(TransformFilterCriteria::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TRANSFORM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransformType").getter(getter(TransformFilterCriteria::transformTypeAsString))
            .setter(setter(Builder::transformType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(TransformFilterCriteria::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> GLUE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GlueVersion").getter(getter(TransformFilterCriteria::glueVersion)).setter(setter(Builder::glueVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlueVersion").build()).build();

    private static final SdkField<Instant> CREATED_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedBefore").getter(getter(TransformFilterCriteria::createdBefore))
            .setter(setter(Builder::createdBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBefore").build()).build();

    private static final SdkField<Instant> CREATED_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAfter").getter(getter(TransformFilterCriteria::createdAfter))
            .setter(setter(Builder::createdAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAfter").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedBefore").getter(getter(TransformFilterCriteria::lastModifiedBefore))
            .setter(setter(Builder::lastModifiedBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedBefore").build())
            .build();

    private static final SdkField<Instant> LAST_MODIFIED_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedAfter").getter(getter(TransformFilterCriteria::lastModifiedAfter))
            .setter(setter(Builder::lastModifiedAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedAfter").build()).build();

    private static final SdkField<List<SchemaColumn>> SCHEMA_FIELD = SdkField
            .<List<SchemaColumn>> builder(MarshallingType.LIST)
            .memberName("Schema")
            .getter(getter(TransformFilterCriteria::schema))
            .setter(setter(Builder::schema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SchemaColumn> builder(MarshallingType.SDK_POJO)
                                            .constructor(SchemaColumn::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            TRANSFORM_TYPE_FIELD, STATUS_FIELD, GLUE_VERSION_FIELD, CREATED_BEFORE_FIELD, CREATED_AFTER_FIELD,
            LAST_MODIFIED_BEFORE_FIELD, LAST_MODIFIED_AFTER_FIELD, SCHEMA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String transformType;

    private final String status;

    private final String glueVersion;

    private final Instant createdBefore;

    private final Instant createdAfter;

    private final Instant lastModifiedBefore;

    private final Instant lastModifiedAfter;

    private final List<SchemaColumn> schema;

    private TransformFilterCriteria(BuilderImpl builder) {
        this.name = builder.name;
        this.transformType = builder.transformType;
        this.status = builder.status;
        this.glueVersion = builder.glueVersion;
        this.createdBefore = builder.createdBefore;
        this.createdAfter = builder.createdAfter;
        this.lastModifiedBefore = builder.lastModifiedBefore;
        this.lastModifiedAfter = builder.lastModifiedAfter;
        this.schema = builder.schema;
    }

    /**
     * <p>
     * A unique transform name that is used to filter the machine learning transforms.
     * </p>
     * 
     * @return A unique transform name that is used to filter the machine learning transforms.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of machine learning transform that is used to filter the machine learning transforms.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transformType}
     * will return {@link TransformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #transformTypeAsString}.
     * </p>
     * 
     * @return The type of machine learning transform that is used to filter the machine learning transforms.
     * @see TransformType
     */
    public final TransformType transformType() {
        return TransformType.fromValue(transformType);
    }

    /**
     * <p>
     * The type of machine learning transform that is used to filter the machine learning transforms.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transformType}
     * will return {@link TransformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #transformTypeAsString}.
     * </p>
     * 
     * @return The type of machine learning transform that is used to filter the machine learning transforms.
     * @see TransformType
     */
    public final String transformTypeAsString() {
        return transformType;
    }

    /**
     * <p>
     * Filters the list of machine learning transforms by the last known status of the transforms (to indicate whether a
     * transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Filters the list of machine learning transforms by the last known status of the transforms (to indicate
     *         whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
     * @see TransformStatusType
     */
    public final TransformStatusType status() {
        return TransformStatusType.fromValue(status);
    }

    /**
     * <p>
     * Filters the list of machine learning transforms by the last known status of the transforms (to indicate whether a
     * transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Filters the list of machine learning transforms by the last known status of the transforms (to indicate
     *         whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
     * @see TransformStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * This value determines which version of Glue this machine learning transform is compatible with. Glue 1.0 is
     * recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue 0.9. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">Glue Versions</a> in
     * the developer guide.
     * </p>
     * 
     * @return This value determines which version of Glue this machine learning transform is compatible with. Glue 1.0
     *         is recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue 0.9.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">Glue
     *         Versions</a> in the developer guide.
     */
    public final String glueVersion() {
        return glueVersion;
    }

    /**
     * <p>
     * The time and date before which the transforms were created.
     * </p>
     * 
     * @return The time and date before which the transforms were created.
     */
    public final Instant createdBefore() {
        return createdBefore;
    }

    /**
     * <p>
     * The time and date after which the transforms were created.
     * </p>
     * 
     * @return The time and date after which the transforms were created.
     */
    public final Instant createdAfter() {
        return createdAfter;
    }

    /**
     * <p>
     * Filter on transforms last modified before this date.
     * </p>
     * 
     * @return Filter on transforms last modified before this date.
     */
    public final Instant lastModifiedBefore() {
        return lastModifiedBefore;
    }

    /**
     * <p>
     * Filter on transforms last modified after this date.
     * </p>
     * 
     * @return Filter on transforms last modified after this date.
     */
    public final Instant lastModifiedAfter() {
        return lastModifiedAfter;
    }

    /**
     * Returns true if the Schema property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSchema() {
        return schema != null && !(schema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an array of
     * key-value pairs representing the schema this transform accepts, where <code>Column</code> is the name of a
     * column, and <code>Type</code> is the type of the data such as an integer or string. Has an upper bound of 100
     * columns.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSchema()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an array
     *         of key-value pairs representing the schema this transform accepts, where <code>Column</code> is the name
     *         of a column, and <code>Type</code> is the type of the data such as an integer or string. Has an upper
     *         bound of 100 columns.
     */
    public final List<SchemaColumn> schema() {
        return schema;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(transformTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(glueVersion());
        hashCode = 31 * hashCode + Objects.hashCode(createdBefore());
        hashCode = 31 * hashCode + Objects.hashCode(createdAfter());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedBefore());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedAfter());
        hashCode = 31 * hashCode + Objects.hashCode(hasSchema() ? schema() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TransformFilterCriteria)) {
            return false;
        }
        TransformFilterCriteria other = (TransformFilterCriteria) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(transformTypeAsString(), other.transformTypeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(glueVersion(), other.glueVersion())
                && Objects.equals(createdBefore(), other.createdBefore()) && Objects.equals(createdAfter(), other.createdAfter())
                && Objects.equals(lastModifiedBefore(), other.lastModifiedBefore())
                && Objects.equals(lastModifiedAfter(), other.lastModifiedAfter()) && hasSchema() == other.hasSchema()
                && Objects.equals(schema(), other.schema());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TransformFilterCriteria").add("Name", name()).add("TransformType", transformTypeAsString())
                .add("Status", statusAsString()).add("GlueVersion", glueVersion()).add("CreatedBefore", createdBefore())
                .add("CreatedAfter", createdAfter()).add("LastModifiedBefore", lastModifiedBefore())
                .add("LastModifiedAfter", lastModifiedAfter()).add("Schema", hasSchema() ? schema() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "TransformType":
            return Optional.ofNullable(clazz.cast(transformTypeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "GlueVersion":
            return Optional.ofNullable(clazz.cast(glueVersion()));
        case "CreatedBefore":
            return Optional.ofNullable(clazz.cast(createdBefore()));
        case "CreatedAfter":
            return Optional.ofNullable(clazz.cast(createdAfter()));
        case "LastModifiedBefore":
            return Optional.ofNullable(clazz.cast(lastModifiedBefore()));
        case "LastModifiedAfter":
            return Optional.ofNullable(clazz.cast(lastModifiedAfter()));
        case "Schema":
            return Optional.ofNullable(clazz.cast(schema()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TransformFilterCriteria, T> g) {
        return obj -> g.apply((TransformFilterCriteria) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TransformFilterCriteria> {
        /**
         * <p>
         * A unique transform name that is used to filter the machine learning transforms.
         * </p>
         * 
         * @param name
         *        A unique transform name that is used to filter the machine learning transforms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of machine learning transform that is used to filter the machine learning transforms.
         * </p>
         * 
         * @param transformType
         *        The type of machine learning transform that is used to filter the machine learning transforms.
         * @see TransformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformType
         */
        Builder transformType(String transformType);

        /**
         * <p>
         * The type of machine learning transform that is used to filter the machine learning transforms.
         * </p>
         * 
         * @param transformType
         *        The type of machine learning transform that is used to filter the machine learning transforms.
         * @see TransformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformType
         */
        Builder transformType(TransformType transformType);

        /**
         * <p>
         * Filters the list of machine learning transforms by the last known status of the transforms (to indicate
         * whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
         * </p>
         * 
         * @param status
         *        Filters the list of machine learning transforms by the last known status of the transforms (to
         *        indicate whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
         * @see TransformStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * Filters the list of machine learning transforms by the last known status of the transforms (to indicate
         * whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
         * </p>
         * 
         * @param status
         *        Filters the list of machine learning transforms by the last known status of the transforms (to
         *        indicate whether a transform can be used or not). One of "NOT_READY", "READY", or "DELETING".
         * @see TransformStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformStatusType
         */
        Builder status(TransformStatusType status);

        /**
         * <p>
         * This value determines which version of Glue this machine learning transform is compatible with. Glue 1.0 is
         * recommended for most customers. If the value is not set, the Glue compatibility defaults to Glue 0.9. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">Glue Versions</a>
         * in the developer guide.
         * </p>
         * 
         * @param glueVersion
         *        This value determines which version of Glue this machine learning transform is compatible with. Glue
         *        1.0 is recommended for most customers. If the value is not set, the Glue compatibility defaults to
         *        Glue 0.9. For more information, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/release-notes.html#release-notes-versions">Glue
         *        Versions</a> in the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueVersion(String glueVersion);

        /**
         * <p>
         * The time and date before which the transforms were created.
         * </p>
         * 
         * @param createdBefore
         *        The time and date before which the transforms were created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBefore(Instant createdBefore);

        /**
         * <p>
         * The time and date after which the transforms were created.
         * </p>
         * 
         * @param createdAfter
         *        The time and date after which the transforms were created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAfter(Instant createdAfter);

        /**
         * <p>
         * Filter on transforms last modified before this date.
         * </p>
         * 
         * @param lastModifiedBefore
         *        Filter on transforms last modified before this date.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedBefore(Instant lastModifiedBefore);

        /**
         * <p>
         * Filter on transforms last modified after this date.
         * </p>
         * 
         * @param lastModifiedAfter
         *        Filter on transforms last modified after this date.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedAfter(Instant lastModifiedAfter);

        /**
         * <p>
         * Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an array of
         * key-value pairs representing the schema this transform accepts, where <code>Column</code> is the name of a
         * column, and <code>Type</code> is the type of the data such as an integer or string. Has an upper bound of 100
         * columns.
         * </p>
         * 
         * @param schema
         *        Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an
         *        array of key-value pairs representing the schema this transform accepts, where <code>Column</code> is
         *        the name of a column, and <code>Type</code> is the type of the data such as an integer or string. Has
         *        an upper bound of 100 columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(Collection<SchemaColumn> schema);

        /**
         * <p>
         * Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an array of
         * key-value pairs representing the schema this transform accepts, where <code>Column</code> is the name of a
         * column, and <code>Type</code> is the type of the data such as an integer or string. Has an upper bound of 100
         * columns.
         * </p>
         * 
         * @param schema
         *        Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an
         *        array of key-value pairs representing the schema this transform accepts, where <code>Column</code> is
         *        the name of a column, and <code>Type</code> is the type of the data such as an integer or string. Has
         *        an upper bound of 100 columns.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(SchemaColumn... schema);

        /**
         * <p>
         * Filters on datasets with a specific schema. The <code>Map&lt;Column, Type&gt;</code> object is an array of
         * key-value pairs representing the schema this transform accepts, where <code>Column</code> is the name of a
         * column, and <code>Type</code> is the type of the data such as an integer or string. Has an upper bound of 100
         * columns.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SchemaColumn>.Builder} avoiding the need to
         * create one manually via {@link List<SchemaColumn>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SchemaColumn>.Builder#build()} is called immediately and its
         * result is passed to {@link #schema(List<SchemaColumn>)}.
         * 
         * @param schema
         *        a consumer that will call methods on {@link List<SchemaColumn>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schema(List<SchemaColumn>)
         */
        Builder schema(Consumer<SchemaColumn.Builder>... schema);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String transformType;

        private String status;

        private String glueVersion;

        private Instant createdBefore;

        private Instant createdAfter;

        private Instant lastModifiedBefore;

        private Instant lastModifiedAfter;

        private List<SchemaColumn> schema = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TransformFilterCriteria model) {
            name(model.name);
            transformType(model.transformType);
            status(model.status);
            glueVersion(model.glueVersion);
            createdBefore(model.createdBefore);
            createdAfter(model.createdAfter);
            lastModifiedBefore(model.lastModifiedBefore);
            lastModifiedAfter(model.lastModifiedAfter);
            schema(model.schema);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getTransformType() {
            return transformType;
        }

        @Override
        public final Builder transformType(String transformType) {
            this.transformType = transformType;
            return this;
        }

        @Override
        public final Builder transformType(TransformType transformType) {
            this.transformType(transformType == null ? null : transformType.toString());
            return this;
        }

        public final void setTransformType(String transformType) {
            this.transformType = transformType;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TransformStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getGlueVersion() {
            return glueVersion;
        }

        @Override
        public final Builder glueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
            return this;
        }

        public final void setGlueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
        }

        public final Instant getCreatedBefore() {
            return createdBefore;
        }

        @Override
        public final Builder createdBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
            return this;
        }

        public final void setCreatedBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
        }

        public final Instant getCreatedAfter() {
            return createdAfter;
        }

        @Override
        public final Builder createdAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
            return this;
        }

        public final void setCreatedAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
        }

        public final Instant getLastModifiedBefore() {
            return lastModifiedBefore;
        }

        @Override
        public final Builder lastModifiedBefore(Instant lastModifiedBefore) {
            this.lastModifiedBefore = lastModifiedBefore;
            return this;
        }

        public final void setLastModifiedBefore(Instant lastModifiedBefore) {
            this.lastModifiedBefore = lastModifiedBefore;
        }

        public final Instant getLastModifiedAfter() {
            return lastModifiedAfter;
        }

        @Override
        public final Builder lastModifiedAfter(Instant lastModifiedAfter) {
            this.lastModifiedAfter = lastModifiedAfter;
            return this;
        }

        public final void setLastModifiedAfter(Instant lastModifiedAfter) {
            this.lastModifiedAfter = lastModifiedAfter;
        }

        public final List<SchemaColumn.Builder> getSchema() {
            List<SchemaColumn.Builder> result = TransformSchemaCopier.copyToBuilder(this.schema);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder schema(Collection<SchemaColumn> schema) {
            this.schema = TransformSchemaCopier.copy(schema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(SchemaColumn... schema) {
            schema(Arrays.asList(schema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(Consumer<SchemaColumn.Builder>... schema) {
            schema(Stream.of(schema).map(c -> SchemaColumn.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSchema(Collection<SchemaColumn.BuilderImpl> schema) {
            this.schema = TransformSchemaCopier.copyFromBuilder(schema);
        }

        @Override
        public TransformFilterCriteria build() {
            return new TransformFilterCriteria(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
