/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a job run.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobRun implements SdkPojo, Serializable, ToCopyableBuilder<JobRun.Builder, JobRun> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(JobRun::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<Integer> ATTEMPT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Attempt").getter(getter(JobRun::attempt)).setter(setter(Builder::attempt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attempt").build()).build();

    private static final SdkField<String> PREVIOUS_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreviousRunId").getter(getter(JobRun::previousRunId)).setter(setter(Builder::previousRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreviousRunId").build()).build();

    private static final SdkField<String> TRIGGER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TriggerName").getter(getter(JobRun::triggerName)).setter(setter(Builder::triggerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TriggerName").build()).build();

    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(JobRun::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<Instant> STARTED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedOn").getter(getter(JobRun::startedOn)).setter(setter(Builder::startedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedOn").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedOn").getter(getter(JobRun::lastModifiedOn)).setter(setter(Builder::lastModifiedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedOn").build()).build();

    private static final SdkField<Instant> COMPLETED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletedOn").getter(getter(JobRun::completedOn)).setter(setter(Builder::completedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedOn").build()).build();

    private static final SdkField<String> JOB_RUN_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobRunState").getter(getter(JobRun::jobRunStateAsString)).setter(setter(Builder::jobRunState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobRunState").build()).build();

    private static final SdkField<Map<String, String>> ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Arguments")
            .getter(getter(JobRun::arguments))
            .setter(setter(Builder::arguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(JobRun::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<List<Predecessor>> PREDECESSOR_RUNS_FIELD = SdkField
            .<List<Predecessor>> builder(MarshallingType.LIST)
            .memberName("PredecessorRuns")
            .getter(getter(JobRun::predecessorRuns))
            .setter(setter(Builder::predecessorRuns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PredecessorRuns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Predecessor> builder(MarshallingType.SDK_POJO)
                                            .constructor(Predecessor::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> ALLOCATED_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocatedCapacity").getter(getter(JobRun::allocatedCapacity)).setter(setter(Builder::allocatedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedCapacity").build()).build();

    private static final SdkField<Integer> EXECUTION_TIME_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExecutionTime").getter(getter(JobRun::executionTime)).setter(setter(Builder::executionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionTime").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(JobRun::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Double> MAX_CAPACITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxCapacity").getter(getter(JobRun::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<String> WORKER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkerType").getter(getter(JobRun::workerTypeAsString)).setter(setter(Builder::workerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkerType").build()).build();

    private static final SdkField<Integer> NUMBER_OF_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfWorkers").getter(getter(JobRun::numberOfWorkers)).setter(setter(Builder::numberOfWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfWorkers").build()).build();

    private static final SdkField<String> SECURITY_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityConfiguration").getter(getter(JobRun::securityConfiguration))
            .setter(setter(Builder::securityConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityConfiguration").build())
            .build();

    private static final SdkField<String> LOG_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogGroupName").getter(getter(JobRun::logGroupName)).setter(setter(Builder::logGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogGroupName").build()).build();

    private static final SdkField<NotificationProperty> NOTIFICATION_PROPERTY_FIELD = SdkField
            .<NotificationProperty> builder(MarshallingType.SDK_POJO).memberName("NotificationProperty")
            .getter(getter(JobRun::notificationProperty)).setter(setter(Builder::notificationProperty))
            .constructor(NotificationProperty::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationProperty").build())
            .build();

    private static final SdkField<String> GLUE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GlueVersion").getter(getter(JobRun::glueVersion)).setter(setter(Builder::glueVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlueVersion").build()).build();

    private static final SdkField<Double> DPU_SECONDS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("DPUSeconds").getter(getter(JobRun::dpuSeconds)).setter(setter(Builder::dpuSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DPUSeconds").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ATTEMPT_FIELD,
            PREVIOUS_RUN_ID_FIELD, TRIGGER_NAME_FIELD, JOB_NAME_FIELD, STARTED_ON_FIELD, LAST_MODIFIED_ON_FIELD,
            COMPLETED_ON_FIELD, JOB_RUN_STATE_FIELD, ARGUMENTS_FIELD, ERROR_MESSAGE_FIELD, PREDECESSOR_RUNS_FIELD,
            ALLOCATED_CAPACITY_FIELD, EXECUTION_TIME_FIELD, TIMEOUT_FIELD, MAX_CAPACITY_FIELD, WORKER_TYPE_FIELD,
            NUMBER_OF_WORKERS_FIELD, SECURITY_CONFIGURATION_FIELD, LOG_GROUP_NAME_FIELD, NOTIFICATION_PROPERTY_FIELD,
            GLUE_VERSION_FIELD, DPU_SECONDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Integer attempt;

    private final String previousRunId;

    private final String triggerName;

    private final String jobName;

    private final Instant startedOn;

    private final Instant lastModifiedOn;

    private final Instant completedOn;

    private final String jobRunState;

    private final Map<String, String> arguments;

    private final String errorMessage;

    private final List<Predecessor> predecessorRuns;

    private final Integer allocatedCapacity;

    private final Integer executionTime;

    private final Integer timeout;

    private final Double maxCapacity;

    private final String workerType;

    private final Integer numberOfWorkers;

    private final String securityConfiguration;

    private final String logGroupName;

    private final NotificationProperty notificationProperty;

    private final String glueVersion;

    private final Double dpuSeconds;

    private JobRun(BuilderImpl builder) {
        this.id = builder.id;
        this.attempt = builder.attempt;
        this.previousRunId = builder.previousRunId;
        this.triggerName = builder.triggerName;
        this.jobName = builder.jobName;
        this.startedOn = builder.startedOn;
        this.lastModifiedOn = builder.lastModifiedOn;
        this.completedOn = builder.completedOn;
        this.jobRunState = builder.jobRunState;
        this.arguments = builder.arguments;
        this.errorMessage = builder.errorMessage;
        this.predecessorRuns = builder.predecessorRuns;
        this.allocatedCapacity = builder.allocatedCapacity;
        this.executionTime = builder.executionTime;
        this.timeout = builder.timeout;
        this.maxCapacity = builder.maxCapacity;
        this.workerType = builder.workerType;
        this.numberOfWorkers = builder.numberOfWorkers;
        this.securityConfiguration = builder.securityConfiguration;
        this.logGroupName = builder.logGroupName;
        this.notificationProperty = builder.notificationProperty;
        this.glueVersion = builder.glueVersion;
        this.dpuSeconds = builder.dpuSeconds;
    }

    /**
     * <p>
     * The ID of this job run.
     * </p>
     * 
     * @return The ID of this job run.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The number of the attempt to run this job.
     * </p>
     * 
     * @return The number of the attempt to run this job.
     */
    public final Integer attempt() {
        return attempt;
    }

    /**
     * <p>
     * The ID of the previous run of this job. For example, the <code>JobRunId</code> specified in the
     * <code>StartJobRun</code> action.
     * </p>
     * 
     * @return The ID of the previous run of this job. For example, the <code>JobRunId</code> specified in the
     *         <code>StartJobRun</code> action.
     */
    public final String previousRunId() {
        return previousRunId;
    }

    /**
     * <p>
     * The name of the trigger that started this job run.
     * </p>
     * 
     * @return The name of the trigger that started this job run.
     */
    public final String triggerName() {
        return triggerName;
    }

    /**
     * <p>
     * The name of the job definition being used in this run.
     * </p>
     * 
     * @return The name of the job definition being used in this run.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The date and time at which this job run was started.
     * </p>
     * 
     * @return The date and time at which this job run was started.
     */
    public final Instant startedOn() {
        return startedOn;
    }

    /**
     * <p>
     * The last time that this job run was modified.
     * </p>
     * 
     * @return The last time that this job run was modified.
     */
    public final Instant lastModifiedOn() {
        return lastModifiedOn;
    }

    /**
     * <p>
     * The date and time that this job run completed.
     * </p>
     * 
     * @return The date and time that this job run completed.
     */
    public final Instant completedOn() {
        return completedOn;
    }

    /**
     * <p>
     * The current state of the job run. For more information about the statuses of jobs that have terminated
     * abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
     * Statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobRunState} will
     * return {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobRunStateAsString}.
     * </p>
     * 
     * @return The current state of the job run. For more information about the statuses of jobs that have terminated
     *         abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
     *         Statuses</a>.
     * @see JobRunState
     */
    public final JobRunState jobRunState() {
        return JobRunState.fromValue(jobRunState);
    }

    /**
     * <p>
     * The current state of the job run. For more information about the statuses of jobs that have terminated
     * abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
     * Statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobRunState} will
     * return {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobRunStateAsString}.
     * </p>
     * 
     * @return The current state of the job run. For more information about the statuses of jobs that have terminated
     *         abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
     *         Statuses</a>.
     * @see JobRunState
     */
    public final String jobRunStateAsString() {
        return jobRunState;
    }

    /**
     * For responses, this returns true if the service returned a value for the Arguments property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasArguments() {
        return arguments != null && !(arguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The job arguments associated with this run. For this job run, they replace the default arguments set in the job
     * definition itself.
     * </p>
     * <p>
     * You can specify arguments here that your own job-execution script consumes, as well as arguments that Glue itself
     * consumes.
     * </p>
     * <p>
     * For information about how to specify and consume your own job arguments, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue APIs in
     * Python</a> topic in the developer guide.
     * </p>
     * <p>
     * For information about the key-value pairs that Glue consumes to set up your job, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special Parameters
     * Used by Glue</a> topic in the developer guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArguments} method.
     * </p>
     * 
     * @return The job arguments associated with this run. For this job run, they replace the default arguments set in
     *         the job definition itself.</p>
     *         <p>
     *         You can specify arguments here that your own job-execution script consumes, as well as arguments that
     *         Glue itself consumes.
     *         </p>
     *         <p>
     *         For information about how to specify and consume your own job arguments, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue
     *         APIs in Python</a> topic in the developer guide.
     *         </p>
     *         <p>
     *         For information about the key-value pairs that Glue consumes to set up your job, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
     *         Parameters Used by Glue</a> topic in the developer guide.
     */
    public final Map<String, String> arguments() {
        return arguments;
    }

    /**
     * <p>
     * An error message associated with this job run.
     * </p>
     * 
     * @return An error message associated with this job run.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the PredecessorRuns property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPredecessorRuns() {
        return predecessorRuns != null && !(predecessorRuns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of predecessors to this job run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPredecessorRuns} method.
     * </p>
     * 
     * @return A list of predecessors to this job run.
     */
    public final List<Predecessor> predecessorRuns() {
        return predecessorRuns;
    }

    /**
     * <p>
     * This field is deprecated. Use <code>MaxCapacity</code> instead.
     * </p>
     * <p>
     * The number of Glue data processing units (DPUs) allocated to this JobRun. From 2 to 100 DPUs can be allocated;
     * the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute capacity
     * and 16 GB of memory. For more information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing
     * page</a>.
     * </p>
     * 
     * @return This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
     *         <p>
     *         The number of Glue data processing units (DPUs) allocated to this JobRun. From 2 to 100 DPUs can be
     *         allocated; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of
     *         compute capacity and 16 GB of memory. For more information, see the <a
     *         href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
     * @deprecated This property is deprecated, use MaxCapacity instead.
     */
    @Deprecated
    public final Integer allocatedCapacity() {
        return allocatedCapacity;
    }

    /**
     * <p>
     * The amount of time (in seconds) that the job run consumed resources.
     * </p>
     * 
     * @return The amount of time (in seconds) that the job run consumed resources.
     */
    public final Integer executionTime() {
        return executionTime;
    }

    /**
     * <p>
     * The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources before
     * it is terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours). This overrides
     * the timeout value set in the parent job.
     * </p>
     * 
     * @return The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources
     *         before it is terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours).
     *         This overrides the timeout value set in the parent job.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * The number of Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a relative
     * measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more
     * information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
     * </p>
     * <p>
     * Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     * </p>
     * <p>
     * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python shell
     * job or an Apache Spark ETL job:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either 0.0625
     * or 1 DPU. The default is 0.0625 DPU.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl"), you can allocate from 2 to 100
     * DPUs. The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a
     *         relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory.
     *         For more information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.</p>
     *         <p>
     *         Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     *         </p>
     *         <p>
     *         The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
     *         shell job or an Apache Spark ETL job:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
     *         0.0625 or 1 DPU. The default is 0.0625 DPU.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl"), you can allocate from
     *         2 to 100 DPUs. The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
     *         </p>
     *         </li>
     */
    public final Double maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
     *         G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final WorkerType workerType() {
        return WorkerType.fromValue(workerType);
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2
     * executors per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
     * executor per worker.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
     *         G.2X.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk,
     *         and 2 executors per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and
     *         1 executor per worker.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final String workerTypeAsString() {
        return workerType;
    }

    /**
     * <p>
     * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
     * </p>
     * <p>
     * The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
     * </p>
     * 
     * @return The number of workers of a defined <code>workerType</code> that are allocated when a job runs.</p>
     *         <p>
     *         The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
     */
    public final Integer numberOfWorkers() {
        return numberOfWorkers;
    }

    /**
     * <p>
     * The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
     * </p>
     * 
     * @return The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
     */
    public final String securityConfiguration() {
        return securityConfiguration;
    }

    /**
     * <p>
     * The name of the log group for secure logging that can be server-side encrypted in Amazon CloudWatch using KMS.
     * This name can be <code>/aws-glue/jobs/</code>, in which case the default encryption is <code>NONE</code>. If you
     * add a role name and <code>SecurityConfiguration</code> name (in other words,
     * <code>/aws-glue/jobs-yourRoleName-yourSecurityConfigurationName/</code>), then that security configuration is
     * used to encrypt the log group.
     * </p>
     * 
     * @return The name of the log group for secure logging that can be server-side encrypted in Amazon CloudWatch using
     *         KMS. This name can be <code>/aws-glue/jobs/</code>, in which case the default encryption is
     *         <code>NONE</code>. If you add a role name and <code>SecurityConfiguration</code> name (in other words,
     *         <code>/aws-glue/jobs-yourRoleName-yourSecurityConfigurationName/</code>), then that security
     *         configuration is used to encrypt the log group.
     */
    public final String logGroupName() {
        return logGroupName;
    }

    /**
     * <p>
     * Specifies configuration properties of a job run notification.
     * </p>
     * 
     * @return Specifies configuration properties of a job run notification.
     */
    public final NotificationProperty notificationProperty() {
        return notificationProperty;
    }

    /**
     * <p>
     * Glue version determines the versions of Apache Spark and Python that Glue supports. The Python version indicates
     * the version supported for jobs of type Spark.
     * </p>
     * <p>
     * For more information about the available Glue versions and corresponding Spark and Python versions, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer guide.
     * </p>
     * <p>
     * Jobs that are created without specifying a Glue version default to Glue 0.9.
     * </p>
     * 
     * @return Glue version determines the versions of Apache Spark and Python that Glue supports. The Python version
     *         indicates the version supported for jobs of type Spark. </p>
     *         <p>
     *         For more information about the available Glue versions and corresponding Spark and Python versions, see
     *         <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer
     *         guide.
     *         </p>
     *         <p>
     *         Jobs that are created without specifying a Glue version default to Glue 0.9.
     */
    public final String glueVersion() {
        return glueVersion;
    }

    /**
     * <p>
     * This field populates only when an Auto Scaling job run completes, and represents the total time each executor ran
     * during the lifecycle of a job run in seconds, multiplied by a DPU factor (1 for <code>G.1X</code> and 2 for
     * <code>G.2X</code> workers). This value may be different than the <code>executionEngineRuntime</code> *
     * <code>MaxCapacity</code> as in the case of Auto Scaling jobs, as the number of executors running at a given time
     * may be less than the <code>MaxCapacity</code>. Therefore, it is possible that the value of
     * <code>DPUSeconds</code> is less than <code>executionEngineRuntime</code> * <code>MaxCapacity</code>.
     * </p>
     * 
     * @return This field populates only when an Auto Scaling job run completes, and represents the total time each
     *         executor ran during the lifecycle of a job run in seconds, multiplied by a DPU factor (1 for
     *         <code>G.1X</code> and 2 for <code>G.2X</code> workers). This value may be different than the
     *         <code>executionEngineRuntime</code> * <code>MaxCapacity</code> as in the case of Auto Scaling jobs, as
     *         the number of executors running at a given time may be less than the <code>MaxCapacity</code>. Therefore,
     *         it is possible that the value of <code>DPUSeconds</code> is less than <code>executionEngineRuntime</code>
     *         * <code>MaxCapacity</code>.
     */
    public final Double dpuSeconds() {
        return dpuSeconds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(attempt());
        hashCode = 31 * hashCode + Objects.hashCode(previousRunId());
        hashCode = 31 * hashCode + Objects.hashCode(triggerName());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(startedOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedOn());
        hashCode = 31 * hashCode + Objects.hashCode(completedOn());
        hashCode = 31 * hashCode + Objects.hashCode(jobRunStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasArguments() ? arguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasPredecessorRuns() ? predecessorRuns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(allocatedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(executionTime());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(workerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(securityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(logGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(notificationProperty());
        hashCode = 31 * hashCode + Objects.hashCode(glueVersion());
        hashCode = 31 * hashCode + Objects.hashCode(dpuSeconds());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobRun)) {
            return false;
        }
        JobRun other = (JobRun) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(attempt(), other.attempt())
                && Objects.equals(previousRunId(), other.previousRunId()) && Objects.equals(triggerName(), other.triggerName())
                && Objects.equals(jobName(), other.jobName()) && Objects.equals(startedOn(), other.startedOn())
                && Objects.equals(lastModifiedOn(), other.lastModifiedOn()) && Objects.equals(completedOn(), other.completedOn())
                && Objects.equals(jobRunStateAsString(), other.jobRunStateAsString()) && hasArguments() == other.hasArguments()
                && Objects.equals(arguments(), other.arguments()) && Objects.equals(errorMessage(), other.errorMessage())
                && hasPredecessorRuns() == other.hasPredecessorRuns()
                && Objects.equals(predecessorRuns(), other.predecessorRuns())
                && Objects.equals(allocatedCapacity(), other.allocatedCapacity())
                && Objects.equals(executionTime(), other.executionTime()) && Objects.equals(timeout(), other.timeout())
                && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(workerTypeAsString(), other.workerTypeAsString())
                && Objects.equals(numberOfWorkers(), other.numberOfWorkers())
                && Objects.equals(securityConfiguration(), other.securityConfiguration())
                && Objects.equals(logGroupName(), other.logGroupName())
                && Objects.equals(notificationProperty(), other.notificationProperty())
                && Objects.equals(glueVersion(), other.glueVersion()) && Objects.equals(dpuSeconds(), other.dpuSeconds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobRun").add("Id", id()).add("Attempt", attempt()).add("PreviousRunId", previousRunId())
                .add("TriggerName", triggerName()).add("JobName", jobName()).add("StartedOn", startedOn())
                .add("LastModifiedOn", lastModifiedOn()).add("CompletedOn", completedOn())
                .add("JobRunState", jobRunStateAsString()).add("Arguments", hasArguments() ? arguments() : null)
                .add("ErrorMessage", errorMessage()).add("PredecessorRuns", hasPredecessorRuns() ? predecessorRuns() : null)
                .add("AllocatedCapacity", allocatedCapacity()).add("ExecutionTime", executionTime()).add("Timeout", timeout())
                .add("MaxCapacity", maxCapacity()).add("WorkerType", workerTypeAsString())
                .add("NumberOfWorkers", numberOfWorkers()).add("SecurityConfiguration", securityConfiguration())
                .add("LogGroupName", logGroupName()).add("NotificationProperty", notificationProperty())
                .add("GlueVersion", glueVersion()).add("DPUSeconds", dpuSeconds()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Attempt":
            return Optional.ofNullable(clazz.cast(attempt()));
        case "PreviousRunId":
            return Optional.ofNullable(clazz.cast(previousRunId()));
        case "TriggerName":
            return Optional.ofNullable(clazz.cast(triggerName()));
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "StartedOn":
            return Optional.ofNullable(clazz.cast(startedOn()));
        case "LastModifiedOn":
            return Optional.ofNullable(clazz.cast(lastModifiedOn()));
        case "CompletedOn":
            return Optional.ofNullable(clazz.cast(completedOn()));
        case "JobRunState":
            return Optional.ofNullable(clazz.cast(jobRunStateAsString()));
        case "Arguments":
            return Optional.ofNullable(clazz.cast(arguments()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "PredecessorRuns":
            return Optional.ofNullable(clazz.cast(predecessorRuns()));
        case "AllocatedCapacity":
            return Optional.ofNullable(clazz.cast(allocatedCapacity()));
        case "ExecutionTime":
            return Optional.ofNullable(clazz.cast(executionTime()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "WorkerType":
            return Optional.ofNullable(clazz.cast(workerTypeAsString()));
        case "NumberOfWorkers":
            return Optional.ofNullable(clazz.cast(numberOfWorkers()));
        case "SecurityConfiguration":
            return Optional.ofNullable(clazz.cast(securityConfiguration()));
        case "LogGroupName":
            return Optional.ofNullable(clazz.cast(logGroupName()));
        case "NotificationProperty":
            return Optional.ofNullable(clazz.cast(notificationProperty()));
        case "GlueVersion":
            return Optional.ofNullable(clazz.cast(glueVersion()));
        case "DPUSeconds":
            return Optional.ofNullable(clazz.cast(dpuSeconds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobRun, T> g) {
        return obj -> g.apply((JobRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobRun> {
        /**
         * <p>
         * The ID of this job run.
         * </p>
         * 
         * @param id
         *        The ID of this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The number of the attempt to run this job.
         * </p>
         * 
         * @param attempt
         *        The number of the attempt to run this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attempt(Integer attempt);

        /**
         * <p>
         * The ID of the previous run of this job. For example, the <code>JobRunId</code> specified in the
         * <code>StartJobRun</code> action.
         * </p>
         * 
         * @param previousRunId
         *        The ID of the previous run of this job. For example, the <code>JobRunId</code> specified in the
         *        <code>StartJobRun</code> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder previousRunId(String previousRunId);

        /**
         * <p>
         * The name of the trigger that started this job run.
         * </p>
         * 
         * @param triggerName
         *        The name of the trigger that started this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggerName(String triggerName);

        /**
         * <p>
         * The name of the job definition being used in this run.
         * </p>
         * 
         * @param jobName
         *        The name of the job definition being used in this run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The date and time at which this job run was started.
         * </p>
         * 
         * @param startedOn
         *        The date and time at which this job run was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedOn(Instant startedOn);

        /**
         * <p>
         * The last time that this job run was modified.
         * </p>
         * 
         * @param lastModifiedOn
         *        The last time that this job run was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedOn(Instant lastModifiedOn);

        /**
         * <p>
         * The date and time that this job run completed.
         * </p>
         * 
         * @param completedOn
         *        The date and time that this job run completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedOn(Instant completedOn);

        /**
         * <p>
         * The current state of the job run. For more information about the statuses of jobs that have terminated
         * abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
         * Statuses</a>.
         * </p>
         * 
         * @param jobRunState
         *        The current state of the job run. For more information about the statuses of jobs that have terminated
         *        abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job
         *        Run Statuses</a>.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder jobRunState(String jobRunState);

        /**
         * <p>
         * The current state of the job run. For more information about the statuses of jobs that have terminated
         * abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job Run
         * Statuses</a>.
         * </p>
         * 
         * @param jobRunState
         *        The current state of the job run. For more information about the statuses of jobs that have terminated
         *        abnormally, see <a href="https://docs.aws.amazon.com/glue/latest/dg/job-run-statuses.html">Glue Job
         *        Run Statuses</a>.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder jobRunState(JobRunState jobRunState);

        /**
         * <p>
         * The job arguments associated with this run. For this job run, they replace the default arguments set in the
         * job definition itself.
         * </p>
         * <p>
         * You can specify arguments here that your own job-execution script consumes, as well as arguments that Glue
         * itself consumes.
         * </p>
         * <p>
         * For information about how to specify and consume your own job arguments, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue APIs
         * in Python</a> topic in the developer guide.
         * </p>
         * <p>
         * For information about the key-value pairs that Glue consumes to set up your job, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
         * Parameters Used by Glue</a> topic in the developer guide.
         * </p>
         * 
         * @param arguments
         *        The job arguments associated with this run. For this job run, they replace the default arguments set
         *        in the job definition itself.</p>
         *        <p>
         *        You can specify arguments here that your own job-execution script consumes, as well as arguments that
         *        Glue itself consumes.
         *        </p>
         *        <p>
         *        For information about how to specify and consume your own job arguments, see the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling
         *        Glue APIs in Python</a> topic in the developer guide.
         *        </p>
         *        <p>
         *        For information about the key-value pairs that Glue consumes to set up your job, see the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
         *        Parameters Used by Glue</a> topic in the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arguments(Map<String, String> arguments);

        /**
         * <p>
         * An error message associated with this job run.
         * </p>
         * 
         * @param errorMessage
         *        An error message associated with this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * A list of predecessors to this job run.
         * </p>
         * 
         * @param predecessorRuns
         *        A list of predecessors to this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predecessorRuns(Collection<Predecessor> predecessorRuns);

        /**
         * <p>
         * A list of predecessors to this job run.
         * </p>
         * 
         * @param predecessorRuns
         *        A list of predecessors to this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder predecessorRuns(Predecessor... predecessorRuns);

        /**
         * <p>
         * A list of predecessors to this job run.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Predecessor>.Builder} avoiding the
         * need to create one manually via {@link List<Predecessor>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Predecessor>.Builder#build()} is called immediately and its
         * result is passed to {@link #predecessorRuns(List<Predecessor>)}.
         * 
         * @param predecessorRuns
         *        a consumer that will call methods on {@link List<Predecessor>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #predecessorRuns(List<Predecessor>)
         */
        Builder predecessorRuns(Consumer<Predecessor.Builder>... predecessorRuns);

        /**
         * <p>
         * This field is deprecated. Use <code>MaxCapacity</code> instead.
         * </p>
         * <p>
         * The number of Glue data processing units (DPUs) allocated to this JobRun. From 2 to 100 DPUs can be
         * allocated; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of
         * compute capacity and 16 GB of memory. For more information, see the <a
         * href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
         * </p>
         * 
         * @param allocatedCapacity
         *        This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
         *        <p>
         *        The number of Glue data processing units (DPUs) allocated to this JobRun. From 2 to 100 DPUs can be
         *        allocated; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs
         *        of compute capacity and 16 GB of memory. For more information, see the <a
         *        href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated, use MaxCapacity instead.
         */
        @Deprecated
        Builder allocatedCapacity(Integer allocatedCapacity);

        /**
         * <p>
         * The amount of time (in seconds) that the job run consumed resources.
         * </p>
         * 
         * @param executionTime
         *        The amount of time (in seconds) that the job run consumed resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionTime(Integer executionTime);

        /**
         * <p>
         * The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources
         * before it is terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes (48 hours). This
         * overrides the timeout value set in the parent job.
         * </p>
         * 
         * @param timeout
         *        The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume
         *        resources before it is terminated and enters <code>TIMEOUT</code> status. The default is 2,880 minutes
         *        (48 hours). This overrides the timeout value set in the parent job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * The number of Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a relative
         * measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more
         * information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
         * </p>
         * <p>
         * Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
         * </p>
         * <p>
         * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
         * shell job or an Apache Spark ETL job:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
         * 0.0625 or 1 DPU. The default is 0.0625 DPU.
         * </p>
         * </li>
         * <li>
         * <p>
         * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl"), you can allocate from 2 to
         * 100 DPUs. The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param maxCapacity
         *        The number of Glue data processing units (DPUs) that can be allocated when this job runs. A DPU is a
         *        relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory.
         *        For more information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing
         *        page</a>.</p>
         *        <p>
         *        Do not set <code>Max Capacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>
         *        .
         *        </p>
         *        <p>
         *        The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a
         *        Python shell job or an Apache Spark ETL job:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate
         *        either 0.0625 or 1 DPU. The default is 0.0625 DPU.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl"), you can allocate
         *        from 2 to 100 DPUs. The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Double maxCapacity);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
         *        G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(String workerType);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or G.2X.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and
         * 2 executors per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk, and 1
         * executor per worker.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of Standard, G.1X, or
         *        G.2X.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>Standard</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 50GB
         *        disk, and 2 executors per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker provides 4 vCPU, 16 GB of memory and a 64GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker provides 8 vCPU, 32 GB of memory and a 128GB disk,
         *        and 1 executor per worker.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(WorkerType workerType);

        /**
         * <p>
         * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
         * </p>
         * <p>
         * The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for <code>G.2X</code>.
         * </p>
         * 
         * @param numberOfWorkers
         *        The number of workers of a defined <code>workerType</code> that are allocated when a job runs.</p>
         *        <p>
         *        The maximum number of workers you can define are 299 for <code>G.1X</code>, and 149 for
         *        <code>G.2X</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfWorkers(Integer numberOfWorkers);

        /**
         * <p>
         * The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
         * </p>
         * 
         * @param securityConfiguration
         *        The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityConfiguration(String securityConfiguration);

        /**
         * <p>
         * The name of the log group for secure logging that can be server-side encrypted in Amazon CloudWatch using
         * KMS. This name can be <code>/aws-glue/jobs/</code>, in which case the default encryption is <code>NONE</code>
         * . If you add a role name and <code>SecurityConfiguration</code> name (in other words,
         * <code>/aws-glue/jobs-yourRoleName-yourSecurityConfigurationName/</code>), then that security configuration is
         * used to encrypt the log group.
         * </p>
         * 
         * @param logGroupName
         *        The name of the log group for secure logging that can be server-side encrypted in Amazon CloudWatch
         *        using KMS. This name can be <code>/aws-glue/jobs/</code>, in which case the default encryption is
         *        <code>NONE</code>. If you add a role name and <code>SecurityConfiguration</code> name (in other words,
         *        <code>/aws-glue/jobs-yourRoleName-yourSecurityConfigurationName/</code>), then that security
         *        configuration is used to encrypt the log group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupName(String logGroupName);

        /**
         * <p>
         * Specifies configuration properties of a job run notification.
         * </p>
         * 
         * @param notificationProperty
         *        Specifies configuration properties of a job run notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationProperty(NotificationProperty notificationProperty);

        /**
         * <p>
         * Specifies configuration properties of a job run notification.
         * </p>
         * This is a convenience method that creates an instance of the {@link NotificationProperty.Builder} avoiding
         * the need to create one manually via {@link NotificationProperty#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #notificationProperty(NotificationProperty)}.
         * 
         * @param notificationProperty
         *        a consumer that will call methods on {@link NotificationProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationProperty(NotificationProperty)
         */
        default Builder notificationProperty(Consumer<NotificationProperty.Builder> notificationProperty) {
            return notificationProperty(NotificationProperty.builder().applyMutation(notificationProperty).build());
        }

        /**
         * <p>
         * Glue version determines the versions of Apache Spark and Python that Glue supports. The Python version
         * indicates the version supported for jobs of type Spark.
         * </p>
         * <p>
         * For more information about the available Glue versions and corresponding Spark and Python versions, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the developer guide.
         * </p>
         * <p>
         * Jobs that are created without specifying a Glue version default to Glue 0.9.
         * </p>
         * 
         * @param glueVersion
         *        Glue version determines the versions of Apache Spark and Python that Glue supports. The Python version
         *        indicates the version supported for jobs of type Spark. </p>
         *        <p>
         *        For more information about the available Glue versions and corresponding Spark and Python versions,
         *        see <a href="https://docs.aws.amazon.com/glue/latest/dg/add-job.html">Glue version</a> in the
         *        developer guide.
         *        </p>
         *        <p>
         *        Jobs that are created without specifying a Glue version default to Glue 0.9.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueVersion(String glueVersion);

        /**
         * <p>
         * This field populates only when an Auto Scaling job run completes, and represents the total time each executor
         * ran during the lifecycle of a job run in seconds, multiplied by a DPU factor (1 for <code>G.1X</code> and 2
         * for <code>G.2X</code> workers). This value may be different than the <code>executionEngineRuntime</code> *
         * <code>MaxCapacity</code> as in the case of Auto Scaling jobs, as the number of executors running at a given
         * time may be less than the <code>MaxCapacity</code>. Therefore, it is possible that the value of
         * <code>DPUSeconds</code> is less than <code>executionEngineRuntime</code> * <code>MaxCapacity</code>.
         * </p>
         * 
         * @param dpuSeconds
         *        This field populates only when an Auto Scaling job run completes, and represents the total time each
         *        executor ran during the lifecycle of a job run in seconds, multiplied by a DPU factor (1 for
         *        <code>G.1X</code> and 2 for <code>G.2X</code> workers). This value may be different than the
         *        <code>executionEngineRuntime</code> * <code>MaxCapacity</code> as in the case of Auto Scaling jobs, as
         *        the number of executors running at a given time may be less than the <code>MaxCapacity</code>.
         *        Therefore, it is possible that the value of <code>DPUSeconds</code> is less than
         *        <code>executionEngineRuntime</code> * <code>MaxCapacity</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dpuSeconds(Double dpuSeconds);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Integer attempt;

        private String previousRunId;

        private String triggerName;

        private String jobName;

        private Instant startedOn;

        private Instant lastModifiedOn;

        private Instant completedOn;

        private String jobRunState;

        private Map<String, String> arguments = DefaultSdkAutoConstructMap.getInstance();

        private String errorMessage;

        private List<Predecessor> predecessorRuns = DefaultSdkAutoConstructList.getInstance();

        private Integer allocatedCapacity;

        private Integer executionTime;

        private Integer timeout;

        private Double maxCapacity;

        private String workerType;

        private Integer numberOfWorkers;

        private String securityConfiguration;

        private String logGroupName;

        private NotificationProperty notificationProperty;

        private String glueVersion;

        private Double dpuSeconds;

        private BuilderImpl() {
        }

        private BuilderImpl(JobRun model) {
            id(model.id);
            attempt(model.attempt);
            previousRunId(model.previousRunId);
            triggerName(model.triggerName);
            jobName(model.jobName);
            startedOn(model.startedOn);
            lastModifiedOn(model.lastModifiedOn);
            completedOn(model.completedOn);
            jobRunState(model.jobRunState);
            arguments(model.arguments);
            errorMessage(model.errorMessage);
            predecessorRuns(model.predecessorRuns);
            allocatedCapacity(model.allocatedCapacity);
            executionTime(model.executionTime);
            timeout(model.timeout);
            maxCapacity(model.maxCapacity);
            workerType(model.workerType);
            numberOfWorkers(model.numberOfWorkers);
            securityConfiguration(model.securityConfiguration);
            logGroupName(model.logGroupName);
            notificationProperty(model.notificationProperty);
            glueVersion(model.glueVersion);
            dpuSeconds(model.dpuSeconds);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Integer getAttempt() {
            return attempt;
        }

        public final void setAttempt(Integer attempt) {
            this.attempt = attempt;
        }

        @Override
        public final Builder attempt(Integer attempt) {
            this.attempt = attempt;
            return this;
        }

        public final String getPreviousRunId() {
            return previousRunId;
        }

        public final void setPreviousRunId(String previousRunId) {
            this.previousRunId = previousRunId;
        }

        @Override
        public final Builder previousRunId(String previousRunId) {
            this.previousRunId = previousRunId;
            return this;
        }

        public final String getTriggerName() {
            return triggerName;
        }

        public final void setTriggerName(String triggerName) {
            this.triggerName = triggerName;
        }

        @Override
        public final Builder triggerName(String triggerName) {
            this.triggerName = triggerName;
            return this;
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final Instant getStartedOn() {
            return startedOn;
        }

        public final void setStartedOn(Instant startedOn) {
            this.startedOn = startedOn;
        }

        @Override
        public final Builder startedOn(Instant startedOn) {
            this.startedOn = startedOn;
            return this;
        }

        public final Instant getLastModifiedOn() {
            return lastModifiedOn;
        }

        public final void setLastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
        }

        @Override
        public final Builder lastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
            return this;
        }

        public final Instant getCompletedOn() {
            return completedOn;
        }

        public final void setCompletedOn(Instant completedOn) {
            this.completedOn = completedOn;
        }

        @Override
        public final Builder completedOn(Instant completedOn) {
            this.completedOn = completedOn;
            return this;
        }

        public final String getJobRunState() {
            return jobRunState;
        }

        public final void setJobRunState(String jobRunState) {
            this.jobRunState = jobRunState;
        }

        @Override
        public final Builder jobRunState(String jobRunState) {
            this.jobRunState = jobRunState;
            return this;
        }

        @Override
        public final Builder jobRunState(JobRunState jobRunState) {
            this.jobRunState(jobRunState == null ? null : jobRunState.toString());
            return this;
        }

        public final Map<String, String> getArguments() {
            if (arguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return arguments;
        }

        public final void setArguments(Map<String, String> arguments) {
            this.arguments = GenericMapCopier.copy(arguments);
        }

        @Override
        public final Builder arguments(Map<String, String> arguments) {
            this.arguments = GenericMapCopier.copy(arguments);
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final List<Predecessor.Builder> getPredecessorRuns() {
            List<Predecessor.Builder> result = PredecessorListCopier.copyToBuilder(this.predecessorRuns);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPredecessorRuns(Collection<Predecessor.BuilderImpl> predecessorRuns) {
            this.predecessorRuns = PredecessorListCopier.copyFromBuilder(predecessorRuns);
        }

        @Override
        public final Builder predecessorRuns(Collection<Predecessor> predecessorRuns) {
            this.predecessorRuns = PredecessorListCopier.copy(predecessorRuns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder predecessorRuns(Predecessor... predecessorRuns) {
            predecessorRuns(Arrays.asList(predecessorRuns));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder predecessorRuns(Consumer<Predecessor.Builder>... predecessorRuns) {
            predecessorRuns(Stream.of(predecessorRuns).map(c -> Predecessor.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Deprecated
        public final Integer getAllocatedCapacity() {
            return allocatedCapacity;
        }

        @Deprecated
        public final void setAllocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
        }

        @Override
        @Deprecated
        public final Builder allocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
            return this;
        }

        public final Integer getExecutionTime() {
            return executionTime;
        }

        public final void setExecutionTime(Integer executionTime) {
            this.executionTime = executionTime;
        }

        @Override
        public final Builder executionTime(Integer executionTime) {
            this.executionTime = executionTime;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Double getMaxCapacity() {
            return maxCapacity;
        }

        public final void setMaxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final String getWorkerType() {
            return workerType;
        }

        public final void setWorkerType(String workerType) {
            this.workerType = workerType;
        }

        @Override
        public final Builder workerType(String workerType) {
            this.workerType = workerType;
            return this;
        }

        @Override
        public final Builder workerType(WorkerType workerType) {
            this.workerType(workerType == null ? null : workerType.toString());
            return this;
        }

        public final Integer getNumberOfWorkers() {
            return numberOfWorkers;
        }

        public final void setNumberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
        }

        @Override
        public final Builder numberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
            return this;
        }

        public final String getSecurityConfiguration() {
            return securityConfiguration;
        }

        public final void setSecurityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
        }

        @Override
        public final Builder securityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return this;
        }

        public final String getLogGroupName() {
            return logGroupName;
        }

        public final void setLogGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
        }

        @Override
        public final Builder logGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
            return this;
        }

        public final NotificationProperty.Builder getNotificationProperty() {
            return notificationProperty != null ? notificationProperty.toBuilder() : null;
        }

        public final void setNotificationProperty(NotificationProperty.BuilderImpl notificationProperty) {
            this.notificationProperty = notificationProperty != null ? notificationProperty.build() : null;
        }

        @Override
        public final Builder notificationProperty(NotificationProperty notificationProperty) {
            this.notificationProperty = notificationProperty;
            return this;
        }

        public final String getGlueVersion() {
            return glueVersion;
        }

        public final void setGlueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
        }

        @Override
        public final Builder glueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
            return this;
        }

        public final Double getDpuSeconds() {
            return dpuSeconds;
        }

        public final void setDpuSeconds(Double dpuSeconds) {
            this.dpuSeconds = dpuSeconds;
        }

        @Override
        public final Builder dpuSeconds(Double dpuSeconds) {
            this.dpuSeconds = dpuSeconds;
            return this;
        }

        @Override
        public JobRun build() {
            return new JobRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
