/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the equivalent of a Hive user-defined function (<code>UDF</code>) definition.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserDefinedFunction implements SdkPojo, Serializable,
        ToCopyableBuilder<UserDefinedFunction.Builder, UserDefinedFunction> {
    private static final SdkField<String> FUNCTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FunctionName").getter(getter(UserDefinedFunction::functionName)).setter(setter(Builder::functionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionName").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(UserDefinedFunction::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> CLASS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClassName").getter(getter(UserDefinedFunction::className)).setter(setter(Builder::className))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClassName").build()).build();

    private static final SdkField<String> OWNER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerName").getter(getter(UserDefinedFunction::ownerName)).setter(setter(Builder::ownerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerName").build()).build();

    private static final SdkField<String> OWNER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerType").getter(getter(UserDefinedFunction::ownerTypeAsString)).setter(setter(Builder::ownerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerType").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(UserDefinedFunction::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<List<ResourceUri>> RESOURCE_URIS_FIELD = SdkField
            .<List<ResourceUri>> builder(MarshallingType.LIST)
            .memberName("ResourceUris")
            .getter(getter(UserDefinedFunction::resourceUris))
            .setter(setter(Builder::resourceUris))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceUris").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceUri> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceUri::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CATALOG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CatalogId").getter(getter(UserDefinedFunction::catalogId)).setter(setter(Builder::catalogId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FUNCTION_NAME_FIELD,
            DATABASE_NAME_FIELD, CLASS_NAME_FIELD, OWNER_NAME_FIELD, OWNER_TYPE_FIELD, CREATE_TIME_FIELD, RESOURCE_URIS_FIELD,
            CATALOG_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String functionName;

    private final String databaseName;

    private final String className;

    private final String ownerName;

    private final String ownerType;

    private final Instant createTime;

    private final List<ResourceUri> resourceUris;

    private final String catalogId;

    private UserDefinedFunction(BuilderImpl builder) {
        this.functionName = builder.functionName;
        this.databaseName = builder.databaseName;
        this.className = builder.className;
        this.ownerName = builder.ownerName;
        this.ownerType = builder.ownerType;
        this.createTime = builder.createTime;
        this.resourceUris = builder.resourceUris;
        this.catalogId = builder.catalogId;
    }

    /**
     * <p>
     * The name of the function.
     * </p>
     * 
     * @return The name of the function.
     */
    public final String functionName() {
        return functionName;
    }

    /**
     * <p>
     * The name of the catalog database that contains the function.
     * </p>
     * 
     * @return The name of the catalog database that contains the function.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * The Java class that contains the function code.
     * </p>
     * 
     * @return The Java class that contains the function code.
     */
    public final String className() {
        return className;
    }

    /**
     * <p>
     * The owner of the function.
     * </p>
     * 
     * @return The owner of the function.
     */
    public final String ownerName() {
        return ownerName;
    }

    /**
     * <p>
     * The owner type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ownerType} will
     * return {@link PrincipalType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerTypeAsString}.
     * </p>
     * 
     * @return The owner type.
     * @see PrincipalType
     */
    public final PrincipalType ownerType() {
        return PrincipalType.fromValue(ownerType);
    }

    /**
     * <p>
     * The owner type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ownerType} will
     * return {@link PrincipalType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerTypeAsString}.
     * </p>
     * 
     * @return The owner type.
     * @see PrincipalType
     */
    public final String ownerTypeAsString() {
        return ownerType;
    }

    /**
     * <p>
     * The time at which the function was created.
     * </p>
     * 
     * @return The time at which the function was created.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceUris property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceUris() {
        return resourceUris != null && !(resourceUris instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource URIs for the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceUris} method.
     * </p>
     * 
     * @return The resource URIs for the function.
     */
    public final List<ResourceUri> resourceUris() {
        return resourceUris;
    }

    /**
     * <p>
     * The ID of the Data Catalog in which the function resides.
     * </p>
     * 
     * @return The ID of the Data Catalog in which the function resides.
     */
    public final String catalogId() {
        return catalogId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(functionName());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(className());
        hashCode = 31 * hashCode + Objects.hashCode(ownerName());
        hashCode = 31 * hashCode + Objects.hashCode(ownerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceUris() ? resourceUris() : null);
        hashCode = 31 * hashCode + Objects.hashCode(catalogId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserDefinedFunction)) {
            return false;
        }
        UserDefinedFunction other = (UserDefinedFunction) obj;
        return Objects.equals(functionName(), other.functionName()) && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(className(), other.className()) && Objects.equals(ownerName(), other.ownerName())
                && Objects.equals(ownerTypeAsString(), other.ownerTypeAsString())
                && Objects.equals(createTime(), other.createTime()) && hasResourceUris() == other.hasResourceUris()
                && Objects.equals(resourceUris(), other.resourceUris()) && Objects.equals(catalogId(), other.catalogId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserDefinedFunction").add("FunctionName", functionName()).add("DatabaseName", databaseName())
                .add("ClassName", className()).add("OwnerName", ownerName()).add("OwnerType", ownerTypeAsString())
                .add("CreateTime", createTime()).add("ResourceUris", hasResourceUris() ? resourceUris() : null)
                .add("CatalogId", catalogId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FunctionName":
            return Optional.ofNullable(clazz.cast(functionName()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "ClassName":
            return Optional.ofNullable(clazz.cast(className()));
        case "OwnerName":
            return Optional.ofNullable(clazz.cast(ownerName()));
        case "OwnerType":
            return Optional.ofNullable(clazz.cast(ownerTypeAsString()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "ResourceUris":
            return Optional.ofNullable(clazz.cast(resourceUris()));
        case "CatalogId":
            return Optional.ofNullable(clazz.cast(catalogId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UserDefinedFunction, T> g) {
        return obj -> g.apply((UserDefinedFunction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserDefinedFunction> {
        /**
         * <p>
         * The name of the function.
         * </p>
         * 
         * @param functionName
         *        The name of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionName(String functionName);

        /**
         * <p>
         * The name of the catalog database that contains the function.
         * </p>
         * 
         * @param databaseName
         *        The name of the catalog database that contains the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * The Java class that contains the function code.
         * </p>
         * 
         * @param className
         *        The Java class that contains the function code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder className(String className);

        /**
         * <p>
         * The owner of the function.
         * </p>
         * 
         * @param ownerName
         *        The owner of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerName(String ownerName);

        /**
         * <p>
         * The owner type.
         * </p>
         * 
         * @param ownerType
         *        The owner type.
         * @see PrincipalType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PrincipalType
         */
        Builder ownerType(String ownerType);

        /**
         * <p>
         * The owner type.
         * </p>
         * 
         * @param ownerType
         *        The owner type.
         * @see PrincipalType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PrincipalType
         */
        Builder ownerType(PrincipalType ownerType);

        /**
         * <p>
         * The time at which the function was created.
         * </p>
         * 
         * @param createTime
         *        The time at which the function was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The resource URIs for the function.
         * </p>
         * 
         * @param resourceUris
         *        The resource URIs for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceUris(Collection<ResourceUri> resourceUris);

        /**
         * <p>
         * The resource URIs for the function.
         * </p>
         * 
         * @param resourceUris
         *        The resource URIs for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceUris(ResourceUri... resourceUris);

        /**
         * <p>
         * The resource URIs for the function.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ResourceUri>.Builder} avoiding the
         * need to create one manually via {@link List<ResourceUri>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceUri>.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceUris(List<ResourceUri>)}.
         * 
         * @param resourceUris
         *        a consumer that will call methods on {@link List<ResourceUri>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceUris(List<ResourceUri>)
         */
        Builder resourceUris(Consumer<ResourceUri.Builder>... resourceUris);

        /**
         * <p>
         * The ID of the Data Catalog in which the function resides.
         * </p>
         * 
         * @param catalogId
         *        The ID of the Data Catalog in which the function resides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogId(String catalogId);
    }

    static final class BuilderImpl implements Builder {
        private String functionName;

        private String databaseName;

        private String className;

        private String ownerName;

        private String ownerType;

        private Instant createTime;

        private List<ResourceUri> resourceUris = DefaultSdkAutoConstructList.getInstance();

        private String catalogId;

        private BuilderImpl() {
        }

        private BuilderImpl(UserDefinedFunction model) {
            functionName(model.functionName);
            databaseName(model.databaseName);
            className(model.className);
            ownerName(model.ownerName);
            ownerType(model.ownerType);
            createTime(model.createTime);
            resourceUris(model.resourceUris);
            catalogId(model.catalogId);
        }

        public final String getFunctionName() {
            return functionName;
        }

        public final void setFunctionName(String functionName) {
            this.functionName = functionName;
        }

        @Override
        public final Builder functionName(String functionName) {
            this.functionName = functionName;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final String getClassName() {
            return className;
        }

        public final void setClassName(String className) {
            this.className = className;
        }

        @Override
        public final Builder className(String className) {
            this.className = className;
            return this;
        }

        public final String getOwnerName() {
            return ownerName;
        }

        public final void setOwnerName(String ownerName) {
            this.ownerName = ownerName;
        }

        @Override
        public final Builder ownerName(String ownerName) {
            this.ownerName = ownerName;
            return this;
        }

        public final String getOwnerType() {
            return ownerType;
        }

        public final void setOwnerType(String ownerType) {
            this.ownerType = ownerType;
        }

        @Override
        public final Builder ownerType(String ownerType) {
            this.ownerType = ownerType;
            return this;
        }

        @Override
        public final Builder ownerType(PrincipalType ownerType) {
            this.ownerType(ownerType == null ? null : ownerType.toString());
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final List<ResourceUri.Builder> getResourceUris() {
            List<ResourceUri.Builder> result = ResourceUriListCopier.copyToBuilder(this.resourceUris);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceUris(Collection<ResourceUri.BuilderImpl> resourceUris) {
            this.resourceUris = ResourceUriListCopier.copyFromBuilder(resourceUris);
        }

        @Override
        public final Builder resourceUris(Collection<ResourceUri> resourceUris) {
            this.resourceUris = ResourceUriListCopier.copy(resourceUris);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceUris(ResourceUri... resourceUris) {
            resourceUris(Arrays.asList(resourceUris));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceUris(Consumer<ResourceUri.Builder>... resourceUris) {
            resourceUris(Stream.of(resourceUris).map(c -> ResourceUri.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getCatalogId() {
            return catalogId;
        }

        public final void setCatalogId(String catalogId) {
            this.catalogId = catalogId;
        }

        @Override
        public final Builder catalogId(String catalogId) {
            this.catalogId = catalogId;
            return this;
        }

        @Override
        public UserDefinedFunction build() {
            return new UserDefinedFunction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
