/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of errors that can occur when registering partition indexes for an existing table.
 * </p>
 * <p>
 * These errors give the details about why an index registration failed and provide a limited number of partitions in
 * the response, so that you can fix the partitions at fault and try registering the index again. The most common set of
 * errors that can occur are categorized as follows:
 * </p>
 * <ul>
 * <li>
 * <p>
 * EncryptedPartitionError: The partitions are encrypted.
 * </p>
 * </li>
 * <li>
 * <p>
 * InvalidPartitionTypeDataError: The partition value doesn't match the data type for that partition column.
 * </p>
 * </li>
 * <li>
 * <p>
 * MissingPartitionValueError: The partitions are encrypted.
 * </p>
 * </li>
 * <li>
 * <p>
 * UnsupportedPartitionCharacterError: Characters inside the partition value are not supported. For example: U+0000 ,
 * U+0001, U+0002.
 * </p>
 * </li>
 * <li>
 * <p>
 * InternalError: Any error which does not belong to other error codes.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackfillError implements SdkPojo, Serializable, ToCopyableBuilder<BackfillError.Builder, BackfillError> {
    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Code")
            .getter(getter(BackfillError::codeAsString)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Code").build()).build();

    private static final SdkField<List<PartitionValueList>> PARTITIONS_FIELD = SdkField
            .<List<PartitionValueList>> builder(MarshallingType.LIST)
            .memberName("Partitions")
            .getter(getter(BackfillError::partitions))
            .setter(setter(Builder::partitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Partitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PartitionValueList> builder(MarshallingType.SDK_POJO)
                                            .constructor(PartitionValueList::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CODE_FIELD, PARTITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String code;

    private final List<PartitionValueList> partitions;

    private BackfillError(BuilderImpl builder) {
        this.code = builder.code;
        this.partitions = builder.partitions;
    }

    /**
     * <p>
     * The error code for an error that occurred when registering partition indexes for an existing table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link BackfillErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return The error code for an error that occurred when registering partition indexes for an existing table.
     * @see BackfillErrorCode
     */
    public final BackfillErrorCode code() {
        return BackfillErrorCode.fromValue(code);
    }

    /**
     * <p>
     * The error code for an error that occurred when registering partition indexes for an existing table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #code} will return
     * {@link BackfillErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #codeAsString}.
     * </p>
     * 
     * @return The error code for an error that occurred when registering partition indexes for an existing table.
     * @see BackfillErrorCode
     */
    public final String codeAsString() {
        return code;
    }

    /**
     * For responses, this returns true if the service returned a value for the Partitions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPartitions() {
        return partitions != null && !(partitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of a limited number of partitions in the response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitions} method.
     * </p>
     * 
     * @return A list of a limited number of partitions in the response.
     */
    public final List<PartitionValueList> partitions() {
        return partitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(codeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitions() ? partitions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackfillError)) {
            return false;
        }
        BackfillError other = (BackfillError) obj;
        return Objects.equals(codeAsString(), other.codeAsString()) && hasPartitions() == other.hasPartitions()
                && Objects.equals(partitions(), other.partitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BackfillError").add("Code", codeAsString())
                .add("Partitions", hasPartitions() ? partitions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Code":
            return Optional.ofNullable(clazz.cast(codeAsString()));
        case "Partitions":
            return Optional.ofNullable(clazz.cast(partitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BackfillError, T> g) {
        return obj -> g.apply((BackfillError) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackfillError> {
        /**
         * <p>
         * The error code for an error that occurred when registering partition indexes for an existing table.
         * </p>
         * 
         * @param code
         *        The error code for an error that occurred when registering partition indexes for an existing table.
         * @see BackfillErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillErrorCode
         */
        Builder code(String code);

        /**
         * <p>
         * The error code for an error that occurred when registering partition indexes for an existing table.
         * </p>
         * 
         * @param code
         *        The error code for an error that occurred when registering partition indexes for an existing table.
         * @see BackfillErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackfillErrorCode
         */
        Builder code(BackfillErrorCode code);

        /**
         * <p>
         * A list of a limited number of partitions in the response.
         * </p>
         * 
         * @param partitions
         *        A list of a limited number of partitions in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitions(Collection<PartitionValueList> partitions);

        /**
         * <p>
         * A list of a limited number of partitions in the response.
         * </p>
         * 
         * @param partitions
         *        A list of a limited number of partitions in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitions(PartitionValueList... partitions);

        /**
         * <p>
         * A list of a limited number of partitions in the response.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<PartitionValueList>.Builder}
         * avoiding the need to create one manually via {@link List<PartitionValueList>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PartitionValueList>.Builder#build()} is called immediately
         * and its result is passed to {@link #partitions(List<PartitionValueList>)}.
         * 
         * @param partitions
         *        a consumer that will call methods on {@link List<PartitionValueList>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #partitions(List<PartitionValueList>)
         */
        Builder partitions(Consumer<PartitionValueList.Builder>... partitions);
    }

    static final class BuilderImpl implements Builder {
        private String code;

        private List<PartitionValueList> partitions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BackfillError model) {
            code(model.code);
            partitions(model.partitions);
        }

        public final String getCode() {
            return code;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        @Override
        public final Builder code(BackfillErrorCode code) {
            this.code(code == null ? null : code.toString());
            return this;
        }

        public final List<PartitionValueList.Builder> getPartitions() {
            List<PartitionValueList.Builder> result = BackfillErroredPartitionsListCopier.copyToBuilder(this.partitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPartitions(Collection<PartitionValueList.BuilderImpl> partitions) {
            this.partitions = BackfillErroredPartitionsListCopier.copyFromBuilder(partitions);
        }

        @Override
        public final Builder partitions(Collection<PartitionValueList> partitions) {
            this.partitions = BackfillErroredPartitionsListCopier.copy(partitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitions(PartitionValueList... partitions) {
            partitions(Arrays.asList(partitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitions(Consumer<PartitionValueList.Builder>... partitions) {
            partitions(Stream.of(partitions).map(c -> PartitionValueList.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public BackfillError build() {
            return new BackfillError(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
