/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a connector to a JDBC data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JDBCConnectorSource implements SdkPojo, Serializable,
        ToCopyableBuilder<JDBCConnectorSource.Builder, JDBCConnectorSource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(JDBCConnectorSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CONNECTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionName").getter(getter(JDBCConnectorSource::connectionName))
            .setter(setter(Builder::connectionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionName").build()).build();

    private static final SdkField<String> CONNECTOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectorName").getter(getter(JDBCConnectorSource::connectorName))
            .setter(setter(Builder::connectorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectorName").build()).build();

    private static final SdkField<String> CONNECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionType").getter(getter(JDBCConnectorSource::connectionType))
            .setter(setter(Builder::connectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionType").build()).build();

    private static final SdkField<JDBCConnectorOptions> ADDITIONAL_OPTIONS_FIELD = SdkField
            .<JDBCConnectorOptions> builder(MarshallingType.SDK_POJO).memberName("AdditionalOptions")
            .getter(getter(JDBCConnectorSource::additionalOptions)).setter(setter(Builder::additionalOptions))
            .constructor(JDBCConnectorOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalOptions").build()).build();

    private static final SdkField<String> CONNECTION_TABLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConnectionTable").getter(getter(JDBCConnectorSource::connectionTable))
            .setter(setter(Builder::connectionTable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionTable").build()).build();

    private static final SdkField<String> QUERY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Query")
            .getter(getter(JDBCConnectorSource::query)).setter(setter(Builder::query))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Query").build()).build();

    private static final SdkField<List<GlueSchema>> OUTPUT_SCHEMAS_FIELD = SdkField
            .<List<GlueSchema>> builder(MarshallingType.LIST)
            .memberName("OutputSchemas")
            .getter(getter(JDBCConnectorSource::outputSchemas))
            .setter(setter(Builder::outputSchemas))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputSchemas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GlueSchema> builder(MarshallingType.SDK_POJO)
                                            .constructor(GlueSchema::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CONNECTION_NAME_FIELD, CONNECTOR_NAME_FIELD, CONNECTION_TYPE_FIELD, ADDITIONAL_OPTIONS_FIELD, CONNECTION_TABLE_FIELD,
            QUERY_FIELD, OUTPUT_SCHEMAS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String connectionName;

    private final String connectorName;

    private final String connectionType;

    private final JDBCConnectorOptions additionalOptions;

    private final String connectionTable;

    private final String query;

    private final List<GlueSchema> outputSchemas;

    private JDBCConnectorSource(BuilderImpl builder) {
        this.name = builder.name;
        this.connectionName = builder.connectionName;
        this.connectorName = builder.connectorName;
        this.connectionType = builder.connectionType;
        this.additionalOptions = builder.additionalOptions;
        this.connectionTable = builder.connectionTable;
        this.query = builder.query;
        this.outputSchemas = builder.outputSchemas;
    }

    /**
     * <p>
     * The name of the data source.
     * </p>
     * 
     * @return The name of the data source.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name of the connection that is associated with the connector.
     * </p>
     * 
     * @return The name of the connection that is associated with the connector.
     */
    public final String connectionName() {
        return connectionName;
    }

    /**
     * <p>
     * The name of a connector that assists with accessing the data store in Glue Studio.
     * </p>
     * 
     * @return The name of a connector that assists with accessing the data store in Glue Studio.
     */
    public final String connectorName() {
        return connectorName;
    }

    /**
     * <p>
     * The type of connection, such as marketplace.jdbc or custom.jdbc, designating a connection to a JDBC data store.
     * </p>
     * 
     * @return The type of connection, such as marketplace.jdbc or custom.jdbc, designating a connection to a JDBC data
     *         store.
     */
    public final String connectionType() {
        return connectionType;
    }

    /**
     * <p>
     * Additional connection options for the connector.
     * </p>
     * 
     * @return Additional connection options for the connector.
     */
    public final JDBCConnectorOptions additionalOptions() {
        return additionalOptions;
    }

    /**
     * <p>
     * The name of the table in the data source.
     * </p>
     * 
     * @return The name of the table in the data source.
     */
    public final String connectionTable() {
        return connectionTable;
    }

    /**
     * <p>
     * The table or SQL query to get the data from. You can specify either <code>ConnectionTable</code> or
     * <code>query</code>, but not both.
     * </p>
     * 
     * @return The table or SQL query to get the data from. You can specify either <code>ConnectionTable</code> or
     *         <code>query</code>, but not both.
     */
    public final String query() {
        return query;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputSchemas property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputSchemas() {
        return outputSchemas != null && !(outputSchemas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the data schema for the custom JDBC source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputSchemas} method.
     * </p>
     * 
     * @return Specifies the data schema for the custom JDBC source.
     */
    public final List<GlueSchema> outputSchemas() {
        return outputSchemas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(connectionName());
        hashCode = 31 * hashCode + Objects.hashCode(connectorName());
        hashCode = 31 * hashCode + Objects.hashCode(connectionType());
        hashCode = 31 * hashCode + Objects.hashCode(additionalOptions());
        hashCode = 31 * hashCode + Objects.hashCode(connectionTable());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputSchemas() ? outputSchemas() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JDBCConnectorSource)) {
            return false;
        }
        JDBCConnectorSource other = (JDBCConnectorSource) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(connectionName(), other.connectionName())
                && Objects.equals(connectorName(), other.connectorName())
                && Objects.equals(connectionType(), other.connectionType())
                && Objects.equals(additionalOptions(), other.additionalOptions())
                && Objects.equals(connectionTable(), other.connectionTable()) && Objects.equals(query(), other.query())
                && hasOutputSchemas() == other.hasOutputSchemas() && Objects.equals(outputSchemas(), other.outputSchemas());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JDBCConnectorSource").add("Name", name()).add("ConnectionName", connectionName())
                .add("ConnectorName", connectorName()).add("ConnectionType", connectionType())
                .add("AdditionalOptions", additionalOptions()).add("ConnectionTable", connectionTable()).add("Query", query())
                .add("OutputSchemas", hasOutputSchemas() ? outputSchemas() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ConnectionName":
            return Optional.ofNullable(clazz.cast(connectionName()));
        case "ConnectorName":
            return Optional.ofNullable(clazz.cast(connectorName()));
        case "ConnectionType":
            return Optional.ofNullable(clazz.cast(connectionType()));
        case "AdditionalOptions":
            return Optional.ofNullable(clazz.cast(additionalOptions()));
        case "ConnectionTable":
            return Optional.ofNullable(clazz.cast(connectionTable()));
        case "Query":
            return Optional.ofNullable(clazz.cast(query()));
        case "OutputSchemas":
            return Optional.ofNullable(clazz.cast(outputSchemas()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JDBCConnectorSource, T> g) {
        return obj -> g.apply((JDBCConnectorSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JDBCConnectorSource> {
        /**
         * <p>
         * The name of the data source.
         * </p>
         * 
         * @param name
         *        The name of the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the connection that is associated with the connector.
         * </p>
         * 
         * @param connectionName
         *        The name of the connection that is associated with the connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionName(String connectionName);

        /**
         * <p>
         * The name of a connector that assists with accessing the data store in Glue Studio.
         * </p>
         * 
         * @param connectorName
         *        The name of a connector that assists with accessing the data store in Glue Studio.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectorName(String connectorName);

        /**
         * <p>
         * The type of connection, such as marketplace.jdbc or custom.jdbc, designating a connection to a JDBC data
         * store.
         * </p>
         * 
         * @param connectionType
         *        The type of connection, such as marketplace.jdbc or custom.jdbc, designating a connection to a JDBC
         *        data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionType(String connectionType);

        /**
         * <p>
         * Additional connection options for the connector.
         * </p>
         * 
         * @param additionalOptions
         *        Additional connection options for the connector.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalOptions(JDBCConnectorOptions additionalOptions);

        /**
         * <p>
         * Additional connection options for the connector.
         * </p>
         * This is a convenience method that creates an instance of the {@link JDBCConnectorOptions.Builder} avoiding
         * the need to create one manually via {@link JDBCConnectorOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link JDBCConnectorOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #additionalOptions(JDBCConnectorOptions)}.
         * 
         * @param additionalOptions
         *        a consumer that will call methods on {@link JDBCConnectorOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #additionalOptions(JDBCConnectorOptions)
         */
        default Builder additionalOptions(Consumer<JDBCConnectorOptions.Builder> additionalOptions) {
            return additionalOptions(JDBCConnectorOptions.builder().applyMutation(additionalOptions).build());
        }

        /**
         * <p>
         * The name of the table in the data source.
         * </p>
         * 
         * @param connectionTable
         *        The name of the table in the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionTable(String connectionTable);

        /**
         * <p>
         * The table or SQL query to get the data from. You can specify either <code>ConnectionTable</code> or
         * <code>query</code>, but not both.
         * </p>
         * 
         * @param query
         *        The table or SQL query to get the data from. You can specify either <code>ConnectionTable</code> or
         *        <code>query</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(String query);

        /**
         * <p>
         * Specifies the data schema for the custom JDBC source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for the custom JDBC source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(Collection<GlueSchema> outputSchemas);

        /**
         * <p>
         * Specifies the data schema for the custom JDBC source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for the custom JDBC source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(GlueSchema... outputSchemas);

        /**
         * <p>
         * Specifies the data schema for the custom JDBC source.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<GlueSchema>.Builder} avoiding the
         * need to create one manually via {@link List<GlueSchema>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GlueSchema>.Builder#build()} is called immediately and its
         * result is passed to {@link #outputSchemas(List<GlueSchema>)}.
         * 
         * @param outputSchemas
         *        a consumer that will call methods on {@link List<GlueSchema>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputSchemas(List<GlueSchema>)
         */
        Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String connectionName;

        private String connectorName;

        private String connectionType;

        private JDBCConnectorOptions additionalOptions;

        private String connectionTable;

        private String query;

        private List<GlueSchema> outputSchemas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JDBCConnectorSource model) {
            name(model.name);
            connectionName(model.connectionName);
            connectorName(model.connectorName);
            connectionType(model.connectionType);
            additionalOptions(model.additionalOptions);
            connectionTable(model.connectionTable);
            query(model.query);
            outputSchemas(model.outputSchemas);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getConnectionName() {
            return connectionName;
        }

        public final void setConnectionName(String connectionName) {
            this.connectionName = connectionName;
        }

        @Override
        public final Builder connectionName(String connectionName) {
            this.connectionName = connectionName;
            return this;
        }

        public final String getConnectorName() {
            return connectorName;
        }

        public final void setConnectorName(String connectorName) {
            this.connectorName = connectorName;
        }

        @Override
        public final Builder connectorName(String connectorName) {
            this.connectorName = connectorName;
            return this;
        }

        public final String getConnectionType() {
            return connectionType;
        }

        public final void setConnectionType(String connectionType) {
            this.connectionType = connectionType;
        }

        @Override
        public final Builder connectionType(String connectionType) {
            this.connectionType = connectionType;
            return this;
        }

        public final JDBCConnectorOptions.Builder getAdditionalOptions() {
            return additionalOptions != null ? additionalOptions.toBuilder() : null;
        }

        public final void setAdditionalOptions(JDBCConnectorOptions.BuilderImpl additionalOptions) {
            this.additionalOptions = additionalOptions != null ? additionalOptions.build() : null;
        }

        @Override
        public final Builder additionalOptions(JDBCConnectorOptions additionalOptions) {
            this.additionalOptions = additionalOptions;
            return this;
        }

        public final String getConnectionTable() {
            return connectionTable;
        }

        public final void setConnectionTable(String connectionTable) {
            this.connectionTable = connectionTable;
        }

        @Override
        public final Builder connectionTable(String connectionTable) {
            this.connectionTable = connectionTable;
            return this;
        }

        public final String getQuery() {
            return query;
        }

        public final void setQuery(String query) {
            this.query = query;
        }

        @Override
        public final Builder query(String query) {
            this.query = query;
            return this;
        }

        public final List<GlueSchema.Builder> getOutputSchemas() {
            List<GlueSchema.Builder> result = GlueSchemasCopier.copyToBuilder(this.outputSchemas);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputSchemas(Collection<GlueSchema.BuilderImpl> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copyFromBuilder(outputSchemas);
        }

        @Override
        public final Builder outputSchemas(Collection<GlueSchema> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copy(outputSchemas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(GlueSchema... outputSchemas) {
            outputSchemas(Arrays.asList(outputSchemas));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas) {
            outputSchemas(Stream.of(outputSchemas).map(c -> GlueSchema.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public JDBCConnectorSource build() {
            return new JDBCConnectorSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
