/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a data target that writes to Amazon S3.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DirectTarget implements SdkPojo, Serializable, ToCopyableBuilder<S3DirectTarget.Builder, S3DirectTarget> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(S3DirectTarget::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> INPUTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Inputs")
            .getter(getter(S3DirectTarget::inputs))
            .setter(setter(Builder::inputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Inputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<List<String>>> PARTITION_KEYS_FIELD = SdkField
            .<List<List<String>>> builder(MarshallingType.LIST)
            .memberName("PartitionKeys")
            .getter(getter(S3DirectTarget::partitionKeys))
            .setter(setter(Builder::partitionKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(S3DirectTarget::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<String> COMPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Compression").getter(getter(S3DirectTarget::compression)).setter(setter(Builder::compression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Compression").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(S3DirectTarget::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<DirectSchemaChangePolicy> SCHEMA_CHANGE_POLICY_FIELD = SdkField
            .<DirectSchemaChangePolicy> builder(MarshallingType.SDK_POJO).memberName("SchemaChangePolicy")
            .getter(getter(S3DirectTarget::schemaChangePolicy)).setter(setter(Builder::schemaChangePolicy))
            .constructor(DirectSchemaChangePolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaChangePolicy").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, INPUTS_FIELD,
            PARTITION_KEYS_FIELD, PATH_FIELD, COMPRESSION_FIELD, FORMAT_FIELD, SCHEMA_CHANGE_POLICY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> inputs;

    private final List<List<String>> partitionKeys;

    private final String path;

    private final String compression;

    private final String format;

    private final DirectSchemaChangePolicy schemaChangePolicy;

    private S3DirectTarget(BuilderImpl builder) {
        this.name = builder.name;
        this.inputs = builder.inputs;
        this.partitionKeys = builder.partitionKeys;
        this.path = builder.path;
        this.compression = builder.compression;
        this.format = builder.format;
        this.schemaChangePolicy = builder.schemaChangePolicy;
    }

    /**
     * <p>
     * The name of the data target.
     * </p>
     * 
     * @return The name of the data target.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Inputs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasInputs() {
        return inputs != null && !(inputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The nodes that are inputs to the data target.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputs} method.
     * </p>
     * 
     * @return The nodes that are inputs to the data target.
     */
    public final List<String> inputs() {
        return inputs;
    }

    /**
     * For responses, this returns true if the service returned a value for the PartitionKeys property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPartitionKeys() {
        return partitionKeys != null && !(partitionKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies native partitioning using a sequence of keys.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPartitionKeys} method.
     * </p>
     * 
     * @return Specifies native partitioning using a sequence of keys.
     */
    public final List<List<String>> partitionKeys() {
        return partitionKeys;
    }

    /**
     * <p>
     * A single Amazon S3 path to write to.
     * </p>
     * 
     * @return A single Amazon S3 path to write to.
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * Specifies how the data is compressed. This is generally not necessary if the data has a standard file extension.
     * Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     * </p>
     * 
     * @return Specifies how the data is compressed. This is generally not necessary if the data has a standard file
     *         extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     */
    public final String compression() {
        return compression;
    }

    /**
     * <p>
     * Specifies the data output format for the target.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TargetFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return Specifies the data output format for the target.
     * @see TargetFormat
     */
    public final TargetFormat format() {
        return TargetFormat.fromValue(format);
    }

    /**
     * <p>
     * Specifies the data output format for the target.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link TargetFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return Specifies the data output format for the target.
     * @see TargetFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * A policy that specifies update behavior for the crawler.
     * </p>
     * 
     * @return A policy that specifies update behavior for the crawler.
     */
    public final DirectSchemaChangePolicy schemaChangePolicy() {
        return schemaChangePolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputs() ? inputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionKeys() ? partitionKeys() : null);
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(compression());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schemaChangePolicy());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DirectTarget)) {
            return false;
        }
        S3DirectTarget other = (S3DirectTarget) obj;
        return Objects.equals(name(), other.name()) && hasInputs() == other.hasInputs()
                && Objects.equals(inputs(), other.inputs()) && hasPartitionKeys() == other.hasPartitionKeys()
                && Objects.equals(partitionKeys(), other.partitionKeys()) && Objects.equals(path(), other.path())
                && Objects.equals(compression(), other.compression()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(schemaChangePolicy(), other.schemaChangePolicy());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DirectTarget").add("Name", name()).add("Inputs", hasInputs() ? inputs() : null)
                .add("PartitionKeys", hasPartitionKeys() ? partitionKeys() : null).add("Path", path())
                .add("Compression", compression()).add("Format", formatAsString())
                .add("SchemaChangePolicy", schemaChangePolicy()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Inputs":
            return Optional.ofNullable(clazz.cast(inputs()));
        case "PartitionKeys":
            return Optional.ofNullable(clazz.cast(partitionKeys()));
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "Compression":
            return Optional.ofNullable(clazz.cast(compression()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "SchemaChangePolicy":
            return Optional.ofNullable(clazz.cast(schemaChangePolicy()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DirectTarget, T> g) {
        return obj -> g.apply((S3DirectTarget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DirectTarget> {
        /**
         * <p>
         * The name of the data target.
         * </p>
         * 
         * @param name
         *        The name of the data target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The nodes that are inputs to the data target.
         * </p>
         * 
         * @param inputs
         *        The nodes that are inputs to the data target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(Collection<String> inputs);

        /**
         * <p>
         * The nodes that are inputs to the data target.
         * </p>
         * 
         * @param inputs
         *        The nodes that are inputs to the data target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(String... inputs);

        /**
         * <p>
         * Specifies native partitioning using a sequence of keys.
         * </p>
         * 
         * @param partitionKeys
         *        Specifies native partitioning using a sequence of keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Collection<? extends Collection<String>> partitionKeys);

        /**
         * <p>
         * Specifies native partitioning using a sequence of keys.
         * </p>
         * 
         * @param partitionKeys
         *        Specifies native partitioning using a sequence of keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionKeys(Collection<String>... partitionKeys);

        /**
         * <p>
         * A single Amazon S3 path to write to.
         * </p>
         * 
         * @param path
         *        A single Amazon S3 path to write to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         * extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * </p>
         * 
         * @param compression
         *        Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         *        extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compression(String compression);

        /**
         * <p>
         * Specifies the data output format for the target.
         * </p>
         * 
         * @param format
         *        Specifies the data output format for the target.
         * @see TargetFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetFormat
         */
        Builder format(String format);

        /**
         * <p>
         * Specifies the data output format for the target.
         * </p>
         * 
         * @param format
         *        Specifies the data output format for the target.
         * @see TargetFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetFormat
         */
        Builder format(TargetFormat format);

        /**
         * <p>
         * A policy that specifies update behavior for the crawler.
         * </p>
         * 
         * @param schemaChangePolicy
         *        A policy that specifies update behavior for the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaChangePolicy(DirectSchemaChangePolicy schemaChangePolicy);

        /**
         * <p>
         * A policy that specifies update behavior for the crawler.
         * </p>
         * This is a convenience method that creates an instance of the {@link DirectSchemaChangePolicy.Builder}
         * avoiding the need to create one manually via {@link DirectSchemaChangePolicy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DirectSchemaChangePolicy.Builder#build()} is called immediately
         * and its result is passed to {@link #schemaChangePolicy(DirectSchemaChangePolicy)}.
         * 
         * @param schemaChangePolicy
         *        a consumer that will call methods on {@link DirectSchemaChangePolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaChangePolicy(DirectSchemaChangePolicy)
         */
        default Builder schemaChangePolicy(Consumer<DirectSchemaChangePolicy.Builder> schemaChangePolicy) {
            return schemaChangePolicy(DirectSchemaChangePolicy.builder().applyMutation(schemaChangePolicy).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> inputs = DefaultSdkAutoConstructList.getInstance();

        private List<List<String>> partitionKeys = DefaultSdkAutoConstructList.getInstance();

        private String path;

        private String compression;

        private String format;

        private DirectSchemaChangePolicy schemaChangePolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DirectTarget model) {
            name(model.name);
            inputs(model.inputs);
            partitionKeys(model.partitionKeys);
            path(model.path);
            compression(model.compression);
            format(model.format);
            schemaChangePolicy(model.schemaChangePolicy);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getInputs() {
            if (inputs instanceof SdkAutoConstructList) {
                return null;
            }
            return inputs;
        }

        public final void setInputs(Collection<String> inputs) {
            this.inputs = OneInputCopier.copy(inputs);
        }

        @Override
        public final Builder inputs(Collection<String> inputs) {
            this.inputs = OneInputCopier.copy(inputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputs(String... inputs) {
            inputs(Arrays.asList(inputs));
            return this;
        }

        public final Collection<? extends Collection<String>> getPartitionKeys() {
            if (partitionKeys instanceof SdkAutoConstructList) {
                return null;
            }
            return partitionKeys;
        }

        public final void setPartitionKeys(Collection<? extends Collection<String>> partitionKeys) {
            this.partitionKeys = GlueStudioPathListCopier.copy(partitionKeys);
        }

        @Override
        public final Builder partitionKeys(Collection<? extends Collection<String>> partitionKeys) {
            this.partitionKeys = GlueStudioPathListCopier.copy(partitionKeys);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionKeys(Collection<String>... partitionKeys) {
            partitionKeys(Arrays.asList(partitionKeys));
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final String getCompression() {
            return compression;
        }

        public final void setCompression(String compression) {
            this.compression = compression;
        }

        @Override
        public final Builder compression(String compression) {
            this.compression = compression;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(TargetFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final DirectSchemaChangePolicy.Builder getSchemaChangePolicy() {
            return schemaChangePolicy != null ? schemaChangePolicy.toBuilder() : null;
        }

        public final void setSchemaChangePolicy(DirectSchemaChangePolicy.BuilderImpl schemaChangePolicy) {
            this.schemaChangePolicy = schemaChangePolicy != null ? schemaChangePolicy.build() : null;
        }

        @Override
        public final Builder schemaChangePolicy(DirectSchemaChangePolicy schemaChangePolicy) {
            this.schemaChangePolicy = schemaChangePolicy;
            return this;
        }

        @Override
        public S3DirectTarget build() {
            return new S3DirectTarget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
