/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The catalog object represents a logical grouping of databases in the Glue Data Catalog or a federated source. You can
 * now create a Redshift-federated catalog or a catalog containing resource links to Redshift databases in another
 * account or region.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Catalog implements SdkPojo, Serializable, ToCopyableBuilder<Catalog.Builder, Catalog> {
    private static final SdkField<String> CATALOG_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CatalogId").getter(getter(Catalog::catalogId)).setter(setter(Builder::catalogId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Catalog::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(Catalog::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Catalog::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(Catalog::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime").getter(getter(Catalog::createTime)).setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime").build()).build();

    private static final SdkField<Instant> UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdateTime").getter(getter(Catalog::updateTime)).setter(setter(Builder::updateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdateTime").build()).build();

    private static final SdkField<TargetRedshiftCatalog> TARGET_REDSHIFT_CATALOG_FIELD = SdkField
            .<TargetRedshiftCatalog> builder(MarshallingType.SDK_POJO).memberName("TargetRedshiftCatalog")
            .getter(getter(Catalog::targetRedshiftCatalog)).setter(setter(Builder::targetRedshiftCatalog))
            .constructor(TargetRedshiftCatalog::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetRedshiftCatalog").build())
            .build();

    private static final SdkField<FederatedCatalog> FEDERATED_CATALOG_FIELD = SdkField
            .<FederatedCatalog> builder(MarshallingType.SDK_POJO).memberName("FederatedCatalog")
            .getter(getter(Catalog::federatedCatalog)).setter(setter(Builder::federatedCatalog))
            .constructor(FederatedCatalog::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FederatedCatalog").build()).build();

    private static final SdkField<CatalogPropertiesOutput> CATALOG_PROPERTIES_FIELD = SdkField
            .<CatalogPropertiesOutput> builder(MarshallingType.SDK_POJO).memberName("CatalogProperties")
            .getter(getter(Catalog::catalogProperties)).setter(setter(Builder::catalogProperties))
            .constructor(CatalogPropertiesOutput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CatalogProperties").build()).build();

    private static final SdkField<List<PrincipalPermissions>> CREATE_TABLE_DEFAULT_PERMISSIONS_FIELD = SdkField
            .<List<PrincipalPermissions>> builder(MarshallingType.LIST)
            .memberName("CreateTableDefaultPermissions")
            .getter(getter(Catalog::createTableDefaultPermissions))
            .setter(setter(Builder::createTableDefaultPermissions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTableDefaultPermissions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PrincipalPermissions> builder(MarshallingType.SDK_POJO)
                                            .constructor(PrincipalPermissions::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PrincipalPermissions>> CREATE_DATABASE_DEFAULT_PERMISSIONS_FIELD = SdkField
            .<List<PrincipalPermissions>> builder(MarshallingType.LIST)
            .memberName("CreateDatabaseDefaultPermissions")
            .getter(getter(Catalog::createDatabaseDefaultPermissions))
            .setter(setter(Builder::createDatabaseDefaultPermissions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDatabaseDefaultPermissions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PrincipalPermissions> builder(MarshallingType.SDK_POJO)
                                            .constructor(PrincipalPermissions::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ALLOW_FULL_TABLE_EXTERNAL_DATA_ACCESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllowFullTableExternalDataAccess")
            .getter(getter(Catalog::allowFullTableExternalDataAccessAsString))
            .setter(setter(Builder::allowFullTableExternalDataAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowFullTableExternalDataAccess")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CATALOG_ID_FIELD, NAME_FIELD,
            RESOURCE_ARN_FIELD, DESCRIPTION_FIELD, PARAMETERS_FIELD, CREATE_TIME_FIELD, UPDATE_TIME_FIELD,
            TARGET_REDSHIFT_CATALOG_FIELD, FEDERATED_CATALOG_FIELD, CATALOG_PROPERTIES_FIELD,
            CREATE_TABLE_DEFAULT_PERMISSIONS_FIELD, CREATE_DATABASE_DEFAULT_PERMISSIONS_FIELD,
            ALLOW_FULL_TABLE_EXTERNAL_DATA_ACCESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String catalogId;

    private final String name;

    private final String resourceArn;

    private final String description;

    private final Map<String, String> parameters;

    private final Instant createTime;

    private final Instant updateTime;

    private final TargetRedshiftCatalog targetRedshiftCatalog;

    private final FederatedCatalog federatedCatalog;

    private final CatalogPropertiesOutput catalogProperties;

    private final List<PrincipalPermissions> createTableDefaultPermissions;

    private final List<PrincipalPermissions> createDatabaseDefaultPermissions;

    private final String allowFullTableExternalDataAccess;

    private Catalog(BuilderImpl builder) {
        this.catalogId = builder.catalogId;
        this.name = builder.name;
        this.resourceArn = builder.resourceArn;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.createTime = builder.createTime;
        this.updateTime = builder.updateTime;
        this.targetRedshiftCatalog = builder.targetRedshiftCatalog;
        this.federatedCatalog = builder.federatedCatalog;
        this.catalogProperties = builder.catalogProperties;
        this.createTableDefaultPermissions = builder.createTableDefaultPermissions;
        this.createDatabaseDefaultPermissions = builder.createDatabaseDefaultPermissions;
        this.allowFullTableExternalDataAccess = builder.allowFullTableExternalDataAccess;
    }

    /**
     * <p>
     * The ID of the catalog. To grant access to the default catalog, this field should not be provided.
     * </p>
     * 
     * @return The ID of the catalog. To grant access to the default catalog, this field should not be provided.
     */
    public final String catalogId() {
        return catalogId;
    }

    /**
     * <p>
     * The name of the catalog. Cannot be the same as the account ID.
     * </p>
     * 
     * @return The name of the catalog. Cannot be the same as the account ID.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) assigned to the catalog resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) assigned to the catalog resource.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * Description string, not more than 2048 bytes long, matching the URI address multi-line string pattern. A
     * description of the catalog.
     * </p>
     * 
     * @return Description string, not more than 2048 bytes long, matching the URI address multi-line string pattern. A
     *         description of the catalog.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map array of key-value pairs that define parameters and properties of the catalog.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return A map array of key-value pairs that define parameters and properties of the catalog.
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The time at which the catalog was created.
     * </p>
     * 
     * @return The time at which the catalog was created.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The time at which the catalog was last updated.
     * </p>
     * 
     * @return The time at which the catalog was last updated.
     */
    public final Instant updateTime() {
        return updateTime;
    }

    /**
     * <p>
     * A <code>TargetRedshiftCatalog</code> object that describes a target catalog for database resource linking.
     * </p>
     * 
     * @return A <code>TargetRedshiftCatalog</code> object that describes a target catalog for database resource
     *         linking.
     */
    public final TargetRedshiftCatalog targetRedshiftCatalog() {
        return targetRedshiftCatalog;
    }

    /**
     * <p>
     * A <code>FederatedCatalog</code> object that points to an entity outside the Glue Data Catalog.
     * </p>
     * 
     * @return A <code>FederatedCatalog</code> object that points to an entity outside the Glue Data Catalog.
     */
    public final FederatedCatalog federatedCatalog() {
        return federatedCatalog;
    }

    /**
     * <p>
     * A <code>CatalogProperties</code> object that specifies data lake access properties and other custom properties.
     * </p>
     * 
     * @return A <code>CatalogProperties</code> object that specifies data lake access properties and other custom
     *         properties.
     */
    public final CatalogPropertiesOutput catalogProperties() {
        return catalogProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the CreateTableDefaultPermissions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCreateTableDefaultPermissions() {
        return createTableDefaultPermissions != null && !(createTableDefaultPermissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the table(s) for
     * principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue operations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCreateTableDefaultPermissions} method.
     * </p>
     * 
     * @return An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
     *         table(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of
     *         Glue operations.
     */
    public final List<PrincipalPermissions> createTableDefaultPermissions() {
        return createTableDefaultPermissions;
    }

    /**
     * For responses, this returns true if the service returned a value for the CreateDatabaseDefaultPermissions
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasCreateDatabaseDefaultPermissions() {
        return createDatabaseDefaultPermissions != null && !(createDatabaseDefaultPermissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the database(s)
     * for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue operations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCreateDatabaseDefaultPermissions}
     * method.
     * </p>
     * 
     * @return An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
     *         database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of
     *         Glue operations.
     */
    public final List<PrincipalPermissions> createDatabaseDefaultPermissions() {
        return createDatabaseDefaultPermissions;
    }

    /**
     * <p>
     * Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowFullTableExternalDataAccess} will return
     * {@link AllowFullTableExternalDataAccessEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #allowFullTableExternalDataAccessAsString}.
     * </p>
     * 
     * @return Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
     * @see AllowFullTableExternalDataAccessEnum
     */
    public final AllowFullTableExternalDataAccessEnum allowFullTableExternalDataAccess() {
        return AllowFullTableExternalDataAccessEnum.fromValue(allowFullTableExternalDataAccess);
    }

    /**
     * <p>
     * Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowFullTableExternalDataAccess} will return
     * {@link AllowFullTableExternalDataAccessEnum#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #allowFullTableExternalDataAccessAsString}.
     * </p>
     * 
     * @return Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
     * @see AllowFullTableExternalDataAccessEnum
     */
    public final String allowFullTableExternalDataAccessAsString() {
        return allowFullTableExternalDataAccess;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(catalogId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(updateTime());
        hashCode = 31 * hashCode + Objects.hashCode(targetRedshiftCatalog());
        hashCode = 31 * hashCode + Objects.hashCode(federatedCatalog());
        hashCode = 31 * hashCode + Objects.hashCode(catalogProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hasCreateTableDefaultPermissions() ? createTableDefaultPermissions() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasCreateDatabaseDefaultPermissions() ? createDatabaseDefaultPermissions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(allowFullTableExternalDataAccessAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Catalog)) {
            return false;
        }
        Catalog other = (Catalog) obj;
        return Objects.equals(catalogId(), other.catalogId()) && Objects.equals(name(), other.name())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(description(), other.description())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(createTime(), other.createTime()) && Objects.equals(updateTime(), other.updateTime())
                && Objects.equals(targetRedshiftCatalog(), other.targetRedshiftCatalog())
                && Objects.equals(federatedCatalog(), other.federatedCatalog())
                && Objects.equals(catalogProperties(), other.catalogProperties())
                && hasCreateTableDefaultPermissions() == other.hasCreateTableDefaultPermissions()
                && Objects.equals(createTableDefaultPermissions(), other.createTableDefaultPermissions())
                && hasCreateDatabaseDefaultPermissions() == other.hasCreateDatabaseDefaultPermissions()
                && Objects.equals(createDatabaseDefaultPermissions(), other.createDatabaseDefaultPermissions())
                && Objects.equals(allowFullTableExternalDataAccessAsString(), other.allowFullTableExternalDataAccessAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Catalog")
                .add("CatalogId", catalogId())
                .add("Name", name())
                .add("ResourceArn", resourceArn())
                .add("Description", description())
                .add("Parameters", hasParameters() ? parameters() : null)
                .add("CreateTime", createTime())
                .add("UpdateTime", updateTime())
                .add("TargetRedshiftCatalog", targetRedshiftCatalog())
                .add("FederatedCatalog", federatedCatalog())
                .add("CatalogProperties", catalogProperties())
                .add("CreateTableDefaultPermissions", hasCreateTableDefaultPermissions() ? createTableDefaultPermissions() : null)
                .add("CreateDatabaseDefaultPermissions",
                        hasCreateDatabaseDefaultPermissions() ? createDatabaseDefaultPermissions() : null)
                .add("AllowFullTableExternalDataAccess", allowFullTableExternalDataAccessAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CatalogId":
            return Optional.ofNullable(clazz.cast(catalogId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "UpdateTime":
            return Optional.ofNullable(clazz.cast(updateTime()));
        case "TargetRedshiftCatalog":
            return Optional.ofNullable(clazz.cast(targetRedshiftCatalog()));
        case "FederatedCatalog":
            return Optional.ofNullable(clazz.cast(federatedCatalog()));
        case "CatalogProperties":
            return Optional.ofNullable(clazz.cast(catalogProperties()));
        case "CreateTableDefaultPermissions":
            return Optional.ofNullable(clazz.cast(createTableDefaultPermissions()));
        case "CreateDatabaseDefaultPermissions":
            return Optional.ofNullable(clazz.cast(createDatabaseDefaultPermissions()));
        case "AllowFullTableExternalDataAccess":
            return Optional.ofNullable(clazz.cast(allowFullTableExternalDataAccessAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CatalogId", CATALOG_ID_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("ResourceArn", RESOURCE_ARN_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        map.put("CreateTime", CREATE_TIME_FIELD);
        map.put("UpdateTime", UPDATE_TIME_FIELD);
        map.put("TargetRedshiftCatalog", TARGET_REDSHIFT_CATALOG_FIELD);
        map.put("FederatedCatalog", FEDERATED_CATALOG_FIELD);
        map.put("CatalogProperties", CATALOG_PROPERTIES_FIELD);
        map.put("CreateTableDefaultPermissions", CREATE_TABLE_DEFAULT_PERMISSIONS_FIELD);
        map.put("CreateDatabaseDefaultPermissions", CREATE_DATABASE_DEFAULT_PERMISSIONS_FIELD);
        map.put("AllowFullTableExternalDataAccess", ALLOW_FULL_TABLE_EXTERNAL_DATA_ACCESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Catalog, T> g) {
        return obj -> g.apply((Catalog) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Catalog> {
        /**
         * <p>
         * The ID of the catalog. To grant access to the default catalog, this field should not be provided.
         * </p>
         * 
         * @param catalogId
         *        The ID of the catalog. To grant access to the default catalog, this field should not be provided.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogId(String catalogId);

        /**
         * <p>
         * The name of the catalog. Cannot be the same as the account ID.
         * </p>
         * 
         * @param name
         *        The name of the catalog. Cannot be the same as the account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) assigned to the catalog resource.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Resource Name (ARN) assigned to the catalog resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * Description string, not more than 2048 bytes long, matching the URI address multi-line string pattern. A
         * description of the catalog.
         * </p>
         * 
         * @param description
         *        Description string, not more than 2048 bytes long, matching the URI address multi-line string pattern.
         *        A description of the catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A map array of key-value pairs that define parameters and properties of the catalog.
         * </p>
         * 
         * @param parameters
         *        A map array of key-value pairs that define parameters and properties of the catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * The time at which the catalog was created.
         * </p>
         * 
         * @param createTime
         *        The time at which the catalog was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The time at which the catalog was last updated.
         * </p>
         * 
         * @param updateTime
         *        The time at which the catalog was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateTime(Instant updateTime);

        /**
         * <p>
         * A <code>TargetRedshiftCatalog</code> object that describes a target catalog for database resource linking.
         * </p>
         * 
         * @param targetRedshiftCatalog
         *        A <code>TargetRedshiftCatalog</code> object that describes a target catalog for database resource
         *        linking.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetRedshiftCatalog(TargetRedshiftCatalog targetRedshiftCatalog);

        /**
         * <p>
         * A <code>TargetRedshiftCatalog</code> object that describes a target catalog for database resource linking.
         * </p>
         * This is a convenience method that creates an instance of the {@link TargetRedshiftCatalog.Builder} avoiding
         * the need to create one manually via {@link TargetRedshiftCatalog#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TargetRedshiftCatalog.Builder#build()} is called immediately and
         * its result is passed to {@link #targetRedshiftCatalog(TargetRedshiftCatalog)}.
         * 
         * @param targetRedshiftCatalog
         *        a consumer that will call methods on {@link TargetRedshiftCatalog.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetRedshiftCatalog(TargetRedshiftCatalog)
         */
        default Builder targetRedshiftCatalog(Consumer<TargetRedshiftCatalog.Builder> targetRedshiftCatalog) {
            return targetRedshiftCatalog(TargetRedshiftCatalog.builder().applyMutation(targetRedshiftCatalog).build());
        }

        /**
         * <p>
         * A <code>FederatedCatalog</code> object that points to an entity outside the Glue Data Catalog.
         * </p>
         * 
         * @param federatedCatalog
         *        A <code>FederatedCatalog</code> object that points to an entity outside the Glue Data Catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder federatedCatalog(FederatedCatalog federatedCatalog);

        /**
         * <p>
         * A <code>FederatedCatalog</code> object that points to an entity outside the Glue Data Catalog.
         * </p>
         * This is a convenience method that creates an instance of the {@link FederatedCatalog.Builder} avoiding the
         * need to create one manually via {@link FederatedCatalog#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FederatedCatalog.Builder#build()} is called immediately and its
         * result is passed to {@link #federatedCatalog(FederatedCatalog)}.
         * 
         * @param federatedCatalog
         *        a consumer that will call methods on {@link FederatedCatalog.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #federatedCatalog(FederatedCatalog)
         */
        default Builder federatedCatalog(Consumer<FederatedCatalog.Builder> federatedCatalog) {
            return federatedCatalog(FederatedCatalog.builder().applyMutation(federatedCatalog).build());
        }

        /**
         * <p>
         * A <code>CatalogProperties</code> object that specifies data lake access properties and other custom
         * properties.
         * </p>
         * 
         * @param catalogProperties
         *        A <code>CatalogProperties</code> object that specifies data lake access properties and other custom
         *        properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder catalogProperties(CatalogPropertiesOutput catalogProperties);

        /**
         * <p>
         * A <code>CatalogProperties</code> object that specifies data lake access properties and other custom
         * properties.
         * </p>
         * This is a convenience method that creates an instance of the {@link CatalogPropertiesOutput.Builder} avoiding
         * the need to create one manually via {@link CatalogPropertiesOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CatalogPropertiesOutput.Builder#build()} is called immediately
         * and its result is passed to {@link #catalogProperties(CatalogPropertiesOutput)}.
         * 
         * @param catalogProperties
         *        a consumer that will call methods on {@link CatalogPropertiesOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #catalogProperties(CatalogPropertiesOutput)
         */
        default Builder catalogProperties(Consumer<CatalogPropertiesOutput.Builder> catalogProperties) {
            return catalogProperties(CatalogPropertiesOutput.builder().applyMutation(catalogProperties).build());
        }

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the table(s)
         * for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue operations.
         * </p>
         * 
         * @param createTableDefaultPermissions
         *        An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         *        table(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of
         *        Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTableDefaultPermissions(Collection<PrincipalPermissions> createTableDefaultPermissions);

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the table(s)
         * for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue operations.
         * </p>
         * 
         * @param createTableDefaultPermissions
         *        An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         *        table(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of
         *        Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTableDefaultPermissions(PrincipalPermissions... createTableDefaultPermissions);

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the table(s)
         * for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue operations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder#build()} is called immediately
         * and its result is passed to {@link #createTableDefaultPermissions(List<PrincipalPermissions>)}.
         * 
         * @param createTableDefaultPermissions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createTableDefaultPermissions(java.util.Collection<PrincipalPermissions>)
         */
        Builder createTableDefaultPermissions(Consumer<PrincipalPermissions.Builder>... createTableDefaultPermissions);

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         * database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue
         * operations.
         * </p>
         * 
         * @param createDatabaseDefaultPermissions
         *        An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         *        database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course
         *        of Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDatabaseDefaultPermissions(Collection<PrincipalPermissions> createDatabaseDefaultPermissions);

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         * database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue
         * operations.
         * </p>
         * 
         * @param createDatabaseDefaultPermissions
         *        An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         *        database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course
         *        of Glue operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDatabaseDefaultPermissions(PrincipalPermissions... createDatabaseDefaultPermissions);

        /**
         * <p>
         * An array of <code>PrincipalPermissions</code> objects. Creates a set of default permissions on the
         * database(s) for principals. Used by Amazon Web Services Lake Formation. Not used in the normal course of Glue
         * operations.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder#build()} is called immediately
         * and its result is passed to {@link #createDatabaseDefaultPermissions(List<PrincipalPermissions>)}.
         * 
         * @param createDatabaseDefaultPermissions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.PrincipalPermissions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createDatabaseDefaultPermissions(java.util.Collection<PrincipalPermissions>)
         */
        Builder createDatabaseDefaultPermissions(Consumer<PrincipalPermissions.Builder>... createDatabaseDefaultPermissions);

        /**
         * <p>
         * Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
         * </p>
         * 
         * @param allowFullTableExternalDataAccess
         *        Allows third-party engines to access data in Amazon S3 locations that are registered with Lake
         *        Formation.
         * @see AllowFullTableExternalDataAccessEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowFullTableExternalDataAccessEnum
         */
        Builder allowFullTableExternalDataAccess(String allowFullTableExternalDataAccess);

        /**
         * <p>
         * Allows third-party engines to access data in Amazon S3 locations that are registered with Lake Formation.
         * </p>
         * 
         * @param allowFullTableExternalDataAccess
         *        Allows third-party engines to access data in Amazon S3 locations that are registered with Lake
         *        Formation.
         * @see AllowFullTableExternalDataAccessEnum
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowFullTableExternalDataAccessEnum
         */
        Builder allowFullTableExternalDataAccess(AllowFullTableExternalDataAccessEnum allowFullTableExternalDataAccess);
    }

    static final class BuilderImpl implements Builder {
        private String catalogId;

        private String name;

        private String resourceArn;

        private String description;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private Instant createTime;

        private Instant updateTime;

        private TargetRedshiftCatalog targetRedshiftCatalog;

        private FederatedCatalog federatedCatalog;

        private CatalogPropertiesOutput catalogProperties;

        private List<PrincipalPermissions> createTableDefaultPermissions = DefaultSdkAutoConstructList.getInstance();

        private List<PrincipalPermissions> createDatabaseDefaultPermissions = DefaultSdkAutoConstructList.getInstance();

        private String allowFullTableExternalDataAccess;

        private BuilderImpl() {
        }

        private BuilderImpl(Catalog model) {
            catalogId(model.catalogId);
            name(model.name);
            resourceArn(model.resourceArn);
            description(model.description);
            parameters(model.parameters);
            createTime(model.createTime);
            updateTime(model.updateTime);
            targetRedshiftCatalog(model.targetRedshiftCatalog);
            federatedCatalog(model.federatedCatalog);
            catalogProperties(model.catalogProperties);
            createTableDefaultPermissions(model.createTableDefaultPermissions);
            createDatabaseDefaultPermissions(model.createDatabaseDefaultPermissions);
            allowFullTableExternalDataAccess(model.allowFullTableExternalDataAccess);
        }

        public final String getCatalogId() {
            return catalogId;
        }

        public final void setCatalogId(String catalogId) {
            this.catalogId = catalogId;
        }

        @Override
        public final Builder catalogId(String catalogId) {
            this.catalogId = catalogId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final Instant getUpdateTime() {
            return updateTime;
        }

        public final void setUpdateTime(Instant updateTime) {
            this.updateTime = updateTime;
        }

        @Override
        public final Builder updateTime(Instant updateTime) {
            this.updateTime = updateTime;
            return this;
        }

        public final TargetRedshiftCatalog.Builder getTargetRedshiftCatalog() {
            return targetRedshiftCatalog != null ? targetRedshiftCatalog.toBuilder() : null;
        }

        public final void setTargetRedshiftCatalog(TargetRedshiftCatalog.BuilderImpl targetRedshiftCatalog) {
            this.targetRedshiftCatalog = targetRedshiftCatalog != null ? targetRedshiftCatalog.build() : null;
        }

        @Override
        public final Builder targetRedshiftCatalog(TargetRedshiftCatalog targetRedshiftCatalog) {
            this.targetRedshiftCatalog = targetRedshiftCatalog;
            return this;
        }

        public final FederatedCatalog.Builder getFederatedCatalog() {
            return federatedCatalog != null ? federatedCatalog.toBuilder() : null;
        }

        public final void setFederatedCatalog(FederatedCatalog.BuilderImpl federatedCatalog) {
            this.federatedCatalog = federatedCatalog != null ? federatedCatalog.build() : null;
        }

        @Override
        public final Builder federatedCatalog(FederatedCatalog federatedCatalog) {
            this.federatedCatalog = federatedCatalog;
            return this;
        }

        public final CatalogPropertiesOutput.Builder getCatalogProperties() {
            return catalogProperties != null ? catalogProperties.toBuilder() : null;
        }

        public final void setCatalogProperties(CatalogPropertiesOutput.BuilderImpl catalogProperties) {
            this.catalogProperties = catalogProperties != null ? catalogProperties.build() : null;
        }

        @Override
        public final Builder catalogProperties(CatalogPropertiesOutput catalogProperties) {
            this.catalogProperties = catalogProperties;
            return this;
        }

        public final List<PrincipalPermissions.Builder> getCreateTableDefaultPermissions() {
            List<PrincipalPermissions.Builder> result = PrincipalPermissionsListCopier
                    .copyToBuilder(this.createTableDefaultPermissions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCreateTableDefaultPermissions(
                Collection<PrincipalPermissions.BuilderImpl> createTableDefaultPermissions) {
            this.createTableDefaultPermissions = PrincipalPermissionsListCopier.copyFromBuilder(createTableDefaultPermissions);
        }

        @Override
        public final Builder createTableDefaultPermissions(Collection<PrincipalPermissions> createTableDefaultPermissions) {
            this.createTableDefaultPermissions = PrincipalPermissionsListCopier.copy(createTableDefaultPermissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder createTableDefaultPermissions(PrincipalPermissions... createTableDefaultPermissions) {
            createTableDefaultPermissions(Arrays.asList(createTableDefaultPermissions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder createTableDefaultPermissions(
                Consumer<PrincipalPermissions.Builder>... createTableDefaultPermissions) {
            createTableDefaultPermissions(Stream.of(createTableDefaultPermissions)
                    .map(c -> PrincipalPermissions.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<PrincipalPermissions.Builder> getCreateDatabaseDefaultPermissions() {
            List<PrincipalPermissions.Builder> result = PrincipalPermissionsListCopier
                    .copyToBuilder(this.createDatabaseDefaultPermissions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCreateDatabaseDefaultPermissions(
                Collection<PrincipalPermissions.BuilderImpl> createDatabaseDefaultPermissions) {
            this.createDatabaseDefaultPermissions = PrincipalPermissionsListCopier
                    .copyFromBuilder(createDatabaseDefaultPermissions);
        }

        @Override
        public final Builder createDatabaseDefaultPermissions(Collection<PrincipalPermissions> createDatabaseDefaultPermissions) {
            this.createDatabaseDefaultPermissions = PrincipalPermissionsListCopier.copy(createDatabaseDefaultPermissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder createDatabaseDefaultPermissions(PrincipalPermissions... createDatabaseDefaultPermissions) {
            createDatabaseDefaultPermissions(Arrays.asList(createDatabaseDefaultPermissions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder createDatabaseDefaultPermissions(
                Consumer<PrincipalPermissions.Builder>... createDatabaseDefaultPermissions) {
            createDatabaseDefaultPermissions(Stream.of(createDatabaseDefaultPermissions)
                    .map(c -> PrincipalPermissions.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getAllowFullTableExternalDataAccess() {
            return allowFullTableExternalDataAccess;
        }

        public final void setAllowFullTableExternalDataAccess(String allowFullTableExternalDataAccess) {
            this.allowFullTableExternalDataAccess = allowFullTableExternalDataAccess;
        }

        @Override
        public final Builder allowFullTableExternalDataAccess(String allowFullTableExternalDataAccess) {
            this.allowFullTableExternalDataAccess = allowFullTableExternalDataAccess;
            return this;
        }

        @Override
        public final Builder allowFullTableExternalDataAccess(
                AllowFullTableExternalDataAccessEnum allowFullTableExternalDataAccess) {
            this.allowFullTableExternalDataAccess(allowFullTableExternalDataAccess == null ? null
                    : allowFullTableExternalDataAccess.toString());
            return this;
        }

        @Override
        public Catalog build() {
            return new Catalog(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
