/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies an Apache Parquet data store stored in Amazon S3.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3ParquetSource implements SdkPojo, Serializable, ToCopyableBuilder<S3ParquetSource.Builder, S3ParquetSource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(S3ParquetSource::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Paths")
            .getter(getter(S3ParquetSource::paths))
            .setter(setter(Builder::paths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Paths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompressionType").getter(getter(S3ParquetSource::compressionTypeAsString))
            .setter(setter(Builder::compressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompressionType").build()).build();

    private static final SdkField<List<String>> EXCLUSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Exclusions")
            .getter(getter(S3ParquetSource::exclusions))
            .setter(setter(Builder::exclusions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Exclusions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> GROUP_SIZE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupSize").getter(getter(S3ParquetSource::groupSize)).setter(setter(Builder::groupSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupSize").build()).build();

    private static final SdkField<String> GROUP_FILES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupFiles").getter(getter(S3ParquetSource::groupFiles)).setter(setter(Builder::groupFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupFiles").build()).build();

    private static final SdkField<Boolean> RECURSE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Recurse").getter(getter(S3ParquetSource::recurse)).setter(setter(Builder::recurse))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recurse").build()).build();

    private static final SdkField<Integer> MAX_BAND_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxBand").getter(getter(S3ParquetSource::maxBand)).setter(setter(Builder::maxBand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxBand").build()).build();

    private static final SdkField<Integer> MAX_FILES_IN_BAND_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxFilesInBand").getter(getter(S3ParquetSource::maxFilesInBand)).setter(setter(Builder::maxFilesInBand))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxFilesInBand").build()).build();

    private static final SdkField<S3DirectSourceAdditionalOptions> ADDITIONAL_OPTIONS_FIELD = SdkField
            .<S3DirectSourceAdditionalOptions> builder(MarshallingType.SDK_POJO).memberName("AdditionalOptions")
            .getter(getter(S3ParquetSource::additionalOptions)).setter(setter(Builder::additionalOptions))
            .constructor(S3DirectSourceAdditionalOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalOptions").build()).build();

    private static final SdkField<List<GlueSchema>> OUTPUT_SCHEMAS_FIELD = SdkField
            .<List<GlueSchema>> builder(MarshallingType.LIST)
            .memberName("OutputSchemas")
            .getter(getter(S3ParquetSource::outputSchemas))
            .setter(setter(Builder::outputSchemas))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputSchemas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GlueSchema> builder(MarshallingType.SDK_POJO)
                                            .constructor(GlueSchema::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, PATHS_FIELD,
            COMPRESSION_TYPE_FIELD, EXCLUSIONS_FIELD, GROUP_SIZE_FIELD, GROUP_FILES_FIELD, RECURSE_FIELD, MAX_BAND_FIELD,
            MAX_FILES_IN_BAND_FIELD, ADDITIONAL_OPTIONS_FIELD, OUTPUT_SCHEMAS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> paths;

    private final String compressionType;

    private final List<String> exclusions;

    private final String groupSize;

    private final String groupFiles;

    private final Boolean recurse;

    private final Integer maxBand;

    private final Integer maxFilesInBand;

    private final S3DirectSourceAdditionalOptions additionalOptions;

    private final List<GlueSchema> outputSchemas;

    private S3ParquetSource(BuilderImpl builder) {
        this.name = builder.name;
        this.paths = builder.paths;
        this.compressionType = builder.compressionType;
        this.exclusions = builder.exclusions;
        this.groupSize = builder.groupSize;
        this.groupFiles = builder.groupFiles;
        this.recurse = builder.recurse;
        this.maxBand = builder.maxBand;
        this.maxFilesInBand = builder.maxFilesInBand;
        this.additionalOptions = builder.additionalOptions;
        this.outputSchemas = builder.outputSchemas;
    }

    /**
     * <p>
     * The name of the data store.
     * </p>
     * 
     * @return The name of the data store.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Paths property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasPaths() {
        return paths != null && !(paths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the Amazon S3 paths to read from.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPaths} method.
     * </p>
     * 
     * @return A list of the Amazon S3 paths to read from.
     */
    public final List<String> paths() {
        return paths;
    }

    /**
     * <p>
     * Specifies how the data is compressed. This is generally not necessary if the data has a standard file extension.
     * Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link ParquetCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return Specifies how the data is compressed. This is generally not necessary if the data has a standard file
     *         extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     * @see ParquetCompressionType
     */
    public final ParquetCompressionType compressionType() {
        return ParquetCompressionType.fromValue(compressionType);
    }

    /**
     * <p>
     * Specifies how the data is compressed. This is generally not necessary if the data has a standard file extension.
     * Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionType}
     * will return {@link ParquetCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #compressionTypeAsString}.
     * </p>
     * 
     * @return Specifies how the data is compressed. This is generally not necessary if the data has a standard file
     *         extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
     * @see ParquetCompressionType
     */
    public final String compressionTypeAsString() {
        return compressionType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Exclusions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExclusions() {
        return exclusions != null && !(exclusions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]" excludes all
     * PDF files.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusions} method.
     * </p>
     * 
     * @return A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]"
     *         excludes all PDF files.
     */
    public final List<String> exclusions() {
        return exclusions;
    }

    /**
     * <p>
     * The target group size in bytes. The default is computed based on the input data size and the size of your
     * cluster. When there are fewer than 50,000 input files, <code>"groupFiles"</code> must be set to
     * <code>"inPartition"</code> for this to take effect.
     * </p>
     * 
     * @return The target group size in bytes. The default is computed based on the input data size and the size of your
     *         cluster. When there are fewer than 50,000 input files, <code>"groupFiles"</code> must be set to
     *         <code>"inPartition"</code> for this to take effect.
     */
    public final String groupSize() {
        return groupSize;
    }

    /**
     * <p>
     * Grouping files is turned on by default when the input contains more than 50,000 files. To turn on grouping with
     * fewer than 50,000 files, set this parameter to "inPartition". To disable grouping when there are more than 50,000
     * files, set this parameter to <code>"none"</code>.
     * </p>
     * 
     * @return Grouping files is turned on by default when the input contains more than 50,000 files. To turn on
     *         grouping with fewer than 50,000 files, set this parameter to "inPartition". To disable grouping when
     *         there are more than 50,000 files, set this parameter to <code>"none"</code>.
     */
    public final String groupFiles() {
        return groupFiles;
    }

    /**
     * <p>
     * If set to true, recursively reads files in all subdirectories under the specified paths.
     * </p>
     * 
     * @return If set to true, recursively reads files in all subdirectories under the specified paths.
     */
    public final Boolean recurse() {
        return recurse;
    }

    /**
     * <p>
     * This option controls the duration in milliseconds after which the s3 listing is likely to be consistent. Files
     * with modification timestamps falling within the last maxBand milliseconds are tracked specially when using
     * JobBookmarks to account for Amazon S3 eventual consistency. Most users don't need to set this option. The default
     * is 900000 milliseconds, or 15 minutes.
     * </p>
     * 
     * @return This option controls the duration in milliseconds after which the s3 listing is likely to be consistent.
     *         Files with modification timestamps falling within the last maxBand milliseconds are tracked specially
     *         when using JobBookmarks to account for Amazon S3 eventual consistency. Most users don't need to set this
     *         option. The default is 900000 milliseconds, or 15 minutes.
     */
    public final Integer maxBand() {
        return maxBand;
    }

    /**
     * <p>
     * This option specifies the maximum number of files to save from the last maxBand seconds. If this number is
     * exceeded, extra files are skipped and only processed in the next job run.
     * </p>
     * 
     * @return This option specifies the maximum number of files to save from the last maxBand seconds. If this number
     *         is exceeded, extra files are skipped and only processed in the next job run.
     */
    public final Integer maxFilesInBand() {
        return maxFilesInBand;
    }

    /**
     * <p>
     * Specifies additional connection options.
     * </p>
     * 
     * @return Specifies additional connection options.
     */
    public final S3DirectSourceAdditionalOptions additionalOptions() {
        return additionalOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputSchemas property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputSchemas() {
        return outputSchemas != null && !(outputSchemas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the data schema for the S3 Parquet source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputSchemas} method.
     * </p>
     * 
     * @return Specifies the data schema for the S3 Parquet source.
     */
    public final List<GlueSchema> outputSchemas() {
        return outputSchemas;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasPaths() ? paths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(compressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusions() ? exclusions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(groupSize());
        hashCode = 31 * hashCode + Objects.hashCode(groupFiles());
        hashCode = 31 * hashCode + Objects.hashCode(recurse());
        hashCode = 31 * hashCode + Objects.hashCode(maxBand());
        hashCode = 31 * hashCode + Objects.hashCode(maxFilesInBand());
        hashCode = 31 * hashCode + Objects.hashCode(additionalOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputSchemas() ? outputSchemas() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3ParquetSource)) {
            return false;
        }
        S3ParquetSource other = (S3ParquetSource) obj;
        return Objects.equals(name(), other.name()) && hasPaths() == other.hasPaths() && Objects.equals(paths(), other.paths())
                && Objects.equals(compressionTypeAsString(), other.compressionTypeAsString())
                && hasExclusions() == other.hasExclusions() && Objects.equals(exclusions(), other.exclusions())
                && Objects.equals(groupSize(), other.groupSize()) && Objects.equals(groupFiles(), other.groupFiles())
                && Objects.equals(recurse(), other.recurse()) && Objects.equals(maxBand(), other.maxBand())
                && Objects.equals(maxFilesInBand(), other.maxFilesInBand())
                && Objects.equals(additionalOptions(), other.additionalOptions())
                && hasOutputSchemas() == other.hasOutputSchemas() && Objects.equals(outputSchemas(), other.outputSchemas());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3ParquetSource").add("Name", name()).add("Paths", hasPaths() ? paths() : null)
                .add("CompressionType", compressionTypeAsString()).add("Exclusions", hasExclusions() ? exclusions() : null)
                .add("GroupSize", groupSize()).add("GroupFiles", groupFiles()).add("Recurse", recurse())
                .add("MaxBand", maxBand()).add("MaxFilesInBand", maxFilesInBand()).add("AdditionalOptions", additionalOptions())
                .add("OutputSchemas", hasOutputSchemas() ? outputSchemas() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Paths":
            return Optional.ofNullable(clazz.cast(paths()));
        case "CompressionType":
            return Optional.ofNullable(clazz.cast(compressionTypeAsString()));
        case "Exclusions":
            return Optional.ofNullable(clazz.cast(exclusions()));
        case "GroupSize":
            return Optional.ofNullable(clazz.cast(groupSize()));
        case "GroupFiles":
            return Optional.ofNullable(clazz.cast(groupFiles()));
        case "Recurse":
            return Optional.ofNullable(clazz.cast(recurse()));
        case "MaxBand":
            return Optional.ofNullable(clazz.cast(maxBand()));
        case "MaxFilesInBand":
            return Optional.ofNullable(clazz.cast(maxFilesInBand()));
        case "AdditionalOptions":
            return Optional.ofNullable(clazz.cast(additionalOptions()));
        case "OutputSchemas":
            return Optional.ofNullable(clazz.cast(outputSchemas()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Paths", PATHS_FIELD);
        map.put("CompressionType", COMPRESSION_TYPE_FIELD);
        map.put("Exclusions", EXCLUSIONS_FIELD);
        map.put("GroupSize", GROUP_SIZE_FIELD);
        map.put("GroupFiles", GROUP_FILES_FIELD);
        map.put("Recurse", RECURSE_FIELD);
        map.put("MaxBand", MAX_BAND_FIELD);
        map.put("MaxFilesInBand", MAX_FILES_IN_BAND_FIELD);
        map.put("AdditionalOptions", ADDITIONAL_OPTIONS_FIELD);
        map.put("OutputSchemas", OUTPUT_SCHEMAS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3ParquetSource, T> g) {
        return obj -> g.apply((S3ParquetSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3ParquetSource> {
        /**
         * <p>
         * The name of the data store.
         * </p>
         * 
         * @param name
         *        The name of the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A list of the Amazon S3 paths to read from.
         * </p>
         * 
         * @param paths
         *        A list of the Amazon S3 paths to read from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(Collection<String> paths);

        /**
         * <p>
         * A list of the Amazon S3 paths to read from.
         * </p>
         * 
         * @param paths
         *        A list of the Amazon S3 paths to read from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(String... paths);

        /**
         * <p>
         * Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         * extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * </p>
         * 
         * @param compressionType
         *        Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         *        extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * @see ParquetCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetCompressionType
         */
        Builder compressionType(String compressionType);

        /**
         * <p>
         * Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         * extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * </p>
         * 
         * @param compressionType
         *        Specifies how the data is compressed. This is generally not necessary if the data has a standard file
         *        extension. Possible values are <code>"gzip"</code> and <code>"bzip"</code>).
         * @see ParquetCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetCompressionType
         */
        Builder compressionType(ParquetCompressionType compressionType);

        /**
         * <p>
         * A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]" excludes
         * all PDF files.
         * </p>
         * 
         * @param exclusions
         *        A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]"
         *        excludes all PDF files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusions(Collection<String> exclusions);

        /**
         * <p>
         * A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]" excludes
         * all PDF files.
         * </p>
         * 
         * @param exclusions
         *        A string containing a JSON list of Unix-style glob patterns to exclude. For example, "[\"**.pdf\"]"
         *        excludes all PDF files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusions(String... exclusions);

        /**
         * <p>
         * The target group size in bytes. The default is computed based on the input data size and the size of your
         * cluster. When there are fewer than 50,000 input files, <code>"groupFiles"</code> must be set to
         * <code>"inPartition"</code> for this to take effect.
         * </p>
         * 
         * @param groupSize
         *        The target group size in bytes. The default is computed based on the input data size and the size of
         *        your cluster. When there are fewer than 50,000 input files, <code>"groupFiles"</code> must be set to
         *        <code>"inPartition"</code> for this to take effect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupSize(String groupSize);

        /**
         * <p>
         * Grouping files is turned on by default when the input contains more than 50,000 files. To turn on grouping
         * with fewer than 50,000 files, set this parameter to "inPartition". To disable grouping when there are more
         * than 50,000 files, set this parameter to <code>"none"</code>.
         * </p>
         * 
         * @param groupFiles
         *        Grouping files is turned on by default when the input contains more than 50,000 files. To turn on
         *        grouping with fewer than 50,000 files, set this parameter to "inPartition". To disable grouping when
         *        there are more than 50,000 files, set this parameter to <code>"none"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupFiles(String groupFiles);

        /**
         * <p>
         * If set to true, recursively reads files in all subdirectories under the specified paths.
         * </p>
         * 
         * @param recurse
         *        If set to true, recursively reads files in all subdirectories under the specified paths.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurse(Boolean recurse);

        /**
         * <p>
         * This option controls the duration in milliseconds after which the s3 listing is likely to be consistent.
         * Files with modification timestamps falling within the last maxBand milliseconds are tracked specially when
         * using JobBookmarks to account for Amazon S3 eventual consistency. Most users don't need to set this option.
         * The default is 900000 milliseconds, or 15 minutes.
         * </p>
         * 
         * @param maxBand
         *        This option controls the duration in milliseconds after which the s3 listing is likely to be
         *        consistent. Files with modification timestamps falling within the last maxBand milliseconds are
         *        tracked specially when using JobBookmarks to account for Amazon S3 eventual consistency. Most users
         *        don't need to set this option. The default is 900000 milliseconds, or 15 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxBand(Integer maxBand);

        /**
         * <p>
         * This option specifies the maximum number of files to save from the last maxBand seconds. If this number is
         * exceeded, extra files are skipped and only processed in the next job run.
         * </p>
         * 
         * @param maxFilesInBand
         *        This option specifies the maximum number of files to save from the last maxBand seconds. If this
         *        number is exceeded, extra files are skipped and only processed in the next job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxFilesInBand(Integer maxFilesInBand);

        /**
         * <p>
         * Specifies additional connection options.
         * </p>
         * 
         * @param additionalOptions
         *        Specifies additional connection options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalOptions(S3DirectSourceAdditionalOptions additionalOptions);

        /**
         * <p>
         * Specifies additional connection options.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3DirectSourceAdditionalOptions.Builder}
         * avoiding the need to create one manually via {@link S3DirectSourceAdditionalOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3DirectSourceAdditionalOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #additionalOptions(S3DirectSourceAdditionalOptions)}.
         * 
         * @param additionalOptions
         *        a consumer that will call methods on {@link S3DirectSourceAdditionalOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #additionalOptions(S3DirectSourceAdditionalOptions)
         */
        default Builder additionalOptions(Consumer<S3DirectSourceAdditionalOptions.Builder> additionalOptions) {
            return additionalOptions(S3DirectSourceAdditionalOptions.builder().applyMutation(additionalOptions).build());
        }

        /**
         * <p>
         * Specifies the data schema for the S3 Parquet source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for the S3 Parquet source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(Collection<GlueSchema> outputSchemas);

        /**
         * <p>
         * Specifies the data schema for the S3 Parquet source.
         * </p>
         * 
         * @param outputSchemas
         *        Specifies the data schema for the S3 Parquet source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputSchemas(GlueSchema... outputSchemas);

        /**
         * <p>
         * Specifies the data schema for the S3 Parquet source.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.glue.model.GlueSchema#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #outputSchemas(List<GlueSchema>)}.
         * 
         * @param outputSchemas
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.GlueSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputSchemas(java.util.Collection<GlueSchema>)
         */
        Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> paths = DefaultSdkAutoConstructList.getInstance();

        private String compressionType;

        private List<String> exclusions = DefaultSdkAutoConstructList.getInstance();

        private String groupSize;

        private String groupFiles;

        private Boolean recurse;

        private Integer maxBand;

        private Integer maxFilesInBand;

        private S3DirectSourceAdditionalOptions additionalOptions;

        private List<GlueSchema> outputSchemas = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3ParquetSource model) {
            name(model.name);
            paths(model.paths);
            compressionType(model.compressionType);
            exclusions(model.exclusions);
            groupSize(model.groupSize);
            groupFiles(model.groupFiles);
            recurse(model.recurse);
            maxBand(model.maxBand);
            maxFilesInBand(model.maxFilesInBand);
            additionalOptions(model.additionalOptions);
            outputSchemas(model.outputSchemas);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getPaths() {
            if (paths instanceof SdkAutoConstructList) {
                return null;
            }
            return paths;
        }

        public final void setPaths(Collection<String> paths) {
            this.paths = EnclosedInStringPropertiesCopier.copy(paths);
        }

        @Override
        public final Builder paths(Collection<String> paths) {
            this.paths = EnclosedInStringPropertiesCopier.copy(paths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder paths(String... paths) {
            paths(Arrays.asList(paths));
            return this;
        }

        public final String getCompressionType() {
            return compressionType;
        }

        public final void setCompressionType(String compressionType) {
            this.compressionType = compressionType;
        }

        @Override
        public final Builder compressionType(String compressionType) {
            this.compressionType = compressionType;
            return this;
        }

        @Override
        public final Builder compressionType(ParquetCompressionType compressionType) {
            this.compressionType(compressionType == null ? null : compressionType.toString());
            return this;
        }

        public final Collection<String> getExclusions() {
            if (exclusions instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusions;
        }

        public final void setExclusions(Collection<String> exclusions) {
            this.exclusions = EnclosedInStringPropertiesCopier.copy(exclusions);
        }

        @Override
        public final Builder exclusions(Collection<String> exclusions) {
            this.exclusions = EnclosedInStringPropertiesCopier.copy(exclusions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusions(String... exclusions) {
            exclusions(Arrays.asList(exclusions));
            return this;
        }

        public final String getGroupSize() {
            return groupSize;
        }

        public final void setGroupSize(String groupSize) {
            this.groupSize = groupSize;
        }

        @Override
        public final Builder groupSize(String groupSize) {
            this.groupSize = groupSize;
            return this;
        }

        public final String getGroupFiles() {
            return groupFiles;
        }

        public final void setGroupFiles(String groupFiles) {
            this.groupFiles = groupFiles;
        }

        @Override
        public final Builder groupFiles(String groupFiles) {
            this.groupFiles = groupFiles;
            return this;
        }

        public final Boolean getRecurse() {
            return recurse;
        }

        public final void setRecurse(Boolean recurse) {
            this.recurse = recurse;
        }

        @Override
        public final Builder recurse(Boolean recurse) {
            this.recurse = recurse;
            return this;
        }

        public final Integer getMaxBand() {
            return maxBand;
        }

        public final void setMaxBand(Integer maxBand) {
            this.maxBand = maxBand;
        }

        @Override
        public final Builder maxBand(Integer maxBand) {
            this.maxBand = maxBand;
            return this;
        }

        public final Integer getMaxFilesInBand() {
            return maxFilesInBand;
        }

        public final void setMaxFilesInBand(Integer maxFilesInBand) {
            this.maxFilesInBand = maxFilesInBand;
        }

        @Override
        public final Builder maxFilesInBand(Integer maxFilesInBand) {
            this.maxFilesInBand = maxFilesInBand;
            return this;
        }

        public final S3DirectSourceAdditionalOptions.Builder getAdditionalOptions() {
            return additionalOptions != null ? additionalOptions.toBuilder() : null;
        }

        public final void setAdditionalOptions(S3DirectSourceAdditionalOptions.BuilderImpl additionalOptions) {
            this.additionalOptions = additionalOptions != null ? additionalOptions.build() : null;
        }

        @Override
        public final Builder additionalOptions(S3DirectSourceAdditionalOptions additionalOptions) {
            this.additionalOptions = additionalOptions;
            return this;
        }

        public final List<GlueSchema.Builder> getOutputSchemas() {
            List<GlueSchema.Builder> result = GlueSchemasCopier.copyToBuilder(this.outputSchemas);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputSchemas(Collection<GlueSchema.BuilderImpl> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copyFromBuilder(outputSchemas);
        }

        @Override
        public final Builder outputSchemas(Collection<GlueSchema> outputSchemas) {
            this.outputSchemas = GlueSchemasCopier.copy(outputSchemas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(GlueSchema... outputSchemas) {
            outputSchemas(Arrays.asList(outputSchemas));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputSchemas(Consumer<GlueSchema.Builder>... outputSchemas) {
            outputSchemas(Stream.of(outputSchemas).map(c -> GlueSchema.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public S3ParquetSource build() {
            return new S3ParquetSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
