/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartJobRunRequest extends GlueRequest implements
        ToCopyableBuilder<StartJobRunRequest.Builder, StartJobRunRequest> {
    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(StartJobRunRequest::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<Boolean> JOB_RUN_QUEUING_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("JobRunQueuingEnabled").getter(getter(StartJobRunRequest::jobRunQueuingEnabled))
            .setter(setter(Builder::jobRunQueuingEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobRunQueuingEnabled").build())
            .build();

    private static final SdkField<String> JOB_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobRunId").getter(getter(StartJobRunRequest::jobRunId)).setter(setter(Builder::jobRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobRunId").build()).build();

    private static final SdkField<Map<String, String>> ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Arguments")
            .getter(getter(StartJobRunRequest::arguments))
            .setter(setter(Builder::arguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Integer> ALLOCATED_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocatedCapacity").getter(getter(StartJobRunRequest::allocatedCapacity))
            .setter(setter(Builder::allocatedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocatedCapacity").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(StartJobRunRequest::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Double> MAX_CAPACITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxCapacity").getter(getter(StartJobRunRequest::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<String> SECURITY_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityConfiguration").getter(getter(StartJobRunRequest::securityConfiguration))
            .setter(setter(Builder::securityConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityConfiguration").build())
            .build();

    private static final SdkField<NotificationProperty> NOTIFICATION_PROPERTY_FIELD = SdkField
            .<NotificationProperty> builder(MarshallingType.SDK_POJO).memberName("NotificationProperty")
            .getter(getter(StartJobRunRequest::notificationProperty)).setter(setter(Builder::notificationProperty))
            .constructor(NotificationProperty::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationProperty").build())
            .build();

    private static final SdkField<String> WORKER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkerType").getter(getter(StartJobRunRequest::workerTypeAsString)).setter(setter(Builder::workerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkerType").build()).build();

    private static final SdkField<Integer> NUMBER_OF_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfWorkers").getter(getter(StartJobRunRequest::numberOfWorkers))
            .setter(setter(Builder::numberOfWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfWorkers").build()).build();

    private static final SdkField<String> EXECUTION_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionClass").getter(getter(StartJobRunRequest::executionClassAsString))
            .setter(setter(Builder::executionClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionClass").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_NAME_FIELD,
            JOB_RUN_QUEUING_ENABLED_FIELD, JOB_RUN_ID_FIELD, ARGUMENTS_FIELD, ALLOCATED_CAPACITY_FIELD, TIMEOUT_FIELD,
            MAX_CAPACITY_FIELD, SECURITY_CONFIGURATION_FIELD, NOTIFICATION_PROPERTY_FIELD, WORKER_TYPE_FIELD,
            NUMBER_OF_WORKERS_FIELD, EXECUTION_CLASS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String jobName;

    private final Boolean jobRunQueuingEnabled;

    private final String jobRunId;

    private final Map<String, String> arguments;

    private final Integer allocatedCapacity;

    private final Integer timeout;

    private final Double maxCapacity;

    private final String securityConfiguration;

    private final NotificationProperty notificationProperty;

    private final String workerType;

    private final Integer numberOfWorkers;

    private final String executionClass;

    private StartJobRunRequest(BuilderImpl builder) {
        super(builder);
        this.jobName = builder.jobName;
        this.jobRunQueuingEnabled = builder.jobRunQueuingEnabled;
        this.jobRunId = builder.jobRunId;
        this.arguments = builder.arguments;
        this.allocatedCapacity = builder.allocatedCapacity;
        this.timeout = builder.timeout;
        this.maxCapacity = builder.maxCapacity;
        this.securityConfiguration = builder.securityConfiguration;
        this.notificationProperty = builder.notificationProperty;
        this.workerType = builder.workerType;
        this.numberOfWorkers = builder.numberOfWorkers;
        this.executionClass = builder.executionClass;
    }

    /**
     * <p>
     * The name of the job definition to use.
     * </p>
     * 
     * @return The name of the job definition to use.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * Specifies whether job run queuing is enabled for the job run.
     * </p>
     * <p>
     * A value of true means job run queuing is enabled for the job run. If false or not populated, the job run will not
     * be considered for queueing.
     * </p>
     * 
     * @return Specifies whether job run queuing is enabled for the job run.</p>
     *         <p>
     *         A value of true means job run queuing is enabled for the job run. If false or not populated, the job run
     *         will not be considered for queueing.
     */
    public final Boolean jobRunQueuingEnabled() {
        return jobRunQueuingEnabled;
    }

    /**
     * <p>
     * The ID of a previous <code>JobRun</code> to retry.
     * </p>
     * 
     * @return The ID of a previous <code>JobRun</code> to retry.
     */
    public final String jobRunId() {
        return jobRunId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Arguments property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasArguments() {
        return arguments != null && !(arguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The job arguments associated with this run. For this job run, they replace the default arguments set in the job
     * definition itself.
     * </p>
     * <p>
     * You can specify arguments here that your own job-execution script consumes, as well as arguments that Glue itself
     * consumes.
     * </p>
     * <p>
     * Job arguments may be logged. Do not pass plaintext secrets as arguments. Retrieve secrets from a Glue Connection,
     * Secrets Manager or other secret management mechanism if you intend to keep them within the Job.
     * </p>
     * <p>
     * For information about how to specify and consume your own Job arguments, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue APIs in
     * Python</a> topic in the developer guide.
     * </p>
     * <p>
     * For information about the arguments you can provide to this field when configuring Spark jobs, see the <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special Parameters
     * Used by Glue</a> topic in the developer guide.
     * </p>
     * <p>
     * For information about the arguments you can provide to this field when configuring Ray jobs, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/author-job-ray-job-parameters.html">Using job parameters in Ray
     * jobs</a> in the developer guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArguments} method.
     * </p>
     * 
     * @return The job arguments associated with this run. For this job run, they replace the default arguments set in
     *         the job definition itself.</p>
     *         <p>
     *         You can specify arguments here that your own job-execution script consumes, as well as arguments that
     *         Glue itself consumes.
     *         </p>
     *         <p>
     *         Job arguments may be logged. Do not pass plaintext secrets as arguments. Retrieve secrets from a Glue
     *         Connection, Secrets Manager or other secret management mechanism if you intend to keep them within the
     *         Job.
     *         </p>
     *         <p>
     *         For information about how to specify and consume your own Job arguments, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue
     *         APIs in Python</a> topic in the developer guide.
     *         </p>
     *         <p>
     *         For information about the arguments you can provide to this field when configuring Spark jobs, see the <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
     *         Parameters Used by Glue</a> topic in the developer guide.
     *         </p>
     *         <p>
     *         For information about the arguments you can provide to this field when configuring Ray jobs, see <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/author-job-ray-job-parameters.html">Using job parameters
     *         in Ray jobs</a> in the developer guide.
     */
    public final Map<String, String> arguments() {
        return arguments;
    }

    /**
     * <p>
     * This field is deprecated. Use <code>MaxCapacity</code> instead.
     * </p>
     * <p>
     * The number of Glue data processing units (DPUs) to allocate to this JobRun. You can allocate a minimum of 2 DPUs;
     * the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute capacity
     * and 16 GB of memory. For more information, see the <a href="https://aws.amazon.com/glue/pricing/">Glue pricing
     * page</a>.
     * </p>
     * 
     * @return This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
     *         <p>
     *         The number of Glue data processing units (DPUs) to allocate to this JobRun. You can allocate a minimum of
     *         2 DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of
     *         compute capacity and 16 GB of memory. For more information, see the <a
     *         href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
     * @deprecated This property is deprecated, use MaxCapacity instead.
     */
    @Deprecated
    public final Integer allocatedCapacity() {
        return allocatedCapacity;
    }

    /**
     * <p>
     * The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources before
     * it is terminated and enters <code>TIMEOUT</code> status. This value overrides the timeout value set in the parent
     * job.
     * </p>
     * <p>
     * Jobs must have timeout values less than 7 days or 10080 minutes. Otherwise, the jobs will throw an exception.
     * </p>
     * <p>
     * When the value is left blank, the timeout is defaulted to 2880 minutes.
     * </p>
     * <p>
     * Any existing Glue jobs that had a timeout value greater than 7 days will be defaulted to 7 days. For instance if
     * you have specified a timeout of 20 days for a batch job, it will be stopped on the 7th day.
     * </p>
     * <p>
     * For streaming jobs, if you have set up a maintenance window, it will be restarted during the maintenance window
     * after 7 days.
     * </p>
     * 
     * @return The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources
     *         before it is terminated and enters <code>TIMEOUT</code> status. This value overrides the timeout value
     *         set in the parent job. </p>
     *         <p>
     *         Jobs must have timeout values less than 7 days or 10080 minutes. Otherwise, the jobs will throw an
     *         exception.
     *         </p>
     *         <p>
     *         When the value is left blank, the timeout is defaulted to 2880 minutes.
     *         </p>
     *         <p>
     *         Any existing Glue jobs that had a timeout value greater than 7 days will be defaulted to 7 days. For
     *         instance if you have specified a timeout of 20 days for a batch job, it will be stopped on the 7th day.
     *         </p>
     *         <p>
     *         For streaming jobs, if you have set up a maintenance window, it will be restarted during the maintenance
     *         window after 7 days.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * For Glue version 1.0 or earlier jobs, using the standard worker type, the number of Glue data processing units
     * (DPUs) that can be allocated when this job runs. A DPU is a relative measure of processing power that consists of
     * 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
     * href="https://aws.amazon.com/glue/pricing/"> Glue pricing page</a>.
     * </p>
     * <p>
     * For Glue version 2.0+ jobs, you cannot specify a <code>Maximum capacity</code>. Instead, you should specify a
     * <code>Worker type</code> and the <code>Number of workers</code>.
     * </p>
     * <p>
     * Do not set <code>MaxCapacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     * </p>
     * <p>
     * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python shell
     * job, an Apache Spark ETL job, or an Apache Spark streaming ETL job:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either 0.0625
     * or 1 DPU. The default is 0.0625 DPU.
     * </p>
     * </li>
     * <li>
     * <p>
     * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark streaming ETL
     * job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs. The default is 10 DPUs.
     * This job type cannot have a fractional DPU allocation.
     * </p>
     * </li>
     * </ul>
     * 
     * @return For Glue version 1.0 or earlier jobs, using the standard worker type, the number of Glue data processing
     *         units (DPUs) that can be allocated when this job runs. A DPU is a relative measure of processing power
     *         that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
     *         href="https://aws.amazon.com/glue/pricing/"> Glue pricing page</a>.</p>
     *         <p>
     *         For Glue version 2.0+ jobs, you cannot specify a <code>Maximum capacity</code>. Instead, you should
     *         specify a <code>Worker type</code> and the <code>Number of workers</code>.
     *         </p>
     *         <p>
     *         Do not set <code>MaxCapacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
     *         </p>
     *         <p>
     *         The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
     *         shell job, an Apache Spark ETL job, or an Apache Spark streaming ETL job:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
     *         0.0625 or 1 DPU. The default is 0.0625 DPU.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark
     *         streaming ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs.
     *         The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
     *         </p>
     *         </li>
     */
    public final Double maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
     * </p>
     * 
     * @return The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
     */
    public final String securityConfiguration() {
        return securityConfiguration;
    }

    /**
     * <p>
     * Specifies configuration properties of a job run notification.
     * </p>
     * 
     * @return Specifies configuration properties of a job run notification.
     */
    public final NotificationProperty notificationProperty() {
        return notificationProperty;
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, G.8X or
     * G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk, and
     * provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N. Virginia), US
     * West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe
     * (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk,
     * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the <code>G.4X</code> worker
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB disk,
     * and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs. This worker type
     * is only available for Glue version 3.0 or later streaming jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB disk,
     * and provides up to 8 Ray workers based on the autoscaler.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
     *         G.8X or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US
     *         East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific
     *         (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
     *         <code>G.4X</code> worker type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs.
     *         This worker type is only available for Glue version 3.0 or later streaming jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
     *         disk, and provides up to 8 Ray workers based on the autoscaler.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final WorkerType workerType() {
        return WorkerType.fromValue(workerType);
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, G.8X or
     * G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk, and
     * provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N. Virginia), US
     * West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe
     * (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk,
     * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the <code>G.4X</code> worker
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB disk,
     * and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs. This worker type
     * is only available for Glue version 3.0 or later streaming jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB disk,
     * and provides up to 8 Ray workers based on the autoscaler.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
     *         G.8X or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US
     *         East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific
     *         (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
     *         <code>G.4X</code> worker type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs.
     *         This worker type is only available for Glue version 3.0 or later streaming jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
     *         disk, and provides up to 8 Ray workers based on the autoscaler.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final String workerTypeAsString() {
        return workerType;
    }

    /**
     * <p>
     * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
     * </p>
     * 
     * @return The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
     */
    public final Integer numberOfWorkers() {
        return numberOfWorkers;
    }

    /**
     * <p>
     * Indicates whether the job is run with a standard or flexible execution class. The standard execution-class is
     * ideal for time-sensitive workloads that require fast job startup and dedicated resources.
     * </p>
     * <p>
     * The flexible execution class is appropriate for time-insensitive jobs whose start and completion times may vary.
     * </p>
     * <p>
     * Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
     * <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark jobs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionClass}
     * will return {@link ExecutionClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionClassAsString}.
     * </p>
     * 
     * @return Indicates whether the job is run with a standard or flexible execution class. The standard
     *         execution-class is ideal for time-sensitive workloads that require fast job startup and dedicated
     *         resources.</p>
     *         <p>
     *         The flexible execution class is appropriate for time-insensitive jobs whose start and completion times
     *         may vary.
     *         </p>
     *         <p>
     *         Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
     *         <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark
     *         jobs.
     * @see ExecutionClass
     */
    public final ExecutionClass executionClass() {
        return ExecutionClass.fromValue(executionClass);
    }

    /**
     * <p>
     * Indicates whether the job is run with a standard or flexible execution class. The standard execution-class is
     * ideal for time-sensitive workloads that require fast job startup and dedicated resources.
     * </p>
     * <p>
     * The flexible execution class is appropriate for time-insensitive jobs whose start and completion times may vary.
     * </p>
     * <p>
     * Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
     * <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark jobs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionClass}
     * will return {@link ExecutionClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionClassAsString}.
     * </p>
     * 
     * @return Indicates whether the job is run with a standard or flexible execution class. The standard
     *         execution-class is ideal for time-sensitive workloads that require fast job startup and dedicated
     *         resources.</p>
     *         <p>
     *         The flexible execution class is appropriate for time-insensitive jobs whose start and completion times
     *         may vary.
     *         </p>
     *         <p>
     *         Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
     *         <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark
     *         jobs.
     * @see ExecutionClass
     */
    public final String executionClassAsString() {
        return executionClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(jobRunQueuingEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(jobRunId());
        hashCode = 31 * hashCode + Objects.hashCode(hasArguments() ? arguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(allocatedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(securityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(notificationProperty());
        hashCode = 31 * hashCode + Objects.hashCode(workerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(executionClassAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartJobRunRequest)) {
            return false;
        }
        StartJobRunRequest other = (StartJobRunRequest) obj;
        return Objects.equals(jobName(), other.jobName()) && Objects.equals(jobRunQueuingEnabled(), other.jobRunQueuingEnabled())
                && Objects.equals(jobRunId(), other.jobRunId()) && hasArguments() == other.hasArguments()
                && Objects.equals(arguments(), other.arguments())
                && Objects.equals(allocatedCapacity(), other.allocatedCapacity()) && Objects.equals(timeout(), other.timeout())
                && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(securityConfiguration(), other.securityConfiguration())
                && Objects.equals(notificationProperty(), other.notificationProperty())
                && Objects.equals(workerTypeAsString(), other.workerTypeAsString())
                && Objects.equals(numberOfWorkers(), other.numberOfWorkers())
                && Objects.equals(executionClassAsString(), other.executionClassAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartJobRunRequest").add("JobName", jobName())
                .add("JobRunQueuingEnabled", jobRunQueuingEnabled()).add("JobRunId", jobRunId())
                .add("Arguments", hasArguments() ? arguments() : null).add("AllocatedCapacity", allocatedCapacity())
                .add("Timeout", timeout()).add("MaxCapacity", maxCapacity())
                .add("SecurityConfiguration", securityConfiguration()).add("NotificationProperty", notificationProperty())
                .add("WorkerType", workerTypeAsString()).add("NumberOfWorkers", numberOfWorkers())
                .add("ExecutionClass", executionClassAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "JobRunQueuingEnabled":
            return Optional.ofNullable(clazz.cast(jobRunQueuingEnabled()));
        case "JobRunId":
            return Optional.ofNullable(clazz.cast(jobRunId()));
        case "Arguments":
            return Optional.ofNullable(clazz.cast(arguments()));
        case "AllocatedCapacity":
            return Optional.ofNullable(clazz.cast(allocatedCapacity()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "SecurityConfiguration":
            return Optional.ofNullable(clazz.cast(securityConfiguration()));
        case "NotificationProperty":
            return Optional.ofNullable(clazz.cast(notificationProperty()));
        case "WorkerType":
            return Optional.ofNullable(clazz.cast(workerTypeAsString()));
        case "NumberOfWorkers":
            return Optional.ofNullable(clazz.cast(numberOfWorkers()));
        case "ExecutionClass":
            return Optional.ofNullable(clazz.cast(executionClassAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("JobName", JOB_NAME_FIELD);
        map.put("JobRunQueuingEnabled", JOB_RUN_QUEUING_ENABLED_FIELD);
        map.put("JobRunId", JOB_RUN_ID_FIELD);
        map.put("Arguments", ARGUMENTS_FIELD);
        map.put("AllocatedCapacity", ALLOCATED_CAPACITY_FIELD);
        map.put("Timeout", TIMEOUT_FIELD);
        map.put("MaxCapacity", MAX_CAPACITY_FIELD);
        map.put("SecurityConfiguration", SECURITY_CONFIGURATION_FIELD);
        map.put("NotificationProperty", NOTIFICATION_PROPERTY_FIELD);
        map.put("WorkerType", WORKER_TYPE_FIELD);
        map.put("NumberOfWorkers", NUMBER_OF_WORKERS_FIELD);
        map.put("ExecutionClass", EXECUTION_CLASS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StartJobRunRequest, T> g) {
        return obj -> g.apply((StartJobRunRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GlueRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartJobRunRequest> {
        /**
         * <p>
         * The name of the job definition to use.
         * </p>
         * 
         * @param jobName
         *        The name of the job definition to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * Specifies whether job run queuing is enabled for the job run.
         * </p>
         * <p>
         * A value of true means job run queuing is enabled for the job run. If false or not populated, the job run will
         * not be considered for queueing.
         * </p>
         * 
         * @param jobRunQueuingEnabled
         *        Specifies whether job run queuing is enabled for the job run.</p>
         *        <p>
         *        A value of true means job run queuing is enabled for the job run. If false or not populated, the job
         *        run will not be considered for queueing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRunQueuingEnabled(Boolean jobRunQueuingEnabled);

        /**
         * <p>
         * The ID of a previous <code>JobRun</code> to retry.
         * </p>
         * 
         * @param jobRunId
         *        The ID of a previous <code>JobRun</code> to retry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRunId(String jobRunId);

        /**
         * <p>
         * The job arguments associated with this run. For this job run, they replace the default arguments set in the
         * job definition itself.
         * </p>
         * <p>
         * You can specify arguments here that your own job-execution script consumes, as well as arguments that Glue
         * itself consumes.
         * </p>
         * <p>
         * Job arguments may be logged. Do not pass plaintext secrets as arguments. Retrieve secrets from a Glue
         * Connection, Secrets Manager or other secret management mechanism if you intend to keep them within the Job.
         * </p>
         * <p>
         * For information about how to specify and consume your own Job arguments, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling Glue APIs
         * in Python</a> topic in the developer guide.
         * </p>
         * <p>
         * For information about the arguments you can provide to this field when configuring Spark jobs, see the <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">Special
         * Parameters Used by Glue</a> topic in the developer guide.
         * </p>
         * <p>
         * For information about the arguments you can provide to this field when configuring Ray jobs, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/author-job-ray-job-parameters.html">Using job parameters in
         * Ray jobs</a> in the developer guide.
         * </p>
         * 
         * @param arguments
         *        The job arguments associated with this run. For this job run, they replace the default arguments set
         *        in the job definition itself.</p>
         *        <p>
         *        You can specify arguments here that your own job-execution script consumes, as well as arguments that
         *        Glue itself consumes.
         *        </p>
         *        <p>
         *        Job arguments may be logged. Do not pass plaintext secrets as arguments. Retrieve secrets from a Glue
         *        Connection, Secrets Manager or other secret management mechanism if you intend to keep them within the
         *        Job.
         *        </p>
         *        <p>
         *        For information about how to specify and consume your own Job arguments, see the <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-python-calling.html">Calling
         *        Glue APIs in Python</a> topic in the developer guide.
         *        </p>
         *        <p>
         *        For information about the arguments you can provide to this field when configuring Spark jobs, see the
         *        <a href="https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html">
         *        Special Parameters Used by Glue</a> topic in the developer guide.
         *        </p>
         *        <p>
         *        For information about the arguments you can provide to this field when configuring Ray jobs, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/author-job-ray-job-parameters.html">Using job
         *        parameters in Ray jobs</a> in the developer guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arguments(Map<String, String> arguments);

        /**
         * <p>
         * This field is deprecated. Use <code>MaxCapacity</code> instead.
         * </p>
         * <p>
         * The number of Glue data processing units (DPUs) to allocate to this JobRun. You can allocate a minimum of 2
         * DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs of compute
         * capacity and 16 GB of memory. For more information, see the <a
         * href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
         * </p>
         * 
         * @param allocatedCapacity
         *        This field is deprecated. Use <code>MaxCapacity</code> instead.</p>
         *        <p>
         *        The number of Glue data processing units (DPUs) to allocate to this JobRun. You can allocate a minimum
         *        of 2 DPUs; the default is 10. A DPU is a relative measure of processing power that consists of 4 vCPUs
         *        of compute capacity and 16 GB of memory. For more information, see the <a
         *        href="https://aws.amazon.com/glue/pricing/">Glue pricing page</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated, use MaxCapacity instead.
         */
        @Deprecated
        Builder allocatedCapacity(Integer allocatedCapacity);

        /**
         * <p>
         * The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume resources
         * before it is terminated and enters <code>TIMEOUT</code> status. This value overrides the timeout value set in
         * the parent job.
         * </p>
         * <p>
         * Jobs must have timeout values less than 7 days or 10080 minutes. Otherwise, the jobs will throw an exception.
         * </p>
         * <p>
         * When the value is left blank, the timeout is defaulted to 2880 minutes.
         * </p>
         * <p>
         * Any existing Glue jobs that had a timeout value greater than 7 days will be defaulted to 7 days. For instance
         * if you have specified a timeout of 20 days for a batch job, it will be stopped on the 7th day.
         * </p>
         * <p>
         * For streaming jobs, if you have set up a maintenance window, it will be restarted during the maintenance
         * window after 7 days.
         * </p>
         * 
         * @param timeout
         *        The <code>JobRun</code> timeout in minutes. This is the maximum time that a job run can consume
         *        resources before it is terminated and enters <code>TIMEOUT</code> status. This value overrides the
         *        timeout value set in the parent job. </p>
         *        <p>
         *        Jobs must have timeout values less than 7 days or 10080 minutes. Otherwise, the jobs will throw an
         *        exception.
         *        </p>
         *        <p>
         *        When the value is left blank, the timeout is defaulted to 2880 minutes.
         *        </p>
         *        <p>
         *        Any existing Glue jobs that had a timeout value greater than 7 days will be defaulted to 7 days. For
         *        instance if you have specified a timeout of 20 days for a batch job, it will be stopped on the 7th
         *        day.
         *        </p>
         *        <p>
         *        For streaming jobs, if you have set up a maintenance window, it will be restarted during the
         *        maintenance window after 7 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * For Glue version 1.0 or earlier jobs, using the standard worker type, the number of Glue data processing
         * units (DPUs) that can be allocated when this job runs. A DPU is a relative measure of processing power that
         * consists of 4 vCPUs of compute capacity and 16 GB of memory. For more information, see the <a
         * href="https://aws.amazon.com/glue/pricing/"> Glue pricing page</a>.
         * </p>
         * <p>
         * For Glue version 2.0+ jobs, you cannot specify a <code>Maximum capacity</code>. Instead, you should specify a
         * <code>Worker type</code> and the <code>Number of workers</code>.
         * </p>
         * <p>
         * Do not set <code>MaxCapacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
         * </p>
         * <p>
         * The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a Python
         * shell job, an Apache Spark ETL job, or an Apache Spark streaming ETL job:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate either
         * 0.0625 or 1 DPU. The default is 0.0625 DPU.
         * </p>
         * </li>
         * <li>
         * <p>
         * When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark streaming
         * ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs. The default is
         * 10 DPUs. This job type cannot have a fractional DPU allocation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param maxCapacity
         *        For Glue version 1.0 or earlier jobs, using the standard worker type, the number of Glue data
         *        processing units (DPUs) that can be allocated when this job runs. A DPU is a relative measure of
         *        processing power that consists of 4 vCPUs of compute capacity and 16 GB of memory. For more
         *        information, see the <a href="https://aws.amazon.com/glue/pricing/"> Glue pricing page</a>.</p>
         *        <p>
         *        For Glue version 2.0+ jobs, you cannot specify a <code>Maximum capacity</code>. Instead, you should
         *        specify a <code>Worker type</code> and the <code>Number of workers</code>.
         *        </p>
         *        <p>
         *        Do not set <code>MaxCapacity</code> if using <code>WorkerType</code> and <code>NumberOfWorkers</code>.
         *        </p>
         *        <p>
         *        The value that can be allocated for <code>MaxCapacity</code> depends on whether you are running a
         *        Python shell job, an Apache Spark ETL job, or an Apache Spark streaming ETL job:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify a Python shell job (<code>JobCommand.Name</code>="pythonshell"), you can allocate
         *        either 0.0625 or 1 DPU. The default is 0.0625 DPU.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        When you specify an Apache Spark ETL job (<code>JobCommand.Name</code>="glueetl") or Apache Spark
         *        streaming ETL job (<code>JobCommand.Name</code>="gluestreaming"), you can allocate from 2 to 100 DPUs.
         *        The default is 10 DPUs. This job type cannot have a fractional DPU allocation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Double maxCapacity);

        /**
         * <p>
         * The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
         * </p>
         * 
         * @param securityConfiguration
         *        The name of the <code>SecurityConfiguration</code> structure to be used with this job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityConfiguration(String securityConfiguration);

        /**
         * <p>
         * Specifies configuration properties of a job run notification.
         * </p>
         * 
         * @param notificationProperty
         *        Specifies configuration properties of a job run notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationProperty(NotificationProperty notificationProperty);

        /**
         * <p>
         * Specifies configuration properties of a job run notification.
         * </p>
         * This is a convenience method that creates an instance of the {@link NotificationProperty.Builder} avoiding
         * the need to create one manually via {@link NotificationProperty#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NotificationProperty.Builder#build()} is called immediately and
         * its result is passed to {@link #notificationProperty(NotificationProperty)}.
         * 
         * @param notificationProperty
         *        a consumer that will call methods on {@link NotificationProperty.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationProperty(NotificationProperty)
         */
        default Builder notificationProperty(Consumer<NotificationProperty.Builder> notificationProperty) {
            return notificationProperty(NotificationProperty.builder().applyMutation(notificationProperty).build());
        }

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, G.8X or
         * G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk,
         * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
         * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version
         * 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N.
         * Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada
         * (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
         * disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your
         * most demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue
         * version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
         * <code>G.4X</code> worker type.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB
         * disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs. This
         * worker type is only available for Glue version 3.0 or later streaming jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
         * disk, and provides up to 8 Ray workers based on the autoscaler.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
         *        G.8X or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with
         *        256GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services
         *        Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia
         *        Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and
         *        Europe (Stockholm).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with
         *        512GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions
         *        as supported for the <code>G.4X</code> worker type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with
         *        84GB disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming
         *        jobs. This worker type is only available for Glue version 3.0 or later streaming jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128
         *        GB disk, and provides up to 8 Ray workers based on the autoscaler.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(String workerType);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, G.8X or
         * G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk,
         * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
         * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version
         * 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N.
         * Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada
         * (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
         * disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your
         * most demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue
         * version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
         * <code>G.4X</code> worker type.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with 84GB
         * disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming jobs. This
         * worker type is only available for Glue version 3.0 or later streaming jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
         * disk, and provides up to 8 Ray workers based on the autoscaler.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
         *        G.8X or G.025X for Spark jobs. Accepts the value Z.2X for Ray jobs.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with
         *        256GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services
         *        Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia
         *        Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and
         *        Europe (Stockholm).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with
         *        512GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions
         *        as supported for the <code>G.4X</code> worker type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.025X</code> worker type, each worker maps to 0.25 DPU (2 vCPUs, 4 GB of memory) with
         *        84GB disk, and provides 1 executor per worker. We recommend this worker type for low volume streaming
         *        jobs. This worker type is only available for Glue version 3.0 or later streaming jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128
         *        GB disk, and provides up to 8 Ray workers based on the autoscaler.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(WorkerType workerType);

        /**
         * <p>
         * The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
         * </p>
         * 
         * @param numberOfWorkers
         *        The number of workers of a defined <code>workerType</code> that are allocated when a job runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfWorkers(Integer numberOfWorkers);

        /**
         * <p>
         * Indicates whether the job is run with a standard or flexible execution class. The standard execution-class is
         * ideal for time-sensitive workloads that require fast job startup and dedicated resources.
         * </p>
         * <p>
         * The flexible execution class is appropriate for time-insensitive jobs whose start and completion times may
         * vary.
         * </p>
         * <p>
         * Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
         * <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark jobs.
         * </p>
         * 
         * @param executionClass
         *        Indicates whether the job is run with a standard or flexible execution class. The standard
         *        execution-class is ideal for time-sensitive workloads that require fast job startup and dedicated
         *        resources.</p>
         *        <p>
         *        The flexible execution class is appropriate for time-insensitive jobs whose start and completion times
         *        may vary.
         *        </p>
         *        <p>
         *        Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
         *        <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark
         *        jobs.
         * @see ExecutionClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionClass
         */
        Builder executionClass(String executionClass);

        /**
         * <p>
         * Indicates whether the job is run with a standard or flexible execution class. The standard execution-class is
         * ideal for time-sensitive workloads that require fast job startup and dedicated resources.
         * </p>
         * <p>
         * The flexible execution class is appropriate for time-insensitive jobs whose start and completion times may
         * vary.
         * </p>
         * <p>
         * Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
         * <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark jobs.
         * </p>
         * 
         * @param executionClass
         *        Indicates whether the job is run with a standard or flexible execution class. The standard
         *        execution-class is ideal for time-sensitive workloads that require fast job startup and dedicated
         *        resources.</p>
         *        <p>
         *        The flexible execution class is appropriate for time-insensitive jobs whose start and completion times
         *        may vary.
         *        </p>
         *        <p>
         *        Only jobs with Glue version 3.0 and above and command type <code>glueetl</code> will be allowed to set
         *        <code>ExecutionClass</code> to <code>FLEX</code>. The flexible execution class is available for Spark
         *        jobs.
         * @see ExecutionClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionClass
         */
        Builder executionClass(ExecutionClass executionClass);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlueRequest.BuilderImpl implements Builder {
        private String jobName;

        private Boolean jobRunQueuingEnabled;

        private String jobRunId;

        private Map<String, String> arguments = DefaultSdkAutoConstructMap.getInstance();

        private Integer allocatedCapacity;

        private Integer timeout;

        private Double maxCapacity;

        private String securityConfiguration;

        private NotificationProperty notificationProperty;

        private String workerType;

        private Integer numberOfWorkers;

        private String executionClass;

        private BuilderImpl() {
        }

        private BuilderImpl(StartJobRunRequest model) {
            super(model);
            jobName(model.jobName);
            jobRunQueuingEnabled(model.jobRunQueuingEnabled);
            jobRunId(model.jobRunId);
            arguments(model.arguments);
            allocatedCapacity(model.allocatedCapacity);
            timeout(model.timeout);
            maxCapacity(model.maxCapacity);
            securityConfiguration(model.securityConfiguration);
            notificationProperty(model.notificationProperty);
            workerType(model.workerType);
            numberOfWorkers(model.numberOfWorkers);
            executionClass(model.executionClass);
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final Boolean getJobRunQueuingEnabled() {
            return jobRunQueuingEnabled;
        }

        public final void setJobRunQueuingEnabled(Boolean jobRunQueuingEnabled) {
            this.jobRunQueuingEnabled = jobRunQueuingEnabled;
        }

        @Override
        public final Builder jobRunQueuingEnabled(Boolean jobRunQueuingEnabled) {
            this.jobRunQueuingEnabled = jobRunQueuingEnabled;
            return this;
        }

        public final String getJobRunId() {
            return jobRunId;
        }

        public final void setJobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
        }

        @Override
        public final Builder jobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
            return this;
        }

        public final Map<String, String> getArguments() {
            if (arguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return arguments;
        }

        public final void setArguments(Map<String, String> arguments) {
            this.arguments = GenericMapCopier.copy(arguments);
        }

        @Override
        public final Builder arguments(Map<String, String> arguments) {
            this.arguments = GenericMapCopier.copy(arguments);
            return this;
        }

        @Deprecated
        public final Integer getAllocatedCapacity() {
            return allocatedCapacity;
        }

        @Deprecated
        public final void setAllocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
        }

        @Override
        @Deprecated
        public final Builder allocatedCapacity(Integer allocatedCapacity) {
            this.allocatedCapacity = allocatedCapacity;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Double getMaxCapacity() {
            return maxCapacity;
        }

        public final void setMaxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final String getSecurityConfiguration() {
            return securityConfiguration;
        }

        public final void setSecurityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
        }

        @Override
        public final Builder securityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return this;
        }

        public final NotificationProperty.Builder getNotificationProperty() {
            return notificationProperty != null ? notificationProperty.toBuilder() : null;
        }

        public final void setNotificationProperty(NotificationProperty.BuilderImpl notificationProperty) {
            this.notificationProperty = notificationProperty != null ? notificationProperty.build() : null;
        }

        @Override
        public final Builder notificationProperty(NotificationProperty notificationProperty) {
            this.notificationProperty = notificationProperty;
            return this;
        }

        public final String getWorkerType() {
            return workerType;
        }

        public final void setWorkerType(String workerType) {
            this.workerType = workerType;
        }

        @Override
        public final Builder workerType(String workerType) {
            this.workerType = workerType;
            return this;
        }

        @Override
        public final Builder workerType(WorkerType workerType) {
            this.workerType(workerType == null ? null : workerType.toString());
            return this;
        }

        public final Integer getNumberOfWorkers() {
            return numberOfWorkers;
        }

        public final void setNumberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
        }

        @Override
        public final Builder numberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
            return this;
        }

        public final String getExecutionClass() {
            return executionClass;
        }

        public final void setExecutionClass(String executionClass) {
            this.executionClass = executionClass;
        }

        @Override
        public final Builder executionClass(String executionClass) {
            this.executionClass = executionClass;
            return this;
        }

        @Override
        public final Builder executionClass(ExecutionClass executionClass) {
            this.executionClass(executionClass == null ? null : executionClass.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartJobRunRequest build() {
            return new StartJobRunRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
