/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A classifier for custom <code>CSV</code> content.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CsvClassifier implements SdkPojo, Serializable, ToCopyableBuilder<CsvClassifier.Builder, CsvClassifier> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CsvClassifier::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(CsvClassifier::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdated").getter(getter(CsvClassifier::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdated").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Version")
            .getter(getter(CsvClassifier::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> DELIMITER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Delimiter").getter(getter(CsvClassifier::delimiter)).setter(setter(Builder::delimiter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Delimiter").build()).build();

    private static final SdkField<String> QUOTE_SYMBOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QuoteSymbol").getter(getter(CsvClassifier::quoteSymbol)).setter(setter(Builder::quoteSymbol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QuoteSymbol").build()).build();

    private static final SdkField<String> CONTAINS_HEADER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainsHeader").getter(getter(CsvClassifier::containsHeaderAsString))
            .setter(setter(Builder::containsHeader))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainsHeader").build()).build();

    private static final SdkField<List<String>> HEADER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Header")
            .getter(getter(CsvClassifier::header))
            .setter(setter(Builder::header))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Header").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> DISABLE_VALUE_TRIMMING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DisableValueTrimming").getter(getter(CsvClassifier::disableValueTrimming))
            .setter(setter(Builder::disableValueTrimming))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableValueTrimming").build())
            .build();

    private static final SdkField<Boolean> ALLOW_SINGLE_COLUMN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllowSingleColumn").getter(getter(CsvClassifier::allowSingleColumn))
            .setter(setter(Builder::allowSingleColumn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowSingleColumn").build()).build();

    private static final SdkField<Boolean> CUSTOM_DATATYPE_CONFIGURED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CustomDatatypeConfigured").getter(getter(CsvClassifier::customDatatypeConfigured))
            .setter(setter(Builder::customDatatypeConfigured))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomDatatypeConfigured").build())
            .build();

    private static final SdkField<List<String>> CUSTOM_DATATYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomDatatypes")
            .getter(getter(CsvClassifier::customDatatypes))
            .setter(setter(Builder::customDatatypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomDatatypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SERDE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Serde")
            .getter(getter(CsvClassifier::serdeAsString)).setter(setter(Builder::serde))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Serde").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            CREATION_TIME_FIELD, LAST_UPDATED_FIELD, VERSION_FIELD, DELIMITER_FIELD, QUOTE_SYMBOL_FIELD, CONTAINS_HEADER_FIELD,
            HEADER_FIELD, DISABLE_VALUE_TRIMMING_FIELD, ALLOW_SINGLE_COLUMN_FIELD, CUSTOM_DATATYPE_CONFIGURED_FIELD,
            CUSTOM_DATATYPES_FIELD, SERDE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Instant creationTime;

    private final Instant lastUpdated;

    private final Long version;

    private final String delimiter;

    private final String quoteSymbol;

    private final String containsHeader;

    private final List<String> header;

    private final Boolean disableValueTrimming;

    private final Boolean allowSingleColumn;

    private final Boolean customDatatypeConfigured;

    private final List<String> customDatatypes;

    private final String serde;

    private CsvClassifier(BuilderImpl builder) {
        this.name = builder.name;
        this.creationTime = builder.creationTime;
        this.lastUpdated = builder.lastUpdated;
        this.version = builder.version;
        this.delimiter = builder.delimiter;
        this.quoteSymbol = builder.quoteSymbol;
        this.containsHeader = builder.containsHeader;
        this.header = builder.header;
        this.disableValueTrimming = builder.disableValueTrimming;
        this.allowSingleColumn = builder.allowSingleColumn;
        this.customDatatypeConfigured = builder.customDatatypeConfigured;
        this.customDatatypes = builder.customDatatypes;
        this.serde = builder.serde;
    }

    /**
     * <p>
     * The name of the classifier.
     * </p>
     * 
     * @return The name of the classifier.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The time that this classifier was registered.
     * </p>
     * 
     * @return The time that this classifier was registered.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time that this classifier was last updated.
     * </p>
     * 
     * @return The time that this classifier was last updated.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    /**
     * <p>
     * The version of this classifier.
     * </p>
     * 
     * @return The version of this classifier.
     */
    public final Long version() {
        return version;
    }

    /**
     * <p>
     * A custom symbol to denote what separates each column entry in the row.
     * </p>
     * 
     * @return A custom symbol to denote what separates each column entry in the row.
     */
    public final String delimiter() {
        return delimiter;
    }

    /**
     * <p>
     * A custom symbol to denote what combines content into a single column value. It must be different from the column
     * delimiter.
     * </p>
     * 
     * @return A custom symbol to denote what combines content into a single column value. It must be different from the
     *         column delimiter.
     */
    public final String quoteSymbol() {
        return quoteSymbol;
    }

    /**
     * <p>
     * Indicates whether the CSV file contains a header.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #containsHeader}
     * will return {@link CsvHeaderOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #containsHeaderAsString}.
     * </p>
     * 
     * @return Indicates whether the CSV file contains a header.
     * @see CsvHeaderOption
     */
    public final CsvHeaderOption containsHeader() {
        return CsvHeaderOption.fromValue(containsHeader);
    }

    /**
     * <p>
     * Indicates whether the CSV file contains a header.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #containsHeader}
     * will return {@link CsvHeaderOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #containsHeaderAsString}.
     * </p>
     * 
     * @return Indicates whether the CSV file contains a header.
     * @see CsvHeaderOption
     */
    public final String containsHeaderAsString() {
        return containsHeader;
    }

    /**
     * For responses, this returns true if the service returned a value for the Header property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasHeader() {
        return header != null && !(header instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of strings representing column names.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHeader} method.
     * </p>
     * 
     * @return A list of strings representing column names.
     */
    public final List<String> header() {
        return header;
    }

    /**
     * <p>
     * Specifies not to trim values before identifying the type of column values. The default value is <code>true</code>
     * .
     * </p>
     * 
     * @return Specifies not to trim values before identifying the type of column values. The default value is
     *         <code>true</code>.
     */
    public final Boolean disableValueTrimming() {
        return disableValueTrimming;
    }

    /**
     * <p>
     * Enables the processing of files that contain only one column.
     * </p>
     * 
     * @return Enables the processing of files that contain only one column.
     */
    public final Boolean allowSingleColumn() {
        return allowSingleColumn;
    }

    /**
     * <p>
     * Enables the custom datatype to be configured.
     * </p>
     * 
     * @return Enables the custom datatype to be configured.
     */
    public final Boolean customDatatypeConfigured() {
        return customDatatypeConfigured;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomDatatypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomDatatypes() {
        return customDatatypes != null && !(customDatatypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT", "LONG",
     * "SHORT", "STRING", "TIMESTAMP".
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomDatatypes} method.
     * </p>
     * 
     * @return A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT",
     *         "LONG", "SHORT", "STRING", "TIMESTAMP".
     */
    public final List<String> customDatatypes() {
        return customDatatypes;
    }

    /**
     * <p>
     * Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid values are
     * <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can specify the
     * <code>None</code> value when you want the crawler to do the detection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serde} will return
     * {@link CsvSerdeOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #serdeAsString}.
     * </p>
     * 
     * @return Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid
     *         values are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can
     *         specify the <code>None</code> value when you want the crawler to do the detection.
     * @see CsvSerdeOption
     */
    public final CsvSerdeOption serde() {
        return CsvSerdeOption.fromValue(serde);
    }

    /**
     * <p>
     * Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid values are
     * <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can specify the
     * <code>None</code> value when you want the crawler to do the detection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serde} will return
     * {@link CsvSerdeOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #serdeAsString}.
     * </p>
     * 
     * @return Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid
     *         values are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can
     *         specify the <code>None</code> value when you want the crawler to do the detection.
     * @see CsvSerdeOption
     */
    public final String serdeAsString() {
        return serde;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(delimiter());
        hashCode = 31 * hashCode + Objects.hashCode(quoteSymbol());
        hashCode = 31 * hashCode + Objects.hashCode(containsHeaderAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasHeader() ? header() : null);
        hashCode = 31 * hashCode + Objects.hashCode(disableValueTrimming());
        hashCode = 31 * hashCode + Objects.hashCode(allowSingleColumn());
        hashCode = 31 * hashCode + Objects.hashCode(customDatatypeConfigured());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomDatatypes() ? customDatatypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serdeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CsvClassifier)) {
            return false;
        }
        CsvClassifier other = (CsvClassifier) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdated(), other.lastUpdated()) && Objects.equals(version(), other.version())
                && Objects.equals(delimiter(), other.delimiter()) && Objects.equals(quoteSymbol(), other.quoteSymbol())
                && Objects.equals(containsHeaderAsString(), other.containsHeaderAsString()) && hasHeader() == other.hasHeader()
                && Objects.equals(header(), other.header())
                && Objects.equals(disableValueTrimming(), other.disableValueTrimming())
                && Objects.equals(allowSingleColumn(), other.allowSingleColumn())
                && Objects.equals(customDatatypeConfigured(), other.customDatatypeConfigured())
                && hasCustomDatatypes() == other.hasCustomDatatypes()
                && Objects.equals(customDatatypes(), other.customDatatypes())
                && Objects.equals(serdeAsString(), other.serdeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CsvClassifier").add("Name", name()).add("CreationTime", creationTime())
                .add("LastUpdated", lastUpdated()).add("Version", version()).add("Delimiter", delimiter())
                .add("QuoteSymbol", quoteSymbol()).add("ContainsHeader", containsHeaderAsString())
                .add("Header", hasHeader() ? header() : null).add("DisableValueTrimming", disableValueTrimming())
                .add("AllowSingleColumn", allowSingleColumn()).add("CustomDatatypeConfigured", customDatatypeConfigured())
                .add("CustomDatatypes", hasCustomDatatypes() ? customDatatypes() : null).add("Serde", serdeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Delimiter":
            return Optional.ofNullable(clazz.cast(delimiter()));
        case "QuoteSymbol":
            return Optional.ofNullable(clazz.cast(quoteSymbol()));
        case "ContainsHeader":
            return Optional.ofNullable(clazz.cast(containsHeaderAsString()));
        case "Header":
            return Optional.ofNullable(clazz.cast(header()));
        case "DisableValueTrimming":
            return Optional.ofNullable(clazz.cast(disableValueTrimming()));
        case "AllowSingleColumn":
            return Optional.ofNullable(clazz.cast(allowSingleColumn()));
        case "CustomDatatypeConfigured":
            return Optional.ofNullable(clazz.cast(customDatatypeConfigured()));
        case "CustomDatatypes":
            return Optional.ofNullable(clazz.cast(customDatatypes()));
        case "Serde":
            return Optional.ofNullable(clazz.cast(serdeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("LastUpdated", LAST_UPDATED_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("Delimiter", DELIMITER_FIELD);
        map.put("QuoteSymbol", QUOTE_SYMBOL_FIELD);
        map.put("ContainsHeader", CONTAINS_HEADER_FIELD);
        map.put("Header", HEADER_FIELD);
        map.put("DisableValueTrimming", DISABLE_VALUE_TRIMMING_FIELD);
        map.put("AllowSingleColumn", ALLOW_SINGLE_COLUMN_FIELD);
        map.put("CustomDatatypeConfigured", CUSTOM_DATATYPE_CONFIGURED_FIELD);
        map.put("CustomDatatypes", CUSTOM_DATATYPES_FIELD);
        map.put("Serde", SERDE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CsvClassifier, T> g) {
        return obj -> g.apply((CsvClassifier) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CsvClassifier> {
        /**
         * <p>
         * The name of the classifier.
         * </p>
         * 
         * @param name
         *        The name of the classifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The time that this classifier was registered.
         * </p>
         * 
         * @param creationTime
         *        The time that this classifier was registered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time that this classifier was last updated.
         * </p>
         * 
         * @param lastUpdated
         *        The time that this classifier was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);

        /**
         * <p>
         * The version of this classifier.
         * </p>
         * 
         * @param version
         *        The version of this classifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);

        /**
         * <p>
         * A custom symbol to denote what separates each column entry in the row.
         * </p>
         * 
         * @param delimiter
         *        A custom symbol to denote what separates each column entry in the row.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delimiter(String delimiter);

        /**
         * <p>
         * A custom symbol to denote what combines content into a single column value. It must be different from the
         * column delimiter.
         * </p>
         * 
         * @param quoteSymbol
         *        A custom symbol to denote what combines content into a single column value. It must be different from
         *        the column delimiter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quoteSymbol(String quoteSymbol);

        /**
         * <p>
         * Indicates whether the CSV file contains a header.
         * </p>
         * 
         * @param containsHeader
         *        Indicates whether the CSV file contains a header.
         * @see CsvHeaderOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CsvHeaderOption
         */
        Builder containsHeader(String containsHeader);

        /**
         * <p>
         * Indicates whether the CSV file contains a header.
         * </p>
         * 
         * @param containsHeader
         *        Indicates whether the CSV file contains a header.
         * @see CsvHeaderOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CsvHeaderOption
         */
        Builder containsHeader(CsvHeaderOption containsHeader);

        /**
         * <p>
         * A list of strings representing column names.
         * </p>
         * 
         * @param header
         *        A list of strings representing column names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder header(Collection<String> header);

        /**
         * <p>
         * A list of strings representing column names.
         * </p>
         * 
         * @param header
         *        A list of strings representing column names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder header(String... header);

        /**
         * <p>
         * Specifies not to trim values before identifying the type of column values. The default value is
         * <code>true</code>.
         * </p>
         * 
         * @param disableValueTrimming
         *        Specifies not to trim values before identifying the type of column values. The default value is
         *        <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableValueTrimming(Boolean disableValueTrimming);

        /**
         * <p>
         * Enables the processing of files that contain only one column.
         * </p>
         * 
         * @param allowSingleColumn
         *        Enables the processing of files that contain only one column.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowSingleColumn(Boolean allowSingleColumn);

        /**
         * <p>
         * Enables the custom datatype to be configured.
         * </p>
         * 
         * @param customDatatypeConfigured
         *        Enables the custom datatype to be configured.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDatatypeConfigured(Boolean customDatatypeConfigured);

        /**
         * <p>
         * A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT",
         * "LONG", "SHORT", "STRING", "TIMESTAMP".
         * </p>
         * 
         * @param customDatatypes
         *        A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT",
         *        "LONG", "SHORT", "STRING", "TIMESTAMP".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDatatypes(Collection<String> customDatatypes);

        /**
         * <p>
         * A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT",
         * "LONG", "SHORT", "STRING", "TIMESTAMP".
         * </p>
         * 
         * @param customDatatypes
         *        A list of custom datatypes including "BINARY", "BOOLEAN", "DATE", "DECIMAL", "DOUBLE", "FLOAT", "INT",
         *        "LONG", "SHORT", "STRING", "TIMESTAMP".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDatatypes(String... customDatatypes);

        /**
         * <p>
         * Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid values
         * are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can specify the
         * <code>None</code> value when you want the crawler to do the detection.
         * </p>
         * 
         * @param serde
         *        Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid
         *        values are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can
         *        specify the <code>None</code> value when you want the crawler to do the detection.
         * @see CsvSerdeOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CsvSerdeOption
         */
        Builder serde(String serde);

        /**
         * <p>
         * Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid values
         * are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can specify the
         * <code>None</code> value when you want the crawler to do the detection.
         * </p>
         * 
         * @param serde
         *        Sets the SerDe for processing CSV in the classifier, which will be applied in the Data Catalog. Valid
         *        values are <code>OpenCSVSerDe</code>, <code>LazySimpleSerDe</code>, and <code>None</code>. You can
         *        specify the <code>None</code> value when you want the crawler to do the detection.
         * @see CsvSerdeOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CsvSerdeOption
         */
        Builder serde(CsvSerdeOption serde);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Instant creationTime;

        private Instant lastUpdated;

        private Long version;

        private String delimiter;

        private String quoteSymbol;

        private String containsHeader;

        private List<String> header = DefaultSdkAutoConstructList.getInstance();

        private Boolean disableValueTrimming;

        private Boolean allowSingleColumn;

        private Boolean customDatatypeConfigured;

        private List<String> customDatatypes = DefaultSdkAutoConstructList.getInstance();

        private String serde;

        private BuilderImpl() {
        }

        private BuilderImpl(CsvClassifier model) {
            name(model.name);
            creationTime(model.creationTime);
            lastUpdated(model.lastUpdated);
            version(model.version);
            delimiter(model.delimiter);
            quoteSymbol(model.quoteSymbol);
            containsHeader(model.containsHeader);
            header(model.header);
            disableValueTrimming(model.disableValueTrimming);
            allowSingleColumn(model.allowSingleColumn);
            customDatatypeConfigured(model.customDatatypeConfigured);
            customDatatypes(model.customDatatypes);
            serde(model.serde);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        public final Long getVersion() {
            return version;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final String getDelimiter() {
            return delimiter;
        }

        public final void setDelimiter(String delimiter) {
            this.delimiter = delimiter;
        }

        @Override
        public final Builder delimiter(String delimiter) {
            this.delimiter = delimiter;
            return this;
        }

        public final String getQuoteSymbol() {
            return quoteSymbol;
        }

        public final void setQuoteSymbol(String quoteSymbol) {
            this.quoteSymbol = quoteSymbol;
        }

        @Override
        public final Builder quoteSymbol(String quoteSymbol) {
            this.quoteSymbol = quoteSymbol;
            return this;
        }

        public final String getContainsHeader() {
            return containsHeader;
        }

        public final void setContainsHeader(String containsHeader) {
            this.containsHeader = containsHeader;
        }

        @Override
        public final Builder containsHeader(String containsHeader) {
            this.containsHeader = containsHeader;
            return this;
        }

        @Override
        public final Builder containsHeader(CsvHeaderOption containsHeader) {
            this.containsHeader(containsHeader == null ? null : containsHeader.toString());
            return this;
        }

        public final Collection<String> getHeader() {
            if (header instanceof SdkAutoConstructList) {
                return null;
            }
            return header;
        }

        public final void setHeader(Collection<String> header) {
            this.header = CsvHeaderCopier.copy(header);
        }

        @Override
        public final Builder header(Collection<String> header) {
            this.header = CsvHeaderCopier.copy(header);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder header(String... header) {
            header(Arrays.asList(header));
            return this;
        }

        public final Boolean getDisableValueTrimming() {
            return disableValueTrimming;
        }

        public final void setDisableValueTrimming(Boolean disableValueTrimming) {
            this.disableValueTrimming = disableValueTrimming;
        }

        @Override
        public final Builder disableValueTrimming(Boolean disableValueTrimming) {
            this.disableValueTrimming = disableValueTrimming;
            return this;
        }

        public final Boolean getAllowSingleColumn() {
            return allowSingleColumn;
        }

        public final void setAllowSingleColumn(Boolean allowSingleColumn) {
            this.allowSingleColumn = allowSingleColumn;
        }

        @Override
        public final Builder allowSingleColumn(Boolean allowSingleColumn) {
            this.allowSingleColumn = allowSingleColumn;
            return this;
        }

        public final Boolean getCustomDatatypeConfigured() {
            return customDatatypeConfigured;
        }

        public final void setCustomDatatypeConfigured(Boolean customDatatypeConfigured) {
            this.customDatatypeConfigured = customDatatypeConfigured;
        }

        @Override
        public final Builder customDatatypeConfigured(Boolean customDatatypeConfigured) {
            this.customDatatypeConfigured = customDatatypeConfigured;
            return this;
        }

        public final Collection<String> getCustomDatatypes() {
            if (customDatatypes instanceof SdkAutoConstructList) {
                return null;
            }
            return customDatatypes;
        }

        public final void setCustomDatatypes(Collection<String> customDatatypes) {
            this.customDatatypes = CustomDatatypesCopier.copy(customDatatypes);
        }

        @Override
        public final Builder customDatatypes(Collection<String> customDatatypes) {
            this.customDatatypes = CustomDatatypesCopier.copy(customDatatypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customDatatypes(String... customDatatypes) {
            customDatatypes(Arrays.asList(customDatatypes));
            return this;
        }

        public final String getSerde() {
            return serde;
        }

        public final void setSerde(String serde) {
            this.serde = serde;
        }

        @Override
        public final Builder serde(String serde) {
            this.serde = serde;
            return this;
        }

        @Override
        public final Builder serde(CsvSerdeOption serde) {
            this.serde(serde == null ? null : serde.toString());
            return this;
        }

        @Override
        public CsvClassifier build() {
            return new CsvClassifier(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
