/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to create a new session.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSessionRequest extends GlueRequest implements
        ToCopyableBuilder<CreateSessionRequest.Builder, CreateSessionRequest> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(CreateSessionRequest::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateSessionRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(CreateSessionRequest::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<SessionCommand> COMMAND_FIELD = SdkField.<SessionCommand> builder(MarshallingType.SDK_POJO)
            .memberName("Command").getter(getter(CreateSessionRequest::command)).setter(setter(Builder::command))
            .constructor(SessionCommand::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Command").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(CreateSessionRequest::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Integer> IDLE_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("IdleTimeout").getter(getter(CreateSessionRequest::idleTimeout)).setter(setter(Builder::idleTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdleTimeout").build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("DefaultArguments")
            .getter(getter(CreateSessionRequest::defaultArguments))
            .setter(setter(Builder::defaultArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultArguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ConnectionsList> CONNECTIONS_FIELD = SdkField
            .<ConnectionsList> builder(MarshallingType.SDK_POJO).memberName("Connections")
            .getter(getter(CreateSessionRequest::connections)).setter(setter(Builder::connections))
            .constructor(ConnectionsList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Connections").build()).build();

    private static final SdkField<Double> MAX_CAPACITY_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("MaxCapacity").getter(getter(CreateSessionRequest::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<Integer> NUMBER_OF_WORKERS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumberOfWorkers").getter(getter(CreateSessionRequest::numberOfWorkers))
            .setter(setter(Builder::numberOfWorkers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfWorkers").build()).build();

    private static final SdkField<String> WORKER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WorkerType").getter(getter(CreateSessionRequest::workerTypeAsString))
            .setter(setter(Builder::workerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkerType").build()).build();

    private static final SdkField<String> SECURITY_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityConfiguration").getter(getter(CreateSessionRequest::securityConfiguration))
            .setter(setter(Builder::securityConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityConfiguration").build())
            .build();

    private static final SdkField<String> GLUE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GlueVersion").getter(getter(CreateSessionRequest::glueVersion)).setter(setter(Builder::glueVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlueVersion").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateSessionRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> REQUEST_ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequestOrigin").getter(getter(CreateSessionRequest::requestOrigin))
            .setter(setter(Builder::requestOrigin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestOrigin").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DESCRIPTION_FIELD,
            ROLE_FIELD, COMMAND_FIELD, TIMEOUT_FIELD, IDLE_TIMEOUT_FIELD, DEFAULT_ARGUMENTS_FIELD, CONNECTIONS_FIELD,
            MAX_CAPACITY_FIELD, NUMBER_OF_WORKERS_FIELD, WORKER_TYPE_FIELD, SECURITY_CONFIGURATION_FIELD, GLUE_VERSION_FIELD,
            TAGS_FIELD, REQUEST_ORIGIN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String id;

    private final String description;

    private final String role;

    private final SessionCommand command;

    private final Integer timeout;

    private final Integer idleTimeout;

    private final Map<String, String> defaultArguments;

    private final ConnectionsList connections;

    private final Double maxCapacity;

    private final Integer numberOfWorkers;

    private final String workerType;

    private final String securityConfiguration;

    private final String glueVersion;

    private final Map<String, String> tags;

    private final String requestOrigin;

    private CreateSessionRequest(BuilderImpl builder) {
        super(builder);
        this.id = builder.id;
        this.description = builder.description;
        this.role = builder.role;
        this.command = builder.command;
        this.timeout = builder.timeout;
        this.idleTimeout = builder.idleTimeout;
        this.defaultArguments = builder.defaultArguments;
        this.connections = builder.connections;
        this.maxCapacity = builder.maxCapacity;
        this.numberOfWorkers = builder.numberOfWorkers;
        this.workerType = builder.workerType;
        this.securityConfiguration = builder.securityConfiguration;
        this.glueVersion = builder.glueVersion;
        this.tags = builder.tags;
        this.requestOrigin = builder.requestOrigin;
    }

    /**
     * <p>
     * The ID of the session request.
     * </p>
     * 
     * @return The ID of the session request.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The description of the session.
     * </p>
     * 
     * @return The description of the session.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The IAM Role ARN
     * </p>
     * 
     * @return The IAM Role ARN
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * The <code>SessionCommand</code> that runs the job.
     * </p>
     * 
     * @return The <code>SessionCommand</code> that runs the job.
     */
    public final SessionCommand command() {
        return command;
    }

    /**
     * <p>
     * The number of minutes before session times out. Default for Spark ETL jobs is 48 hours (2880 minutes). Consult
     * the documentation for other job types.
     * </p>
     * 
     * @return The number of minutes before session times out. Default for Spark ETL jobs is 48 hours (2880 minutes).
     *         Consult the documentation for other job types.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * <p>
     * The number of minutes when idle before session times out. Default for Spark ETL jobs is value of Timeout. Consult
     * the documentation for other job types.
     * </p>
     * 
     * @return The number of minutes when idle before session times out. Default for Spark ETL jobs is value of Timeout.
     *         Consult the documentation for other job types.
     */
    public final Integer idleTimeout() {
        return idleTimeout;
    }

    /**
     * For responses, this returns true if the service returned a value for the DefaultArguments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDefaultArguments() {
        return defaultArguments != null && !(defaultArguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map array of key-value pairs. Max is 75 pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDefaultArguments} method.
     * </p>
     * 
     * @return A map array of key-value pairs. Max is 75 pairs.
     */
    public final Map<String, String> defaultArguments() {
        return defaultArguments;
    }

    /**
     * <p>
     * The number of connections to use for the session.
     * </p>
     * 
     * @return The number of connections to use for the session.
     */
    public final ConnectionsList connections() {
        return connections;
    }

    /**
     * <p>
     * The number of Glue data processing units (DPUs) that can be allocated when the job runs. A DPU is a relative
     * measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB memory.
     * </p>
     * 
     * @return The number of Glue data processing units (DPUs) that can be allocated when the job runs. A DPU is a
     *         relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB memory.
     */
    public final Double maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The number of workers of a defined <code>WorkerType</code> to use for the session.
     * </p>
     * 
     * @return The number of workers of a defined <code>WorkerType</code> to use for the session.
     */
    public final Integer numberOfWorkers() {
        return numberOfWorkers;
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or G.8X for
     * Spark jobs. Accepts the value Z.2X for Ray notebooks.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk, and
     * provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N. Virginia), US
     * West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe
     * (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk,
     * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the <code>G.4X</code> worker
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB disk,
     * and provides up to 8 Ray workers based on the autoscaler.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or
     *         G.8X for Spark jobs. Accepts the value Z.2X for Ray notebooks.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US
     *         East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific
     *         (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
     *         <code>G.4X</code> worker type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
     *         disk, and provides up to 8 Ray workers based on the autoscaler.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final WorkerType workerType() {
        return WorkerType.fromValue(workerType);
    }

    /**
     * <p>
     * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or G.8X for
     * Spark jobs. Accepts the value Z.2X for Ray notebooks.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk, and
     * provides 1 executor per worker. We recommend this worker type for workloads such as data transforms, joins, and
     * queries, to offers a scalable and cost effective way to run most jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk, and
     * provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N. Virginia), US
     * West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe
     * (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB disk,
     * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
     * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version 3.0
     * or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the <code>G.4X</code> worker
     * type.
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB disk,
     * and provides up to 8 Ray workers based on the autoscaler.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #workerType} will
     * return {@link WorkerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #workerTypeAsString}.
     * </p>
     * 
     * @return The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or
     *         G.8X for Spark jobs. Accepts the value Z.2X for Ray notebooks.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
     *         transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US
     *         East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific
     *         (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
     *         disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain
     *         your most demanding transforms, aggregations, joins, and queries. This worker type is available only for
     *         Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
     *         <code>G.4X</code> worker type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
     *         disk, and provides up to 8 Ray workers based on the autoscaler.
     *         </p>
     *         </li>
     * @see WorkerType
     */
    public final String workerTypeAsString() {
        return workerType;
    }

    /**
     * <p>
     * The name of the SecurityConfiguration structure to be used with the session
     * </p>
     * 
     * @return The name of the SecurityConfiguration structure to be used with the session
     */
    public final String securityConfiguration() {
        return securityConfiguration;
    }

    /**
     * <p>
     * The Glue version determines the versions of Apache Spark and Python that Glue supports. The GlueVersion must be
     * greater than 2.0.
     * </p>
     * 
     * @return The Glue version determines the versions of Apache Spark and Python that Glue supports. The GlueVersion
     *         must be greater than 2.0.
     */
    public final String glueVersion() {
        return glueVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The map of key value pairs (tags) belonging to the session.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The map of key value pairs (tags) belonging to the session.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The origin of the request.
     * </p>
     * 
     * @return The origin of the request.
     */
    public final String requestOrigin() {
        return requestOrigin;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(idleTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(hasDefaultArguments() ? defaultArguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connections());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfWorkers());
        hashCode = 31 * hashCode + Objects.hashCode(workerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(securityConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(glueVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requestOrigin());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSessionRequest)) {
            return false;
        }
        CreateSessionRequest other = (CreateSessionRequest) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(description(), other.description())
                && Objects.equals(role(), other.role()) && Objects.equals(command(), other.command())
                && Objects.equals(timeout(), other.timeout()) && Objects.equals(idleTimeout(), other.idleTimeout())
                && hasDefaultArguments() == other.hasDefaultArguments()
                && Objects.equals(defaultArguments(), other.defaultArguments())
                && Objects.equals(connections(), other.connections()) && Objects.equals(maxCapacity(), other.maxCapacity())
                && Objects.equals(numberOfWorkers(), other.numberOfWorkers())
                && Objects.equals(workerTypeAsString(), other.workerTypeAsString())
                && Objects.equals(securityConfiguration(), other.securityConfiguration())
                && Objects.equals(glueVersion(), other.glueVersion()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(requestOrigin(), other.requestOrigin());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateSessionRequest").add("Id", id()).add("Description", description()).add("Role", role())
                .add("Command", command()).add("Timeout", timeout()).add("IdleTimeout", idleTimeout())
                .add("DefaultArguments", hasDefaultArguments() ? defaultArguments() : null).add("Connections", connections())
                .add("MaxCapacity", maxCapacity()).add("NumberOfWorkers", numberOfWorkers())
                .add("WorkerType", workerTypeAsString()).add("SecurityConfiguration", securityConfiguration())
                .add("GlueVersion", glueVersion()).add("Tags", hasTags() ? tags() : null).add("RequestOrigin", requestOrigin())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "Command":
            return Optional.ofNullable(clazz.cast(command()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "IdleTimeout":
            return Optional.ofNullable(clazz.cast(idleTimeout()));
        case "DefaultArguments":
            return Optional.ofNullable(clazz.cast(defaultArguments()));
        case "Connections":
            return Optional.ofNullable(clazz.cast(connections()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "NumberOfWorkers":
            return Optional.ofNullable(clazz.cast(numberOfWorkers()));
        case "WorkerType":
            return Optional.ofNullable(clazz.cast(workerTypeAsString()));
        case "SecurityConfiguration":
            return Optional.ofNullable(clazz.cast(securityConfiguration()));
        case "GlueVersion":
            return Optional.ofNullable(clazz.cast(glueVersion()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "RequestOrigin":
            return Optional.ofNullable(clazz.cast(requestOrigin()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Id", ID_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Role", ROLE_FIELD);
        map.put("Command", COMMAND_FIELD);
        map.put("Timeout", TIMEOUT_FIELD);
        map.put("IdleTimeout", IDLE_TIMEOUT_FIELD);
        map.put("DefaultArguments", DEFAULT_ARGUMENTS_FIELD);
        map.put("Connections", CONNECTIONS_FIELD);
        map.put("MaxCapacity", MAX_CAPACITY_FIELD);
        map.put("NumberOfWorkers", NUMBER_OF_WORKERS_FIELD);
        map.put("WorkerType", WORKER_TYPE_FIELD);
        map.put("SecurityConfiguration", SECURITY_CONFIGURATION_FIELD);
        map.put("GlueVersion", GLUE_VERSION_FIELD);
        map.put("Tags", TAGS_FIELD);
        map.put("RequestOrigin", REQUEST_ORIGIN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateSessionRequest, T> g) {
        return obj -> g.apply((CreateSessionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GlueRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateSessionRequest> {
        /**
         * <p>
         * The ID of the session request.
         * </p>
         * 
         * @param id
         *        The ID of the session request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The description of the session.
         * </p>
         * 
         * @param description
         *        The description of the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IAM Role ARN
         * </p>
         * 
         * @param role
         *        The IAM Role ARN
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * The <code>SessionCommand</code> that runs the job.
         * </p>
         * 
         * @param command
         *        The <code>SessionCommand</code> that runs the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(SessionCommand command);

        /**
         * <p>
         * The <code>SessionCommand</code> that runs the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link SessionCommand.Builder} avoiding the need
         * to create one manually via {@link SessionCommand#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SessionCommand.Builder#build()} is called immediately and its
         * result is passed to {@link #command(SessionCommand)}.
         * 
         * @param command
         *        a consumer that will call methods on {@link SessionCommand.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #command(SessionCommand)
         */
        default Builder command(Consumer<SessionCommand.Builder> command) {
            return command(SessionCommand.builder().applyMutation(command).build());
        }

        /**
         * <p>
         * The number of minutes before session times out. Default for Spark ETL jobs is 48 hours (2880 minutes).
         * Consult the documentation for other job types.
         * </p>
         * 
         * @param timeout
         *        The number of minutes before session times out. Default for Spark ETL jobs is 48 hours (2880 minutes).
         *        Consult the documentation for other job types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * The number of minutes when idle before session times out. Default for Spark ETL jobs is value of Timeout.
         * Consult the documentation for other job types.
         * </p>
         * 
         * @param idleTimeout
         *        The number of minutes when idle before session times out. Default for Spark ETL jobs is value of
         *        Timeout. Consult the documentation for other job types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleTimeout(Integer idleTimeout);

        /**
         * <p>
         * A map array of key-value pairs. Max is 75 pairs.
         * </p>
         * 
         * @param defaultArguments
         *        A map array of key-value pairs. Max is 75 pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultArguments(Map<String, String> defaultArguments);

        /**
         * <p>
         * The number of connections to use for the session.
         * </p>
         * 
         * @param connections
         *        The number of connections to use for the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connections(ConnectionsList connections);

        /**
         * <p>
         * The number of connections to use for the session.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConnectionsList.Builder} avoiding the
         * need to create one manually via {@link ConnectionsList#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectionsList.Builder#build()} is called immediately and its
         * result is passed to {@link #connections(ConnectionsList)}.
         * 
         * @param connections
         *        a consumer that will call methods on {@link ConnectionsList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connections(ConnectionsList)
         */
        default Builder connections(Consumer<ConnectionsList.Builder> connections) {
            return connections(ConnectionsList.builder().applyMutation(connections).build());
        }

        /**
         * <p>
         * The number of Glue data processing units (DPUs) that can be allocated when the job runs. A DPU is a relative
         * measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB memory.
         * </p>
         * 
         * @param maxCapacity
         *        The number of Glue data processing units (DPUs) that can be allocated when the job runs. A DPU is a
         *        relative measure of processing power that consists of 4 vCPUs of compute capacity and 16 GB memory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Double maxCapacity);

        /**
         * <p>
         * The number of workers of a defined <code>WorkerType</code> to use for the session.
         * </p>
         * 
         * @param numberOfWorkers
         *        The number of workers of a defined <code>WorkerType</code> to use for the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfWorkers(Integer numberOfWorkers);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or G.8X
         * for Spark jobs. Accepts the value Z.2X for Ray notebooks.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk,
         * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
         * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version
         * 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N.
         * Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada
         * (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
         * disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your
         * most demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue
         * version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
         * <code>G.4X</code> worker type.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
         * disk, and provides up to 8 Ray workers based on the autoscaler.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
         *        or G.8X for Spark jobs. Accepts the value Z.2X for Ray notebooks.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with
         *        256GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services
         *        Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia
         *        Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and
         *        Europe (Stockholm).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with
         *        512GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions
         *        as supported for the <code>G.4X</code> worker type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128
         *        GB disk, and provides up to 8 Ray workers based on the autoscaler.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(String workerType);

        /**
         * <p>
         * The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X, or G.8X
         * for Spark jobs. Accepts the value Z.2X for Ray notebooks.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB disk,
         * and provides 1 executor per worker. We recommend this worker type for workloads such as data transforms,
         * joins, and queries, to offers a scalable and cost effective way to run most jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with 256GB disk,
         * and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your most
         * demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue version
         * 3.0 or later Spark ETL jobs in the following Amazon Web Services Regions: US East (Ohio), US East (N.
         * Virginia), US West (Oregon), Asia Pacific (Singapore), Asia Pacific (Sydney), Asia Pacific (Tokyo), Canada
         * (Central), Europe (Frankfurt), Europe (Ireland), and Europe (Stockholm).
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with 512GB
         * disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads contain your
         * most demanding transforms, aggregations, joins, and queries. This worker type is available only for Glue
         * version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions as supported for the
         * <code>G.4X</code> worker type.
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128 GB
         * disk, and provides up to 8 Ray workers based on the autoscaler.
         * </p>
         * </li>
         * </ul>
         * 
         * @param workerType
         *        The type of predefined worker that is allocated when a job runs. Accepts a value of G.1X, G.2X, G.4X,
         *        or G.8X for Spark jobs. Accepts the value Z.2X for Ray notebooks.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>G.1X</code> worker type, each worker maps to 1 DPU (4 vCPUs, 16 GB of memory) with 94GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.2X</code> worker type, each worker maps to 2 DPU (8 vCPUs, 32 GB of memory) with 138GB
         *        disk, and provides 1 executor per worker. We recommend this worker type for workloads such as data
         *        transforms, joins, and queries, to offers a scalable and cost effective way to run most jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.4X</code> worker type, each worker maps to 4 DPU (16 vCPUs, 64 GB of memory) with
         *        256GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs in the following Amazon Web Services
         *        Regions: US East (Ohio), US East (N. Virginia), US West (Oregon), Asia Pacific (Singapore), Asia
         *        Pacific (Sydney), Asia Pacific (Tokyo), Canada (Central), Europe (Frankfurt), Europe (Ireland), and
         *        Europe (Stockholm).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>G.8X</code> worker type, each worker maps to 8 DPU (32 vCPUs, 128 GB of memory) with
         *        512GB disk, and provides 1 executor per worker. We recommend this worker type for jobs whose workloads
         *        contain your most demanding transforms, aggregations, joins, and queries. This worker type is
         *        available only for Glue version 3.0 or later Spark ETL jobs, in the same Amazon Web Services Regions
         *        as supported for the <code>G.4X</code> worker type.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>Z.2X</code> worker type, each worker maps to 2 M-DPU (8vCPUs, 64 GB of memory) with 128
         *        GB disk, and provides up to 8 Ray workers based on the autoscaler.
         *        </p>
         *        </li>
         * @see WorkerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkerType
         */
        Builder workerType(WorkerType workerType);

        /**
         * <p>
         * The name of the SecurityConfiguration structure to be used with the session
         * </p>
         * 
         * @param securityConfiguration
         *        The name of the SecurityConfiguration structure to be used with the session
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityConfiguration(String securityConfiguration);

        /**
         * <p>
         * The Glue version determines the versions of Apache Spark and Python that Glue supports. The GlueVersion must
         * be greater than 2.0.
         * </p>
         * 
         * @param glueVersion
         *        The Glue version determines the versions of Apache Spark and Python that Glue supports. The
         *        GlueVersion must be greater than 2.0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueVersion(String glueVersion);

        /**
         * <p>
         * The map of key value pairs (tags) belonging to the session.
         * </p>
         * 
         * @param tags
         *        The map of key value pairs (tags) belonging to the session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The origin of the request.
         * </p>
         * 
         * @param requestOrigin
         *        The origin of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestOrigin(String requestOrigin);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlueRequest.BuilderImpl implements Builder {
        private String id;

        private String description;

        private String role;

        private SessionCommand command;

        private Integer timeout;

        private Integer idleTimeout;

        private Map<String, String> defaultArguments = DefaultSdkAutoConstructMap.getInstance();

        private ConnectionsList connections;

        private Double maxCapacity;

        private Integer numberOfWorkers;

        private String workerType;

        private String securityConfiguration;

        private String glueVersion;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String requestOrigin;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSessionRequest model) {
            super(model);
            id(model.id);
            description(model.description);
            role(model.role);
            command(model.command);
            timeout(model.timeout);
            idleTimeout(model.idleTimeout);
            defaultArguments(model.defaultArguments);
            connections(model.connections);
            maxCapacity(model.maxCapacity);
            numberOfWorkers(model.numberOfWorkers);
            workerType(model.workerType);
            securityConfiguration(model.securityConfiguration);
            glueVersion(model.glueVersion);
            tags(model.tags);
            requestOrigin(model.requestOrigin);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRole() {
            return role;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final SessionCommand.Builder getCommand() {
            return command != null ? command.toBuilder() : null;
        }

        public final void setCommand(SessionCommand.BuilderImpl command) {
            this.command = command != null ? command.build() : null;
        }

        @Override
        public final Builder command(SessionCommand command) {
            this.command = command;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Integer getIdleTimeout() {
            return idleTimeout;
        }

        public final void setIdleTimeout(Integer idleTimeout) {
            this.idleTimeout = idleTimeout;
        }

        @Override
        public final Builder idleTimeout(Integer idleTimeout) {
            this.idleTimeout = idleTimeout;
            return this;
        }

        public final Map<String, String> getDefaultArguments() {
            if (defaultArguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return defaultArguments;
        }

        public final void setDefaultArguments(Map<String, String> defaultArguments) {
            this.defaultArguments = OrchestrationArgumentsMapCopier.copy(defaultArguments);
        }

        @Override
        public final Builder defaultArguments(Map<String, String> defaultArguments) {
            this.defaultArguments = OrchestrationArgumentsMapCopier.copy(defaultArguments);
            return this;
        }

        public final ConnectionsList.Builder getConnections() {
            return connections != null ? connections.toBuilder() : null;
        }

        public final void setConnections(ConnectionsList.BuilderImpl connections) {
            this.connections = connections != null ? connections.build() : null;
        }

        @Override
        public final Builder connections(ConnectionsList connections) {
            this.connections = connections;
            return this;
        }

        public final Double getMaxCapacity() {
            return maxCapacity;
        }

        public final void setMaxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Double maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final Integer getNumberOfWorkers() {
            return numberOfWorkers;
        }

        public final void setNumberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
        }

        @Override
        public final Builder numberOfWorkers(Integer numberOfWorkers) {
            this.numberOfWorkers = numberOfWorkers;
            return this;
        }

        public final String getWorkerType() {
            return workerType;
        }

        public final void setWorkerType(String workerType) {
            this.workerType = workerType;
        }

        @Override
        public final Builder workerType(String workerType) {
            this.workerType = workerType;
            return this;
        }

        @Override
        public final Builder workerType(WorkerType workerType) {
            this.workerType(workerType == null ? null : workerType.toString());
            return this;
        }

        public final String getSecurityConfiguration() {
            return securityConfiguration;
        }

        public final void setSecurityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
        }

        @Override
        public final Builder securityConfiguration(String securityConfiguration) {
            this.securityConfiguration = securityConfiguration;
            return this;
        }

        public final String getGlueVersion() {
            return glueVersion;
        }

        public final void setGlueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
        }

        @Override
        public final Builder glueVersion(String glueVersion) {
            this.glueVersion = glueVersion;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
            return this;
        }

        public final String getRequestOrigin() {
            return requestOrigin;
        }

        public final void setRequestOrigin(String requestOrigin) {
            this.requestOrigin = requestOrigin;
        }

        @Override
        public final Builder requestOrigin(String requestOrigin) {
            this.requestOrigin = requestOrigin;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSessionRequest build() {
            return new CreateSessionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
