/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.grafana;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.grafana.model.AccessDeniedException;
import software.amazon.awssdk.services.grafana.model.AssociateLicenseRequest;
import software.amazon.awssdk.services.grafana.model.AssociateLicenseResponse;
import software.amazon.awssdk.services.grafana.model.ConflictException;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceApiKeyRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceApiKeyResponse;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceApiKeyRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceApiKeyResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceAuthenticationRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceAuthenticationResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceConfigurationRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceConfigurationResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.DisassociateLicenseRequest;
import software.amazon.awssdk.services.grafana.model.DisassociateLicenseResponse;
import software.amazon.awssdk.services.grafana.model.GrafanaException;
import software.amazon.awssdk.services.grafana.model.GrafanaRequest;
import software.amazon.awssdk.services.grafana.model.InternalServerException;
import software.amazon.awssdk.services.grafana.model.ListPermissionsRequest;
import software.amazon.awssdk.services.grafana.model.ListPermissionsResponse;
import software.amazon.awssdk.services.grafana.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.grafana.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.grafana.model.ListWorkspacesRequest;
import software.amazon.awssdk.services.grafana.model.ListWorkspacesResponse;
import software.amazon.awssdk.services.grafana.model.ResourceNotFoundException;
import software.amazon.awssdk.services.grafana.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.grafana.model.TagResourceRequest;
import software.amazon.awssdk.services.grafana.model.TagResourceResponse;
import software.amazon.awssdk.services.grafana.model.ThrottlingException;
import software.amazon.awssdk.services.grafana.model.UntagResourceRequest;
import software.amazon.awssdk.services.grafana.model.UntagResourceResponse;
import software.amazon.awssdk.services.grafana.model.UpdatePermissionsRequest;
import software.amazon.awssdk.services.grafana.model.UpdatePermissionsResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceAuthenticationRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceAuthenticationResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceConfigurationRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceConfigurationResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.ValidationException;
import software.amazon.awssdk.services.grafana.paginators.ListPermissionsIterable;
import software.amazon.awssdk.services.grafana.paginators.ListWorkspacesIterable;
import software.amazon.awssdk.services.grafana.transform.AssociateLicenseRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceApiKeyRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceApiKeyRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DisassociateLicenseRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListWorkspacesRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdatePermissionsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GrafanaClient}.
 *
 * @see GrafanaClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGrafanaClient implements GrafanaClient {
    private static final Logger log = Logger.loggerFor(DefaultGrafanaClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final GrafanaServiceClientConfiguration serviceClientConfiguration;

    protected DefaultGrafanaClient(GrafanaServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Assigns a Grafana Enterprise license to a workspace. Upgrading to Grafana Enterprise incurs additional fees. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html">Upgrade a
     * workspace to Grafana Enterprise</a>.
     * </p>
     *
     * @param associateLicenseRequest
     * @return Result of the AssociateLicense operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.AssociateLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/AssociateLicense" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AssociateLicenseResponse associateLicense(AssociateLicenseRequest associateLicenseRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AssociateLicenseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateLicense");

            return clientHandler.execute(new ClientExecutionParams<AssociateLicenseRequest, AssociateLicenseResponse>()
                    .withOperationName("AssociateLicense").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateLicenseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateLicenseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <i>workspace</i>. In a workspace, you can create Grafana dashboards and visualizations to analyze your
     * metrics, logs, and traces. You don't have to build, package, or deploy any hardware to run the Grafana server.
     * </p>
     * <p>
     * Don't use <code>CreateWorkspace</code> to modify an existing workspace. Instead, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspace.html">UpdateWorkspace</a>.
     * </p>
     *
     * @param createWorkspaceRequest
     * @return Result of the CreateWorkspace operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.CreateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateWorkspaceResponse createWorkspace(CreateWorkspaceRequest createWorkspaceRequest) throws ThrottlingException,
            ConflictException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspace");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkspaceRequest, CreateWorkspaceResponse>()
                    .withOperationName("CreateWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Grafana API key for the workspace. This key can be used to authenticate requests sent to the
     * workspace's HTTP API. See <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/Using-Grafana-APIs.html">https
     * ://docs.aws.amazon.com/grafana/latest/userguide/Using-Grafana-APIs.html</a> for available APIs and example
     * requests.
     * </p>
     *
     * @param createWorkspaceApiKeyRequest
     * @return Result of the CreateWorkspaceApiKey operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.CreateWorkspaceApiKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspaceApiKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateWorkspaceApiKeyResponse createWorkspaceApiKey(CreateWorkspaceApiKeyRequest createWorkspaceApiKeyRequest)
            throws ResourceNotFoundException, ThrottlingException, ConflictException, ValidationException, AccessDeniedException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkspaceApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWorkspaceApiKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspaceApiKey");

            return clientHandler.execute(new ClientExecutionParams<CreateWorkspaceApiKeyRequest, CreateWorkspaceApiKeyResponse>()
                    .withOperationName("CreateWorkspaceApiKey").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createWorkspaceApiKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateWorkspaceApiKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Managed Grafana workspace.
     * </p>
     *
     * @param deleteWorkspaceRequest
     * @return Result of the DeleteWorkspace operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DeleteWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteWorkspaceResponse deleteWorkspace(DeleteWorkspaceRequest deleteWorkspaceRequest)
            throws ResourceNotFoundException, ThrottlingException, ConflictException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspace");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkspaceRequest, DeleteWorkspaceResponse>()
                    .withOperationName("DeleteWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Grafana API key for the workspace.
     * </p>
     *
     * @param deleteWorkspaceApiKeyRequest
     * @return Result of the DeleteWorkspaceApiKey operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DeleteWorkspaceApiKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspaceApiKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteWorkspaceApiKeyResponse deleteWorkspaceApiKey(DeleteWorkspaceApiKeyRequest deleteWorkspaceApiKeyRequest)
            throws ResourceNotFoundException, ThrottlingException, ConflictException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkspaceApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteWorkspaceApiKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspaceApiKey");

            return clientHandler.execute(new ClientExecutionParams<DeleteWorkspaceApiKeyRequest, DeleteWorkspaceApiKeyResponse>()
                    .withOperationName("DeleteWorkspaceApiKey").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteWorkspaceApiKeyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteWorkspaceApiKeyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays information about one Amazon Managed Grafana workspace.
     * </p>
     *
     * @param describeWorkspaceRequest
     * @return Result of the DescribeWorkspace operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DescribeWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeWorkspaceResponse describeWorkspace(DescribeWorkspaceRequest describeWorkspaceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspace");

            return clientHandler.execute(new ClientExecutionParams<DescribeWorkspaceRequest, DescribeWorkspaceResponse>()
                    .withOperationName("DescribeWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays information about the authentication methods used in one Amazon Managed Grafana workspace.
     * </p>
     *
     * @param describeWorkspaceAuthenticationRequest
     * @return Result of the DescribeWorkspaceAuthentication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DescribeWorkspaceAuthentication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspaceAuthentication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeWorkspaceAuthenticationResponse describeWorkspaceAuthentication(
            DescribeWorkspaceAuthenticationRequest describeWorkspaceAuthenticationRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWorkspaceAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeWorkspaceAuthenticationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeWorkspaceAuthenticationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspaceAuthentication");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceAuthenticationRequest, DescribeWorkspaceAuthenticationResponse>()
                            .withOperationName("DescribeWorkspaceAuthentication").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeWorkspaceAuthenticationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeWorkspaceAuthenticationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the current configuration string for the given workspace.
     * </p>
     *
     * @param describeWorkspaceConfigurationRequest
     * @return Result of the DescribeWorkspaceConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DescribeWorkspaceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspaceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeWorkspaceConfigurationResponse describeWorkspaceConfiguration(
            DescribeWorkspaceConfigurationRequest describeWorkspaceConfigurationRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWorkspaceConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeWorkspaceConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeWorkspaceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspaceConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceConfigurationRequest, DescribeWorkspaceConfigurationResponse>()
                            .withOperationName("DescribeWorkspaceConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeWorkspaceConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeWorkspaceConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the Grafana Enterprise license from a workspace.
     * </p>
     *
     * @param disassociateLicenseRequest
     * @return Result of the DisassociateLicense operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.DisassociateLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DisassociateLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateLicenseResponse disassociateLicense(DisassociateLicenseRequest disassociateLicenseRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateLicenseResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateLicense");

            return clientHandler.execute(new ClientExecutionParams<DisassociateLicenseRequest, DisassociateLicenseResponse>()
                    .withOperationName("DisassociateLicense").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disassociateLicenseRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisassociateLicenseRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the users and groups who have the Grafana <code>Admin</code> and <code>Editor</code> roles in this
     * workspace. If you use this operation without specifying <code>userId</code> or <code>groupId</code>, the
     * operation returns the roles of all users and groups. If you specify a <code>userId</code> or a
     * <code>groupId</code>, only the roles for that user or group are returned. If you do this, you can specify only
     * one <code>userId</code> or one <code>groupId</code>.
     * </p>
     *
     * @param listPermissionsRequest
     * @return Result of the ListPermissions operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPermissionsResponse listPermissions(ListPermissionsRequest listPermissionsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");

            return clientHandler.execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                    .withOperationName("ListPermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the users and groups who have the Grafana <code>Admin</code> and <code>Editor</code> roles in this
     * workspace. If you use this operation without specifying <code>userId</code> or <code>groupId</code>, the
     * operation returns the roles of all users and groups. If you specify a <code>userId</code> or a
     * <code>groupId</code>, only the roles for that user or group are returned. If you do this, you can specify only
     * one <code>userId</code> or one <code>groupId</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPermissions(software.amazon.awssdk.services.grafana.model.ListPermissionsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.grafana.paginators.ListPermissionsIterable responses = client.listPermissionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.grafana.paginators.ListPermissionsIterable responses = client
     *             .listPermissionsPaginator(request);
     *     for (software.amazon.awssdk.services.grafana.model.ListPermissionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.grafana.paginators.ListPermissionsIterable responses = client.listPermissionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissions(software.amazon.awssdk.services.grafana.model.ListPermissionsRequest)} operation.</b>
     * </p>
     *
     * @param listPermissionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPermissionsIterable listPermissionsPaginator(ListPermissionsRequest listPermissionsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        return new ListPermissionsIterable(this, applyPaginatorUserAgent(listPermissionsRequest));
    }

    /**
     * <p>
     * The <code>ListTagsForResource</code> operation returns the tags that are associated with the Amazon Managed
     * Service for Grafana resource specified by the <code>resourceArn</code>. Currently, the only resource that can be
     * tagged is a workspace.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Amazon Managed Grafana workspaces in the account, with some information about each workspace.
     * For more complete information about one workspace, use <a
     * href="https://docs.aws.amazon.com/AAMG/latest/APIReference/API_DescribeWorkspace.html">DescribeWorkspace</a>.
     * </p>
     *
     * @param listWorkspacesRequest
     * @return Result of the ListWorkspaces operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWorkspacesResponse listWorkspaces(ListWorkspacesRequest listWorkspacesRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkspacesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListWorkspacesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaces");

            return clientHandler.execute(new ClientExecutionParams<ListWorkspacesRequest, ListWorkspacesResponse>()
                    .withOperationName("ListWorkspaces").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listWorkspacesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListWorkspacesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of Amazon Managed Grafana workspaces in the account, with some information about each workspace.
     * For more complete information about one workspace, use <a
     * href="https://docs.aws.amazon.com/AAMG/latest/APIReference/API_DescribeWorkspace.html">DescribeWorkspace</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkspaces(software.amazon.awssdk.services.grafana.model.ListWorkspacesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.grafana.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.grafana.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     *     for (software.amazon.awssdk.services.grafana.model.ListWorkspacesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.grafana.paginators.ListWorkspacesIterable responses = client.listWorkspacesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkspaces(software.amazon.awssdk.services.grafana.model.ListWorkspacesRequest)} operation.</b>
     * </p>
     *
     * @param listWorkspacesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListWorkspacesIterable listWorkspacesPaginator(ListWorkspacesRequest listWorkspacesRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            GrafanaException {
        return new ListWorkspacesIterable(this, applyPaginatorUserAgent(listWorkspacesRequest));
    }

    /**
     * <p>
     * The <code>TagResource</code> operation associates tags with an Amazon Managed Grafana resource. Currently, the
     * only resource that can be tagged is workspaces.
     * </p>
     * <p>
     * If you specify a new tag key for the resource, this tag is appended to the list of tags associated with the
     * resource. If you specify a tag key that is already associated with the resource, the new tag value that you
     * specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UntagResource</code> operation removes the association of the tag with the Amazon Managed Grafana
     * resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates which users in a workspace have the Grafana <code>Admin</code> or <code>Editor</code> roles.
     * </p>
     *
     * @param updatePermissionsRequest
     * @return Result of the UpdatePermissions operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.UpdatePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdatePermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePermissionsResponse updatePermissions(UpdatePermissionsRequest updatePermissionsRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePermissionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePermissions");

            return clientHandler.execute(new ClientExecutionParams<UpdatePermissionsRequest, UpdatePermissionsResponse>()
                    .withOperationName("UpdatePermissions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updatePermissionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePermissionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies an existing Amazon Managed Grafana workspace. If you use this operation and omit any optional
     * parameters, the existing values of those parameters are not changed.
     * </p>
     * <p>
     * To modify the user authentication methods that the workspace uses, such as SAML or IAM Identity Center, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspaceAuthentication.html">
     * UpdateWorkspaceAuthentication</a>.
     * </p>
     * <p>
     * To modify which users in the workspace have the <code>Admin</code> and <code>Editor</code> Grafana roles, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdatePermissions.html">UpdatePermissions</a>.
     * </p>
     *
     * @param updateWorkspaceRequest
     * @return Result of the UpdateWorkspace operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.UpdateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateWorkspaceResponse updateWorkspace(UpdateWorkspaceRequest updateWorkspaceRequest)
            throws ResourceNotFoundException, ThrottlingException, ConflictException, ValidationException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateWorkspaceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspace");

            return clientHandler.execute(new ClientExecutionParams<UpdateWorkspaceRequest, UpdateWorkspaceResponse>()
                    .withOperationName("UpdateWorkspace").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkspaceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateWorkspaceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to define the identity provider (IdP) that this workspace authenticates users from, using
     * SAML. You can also map SAML assertion attributes to workspace user information and define which groups in the
     * assertion attribute are to have the <code>Admin</code> and <code>Editor</code> roles in the workspace.
     * </p>
     * <note>
     * <p>
     * Changes to the authentication method for a workspace may take a few minutes to take effect.
     * </p>
     * </note>
     *
     * @param updateWorkspaceAuthenticationRequest
     * @return Result of the UpdateWorkspaceAuthentication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.UpdateWorkspaceAuthentication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspaceAuthentication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateWorkspaceAuthenticationResponse updateWorkspaceAuthentication(
            UpdateWorkspaceAuthenticationRequest updateWorkspaceAuthenticationRequest) throws ResourceNotFoundException,
            ThrottlingException, ConflictException, ValidationException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkspaceAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateWorkspaceAuthenticationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateWorkspaceAuthenticationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceAuthentication");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceAuthenticationRequest, UpdateWorkspaceAuthenticationResponse>()
                            .withOperationName("UpdateWorkspaceAuthentication").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkspaceAuthenticationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateWorkspaceAuthenticationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration string for the given workspace
     * </p>
     *
     * @param updateWorkspaceConfigurationRequest
     * @return Result of the UpdateWorkspaceConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied because of request throttling. Retry the request.
     * @throws ConflictException
     *         A resource was in an inconsistent state during an update or a deletion.
     * @throws ValidationException
     *         The value of a parameter in the request caused an error.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action.
     * @throws InternalServerException
     *         Unexpected error while processing the request. Retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GrafanaException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GrafanaClient.UpdateWorkspaceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspaceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateWorkspaceConfigurationResponse updateWorkspaceConfiguration(
            UpdateWorkspaceConfigurationRequest updateWorkspaceConfigurationRequest) throws ResourceNotFoundException,
            ThrottlingException, ConflictException, ValidationException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, GrafanaException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateWorkspaceConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateWorkspaceConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceConfigurationRequest, UpdateWorkspaceConfigurationResponse>()
                            .withOperationName("UpdateWorkspaceConfiguration").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateWorkspaceConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateWorkspaceConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends GrafanaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GrafanaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final GrafanaServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
