/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.grafana.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains some information about one workspace in the account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WorkspaceSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<WorkspaceSummary.Builder, WorkspaceSummary> {
    private static final SdkField<AuthenticationSummary> AUTHENTICATION_FIELD = SdkField
            .<AuthenticationSummary> builder(MarshallingType.SDK_POJO).memberName("authentication")
            .getter(getter(WorkspaceSummary::authentication)).setter(setter(Builder::authentication))
            .constructor(AuthenticationSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authentication").build()).build();

    private static final SdkField<Instant> CREATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("created").getter(getter(WorkspaceSummary::created)).setter(setter(Builder::created))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("created").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(WorkspaceSummary::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("endpoint").getter(getter(WorkspaceSummary::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpoint").build()).build();

    private static final SdkField<String> GRAFANA_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("grafanaToken").getter(getter(WorkspaceSummary::grafanaToken)).setter(setter(Builder::grafanaToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("grafanaToken").build()).build();

    private static final SdkField<String> GRAFANA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("grafanaVersion").getter(getter(WorkspaceSummary::grafanaVersion))
            .setter(setter(Builder::grafanaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("grafanaVersion").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(WorkspaceSummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> LICENSE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("licenseType").getter(getter(WorkspaceSummary::licenseTypeAsString)).setter(setter(Builder::licenseType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("licenseType").build()).build();

    private static final SdkField<Instant> MODIFIED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modified").getter(getter(WorkspaceSummary::modified)).setter(setter(Builder::modified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modified").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(WorkspaceSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> NOTIFICATION_DESTINATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("notificationDestinations")
            .getter(getter(WorkspaceSummary::notificationDestinationsAsStrings))
            .setter(setter(Builder::notificationDestinationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationDestinations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(WorkspaceSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(WorkspaceSummary::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTHENTICATION_FIELD,
            CREATED_FIELD, DESCRIPTION_FIELD, ENDPOINT_FIELD, GRAFANA_TOKEN_FIELD, GRAFANA_VERSION_FIELD, ID_FIELD,
            LICENSE_TYPE_FIELD, MODIFIED_FIELD, NAME_FIELD, NOTIFICATION_DESTINATIONS_FIELD, STATUS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final AuthenticationSummary authentication;

    private final Instant created;

    private final String description;

    private final String endpoint;

    private final String grafanaToken;

    private final String grafanaVersion;

    private final String id;

    private final String licenseType;

    private final Instant modified;

    private final String name;

    private final List<String> notificationDestinations;

    private final String status;

    private final Map<String, String> tags;

    private WorkspaceSummary(BuilderImpl builder) {
        this.authentication = builder.authentication;
        this.created = builder.created;
        this.description = builder.description;
        this.endpoint = builder.endpoint;
        this.grafanaToken = builder.grafanaToken;
        this.grafanaVersion = builder.grafanaVersion;
        this.id = builder.id;
        this.licenseType = builder.licenseType;
        this.modified = builder.modified;
        this.name = builder.name;
        this.notificationDestinations = builder.notificationDestinations;
        this.status = builder.status;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * A structure containing information about the authentication methods used in the workspace.
     * </p>
     * 
     * @return A structure containing information about the authentication methods used in the workspace.
     */
    public final AuthenticationSummary authentication() {
        return authentication;
    }

    /**
     * <p>
     * The date that the workspace was created.
     * </p>
     * 
     * @return The date that the workspace was created.
     */
    public final Instant created() {
        return created;
    }

    /**
     * <p>
     * The customer-entered description of the workspace.
     * </p>
     * 
     * @return The customer-entered description of the workspace.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The URL endpoint to use to access the Grafana console in the workspace.
     * </p>
     * 
     * @return The URL endpoint to use to access the Grafana console in the workspace.
     */
    public final String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The token that ties this workspace to a Grafana Labs account. For more information, see <a href=
     * "https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html#AMG-workspace-register-enterprise"
     * >Register with Grafana Labs</a>.
     * </p>
     * 
     * @return The token that ties this workspace to a Grafana Labs account. For more information, see <a href=
     *         "https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html#AMG-workspace-register-enterprise"
     *         >Register with Grafana Labs</a>.
     */
    public final String grafanaToken() {
        return grafanaToken;
    }

    /**
     * <p>
     * The Grafana version that the workspace is running.
     * </p>
     * 
     * @return The Grafana version that the workspace is running.
     */
    public final String grafanaVersion() {
        return grafanaVersion;
    }

    /**
     * <p>
     * The unique ID of the workspace.
     * </p>
     * 
     * @return The unique ID of the workspace.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * Specifies whether this workspace has a full Grafana Enterprise license.
     * </p>
     * <note>
     * <p>
     * Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #licenseType} will
     * return {@link LicenseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #licenseTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this workspace has a full Grafana Enterprise license.</p> <note>
     *         <p>
     *         Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
     *         </p>
     * @see LicenseType
     */
    public final LicenseType licenseType() {
        return LicenseType.fromValue(licenseType);
    }

    /**
     * <p>
     * Specifies whether this workspace has a full Grafana Enterprise license.
     * </p>
     * <note>
     * <p>
     * Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #licenseType} will
     * return {@link LicenseType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #licenseTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this workspace has a full Grafana Enterprise license.</p> <note>
     *         <p>
     *         Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
     *         </p>
     * @see LicenseType
     */
    public final String licenseTypeAsString() {
        return licenseType;
    }

    /**
     * <p>
     * The most recent date that the workspace was modified.
     * </p>
     * 
     * @return The most recent date that the workspace was modified.
     */
    public final Instant modified() {
        return modified;
    }

    /**
     * <p>
     * The name of the workspace.
     * </p>
     * 
     * @return The name of the workspace.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles and
     * permissions for, which allows Amazon Managed Grafana to use these channels.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationDestinations} method.
     * </p>
     * 
     * @return The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
     *         roles and permissions for, which allows Amazon Managed Grafana to use these channels.
     */
    public final List<NotificationDestinationType> notificationDestinations() {
        return NotificationDestinationsListCopier.copyStringToEnum(notificationDestinations);
    }

    /**
     * For responses, this returns true if the service returned a value for the NotificationDestinations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasNotificationDestinations() {
        return notificationDestinations != null && !(notificationDestinations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles and
     * permissions for, which allows Amazon Managed Grafana to use these channels.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationDestinations} method.
     * </p>
     * 
     * @return The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
     *         roles and permissions for, which allows Amazon Managed Grafana to use these channels.
     */
    public final List<String> notificationDestinationsAsStrings() {
        return notificationDestinations;
    }

    /**
     * <p>
     * The current status of the workspace.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkspaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the workspace.
     * @see WorkspaceStatus
     */
    public final WorkspaceStatus status() {
        return WorkspaceStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the workspace.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link WorkspaceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the workspace.
     * @see WorkspaceStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The list of tags associated with the workspace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The list of tags associated with the workspace.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(authentication());
        hashCode = 31 * hashCode + Objects.hashCode(created());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(grafanaToken());
        hashCode = 31 * hashCode + Objects.hashCode(grafanaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(licenseTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modified());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasNotificationDestinations() ? notificationDestinationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WorkspaceSummary)) {
            return false;
        }
        WorkspaceSummary other = (WorkspaceSummary) obj;
        return Objects.equals(authentication(), other.authentication()) && Objects.equals(created(), other.created())
                && Objects.equals(description(), other.description()) && Objects.equals(endpoint(), other.endpoint())
                && Objects.equals(grafanaToken(), other.grafanaToken())
                && Objects.equals(grafanaVersion(), other.grafanaVersion()) && Objects.equals(id(), other.id())
                && Objects.equals(licenseTypeAsString(), other.licenseTypeAsString())
                && Objects.equals(modified(), other.modified()) && Objects.equals(name(), other.name())
                && hasNotificationDestinations() == other.hasNotificationDestinations()
                && Objects.equals(notificationDestinationsAsStrings(), other.notificationDestinationsAsStrings())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WorkspaceSummary").add("Authentication", authentication()).add("Created", created())
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***").add("Endpoint", endpoint())
                .add("GrafanaToken", grafanaToken()).add("GrafanaVersion", grafanaVersion()).add("Id", id())
                .add("LicenseType", licenseTypeAsString()).add("Modified", modified())
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("NotificationDestinations", hasNotificationDestinations() ? notificationDestinationsAsStrings() : null)
                .add("Status", statusAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "authentication":
            return Optional.ofNullable(clazz.cast(authentication()));
        case "created":
            return Optional.ofNullable(clazz.cast(created()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "grafanaToken":
            return Optional.ofNullable(clazz.cast(grafanaToken()));
        case "grafanaVersion":
            return Optional.ofNullable(clazz.cast(grafanaVersion()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "licenseType":
            return Optional.ofNullable(clazz.cast(licenseTypeAsString()));
        case "modified":
            return Optional.ofNullable(clazz.cast(modified()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "notificationDestinations":
            return Optional.ofNullable(clazz.cast(notificationDestinationsAsStrings()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WorkspaceSummary, T> g) {
        return obj -> g.apply((WorkspaceSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WorkspaceSummary> {
        /**
         * <p>
         * A structure containing information about the authentication methods used in the workspace.
         * </p>
         * 
         * @param authentication
         *        A structure containing information about the authentication methods used in the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authentication(AuthenticationSummary authentication);

        /**
         * <p>
         * A structure containing information about the authentication methods used in the workspace.
         * </p>
         * This is a convenience method that creates an instance of the {@link AuthenticationSummary.Builder} avoiding
         * the need to create one manually via {@link AuthenticationSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AuthenticationSummary.Builder#build()} is called immediately and
         * its result is passed to {@link #authentication(AuthenticationSummary)}.
         * 
         * @param authentication
         *        a consumer that will call methods on {@link AuthenticationSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authentication(AuthenticationSummary)
         */
        default Builder authentication(Consumer<AuthenticationSummary.Builder> authentication) {
            return authentication(AuthenticationSummary.builder().applyMutation(authentication).build());
        }

        /**
         * <p>
         * The date that the workspace was created.
         * </p>
         * 
         * @param created
         *        The date that the workspace was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder created(Instant created);

        /**
         * <p>
         * The customer-entered description of the workspace.
         * </p>
         * 
         * @param description
         *        The customer-entered description of the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The URL endpoint to use to access the Grafana console in the workspace.
         * </p>
         * 
         * @param endpoint
         *        The URL endpoint to use to access the Grafana console in the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The token that ties this workspace to a Grafana Labs account. For more information, see <a href=
         * "https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html#AMG-workspace-register-enterprise"
         * >Register with Grafana Labs</a>.
         * </p>
         * 
         * @param grafanaToken
         *        The token that ties this workspace to a Grafana Labs account. For more information, see <a href=
         *        "https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html#AMG-workspace-register-enterprise"
         *        >Register with Grafana Labs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grafanaToken(String grafanaToken);

        /**
         * <p>
         * The Grafana version that the workspace is running.
         * </p>
         * 
         * @param grafanaVersion
         *        The Grafana version that the workspace is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grafanaVersion(String grafanaVersion);

        /**
         * <p>
         * The unique ID of the workspace.
         * </p>
         * 
         * @param id
         *        The unique ID of the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * Specifies whether this workspace has a full Grafana Enterprise license.
         * </p>
         * <note>
         * <p>
         * Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
         * </p>
         * </note>
         * 
         * @param licenseType
         *        Specifies whether this workspace has a full Grafana Enterprise license.</p> <note>
         *        <p>
         *        Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
         *        </p>
         * @see LicenseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseType
         */
        Builder licenseType(String licenseType);

        /**
         * <p>
         * Specifies whether this workspace has a full Grafana Enterprise license.
         * </p>
         * <note>
         * <p>
         * Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
         * </p>
         * </note>
         * 
         * @param licenseType
         *        Specifies whether this workspace has a full Grafana Enterprise license.</p> <note>
         *        <p>
         *        Amazon Managed Grafana workspaces no longer support Grafana Enterprise free trials.
         *        </p>
         * @see LicenseType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LicenseType
         */
        Builder licenseType(LicenseType licenseType);

        /**
         * <p>
         * The most recent date that the workspace was modified.
         * </p>
         * 
         * @param modified
         *        The most recent date that the workspace was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modified(Instant modified);

        /**
         * <p>
         * The name of the workspace.
         * </p>
         * 
         * @param name
         *        The name of the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, which allows Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, which allows Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinationsWithStrings(Collection<String> notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, which allows Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, which allows Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinationsWithStrings(String... notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, which allows Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, which allows Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinations(Collection<NotificationDestinationType> notificationDestinations);

        /**
         * <p>
         * The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM roles
         * and permissions for, which allows Amazon Managed Grafana to use these channels.
         * </p>
         * 
         * @param notificationDestinations
         *        The Amazon Web Services notification channels that Amazon Managed Grafana can automatically create IAM
         *        roles and permissions for, which allows Amazon Managed Grafana to use these channels.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationDestinations(NotificationDestinationType... notificationDestinations);

        /**
         * <p>
         * The current status of the workspace.
         * </p>
         * 
         * @param status
         *        The current status of the workspace.
         * @see WorkspaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the workspace.
         * </p>
         * 
         * @param status
         *        The current status of the workspace.
         * @see WorkspaceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see WorkspaceStatus
         */
        Builder status(WorkspaceStatus status);

        /**
         * <p>
         * The list of tags associated with the workspace.
         * </p>
         * 
         * @param tags
         *        The list of tags associated with the workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private AuthenticationSummary authentication;

        private Instant created;

        private String description;

        private String endpoint;

        private String grafanaToken;

        private String grafanaVersion;

        private String id;

        private String licenseType;

        private Instant modified;

        private String name;

        private List<String> notificationDestinations = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(WorkspaceSummary model) {
            authentication(model.authentication);
            created(model.created);
            description(model.description);
            endpoint(model.endpoint);
            grafanaToken(model.grafanaToken);
            grafanaVersion(model.grafanaVersion);
            id(model.id);
            licenseType(model.licenseType);
            modified(model.modified);
            name(model.name);
            notificationDestinationsWithStrings(model.notificationDestinations);
            status(model.status);
            tags(model.tags);
        }

        public final AuthenticationSummary.Builder getAuthentication() {
            return authentication != null ? authentication.toBuilder() : null;
        }

        public final void setAuthentication(AuthenticationSummary.BuilderImpl authentication) {
            this.authentication = authentication != null ? authentication.build() : null;
        }

        @Override
        public final Builder authentication(AuthenticationSummary authentication) {
            this.authentication = authentication;
            return this;
        }

        public final Instant getCreated() {
            return created;
        }

        public final void setCreated(Instant created) {
            this.created = created;
        }

        @Override
        public final Builder created(Instant created) {
            this.created = created;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final String getGrafanaToken() {
            return grafanaToken;
        }

        public final void setGrafanaToken(String grafanaToken) {
            this.grafanaToken = grafanaToken;
        }

        @Override
        public final Builder grafanaToken(String grafanaToken) {
            this.grafanaToken = grafanaToken;
            return this;
        }

        public final String getGrafanaVersion() {
            return grafanaVersion;
        }

        public final void setGrafanaVersion(String grafanaVersion) {
            this.grafanaVersion = grafanaVersion;
        }

        @Override
        public final Builder grafanaVersion(String grafanaVersion) {
            this.grafanaVersion = grafanaVersion;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getLicenseType() {
            return licenseType;
        }

        public final void setLicenseType(String licenseType) {
            this.licenseType = licenseType;
        }

        @Override
        public final Builder licenseType(String licenseType) {
            this.licenseType = licenseType;
            return this;
        }

        @Override
        public final Builder licenseType(LicenseType licenseType) {
            this.licenseType(licenseType == null ? null : licenseType.toString());
            return this;
        }

        public final Instant getModified() {
            return modified;
        }

        public final void setModified(Instant modified) {
            this.modified = modified;
        }

        @Override
        public final Builder modified(Instant modified) {
            this.modified = modified;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getNotificationDestinations() {
            if (notificationDestinations instanceof SdkAutoConstructList) {
                return null;
            }
            return notificationDestinations;
        }

        public final void setNotificationDestinations(Collection<String> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copy(notificationDestinations);
        }

        @Override
        public final Builder notificationDestinationsWithStrings(Collection<String> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copy(notificationDestinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationDestinationsWithStrings(String... notificationDestinations) {
            notificationDestinationsWithStrings(Arrays.asList(notificationDestinations));
            return this;
        }

        @Override
        public final Builder notificationDestinations(Collection<NotificationDestinationType> notificationDestinations) {
            this.notificationDestinations = NotificationDestinationsListCopier.copyEnumToString(notificationDestinations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationDestinations(NotificationDestinationType... notificationDestinations) {
            notificationDestinations(Arrays.asList(notificationDestinations));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(WorkspaceStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public WorkspaceSummary build() {
            return new WorkspaceSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
