/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.grafana;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.grafana.internal.GrafanaServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.grafana.model.AccessDeniedException;
import software.amazon.awssdk.services.grafana.model.AssociateLicenseRequest;
import software.amazon.awssdk.services.grafana.model.AssociateLicenseResponse;
import software.amazon.awssdk.services.grafana.model.ConflictException;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceApiKeyRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceApiKeyResponse;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceServiceAccountRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceServiceAccountResponse;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceServiceAccountTokenRequest;
import software.amazon.awssdk.services.grafana.model.CreateWorkspaceServiceAccountTokenResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceApiKeyRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceApiKeyResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceServiceAccountRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceServiceAccountResponse;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceServiceAccountTokenRequest;
import software.amazon.awssdk.services.grafana.model.DeleteWorkspaceServiceAccountTokenResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceAuthenticationRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceAuthenticationResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceConfigurationRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceConfigurationResponse;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.DisassociateLicenseRequest;
import software.amazon.awssdk.services.grafana.model.DisassociateLicenseResponse;
import software.amazon.awssdk.services.grafana.model.GrafanaException;
import software.amazon.awssdk.services.grafana.model.InternalServerException;
import software.amazon.awssdk.services.grafana.model.ListPermissionsRequest;
import software.amazon.awssdk.services.grafana.model.ListPermissionsResponse;
import software.amazon.awssdk.services.grafana.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.grafana.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.grafana.model.ListVersionsRequest;
import software.amazon.awssdk.services.grafana.model.ListVersionsResponse;
import software.amazon.awssdk.services.grafana.model.ListWorkspaceServiceAccountTokensRequest;
import software.amazon.awssdk.services.grafana.model.ListWorkspaceServiceAccountTokensResponse;
import software.amazon.awssdk.services.grafana.model.ListWorkspaceServiceAccountsRequest;
import software.amazon.awssdk.services.grafana.model.ListWorkspaceServiceAccountsResponse;
import software.amazon.awssdk.services.grafana.model.ListWorkspacesRequest;
import software.amazon.awssdk.services.grafana.model.ListWorkspacesResponse;
import software.amazon.awssdk.services.grafana.model.ResourceNotFoundException;
import software.amazon.awssdk.services.grafana.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.grafana.model.TagResourceRequest;
import software.amazon.awssdk.services.grafana.model.TagResourceResponse;
import software.amazon.awssdk.services.grafana.model.ThrottlingException;
import software.amazon.awssdk.services.grafana.model.UntagResourceRequest;
import software.amazon.awssdk.services.grafana.model.UntagResourceResponse;
import software.amazon.awssdk.services.grafana.model.UpdatePermissionsRequest;
import software.amazon.awssdk.services.grafana.model.UpdatePermissionsResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceAuthenticationRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceAuthenticationResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceConfigurationRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceConfigurationResponse;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceRequest;
import software.amazon.awssdk.services.grafana.model.UpdateWorkspaceResponse;
import software.amazon.awssdk.services.grafana.model.ValidationException;
import software.amazon.awssdk.services.grafana.transform.AssociateLicenseRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceApiKeyRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceServiceAccountRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.CreateWorkspaceServiceAccountTokenRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceApiKeyRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceServiceAccountRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DeleteWorkspaceServiceAccountTokenRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DescribeWorkspaceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.DisassociateLicenseRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListVersionsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListWorkspaceServiceAccountTokensRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListWorkspaceServiceAccountsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.ListWorkspacesRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdatePermissionsRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceConfigurationRequestMarshaller;
import software.amazon.awssdk.services.grafana.transform.UpdateWorkspaceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GrafanaAsyncClient}.
 *
 * @see GrafanaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGrafanaAsyncClient implements GrafanaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGrafanaAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGrafanaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Assigns a Grafana Enterprise license to a workspace. To upgrade, you must use <code>ENTERPRISE</code> for the
     * <code>licenseType</code>, and pass in a valid Grafana Labs token for the <code>grafanaToken</code>. Upgrading to
     * Grafana Enterprise incurs additional fees. For more information, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/upgrade-to-Grafana-Enterprise.html">Upgrade a
     * workspace to Grafana Enterprise</a>.
     * </p>
     *
     * @param associateLicenseRequest
     * @return A Java Future containing the result of the AssociateLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.AssociateLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/AssociateLicense" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateLicenseResponse> associateLicense(AssociateLicenseRequest associateLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateLicenseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateLicenseRequest, AssociateLicenseResponse>()
                            .withOperationName("AssociateLicense").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateLicenseRequest));
            CompletableFuture<AssociateLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <i>workspace</i>. In a workspace, you can create Grafana dashboards and visualizations to analyze your
     * metrics, logs, and traces. You don't have to build, package, or deploy any hardware to run the Grafana server.
     * </p>
     * <p>
     * Don't use <code>CreateWorkspace</code> to modify an existing workspace. Instead, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspace.html">UpdateWorkspace</a>.
     * </p>
     *
     * @param createWorkspaceRequest
     * @return A Java Future containing the result of the CreateWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.CreateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkspaceResponse> createWorkspace(CreateWorkspaceRequest createWorkspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkspaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceRequest, CreateWorkspaceResponse>()
                            .withOperationName("CreateWorkspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkspaceRequest));
            CompletableFuture<CreateWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Grafana API key for the workspace. This key can be used to authenticate requests sent to the
     * workspace's HTTP API. See <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/Using-Grafana-APIs.html">https
     * ://docs.aws.amazon.com/grafana/latest/userguide/Using-Grafana-APIs.html</a> for available APIs and example
     * requests.
     * </p>
     * <note>
     * <p>
     * In workspaces compatible with Grafana version 9 or above, use workspace service accounts instead of API keys. API
     * keys will be removed in a future release.
     * </p>
     * </note>
     *
     * @param createWorkspaceApiKeyRequest
     * @return A Java Future containing the result of the CreateWorkspaceApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.CreateWorkspaceApiKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspaceApiKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkspaceApiKeyResponse> createWorkspaceApiKey(
            CreateWorkspaceApiKeyRequest createWorkspaceApiKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkspaceApiKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkspaceApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspaceApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkspaceApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkspaceApiKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkspaceApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceApiKeyRequest, CreateWorkspaceApiKeyResponse>()
                            .withOperationName("CreateWorkspaceApiKey").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkspaceApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkspaceApiKeyRequest));
            CompletableFuture<CreateWorkspaceApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a service account for the workspace. A service account can be used to call Grafana HTTP APIs, and run
     * automated workloads. After creating the service account with the correct <code>GrafanaRole</code> for your use
     * case, use <code>CreateWorkspaceServiceAccountToken</code> to create a token that can be used to authenticate and
     * authorize Grafana HTTP API calls.
     * </p>
     * <p>
     * You can only create service accounts for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     * <note>
     * <p>
     * For more information about service accounts, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/service-accounts.html">Service accounts</a> in the
     * <i>Amazon Managed Grafana User Guide</i>.
     * </p>
     * <p>
     * For more information about the Grafana HTTP APIs, see <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/Using-Grafana-APIs.html">Using Grafana HTTP APIs</a>
     * in the <i>Amazon Managed Grafana User Guide</i>.
     * </p>
     * </note>
     *
     * @param createWorkspaceServiceAccountRequest
     * @return A Java Future containing the result of the CreateWorkspaceServiceAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.CreateWorkspaceServiceAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspaceServiceAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkspaceServiceAccountResponse> createWorkspaceServiceAccount(
            CreateWorkspaceServiceAccountRequest createWorkspaceServiceAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkspaceServiceAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createWorkspaceServiceAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspaceServiceAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkspaceServiceAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkspaceServiceAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkspaceServiceAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceServiceAccountRequest, CreateWorkspaceServiceAccountResponse>()
                            .withOperationName("CreateWorkspaceServiceAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkspaceServiceAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkspaceServiceAccountRequest));
            CompletableFuture<CreateWorkspaceServiceAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a token that can be used to authenticate and authorize Grafana HTTP API operations for the given <a
     * href="https://docs.aws.amazon.com/grafana/latest/userguide/service-accounts.html">workspace service account</a>.
     * The service account acts as a user for the API operations, and defines the permissions that are used by the API.
     * </p>
     * <important>
     * <p>
     * When you create the service account token, you will receive a key that is used when calling Grafana APIs. Do not
     * lose this key, as it will not be retrievable again.
     * </p>
     * <p>
     * If you do lose the key, you can delete the token and recreate it to receive a new key. This will disable the
     * initial key.
     * </p>
     * </important>
     * <p>
     * Service accounts are only available for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     *
     * @param createWorkspaceServiceAccountTokenRequest
     * @return A Java Future containing the result of the CreateWorkspaceServiceAccountToken operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>ServiceQuotaExceededException The request would cause a service quota to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.CreateWorkspaceServiceAccountToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/CreateWorkspaceServiceAccountToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkspaceServiceAccountTokenResponse> createWorkspaceServiceAccountToken(
            CreateWorkspaceServiceAccountTokenRequest createWorkspaceServiceAccountTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkspaceServiceAccountTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createWorkspaceServiceAccountTokenRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkspaceServiceAccountToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkspaceServiceAccountTokenResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateWorkspaceServiceAccountTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkspaceServiceAccountTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkspaceServiceAccountTokenRequest, CreateWorkspaceServiceAccountTokenResponse>()
                            .withOperationName("CreateWorkspaceServiceAccountToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkspaceServiceAccountTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkspaceServiceAccountTokenRequest));
            CompletableFuture<CreateWorkspaceServiceAccountTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Managed Grafana workspace.
     * </p>
     *
     * @param deleteWorkspaceRequest
     * @return A Java Future containing the result of the DeleteWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DeleteWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkspaceResponse> deleteWorkspace(DeleteWorkspaceRequest deleteWorkspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkspaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceRequest, DeleteWorkspaceResponse>()
                            .withOperationName("DeleteWorkspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkspaceRequest));
            CompletableFuture<DeleteWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Grafana API key for the workspace.
     * </p>
     * <note>
     * <p>
     * In workspaces compatible with Grafana version 9 or above, use workspace service accounts instead of API keys. API
     * keys will be removed in a future release.
     * </p>
     * </note>
     *
     * @param deleteWorkspaceApiKeyRequest
     * @return A Java Future containing the result of the DeleteWorkspaceApiKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DeleteWorkspaceApiKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspaceApiKey" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkspaceApiKeyResponse> deleteWorkspaceApiKey(
            DeleteWorkspaceApiKeyRequest deleteWorkspaceApiKeyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkspaceApiKeyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkspaceApiKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspaceApiKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkspaceApiKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkspaceApiKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkspaceApiKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceApiKeyRequest, DeleteWorkspaceApiKeyResponse>()
                            .withOperationName("DeleteWorkspaceApiKey").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkspaceApiKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkspaceApiKeyRequest));
            CompletableFuture<DeleteWorkspaceApiKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a workspace service account from the workspace.
     * </p>
     * <p>
     * This will delete any tokens created for the service account, as well. If the tokens are currently in use, the
     * will fail to authenticate / authorize after they are deleted.
     * </p>
     * <p>
     * Service accounts are only available for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     *
     * @param deleteWorkspaceServiceAccountRequest
     * @return A Java Future containing the result of the DeleteWorkspaceServiceAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DeleteWorkspaceServiceAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspaceServiceAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkspaceServiceAccountResponse> deleteWorkspaceServiceAccount(
            DeleteWorkspaceServiceAccountRequest deleteWorkspaceServiceAccountRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkspaceServiceAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteWorkspaceServiceAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspaceServiceAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkspaceServiceAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkspaceServiceAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkspaceServiceAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceServiceAccountRequest, DeleteWorkspaceServiceAccountResponse>()
                            .withOperationName("DeleteWorkspaceServiceAccount").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkspaceServiceAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkspaceServiceAccountRequest));
            CompletableFuture<DeleteWorkspaceServiceAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a token for the workspace service account.
     * </p>
     * <p>
     * This will disable the key associated with the token. If any automation is currently using the key, it will no
     * longer be authenticated or authorized to perform actions with the Grafana HTTP APIs.
     * </p>
     * <p>
     * Service accounts are only available for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     *
     * @param deleteWorkspaceServiceAccountTokenRequest
     * @return A Java Future containing the result of the DeleteWorkspaceServiceAccountToken operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DeleteWorkspaceServiceAccountToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DeleteWorkspaceServiceAccountToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkspaceServiceAccountTokenResponse> deleteWorkspaceServiceAccountToken(
            DeleteWorkspaceServiceAccountTokenRequest deleteWorkspaceServiceAccountTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkspaceServiceAccountTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteWorkspaceServiceAccountTokenRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkspaceServiceAccountToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkspaceServiceAccountTokenResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteWorkspaceServiceAccountTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkspaceServiceAccountTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkspaceServiceAccountTokenRequest, DeleteWorkspaceServiceAccountTokenResponse>()
                            .withOperationName("DeleteWorkspaceServiceAccountToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkspaceServiceAccountTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkspaceServiceAccountTokenRequest));
            CompletableFuture<DeleteWorkspaceServiceAccountTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays information about one Amazon Managed Grafana workspace.
     * </p>
     *
     * @param describeWorkspaceRequest
     * @return A Java Future containing the result of the DescribeWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DescribeWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkspaceResponse> describeWorkspace(DescribeWorkspaceRequest describeWorkspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWorkspaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceRequest, DescribeWorkspaceResponse>()
                            .withOperationName("DescribeWorkspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWorkspaceRequest));
            CompletableFuture<DescribeWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays information about the authentication methods used in one Amazon Managed Grafana workspace.
     * </p>
     *
     * @param describeWorkspaceAuthenticationRequest
     * @return A Java Future containing the result of the DescribeWorkspaceAuthentication operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DescribeWorkspaceAuthentication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspaceAuthentication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkspaceAuthenticationResponse> describeWorkspaceAuthentication(
            DescribeWorkspaceAuthenticationRequest describeWorkspaceAuthenticationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWorkspaceAuthenticationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeWorkspaceAuthenticationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspaceAuthentication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkspaceAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkspaceAuthenticationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkspaceAuthenticationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceAuthenticationRequest, DescribeWorkspaceAuthenticationResponse>()
                            .withOperationName("DescribeWorkspaceAuthentication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWorkspaceAuthenticationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWorkspaceAuthenticationRequest));
            CompletableFuture<DescribeWorkspaceAuthenticationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the current configuration string for the given workspace.
     * </p>
     *
     * @param describeWorkspaceConfigurationRequest
     * @return A Java Future containing the result of the DescribeWorkspaceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DescribeWorkspaceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DescribeWorkspaceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkspaceConfigurationResponse> describeWorkspaceConfiguration(
            DescribeWorkspaceConfigurationRequest describeWorkspaceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWorkspaceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeWorkspaceConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkspaceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkspaceConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkspaceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkspaceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkspaceConfigurationRequest, DescribeWorkspaceConfigurationResponse>()
                            .withOperationName("DescribeWorkspaceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWorkspaceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWorkspaceConfigurationRequest));
            CompletableFuture<DescribeWorkspaceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the Grafana Enterprise license from a workspace.
     * </p>
     *
     * @param disassociateLicenseRequest
     * @return A Java Future containing the result of the DisassociateLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.DisassociateLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/DisassociateLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateLicenseResponse> disassociateLicense(
            DisassociateLicenseRequest disassociateLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateLicenseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateLicenseRequest, DisassociateLicenseResponse>()
                            .withOperationName("DisassociateLicense").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateLicenseRequest));
            CompletableFuture<DisassociateLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the users and groups who have the Grafana <code>Admin</code> and <code>Editor</code> roles in this
     * workspace. If you use this operation without specifying <code>userId</code> or <code>groupId</code>, the
     * operation returns the roles of all users and groups. If you specify a <code>userId</code> or a
     * <code>groupId</code>, only the roles for that user or group are returned. If you do this, you can specify only
     * one <code>userId</code> or one <code>groupId</code>.
     * </p>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListPermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPermissionsRequest));
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>ListTagsForResource</code> operation returns the tags that are associated with the Amazon Managed
     * Service for Grafana resource specified by the <code>resourceArn</code>. Currently, the only resource that can be
     * tagged is a workspace.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists available versions of Grafana. These are available when calling <code>CreateWorkspace</code>. Optionally,
     * include a workspace to list the versions to which it can be upgraded.
     * </p>
     *
     * @param listVersionsRequest
     * @return A Java Future containing the result of the ListVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListVersions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListVersionsResponse> listVersions(ListVersionsRequest listVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVersionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVersionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVersionsRequest, ListVersionsResponse>()
                            .withOperationName("ListVersions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVersionsRequest));
            CompletableFuture<ListVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tokens for a workspace service account.
     * </p>
     * <note>
     * <p>
     * This does not return the key for each token. You cannot access keys after they are created. To create a new key,
     * delete the token and recreate it.
     * </p>
     * </note>
     * <p>
     * Service accounts are only available for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     *
     * @param listWorkspaceServiceAccountTokensRequest
     * @return A Java Future containing the result of the ListWorkspaceServiceAccountTokens operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListWorkspaceServiceAccountTokens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListWorkspaceServiceAccountTokens"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkspaceServiceAccountTokensResponse> listWorkspaceServiceAccountTokens(
            ListWorkspaceServiceAccountTokensRequest listWorkspaceServiceAccountTokensRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkspaceServiceAccountTokensRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listWorkspaceServiceAccountTokensRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaceServiceAccountTokens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkspaceServiceAccountTokensResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListWorkspaceServiceAccountTokensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkspaceServiceAccountTokensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkspaceServiceAccountTokensRequest, ListWorkspaceServiceAccountTokensResponse>()
                            .withOperationName("ListWorkspaceServiceAccountTokens").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkspaceServiceAccountTokensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkspaceServiceAccountTokensRequest));
            CompletableFuture<ListWorkspaceServiceAccountTokensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of service accounts for a workspace.
     * </p>
     * <p>
     * Service accounts are only available for workspaces that are compatible with Grafana version 9 and above.
     * </p>
     *
     * @param listWorkspaceServiceAccountsRequest
     * @return A Java Future containing the result of the ListWorkspaceServiceAccounts operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListWorkspaceServiceAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListWorkspaceServiceAccounts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkspaceServiceAccountsResponse> listWorkspaceServiceAccounts(
            ListWorkspaceServiceAccountsRequest listWorkspaceServiceAccountsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkspaceServiceAccountsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspaceServiceAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaceServiceAccounts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkspaceServiceAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkspaceServiceAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkspaceServiceAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkspaceServiceAccountsRequest, ListWorkspaceServiceAccountsResponse>()
                            .withOperationName("ListWorkspaceServiceAccounts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkspaceServiceAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkspaceServiceAccountsRequest));
            CompletableFuture<ListWorkspaceServiceAccountsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Amazon Managed Grafana workspaces in the account, with some information about each workspace.
     * For more complete information about one workspace, use <a
     * href="https://docs.aws.amazon.com/AAMG/latest/APIReference/API_DescribeWorkspace.html">DescribeWorkspace</a>.
     * </p>
     *
     * @param listWorkspacesRequest
     * @return A Java Future containing the result of the ListWorkspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.ListWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/ListWorkspaces" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkspacesResponse> listWorkspaces(ListWorkspacesRequest listWorkspacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkspacesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkspaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkspacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkspacesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkspacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkspacesRequest, ListWorkspacesResponse>()
                            .withOperationName("ListWorkspaces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkspacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkspacesRequest));
            CompletableFuture<ListWorkspacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>TagResource</code> operation associates tags with an Amazon Managed Grafana resource. Currently, the
     * only resource that can be tagged is workspaces.
     * </p>
     * <p>
     * If you specify a new tag key for the resource, this tag is appended to the list of tags associated with the
     * resource. If you specify a tag key that is already associated with the resource, the new tag value that you
     * specify replaces the previous value for that tag.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UntagResource</code> operation removes the association of the tag with the Amazon Managed Grafana
     * resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates which users in a workspace have the Grafana <code>Admin</code> or <code>Editor</code> roles.
     * </p>
     *
     * @param updatePermissionsRequest
     * @return A Java Future containing the result of the UpdatePermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.UpdatePermissions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdatePermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePermissionsResponse> updatePermissions(UpdatePermissionsRequest updatePermissionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePermissionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePermissionsRequest, UpdatePermissionsResponse>()
                            .withOperationName("UpdatePermissions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePermissionsRequest));
            CompletableFuture<UpdatePermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies an existing Amazon Managed Grafana workspace. If you use this operation and omit any optional
     * parameters, the existing values of those parameters are not changed.
     * </p>
     * <p>
     * To modify the user authentication methods that the workspace uses, such as SAML or IAM Identity Center, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdateWorkspaceAuthentication.html">
     * UpdateWorkspaceAuthentication</a>.
     * </p>
     * <p>
     * To modify which users in the workspace have the <code>Admin</code> and <code>Editor</code> Grafana roles, use <a
     * href="https://docs.aws.amazon.com/grafana/latest/APIReference/API_UpdatePermissions.html">UpdatePermissions</a>.
     * </p>
     *
     * @param updateWorkspaceRequest
     * @return A Java Future containing the result of the UpdateWorkspace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.UpdateWorkspace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkspaceResponse> updateWorkspace(UpdateWorkspaceRequest updateWorkspaceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkspaceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkspaceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkspaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkspaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceRequest, UpdateWorkspaceResponse>()
                            .withOperationName("UpdateWorkspace").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWorkspaceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWorkspaceRequest));
            CompletableFuture<UpdateWorkspaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Use this operation to define the identity provider (IdP) that this workspace authenticates users from, using
     * SAML. You can also map SAML assertion attributes to workspace user information and define which groups in the
     * assertion attribute are to have the <code>Admin</code> and <code>Editor</code> roles in the workspace.
     * </p>
     * <note>
     * <p>
     * Changes to the authentication method for a workspace may take a few minutes to take effect.
     * </p>
     * </note>
     *
     * @param updateWorkspaceAuthenticationRequest
     * @return A Java Future containing the result of the UpdateWorkspaceAuthentication operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.UpdateWorkspaceAuthentication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspaceAuthentication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkspaceAuthenticationResponse> updateWorkspaceAuthentication(
            UpdateWorkspaceAuthenticationRequest updateWorkspaceAuthenticationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkspaceAuthenticationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateWorkspaceAuthenticationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceAuthentication");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkspaceAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkspaceAuthenticationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkspaceAuthenticationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceAuthenticationRequest, UpdateWorkspaceAuthenticationResponse>()
                            .withOperationName("UpdateWorkspaceAuthentication").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWorkspaceAuthenticationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWorkspaceAuthenticationRequest));
            CompletableFuture<UpdateWorkspaceAuthenticationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the configuration string for the given workspace
     * </p>
     *
     * @param updateWorkspaceConfigurationRequest
     * @return A Java Future containing the result of the UpdateWorkspaceConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The request references a resource that does not exist.</li>
     *         <li>ThrottlingException The request was denied because of request throttling. Retry the request.</li>
     *         <li>ConflictException A resource was in an inconsistent state during an update or a deletion.</li>
     *         <li>ValidationException The value of a parameter in the request caused an error.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException Unexpected error while processing the request. Retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GrafanaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GrafanaAsyncClient.UpdateWorkspaceConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/grafana-2020-08-18/UpdateWorkspaceConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkspaceConfigurationResponse> updateWorkspaceConfiguration(
            UpdateWorkspaceConfigurationRequest updateWorkspaceConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateWorkspaceConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkspaceConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "grafana");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkspaceConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkspaceConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkspaceConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkspaceConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkspaceConfigurationRequest, UpdateWorkspaceConfigurationResponse>()
                            .withOperationName("UpdateWorkspaceConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateWorkspaceConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateWorkspaceConfigurationRequest));
            CompletableFuture<UpdateWorkspaceConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final GrafanaServiceClientConfiguration serviceClientConfiguration() {
        return new GrafanaServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GrafanaException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        GrafanaServiceClientConfigurationBuilder serviceConfigBuilder = new GrafanaServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
