/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.greengrassv2.model.AccessDeniedException;
import software.amazon.awssdk.services.greengrassv2.model.CancelDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.CancelDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.ConflictException;
import software.amazon.awssdk.services.greengrassv2.model.CreateComponentVersionRequest;
import software.amazon.awssdk.services.greengrassv2.model.CreateComponentVersionResponse;
import software.amazon.awssdk.services.greengrassv2.model.CreateDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.CreateDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DeleteComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.DeleteComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.DeleteCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.DeleteCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.DescribeComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.DescribeComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentVersionArtifactRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetComponentVersionArtifactResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetCoreDeviceRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetCoreDeviceResponse;
import software.amazon.awssdk.services.greengrassv2.model.GetDeploymentRequest;
import software.amazon.awssdk.services.greengrassv2.model.GetDeploymentResponse;
import software.amazon.awssdk.services.greengrassv2.model.GreengrassV2Exception;
import software.amazon.awssdk.services.greengrassv2.model.GreengrassV2Request;
import software.amazon.awssdk.services.greengrassv2.model.InternalServerException;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListComponentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsResponse;
import software.amazon.awssdk.services.greengrassv2.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.ResolveComponentCandidatesRequest;
import software.amazon.awssdk.services.greengrassv2.model.ResolveComponentCandidatesResponse;
import software.amazon.awssdk.services.greengrassv2.model.ResourceNotFoundException;
import software.amazon.awssdk.services.greengrassv2.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.greengrassv2.model.TagResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.TagResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.ThrottlingException;
import software.amazon.awssdk.services.greengrassv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.greengrassv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.greengrassv2.model.ValidationException;
import software.amazon.awssdk.services.greengrassv2.paginators.ListComponentVersionsPublisher;
import software.amazon.awssdk.services.greengrassv2.paginators.ListComponentsPublisher;
import software.amazon.awssdk.services.greengrassv2.paginators.ListCoreDevicesPublisher;
import software.amazon.awssdk.services.greengrassv2.paginators.ListDeploymentsPublisher;
import software.amazon.awssdk.services.greengrassv2.paginators.ListEffectiveDeploymentsPublisher;
import software.amazon.awssdk.services.greengrassv2.paginators.ListInstalledComponentsPublisher;
import software.amazon.awssdk.services.greengrassv2.transform.CancelDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.CreateComponentVersionRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.CreateDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DeleteComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DeleteCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.DescribeComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetComponentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetComponentVersionArtifactRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetCoreDeviceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListComponentVersionsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListComponentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListCoreDevicesRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListEffectiveDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListInstalledComponentsRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.ResolveComponentCandidatesRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.greengrassv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GreengrassV2AsyncClient}.
 *
 * @see GreengrassV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGreengrassV2AsyncClient implements GreengrassV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGreengrassV2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGreengrassV2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a deployment. This operation cancels the deployment for devices that haven't yet received it. If a device
     * already received the deployment, this operation doesn't change anything for that device.
     * </p>
     *
     * @param cancelDeploymentRequest
     * @return A Java Future containing the result of the CancelDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CancelDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CancelDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelDeploymentResponse> cancelDeployment(CancelDeploymentRequest cancelDeploymentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelDeploymentRequest, CancelDeploymentResponse>()
                            .withOperationName("CancelDeployment")
                            .withMarshaller(new CancelDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = cancelDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CancelDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a component. Components are software that run on AWS IoT Greengrass core devices. After you develop and
     * test a component on your core device, you can use this operation to upload your component to AWS IoT Greengrass.
     * Then, you can deploy the component to other core devices.
     * </p>
     * <p>
     * You can use this operation to do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>Create components from recipes</b>
     * </p>
     * <p>
     * Create a component from a recipe, which is a file that defines the component's metadata, parameters,
     * dependencies, lifecycle, artifacts, and platform capability. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/component-recipe-reference.html">AWS IoT
     * Greengrass component recipe reference</a> in the <i>AWS IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * <p>
     * To create a component from a recipe, specify <code>inlineRecipe</code> when you call this operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>Create components from Lambda functions</b>
     * </p>
     * <p>
     * Create a component from an AWS Lambda function that runs on AWS IoT Greengrass. This creates a recipe and
     * artifacts from the Lambda function's deployment package. You can use this operation to migrate Lambda functions
     * from AWS IoT Greengrass V1 to AWS IoT Greengrass V2.
     * </p>
     * <p>
     * This function only accepts Lambda functions that use the following runtimes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Python 2.7 – <code>python2.7</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Python 3.7 – <code>python3.7</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Python 3.8 – <code>python3.8</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Java 8 – <code>java8</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Node.js 10 – <code>nodejs10.x</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Node.js 12 – <code>nodejs12.x</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To create a component from a Lambda function, specify <code>lambdaFunction</code> when you call this operation.
     * </p>
     * </li>
     * </ul>
     *
     * @param createComponentVersionRequest
     * @return A Java Future containing the result of the CreateComponentVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota. For example, you might have the
     *         maximum number of components that you can create.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CreateComponentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CreateComponentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComponentVersionResponse> createComponentVersion(
            CreateComponentVersionRequest createComponentVersionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComponentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComponentVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComponentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComponentVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComponentVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComponentVersionRequest, CreateComponentVersionResponse>()
                            .withOperationName("CreateComponentVersion")
                            .withMarshaller(new CreateComponentVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createComponentVersionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createComponentVersionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<CreateComponentVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a continuous deployment for a target, which is a AWS IoT Greengrass core device or group of core devices.
     * When you add a new core device to a group of core devices that has a deployment, AWS IoT Greengrass deploys that
     * group's deployment to the new device.
     * </p>
     * <p>
     * You can define one deployment for each target. When you create a new deployment for a target that has an existing
     * deployment, you replace the previous deployment. AWS IoT Greengrass applies the new deployment to the target
     * devices.
     * </p>
     * <p>
     * Every deployment has a revision number that indicates how many deployment revisions you define for a target. Use
     * this operation to create a new revision of an existing deployment. This operation returns the revision number of
     * the new deployment when you create it.
     * </p>
     * <p>
     * For more information, see the <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create deployments</a> in
     * the <i>AWS IoT Greengrass V2 Developer Guide</i>.
     * </p>
     *
     * @param createDeploymentRequest
     * @return A Java Future containing the result of the CreateDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.CreateDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/CreateDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDeploymentResponse> createDeployment(CreateDeploymentRequest createDeploymentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentRequest, CreateDeploymentResponse>()
                            .withOperationName("CreateDeployment")
                            .withMarshaller(new CreateDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a version of a component from AWS IoT Greengrass.
     * </p>
     * <note>
     * <p>
     * This operation deletes the component's recipe and artifacts. As a result, deployments that refer to this
     * component version will fail. If you have deployments that use this component version, you can remove the
     * component from the deployment or update the deployment to use a valid version.
     * </p>
     * </note>
     *
     * @param deleteComponentRequest
     * @return A Java Future containing the result of the DeleteComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DeleteComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DeleteComponent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComponentResponse> deleteComponent(DeleteComponentRequest deleteComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComponentRequest, DeleteComponentResponse>()
                            .withOperationName("DeleteComponent")
                            .withMarshaller(new DeleteComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteComponentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteComponentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a AWS IoT Greengrass core device, which is an AWS IoT thing. This operation removes the core device from
     * the list of core devices. This operation doesn't delete the AWS IoT thing. For more information about how to
     * delete the AWS IoT thing, see <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html">DeleteThing</a> in the <i>AWS IoT
     * API Reference</i>.
     * </p>
     *
     * @param deleteCoreDeviceRequest
     * @return A Java Future containing the result of the DeleteCoreDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DeleteCoreDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DeleteCoreDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCoreDeviceResponse> deleteCoreDevice(DeleteCoreDeviceRequest deleteCoreDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCoreDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCoreDeviceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCoreDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCoreDeviceRequest, DeleteCoreDeviceResponse>()
                            .withOperationName("DeleteCoreDevice")
                            .withMarshaller(new DeleteCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCoreDeviceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteCoreDeviceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteCoreDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves metadata for a version of a component.
     * </p>
     *
     * @param describeComponentRequest
     * @return A Java Future containing the result of the DescribeComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.DescribeComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/DescribeComponent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComponentResponse> describeComponent(DescribeComponentRequest describeComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComponentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComponentRequest, DescribeComponentResponse>()
                            .withOperationName("DescribeComponent")
                            .withMarshaller(new DescribeComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeComponentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeComponentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the recipe for a version of a component. Core devices can call this operation to identify the artifacts and
     * requirements to install a component.
     * </p>
     *
     * @param getComponentRequest
     * @return A Java Future containing the result of the GetComponent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetComponent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetComponent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentResponse> getComponent(GetComponentRequest getComponentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetComponentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComponentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentRequest, GetComponentResponse>()
                            .withOperationName("GetComponent").withMarshaller(new GetComponentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getComponentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getComponentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetComponentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the pre-signed URL to download a public component artifact. Core devices call this operation to identify the
     * URL that they can use to download an artifact to install.
     * </p>
     *
     * @param getComponentVersionArtifactRequest
     * @return A Java Future containing the result of the GetComponentVersionArtifact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetComponentVersionArtifact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetComponentVersionArtifact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetComponentVersionArtifactResponse> getComponentVersionArtifact(
            GetComponentVersionArtifactRequest getComponentVersionArtifactRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getComponentVersionArtifactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComponentVersionArtifact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetComponentVersionArtifactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetComponentVersionArtifactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetComponentVersionArtifactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetComponentVersionArtifactRequest, GetComponentVersionArtifactResponse>()
                            .withOperationName("GetComponentVersionArtifact")
                            .withMarshaller(new GetComponentVersionArtifactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getComponentVersionArtifactRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getComponentVersionArtifactRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetComponentVersionArtifactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves metadata for a AWS IoT Greengrass core device.
     * </p>
     *
     * @param getCoreDeviceRequest
     * @return A Java Future containing the result of the GetCoreDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetCoreDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetCoreDevice" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCoreDeviceResponse> getCoreDevice(GetCoreDeviceRequest getCoreDeviceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCoreDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCoreDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCoreDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCoreDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCoreDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCoreDeviceRequest, GetCoreDeviceResponse>()
                            .withOperationName("GetCoreDevice")
                            .withMarshaller(new GetCoreDeviceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCoreDeviceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCoreDeviceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetCoreDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a deployment. Deployments define the components that run on AWS IoT Greengrass core devices.
     * </p>
     *
     * @param getDeploymentRequest
     * @return A Java Future containing the result of the GetDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.GetDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/GetDeployment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeploymentResponse> getDeployment(GetDeploymentRequest getDeploymentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                            .withOperationName("GetDeployment")
                            .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDeploymentRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDeploymentRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of all versions for a component.
     * </p>
     *
     * @param listComponentVersionsRequest
     * @return A Java Future containing the result of the ListComponentVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentVersionsResponse> listComponentVersions(
            ListComponentVersionsRequest listComponentVersionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponentVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComponentVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentVersionsRequest, ListComponentVersionsResponse>()
                            .withOperationName("ListComponentVersions")
                            .withMarshaller(new ListComponentVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listComponentVersionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listComponentVersionsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListComponentVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of all versions for a component.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listComponentVersions(software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListComponentVersionsPublisher publisher = client.listComponentVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListComponentVersionsPublisher publisher = client.listComponentVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComponentVersions(software.amazon.awssdk.services.greengrassv2.model.ListComponentVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listComponentVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponentVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListComponentVersionsPublisher listComponentVersionsPaginator(ListComponentVersionsRequest listComponentVersionsRequest) {
        return new ListComponentVersionsPublisher(this, applyPaginatorUserAgent(listComponentVersionsRequest));
    }

    /**
     * <p>
     * Retrieves a paginated list of component summaries. This list includes components that you have permission to
     * view.
     * </p>
     *
     * @param listComponentsRequest
     * @return A Java Future containing the result of the ListComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListComponentsResponse> listComponents(ListComponentsRequest listComponentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListComponentsRequest, ListComponentsResponse>()
                            .withOperationName("ListComponents")
                            .withMarshaller(new ListComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listComponentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listComponentsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of component summaries. This list includes components that you have permission to
     * view.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listComponents(software.amazon.awssdk.services.greengrassv2.model.ListComponentsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListComponentsPublisher publisher = client.listComponentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListComponentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListComponentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listComponents(software.amazon.awssdk.services.greengrassv2.model.ListComponentsRequest)} operation.</b>
     * </p>
     *
     * @param listComponentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListComponents" target="_top">AWS
     *      API Documentation</a>
     */
    public ListComponentsPublisher listComponentsPaginator(ListComponentsRequest listComponentsRequest) {
        return new ListComponentsPublisher(this, applyPaginatorUserAgent(listComponentsRequest));
    }

    /**
     * <p>
     * Retrieves a paginated list of AWS IoT Greengrass core devices.
     * </p>
     *
     * @param listCoreDevicesRequest
     * @return A Java Future containing the result of the ListCoreDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListCoreDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListCoreDevices" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCoreDevicesResponse> listCoreDevices(ListCoreDevicesRequest listCoreDevicesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCoreDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCoreDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCoreDevicesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCoreDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCoreDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCoreDevicesRequest, ListCoreDevicesResponse>()
                            .withOperationName("ListCoreDevices")
                            .withMarshaller(new ListCoreDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCoreDevicesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listCoreDevicesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListCoreDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of AWS IoT Greengrass core devices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCoreDevices(software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListCoreDevicesPublisher publisher = client.listCoreDevicesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListCoreDevicesPublisher publisher = client.listCoreDevicesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCoreDevices(software.amazon.awssdk.services.greengrassv2.model.ListCoreDevicesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCoreDevicesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListCoreDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListCoreDevices" target="_top">AWS
     *      API Documentation</a>
     */
    public ListCoreDevicesPublisher listCoreDevicesPaginator(ListCoreDevicesRequest listCoreDevicesRequest) {
        return new ListCoreDevicesPublisher(this, applyPaginatorUserAgent(listCoreDevicesRequest));
    }

    /**
     * <p>
     * Retrieves a paginated list of deployments.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A Java Future containing the result of the ListDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListDeployments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDeploymentsResponse> listDeployments(ListDeploymentsRequest listDeploymentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDeploymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                            .withOperationName("ListDeployments")
                            .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDeploymentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDeploymentsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of deployments.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDeployments(software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListDeploymentsPublisher publisher = client.listDeploymentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListDeploymentsPublisher publisher = client.listDeploymentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDeployments(software.amazon.awssdk.services.greengrassv2.model.ListDeploymentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDeploymentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListDeployments" target="_top">AWS
     *      API Documentation</a>
     */
    public ListDeploymentsPublisher listDeploymentsPaginator(ListDeploymentsRequest listDeploymentsRequest) {
        return new ListDeploymentsPublisher(this, applyPaginatorUserAgent(listDeploymentsRequest));
    }

    /**
     * <p>
     * Retrieves a paginated list of deployment jobs that AWS IoT Greengrass sends to AWS IoT Greengrass core devices.
     * </p>
     *
     * @param listEffectiveDeploymentsRequest
     * @return A Java Future containing the result of the ListEffectiveDeployments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListEffectiveDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListEffectiveDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEffectiveDeploymentsResponse> listEffectiveDeployments(
            ListEffectiveDeploymentsRequest listEffectiveDeploymentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEffectiveDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEffectiveDeployments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEffectiveDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEffectiveDeploymentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEffectiveDeploymentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEffectiveDeploymentsRequest, ListEffectiveDeploymentsResponse>()
                            .withOperationName("ListEffectiveDeployments")
                            .withMarshaller(new ListEffectiveDeploymentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEffectiveDeploymentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEffectiveDeploymentsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListEffectiveDeploymentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of deployment jobs that AWS IoT Greengrass sends to AWS IoT Greengrass core devices.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEffectiveDeployments(software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListEffectiveDeploymentsPublisher publisher = client.listEffectiveDeploymentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListEffectiveDeploymentsPublisher publisher = client.listEffectiveDeploymentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEffectiveDeployments(software.amazon.awssdk.services.greengrassv2.model.ListEffectiveDeploymentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEffectiveDeploymentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListEffectiveDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListEffectiveDeployments"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEffectiveDeploymentsPublisher listEffectiveDeploymentsPaginator(
            ListEffectiveDeploymentsRequest listEffectiveDeploymentsRequest) {
        return new ListEffectiveDeploymentsPublisher(this, applyPaginatorUserAgent(listEffectiveDeploymentsRequest));
    }

    /**
     * <p>
     * Retrieves a paginated list of the components that a AWS IoT Greengrass core device runs.
     * </p>
     *
     * @param listInstalledComponentsRequest
     * @return A Java Future containing the result of the ListInstalledComponents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListInstalledComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListInstalledComponents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInstalledComponentsResponse> listInstalledComponents(
            ListInstalledComponentsRequest listInstalledComponentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInstalledComponentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInstalledComponents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInstalledComponentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInstalledComponentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInstalledComponentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInstalledComponentsRequest, ListInstalledComponentsResponse>()
                            .withOperationName("ListInstalledComponents")
                            .withMarshaller(new ListInstalledComponentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInstalledComponentsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listInstalledComponentsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListInstalledComponentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a paginated list of the components that a AWS IoT Greengrass core device runs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInstalledComponents(software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListInstalledComponentsPublisher publisher = client.listInstalledComponentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.greengrassv2.paginators.ListInstalledComponentsPublisher publisher = client.listInstalledComponentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInstalledComponents(software.amazon.awssdk.services.greengrassv2.model.ListInstalledComponentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInstalledComponentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListInstalledComponents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListInstalledComponents"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInstalledComponentsPublisher listInstalledComponentsPaginator(
            ListInstalledComponentsRequest listInstalledComponentsRequest) {
        return new ListInstalledComponentsPublisher(this, applyPaginatorUserAgent(listInstalledComponentsRequest));
    }

    /**
     * <p>
     * Retrieves the list of tags for an AWS IoT Greengrass resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of components that meet the component, version, and platform requirements of a deployment. AWS
     * IoT Greengrass core devices call this operation when they receive a deployment to identify the components to
     * install.
     * </p>
     * <p>
     * This operation identifies components that meet all dependency requirements for a deployment. If the requirements
     * conflict, then this operation returns an error and the deployment fails. For example, this occurs if component
     * <code>A</code> requires version <code>&gt;2.0.0</code> and component <code>B</code> requires version
     * <code>&lt;2.0.0</code> of a component dependency.
     * </p>
     * <p>
     * When you specify the component candidates to resolve, AWS IoT Greengrass compares each component's digest from
     * the core device with the component's digest in the AWS Cloud. If the digests don't match, then AWS IoT Greengrass
     * specifies to use the version from the AWS Cloud.
     * </p>
     * <important>
     * <p>
     * To use this operation, you must use the data plane API endpoint and authenticate with an AWS IoT device
     * certificate. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/greengrass.html">AWS IoT Greengrass endpoints and quotas</a>.
     * </p>
     * </important>
     *
     * @param resolveComponentCandidatesRequest
     * @return A Java Future containing the result of the ResolveComponentCandidates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>AccessDeniedException You don't have permission to perform the action.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>ThrottlingException Your request exceeded a request rate quota. For example, you might have exceeded
     *         the amount of times that you can retrieve device or deployment status per second.</li>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ConflictException Your request has conflicting operations. This can occur if you're trying to perform
     *         more than one operation on the same resource at the same time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.ResolveComponentCandidates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/ResolveComponentCandidates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ResolveComponentCandidatesResponse> resolveComponentCandidates(
            ResolveComponentCandidatesRequest resolveComponentCandidatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resolveComponentCandidatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResolveComponentCandidates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResolveComponentCandidatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ResolveComponentCandidatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResolveComponentCandidatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResolveComponentCandidatesRequest, ResolveComponentCandidatesResponse>()
                            .withOperationName("ResolveComponentCandidates")
                            .withMarshaller(new ResolveComponentCandidatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(resolveComponentCandidatesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = resolveComponentCandidatesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ResolveComponentCandidatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to an AWS IoT Greengrass resource. If a tag already exists for the resource, this operation updates the
     * tag's value.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from an AWS IoT Greengrass resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException AWS IoT Greengrass can't process your request right now. Try again later.</li>
     *         <li>ValidationException The request isn't valid. This can occur if your request contains malformed JSON
     *         or unsupported characters.</li>
     *         <li>ResourceNotFoundException The requested resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GreengrassV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GreengrassV2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/greengrassv2-2020-11-30/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GreengrassV2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GreengrassV2Exception::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends GreengrassV2Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
