/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a deployment's policy that defines when components are safe to update.
 * </p>
 * <p>
 * Each component on a device can report whether or not it's ready to update. After a component and its dependencies are
 * ready, they can apply the update in the deployment. You can configure whether or not the deployment notifies
 * components of an update and waits for a response. You specify the amount of time each component has to respond to the
 * update notification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentComponentUpdatePolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentComponentUpdatePolicy.Builder, DeploymentComponentUpdatePolicy> {
    private static final SdkField<Integer> TIMEOUT_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInSeconds").getter(getter(DeploymentComponentUpdatePolicy::timeoutInSeconds))
            .setter(setter(Builder::timeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInSeconds").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("action")
            .getter(getter(DeploymentComponentUpdatePolicy::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("action").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMEOUT_IN_SECONDS_FIELD,
            ACTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer timeoutInSeconds;

    private final String action;

    private DeploymentComponentUpdatePolicy(BuilderImpl builder) {
        this.timeoutInSeconds = builder.timeoutInSeconds;
        this.action = builder.action;
    }

    /**
     * <p>
     * The amount of time in seconds that each component on a device has to report that it's safe to update. If the
     * component waits for longer than this timeout, then the deployment proceeds on the device.
     * </p>
     * <p>
     * Default: <code>60</code>
     * </p>
     * 
     * @return The amount of time in seconds that each component on a device has to report that it's safe to update. If
     *         the component waits for longer than this timeout, then the deployment proceeds on the device.</p>
     *         <p>
     *         Default: <code>60</code>
     */
    public final Integer timeoutInSeconds() {
        return timeoutInSeconds;
    }

    /**
     * <p>
     * Whether or not to notify components and wait for components to become safe to update. Choose from the following
     * options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
     * component. Components can use the <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
     * >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can respond with
     * the <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
     * >DeferComponentUpdate</a> IPC operation. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create deployments</a> in
     * the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be safe to
     * update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>NOTIFY_COMPONENTS</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link DeploymentComponentUpdatePolicyAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #actionAsString}.
     * </p>
     * 
     * @return Whether or not to notify components and wait for components to become safe to update. Choose from the
     *         following options:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
     *         component. Components can use the <a href=
     *         "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
     *         >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can
     *         respond with the <a href=
     *         "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
     *         >DeferComponentUpdate</a> IPC operation. For more information, see <a
     *         href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
     *         deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be
     *         safe to update.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>NOTIFY_COMPONENTS</code>
     * @see DeploymentComponentUpdatePolicyAction
     */
    public final DeploymentComponentUpdatePolicyAction action() {
        return DeploymentComponentUpdatePolicyAction.fromValue(action);
    }

    /**
     * <p>
     * Whether or not to notify components and wait for components to become safe to update. Choose from the following
     * options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
     * component. Components can use the <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
     * >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can respond with
     * the <a href=
     * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
     * >DeferComponentUpdate</a> IPC operation. For more information, see <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create deployments</a> in
     * the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be safe to
     * update.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>NOTIFY_COMPONENTS</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link DeploymentComponentUpdatePolicyAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #actionAsString}.
     * </p>
     * 
     * @return Whether or not to notify components and wait for components to become safe to update. Choose from the
     *         following options:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
     *         component. Components can use the <a href=
     *         "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
     *         >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can
     *         respond with the <a href=
     *         "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
     *         >DeferComponentUpdate</a> IPC operation. For more information, see <a
     *         href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
     *         deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be
     *         safe to update.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>NOTIFY_COMPONENTS</code>
     * @see DeploymentComponentUpdatePolicyAction
     */
    public final String actionAsString() {
        return action;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentComponentUpdatePolicy)) {
            return false;
        }
        DeploymentComponentUpdatePolicy other = (DeploymentComponentUpdatePolicy) obj;
        return Objects.equals(timeoutInSeconds(), other.timeoutInSeconds())
                && Objects.equals(actionAsString(), other.actionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentComponentUpdatePolicy").add("TimeoutInSeconds", timeoutInSeconds())
                .add("Action", actionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "timeoutInSeconds":
            return Optional.ofNullable(clazz.cast(timeoutInSeconds()));
        case "action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentComponentUpdatePolicy, T> g) {
        return obj -> g.apply((DeploymentComponentUpdatePolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentComponentUpdatePolicy> {
        /**
         * <p>
         * The amount of time in seconds that each component on a device has to report that it's safe to update. If the
         * component waits for longer than this timeout, then the deployment proceeds on the device.
         * </p>
         * <p>
         * Default: <code>60</code>
         * </p>
         * 
         * @param timeoutInSeconds
         *        The amount of time in seconds that each component on a device has to report that it's safe to update.
         *        If the component waits for longer than this timeout, then the deployment proceeds on the device.</p>
         *        <p>
         *        Default: <code>60</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInSeconds(Integer timeoutInSeconds);

        /**
         * <p>
         * Whether or not to notify components and wait for components to become safe to update. Choose from the
         * following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
         * component. Components can use the <a href=
         * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
         * >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can respond
         * with the <a href=
         * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
         * >DeferComponentUpdate</a> IPC operation. For more information, see <a
         * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
         * deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be safe to
         * update.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>NOTIFY_COMPONENTS</code>
         * </p>
         * 
         * @param action
         *        Whether or not to notify components and wait for components to become safe to update. Choose from the
         *        following options:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates
         *        that component. Components can use the <a href=
         *        "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
         *        >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can
         *        respond with the <a href=
         *        "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
         *        >DeferComponentUpdate</a> IPC operation. For more information, see <a
         *        href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
         *        deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be
         *        safe to update.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>NOTIFY_COMPONENTS</code>
         * @see DeploymentComponentUpdatePolicyAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentComponentUpdatePolicyAction
         */
        Builder action(String action);

        /**
         * <p>
         * Whether or not to notify components and wait for components to become safe to update. Choose from the
         * following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates that
         * component. Components can use the <a href=
         * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
         * >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can respond
         * with the <a href=
         * "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
         * >DeferComponentUpdate</a> IPC operation. For more information, see <a
         * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
         * deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be safe to
         * update.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>NOTIFY_COMPONENTS</code>
         * </p>
         * 
         * @param action
         *        Whether or not to notify components and wait for components to become safe to update. Choose from the
         *        following options:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NOTIFY_COMPONENTS</code> – The deployment notifies each component before it stops and updates
         *        that component. Components can use the <a href=
         *        "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-subscribetocomponentupdates"
         *        >SubscribeToComponentUpdates</a> IPC operation to receive these notifications. Then, components can
         *        respond with the <a href=
         *        "https://docs.aws.amazon.com/greengrass/v2/developerguide/interprocess-communication.html#ipc-operation-defercomponentupdate"
         *        >DeferComponentUpdate</a> IPC operation. For more information, see <a
         *        href="https://docs.aws.amazon.com/greengrass/v2/developerguide/create-deployments.html">Create
         *        deployments</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIP_NOTIFY_COMPONENTS</code> – The deployment doesn't notify components or wait for them to be
         *        safe to update.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>NOTIFY_COMPONENTS</code>
         * @see DeploymentComponentUpdatePolicyAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentComponentUpdatePolicyAction
         */
        Builder action(DeploymentComponentUpdatePolicyAction action);
    }

    static final class BuilderImpl implements Builder {
        private Integer timeoutInSeconds;

        private String action;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentComponentUpdatePolicy model) {
            timeoutInSeconds(model.timeoutInSeconds);
            action(model.action);
        }

        public final Integer getTimeoutInSeconds() {
            return timeoutInSeconds;
        }

        public final void setTimeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
        }

        @Override
        @Transient
        public final Builder timeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        @Transient
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        @Transient
        public final Builder action(DeploymentComponentUpdatePolicyAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        @Override
        public DeploymentComponentUpdatePolicy build() {
            return new DeploymentComponentUpdatePolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
