/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a platform that a component supports.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComponentPlatform implements SdkPojo, Serializable,
        ToCopyableBuilder<ComponentPlatform.Builder, ComponentPlatform> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ComponentPlatform::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("attributes")
            .getter(getter(ComponentPlatform::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ATTRIBUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Map<String, String> attributes;

    private ComponentPlatform(BuilderImpl builder) {
        this.name = builder.name;
        this.attributes = builder.attributes;
    }

    /**
     * <p>
     * The friendly name of the platform. This name helps you identify the platform.
     * </p>
     * <p>
     * If you omit this parameter, IoT Greengrass creates a friendly name from the <code>os</code> and
     * <code>architecture</code> of the platform.
     * </p>
     * 
     * @return The friendly name of the platform. This name helps you identify the platform.</p>
     *         <p>
     *         If you omit this parameter, IoT Greengrass creates a friendly name from the <code>os</code> and
     *         <code>architecture</code> of the platform.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A dictionary of attributes for the platform. The IoT Greengrass Core software defines the <code>os</code> and
     * <code>platform</code> by default. You can specify additional platform attributes for a core device when you
     * deploy the Greengrass nucleus component. For more information, see the <a
     * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html">Greengrass
     * nucleus component</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return A dictionary of attributes for the platform. The IoT Greengrass Core software defines the <code>os</code>
     *         and <code>platform</code> by default. You can specify additional platform attributes for a core device
     *         when you deploy the Greengrass nucleus component. For more information, see the <a
     *         href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html"
     *         >Greengrass nucleus component</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
     */
    public final Map<String, String> attributes() {
        return attributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComponentPlatform)) {
            return false;
        }
        ComponentPlatform other = (ComponentPlatform) obj;
        return Objects.equals(name(), other.name()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ComponentPlatform").add("Name", name()).add("Attributes", hasAttributes() ? attributes() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComponentPlatform, T> g) {
        return obj -> g.apply((ComponentPlatform) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComponentPlatform> {
        /**
         * <p>
         * The friendly name of the platform. This name helps you identify the platform.
         * </p>
         * <p>
         * If you omit this parameter, IoT Greengrass creates a friendly name from the <code>os</code> and
         * <code>architecture</code> of the platform.
         * </p>
         * 
         * @param name
         *        The friendly name of the platform. This name helps you identify the platform.</p>
         *        <p>
         *        If you omit this parameter, IoT Greengrass creates a friendly name from the <code>os</code> and
         *        <code>architecture</code> of the platform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A dictionary of attributes for the platform. The IoT Greengrass Core software defines the <code>os</code> and
         * <code>platform</code> by default. You can specify additional platform attributes for a core device when you
         * deploy the Greengrass nucleus component. For more information, see the <a
         * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html">Greengrass
         * nucleus component</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         * </p>
         * 
         * @param attributes
         *        A dictionary of attributes for the platform. The IoT Greengrass Core software defines the
         *        <code>os</code> and <code>platform</code> by default. You can specify additional platform attributes
         *        for a core device when you deploy the Greengrass nucleus component. For more information, see the <a
         *        href="https://docs.aws.amazon.com/greengrass/v2/developerguide/greengrass-nucleus-component.html">
         *        Greengrass nucleus component</a> in the <i>IoT Greengrass V2 Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Map<String, String> attributes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Map<String, String> attributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComponentPlatform model) {
            name(model.name);
            attributes(model.attributes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, String> getAttributes() {
            if (attributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributes;
        }

        public final void setAttributes(Map<String, String> attributes) {
            this.attributes = PlatformAttributesMapCopier.copy(attributes);
        }

        @Override
        public final Builder attributes(Map<String, String> attributes) {
            this.attributes = PlatformAttributesMapCopier.copy(attributes);
            return this;
        }

        @Override
        public ComponentPlatform build() {
            return new ComponentPlatform(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
