/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.greengrassv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains parameters for a Lambda function that runs on IoT Greengrass.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaExecutionParameters implements SdkPojo, Serializable,
        ToCopyableBuilder<LambdaExecutionParameters.Builder, LambdaExecutionParameters> {
    private static final SdkField<List<LambdaEventSource>> EVENT_SOURCES_FIELD = SdkField
            .<List<LambdaEventSource>> builder(MarshallingType.LIST)
            .memberName("eventSources")
            .getter(getter(LambdaExecutionParameters::eventSources))
            .setter(setter(Builder::eventSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventSources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LambdaEventSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(LambdaEventSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_QUEUE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxQueueSize").getter(getter(LambdaExecutionParameters::maxQueueSize))
            .setter(setter(Builder::maxQueueSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxQueueSize").build()).build();

    private static final SdkField<Integer> MAX_INSTANCES_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxInstancesCount").getter(getter(LambdaExecutionParameters::maxInstancesCount))
            .setter(setter(Builder::maxInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxInstancesCount").build()).build();

    private static final SdkField<Integer> MAX_IDLE_TIME_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxIdleTimeInSeconds").getter(getter(LambdaExecutionParameters::maxIdleTimeInSeconds))
            .setter(setter(Builder::maxIdleTimeInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxIdleTimeInSeconds").build())
            .build();

    private static final SdkField<Integer> TIMEOUT_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInSeconds").getter(getter(LambdaExecutionParameters::timeoutInSeconds))
            .setter(setter(Builder::timeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInSeconds").build()).build();

    private static final SdkField<Integer> STATUS_TIMEOUT_IN_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("statusTimeoutInSeconds").getter(getter(LambdaExecutionParameters::statusTimeoutInSeconds))
            .setter(setter(Builder::statusTimeoutInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusTimeoutInSeconds").build())
            .build();

    private static final SdkField<Boolean> PINNED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("pinned").getter(getter(LambdaExecutionParameters::pinned)).setter(setter(Builder::pinned))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pinned").build()).build();

    private static final SdkField<String> INPUT_PAYLOAD_ENCODING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inputPayloadEncodingType").getter(getter(LambdaExecutionParameters::inputPayloadEncodingTypeAsString))
            .setter(setter(Builder::inputPayloadEncodingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputPayloadEncodingType").build())
            .build();

    private static final SdkField<List<String>> EXEC_ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("execArgs")
            .getter(getter(LambdaExecutionParameters::execArgs))
            .setter(setter(Builder::execArgs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("execArgs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_VARIABLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("environmentVariables")
            .getter(getter(LambdaExecutionParameters::environmentVariables))
            .setter(setter(Builder::environmentVariables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environmentVariables").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<LambdaLinuxProcessParams> LINUX_PROCESS_PARAMS_FIELD = SdkField
            .<LambdaLinuxProcessParams> builder(MarshallingType.SDK_POJO).memberName("linuxProcessParams")
            .getter(getter(LambdaExecutionParameters::linuxProcessParams)).setter(setter(Builder::linuxProcessParams))
            .constructor(LambdaLinuxProcessParams::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linuxProcessParams").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_SOURCES_FIELD,
            MAX_QUEUE_SIZE_FIELD, MAX_INSTANCES_COUNT_FIELD, MAX_IDLE_TIME_IN_SECONDS_FIELD, TIMEOUT_IN_SECONDS_FIELD,
            STATUS_TIMEOUT_IN_SECONDS_FIELD, PINNED_FIELD, INPUT_PAYLOAD_ENCODING_TYPE_FIELD, EXEC_ARGS_FIELD,
            ENVIRONMENT_VARIABLES_FIELD, LINUX_PROCESS_PARAMS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<LambdaEventSource> eventSources;

    private final Integer maxQueueSize;

    private final Integer maxInstancesCount;

    private final Integer maxIdleTimeInSeconds;

    private final Integer timeoutInSeconds;

    private final Integer statusTimeoutInSeconds;

    private final Boolean pinned;

    private final String inputPayloadEncodingType;

    private final List<String> execArgs;

    private final Map<String, String> environmentVariables;

    private final LambdaLinuxProcessParams linuxProcessParams;

    private LambdaExecutionParameters(BuilderImpl builder) {
        this.eventSources = builder.eventSources;
        this.maxQueueSize = builder.maxQueueSize;
        this.maxInstancesCount = builder.maxInstancesCount;
        this.maxIdleTimeInSeconds = builder.maxIdleTimeInSeconds;
        this.timeoutInSeconds = builder.timeoutInSeconds;
        this.statusTimeoutInSeconds = builder.statusTimeoutInSeconds;
        this.pinned = builder.pinned;
        this.inputPayloadEncodingType = builder.inputPayloadEncodingType;
        this.execArgs = builder.execArgs;
        this.environmentVariables = builder.environmentVariables;
        this.linuxProcessParams = builder.linuxProcessParams;
    }

    /**
     * For responses, this returns true if the service returned a value for the EventSources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEventSources() {
        return eventSources != null && !(eventSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of event sources to which to subscribe to receive work messages. The Lambda function runs when it
     * receives a message from an event source. You can subscribe this function to local publish/subscribe messages and
     * Amazon Web Services IoT Core MQTT messages.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEventSources} method.
     * </p>
     * 
     * @return The list of event sources to which to subscribe to receive work messages. The Lambda function runs when
     *         it receives a message from an event source. You can subscribe this function to local publish/subscribe
     *         messages and Amazon Web Services IoT Core MQTT messages.
     */
    public final List<LambdaEventSource> eventSources() {
        return eventSources;
    }

    /**
     * <p>
     * The maximum size of the message queue for the Lambda function component. The IoT Greengrass core stores messages
     * in a FIFO (first-in-first-out) queue until it can run the Lambda function to consume each message.
     * </p>
     * 
     * @return The maximum size of the message queue for the Lambda function component. The IoT Greengrass core stores
     *         messages in a FIFO (first-in-first-out) queue until it can run the Lambda function to consume each
     *         message.
     */
    public final Integer maxQueueSize() {
        return maxQueueSize;
    }

    /**
     * <p>
     * The maximum number of instances that a non-pinned Lambda function can run at the same time.
     * </p>
     * 
     * @return The maximum number of instances that a non-pinned Lambda function can run at the same time.
     */
    public final Integer maxInstancesCount() {
        return maxInstancesCount;
    }

    /**
     * <p>
     * The maximum amount of time in seconds that a non-pinned Lambda function can idle before the IoT Greengrass Core
     * software stops its process.
     * </p>
     * 
     * @return The maximum amount of time in seconds that a non-pinned Lambda function can idle before the IoT
     *         Greengrass Core software stops its process.
     */
    public final Integer maxIdleTimeInSeconds() {
        return maxIdleTimeInSeconds;
    }

    /**
     * <p>
     * The maximum amount of time in seconds that the Lambda function can process a work item.
     * </p>
     * 
     * @return The maximum amount of time in seconds that the Lambda function can process a work item.
     */
    public final Integer timeoutInSeconds() {
        return timeoutInSeconds;
    }

    /**
     * <p>
     * The interval in seconds at which a pinned (also known as long-lived) Lambda function component sends status
     * updates to the Lambda manager component.
     * </p>
     * 
     * @return The interval in seconds at which a pinned (also known as long-lived) Lambda function component sends
     *         status updates to the Lambda manager component.
     */
    public final Integer statusTimeoutInSeconds() {
        return statusTimeoutInSeconds;
    }

    /**
     * <p>
     * Whether or not the Lambda function is pinned, or long-lived.
     * </p>
     * <ul>
     * <li>
     * <p>
     * A pinned Lambda function starts when IoT Greengrass starts and keeps running in its own container.
     * </p>
     * </li>
     * <li>
     * <p>
     * A non-pinned Lambda function starts only when it receives a work item and exists after it idles for
     * <code>maxIdleTimeInSeconds</code>. If the function has multiple work items, the IoT Greengrass Core software
     * creates multiple instances of the function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>true</code>
     * </p>
     * 
     * @return Whether or not the Lambda function is pinned, or long-lived.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         A pinned Lambda function starts when IoT Greengrass starts and keeps running in its own container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A non-pinned Lambda function starts only when it receives a work item and exists after it idles for
     *         <code>maxIdleTimeInSeconds</code>. If the function has multiple work items, the IoT Greengrass Core
     *         software creates multiple instances of the function.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>true</code>
     */
    public final Boolean pinned() {
        return pinned;
    }

    /**
     * <p>
     * The encoding type that the Lambda function supports.
     * </p>
     * <p>
     * Default: <code>json</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inputPayloadEncodingType} will return {@link LambdaInputPayloadEncodingType#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inputPayloadEncodingTypeAsString}.
     * </p>
     * 
     * @return The encoding type that the Lambda function supports.</p>
     *         <p>
     *         Default: <code>json</code>
     * @see LambdaInputPayloadEncodingType
     */
    public final LambdaInputPayloadEncodingType inputPayloadEncodingType() {
        return LambdaInputPayloadEncodingType.fromValue(inputPayloadEncodingType);
    }

    /**
     * <p>
     * The encoding type that the Lambda function supports.
     * </p>
     * <p>
     * Default: <code>json</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #inputPayloadEncodingType} will return {@link LambdaInputPayloadEncodingType#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #inputPayloadEncodingTypeAsString}.
     * </p>
     * 
     * @return The encoding type that the Lambda function supports.</p>
     *         <p>
     *         Default: <code>json</code>
     * @see LambdaInputPayloadEncodingType
     */
    public final String inputPayloadEncodingTypeAsString() {
        return inputPayloadEncodingType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExecArgs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasExecArgs() {
        return execArgs != null && !(execArgs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of arguments to pass to the Lambda function when it runs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExecArgs} method.
     * </p>
     * 
     * @return The list of arguments to pass to the Lambda function when it runs.
     */
    public final List<String> execArgs() {
        return execArgs;
    }

    /**
     * For responses, this returns true if the service returned a value for the EnvironmentVariables property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironmentVariables() {
        return environmentVariables != null && !(environmentVariables instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The map of environment variables that are available to the Lambda function when it runs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironmentVariables} method.
     * </p>
     * 
     * @return The map of environment variables that are available to the Lambda function when it runs.
     */
    public final Map<String, String> environmentVariables() {
        return environmentVariables;
    }

    /**
     * <p>
     * The parameters for the Linux process that contains the Lambda function.
     * </p>
     * 
     * @return The parameters for the Linux process that contains the Lambda function.
     */
    public final LambdaLinuxProcessParams linuxProcessParams() {
        return linuxProcessParams;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasEventSources() ? eventSources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxQueueSize());
        hashCode = 31 * hashCode + Objects.hashCode(maxInstancesCount());
        hashCode = 31 * hashCode + Objects.hashCode(maxIdleTimeInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(statusTimeoutInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(pinned());
        hashCode = 31 * hashCode + Objects.hashCode(inputPayloadEncodingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasExecArgs() ? execArgs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironmentVariables() ? environmentVariables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linuxProcessParams());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaExecutionParameters)) {
            return false;
        }
        LambdaExecutionParameters other = (LambdaExecutionParameters) obj;
        return hasEventSources() == other.hasEventSources() && Objects.equals(eventSources(), other.eventSources())
                && Objects.equals(maxQueueSize(), other.maxQueueSize())
                && Objects.equals(maxInstancesCount(), other.maxInstancesCount())
                && Objects.equals(maxIdleTimeInSeconds(), other.maxIdleTimeInSeconds())
                && Objects.equals(timeoutInSeconds(), other.timeoutInSeconds())
                && Objects.equals(statusTimeoutInSeconds(), other.statusTimeoutInSeconds())
                && Objects.equals(pinned(), other.pinned())
                && Objects.equals(inputPayloadEncodingTypeAsString(), other.inputPayloadEncodingTypeAsString())
                && hasExecArgs() == other.hasExecArgs() && Objects.equals(execArgs(), other.execArgs())
                && hasEnvironmentVariables() == other.hasEnvironmentVariables()
                && Objects.equals(environmentVariables(), other.environmentVariables())
                && Objects.equals(linuxProcessParams(), other.linuxProcessParams());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LambdaExecutionParameters").add("EventSources", hasEventSources() ? eventSources() : null)
                .add("MaxQueueSize", maxQueueSize()).add("MaxInstancesCount", maxInstancesCount())
                .add("MaxIdleTimeInSeconds", maxIdleTimeInSeconds()).add("TimeoutInSeconds", timeoutInSeconds())
                .add("StatusTimeoutInSeconds", statusTimeoutInSeconds()).add("Pinned", pinned())
                .add("InputPayloadEncodingType", inputPayloadEncodingTypeAsString())
                .add("ExecArgs", hasExecArgs() ? execArgs() : null)
                .add("EnvironmentVariables", hasEnvironmentVariables() ? environmentVariables() : null)
                .add("LinuxProcessParams", linuxProcessParams()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventSources":
            return Optional.ofNullable(clazz.cast(eventSources()));
        case "maxQueueSize":
            return Optional.ofNullable(clazz.cast(maxQueueSize()));
        case "maxInstancesCount":
            return Optional.ofNullable(clazz.cast(maxInstancesCount()));
        case "maxIdleTimeInSeconds":
            return Optional.ofNullable(clazz.cast(maxIdleTimeInSeconds()));
        case "timeoutInSeconds":
            return Optional.ofNullable(clazz.cast(timeoutInSeconds()));
        case "statusTimeoutInSeconds":
            return Optional.ofNullable(clazz.cast(statusTimeoutInSeconds()));
        case "pinned":
            return Optional.ofNullable(clazz.cast(pinned()));
        case "inputPayloadEncodingType":
            return Optional.ofNullable(clazz.cast(inputPayloadEncodingTypeAsString()));
        case "execArgs":
            return Optional.ofNullable(clazz.cast(execArgs()));
        case "environmentVariables":
            return Optional.ofNullable(clazz.cast(environmentVariables()));
        case "linuxProcessParams":
            return Optional.ofNullable(clazz.cast(linuxProcessParams()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("eventSources", EVENT_SOURCES_FIELD);
        map.put("maxQueueSize", MAX_QUEUE_SIZE_FIELD);
        map.put("maxInstancesCount", MAX_INSTANCES_COUNT_FIELD);
        map.put("maxIdleTimeInSeconds", MAX_IDLE_TIME_IN_SECONDS_FIELD);
        map.put("timeoutInSeconds", TIMEOUT_IN_SECONDS_FIELD);
        map.put("statusTimeoutInSeconds", STATUS_TIMEOUT_IN_SECONDS_FIELD);
        map.put("pinned", PINNED_FIELD);
        map.put("inputPayloadEncodingType", INPUT_PAYLOAD_ENCODING_TYPE_FIELD);
        map.put("execArgs", EXEC_ARGS_FIELD);
        map.put("environmentVariables", ENVIRONMENT_VARIABLES_FIELD);
        map.put("linuxProcessParams", LINUX_PROCESS_PARAMS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LambdaExecutionParameters, T> g) {
        return obj -> g.apply((LambdaExecutionParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaExecutionParameters> {
        /**
         * <p>
         * The list of event sources to which to subscribe to receive work messages. The Lambda function runs when it
         * receives a message from an event source. You can subscribe this function to local publish/subscribe messages
         * and Amazon Web Services IoT Core MQTT messages.
         * </p>
         * 
         * @param eventSources
         *        The list of event sources to which to subscribe to receive work messages. The Lambda function runs
         *        when it receives a message from an event source. You can subscribe this function to local
         *        publish/subscribe messages and Amazon Web Services IoT Core MQTT messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSources(Collection<LambdaEventSource> eventSources);

        /**
         * <p>
         * The list of event sources to which to subscribe to receive work messages. The Lambda function runs when it
         * receives a message from an event source. You can subscribe this function to local publish/subscribe messages
         * and Amazon Web Services IoT Core MQTT messages.
         * </p>
         * 
         * @param eventSources
         *        The list of event sources to which to subscribe to receive work messages. The Lambda function runs
         *        when it receives a message from an event source. You can subscribe this function to local
         *        publish/subscribe messages and Amazon Web Services IoT Core MQTT messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSources(LambdaEventSource... eventSources);

        /**
         * <p>
         * The list of event sources to which to subscribe to receive work messages. The Lambda function runs when it
         * receives a message from an event source. You can subscribe this function to local publish/subscribe messages
         * and Amazon Web Services IoT Core MQTT messages.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.greengrassv2.model.LambdaEventSource.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.greengrassv2.model.LambdaEventSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.greengrassv2.model.LambdaEventSource.Builder#build()} is called
         * immediately and its result is passed to {@link #eventSources(List<LambdaEventSource>)}.
         * 
         * @param eventSources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.greengrassv2.model.LambdaEventSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eventSources(java.util.Collection<LambdaEventSource>)
         */
        Builder eventSources(Consumer<LambdaEventSource.Builder>... eventSources);

        /**
         * <p>
         * The maximum size of the message queue for the Lambda function component. The IoT Greengrass core stores
         * messages in a FIFO (first-in-first-out) queue until it can run the Lambda function to consume each message.
         * </p>
         * 
         * @param maxQueueSize
         *        The maximum size of the message queue for the Lambda function component. The IoT Greengrass core
         *        stores messages in a FIFO (first-in-first-out) queue until it can run the Lambda function to consume
         *        each message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxQueueSize(Integer maxQueueSize);

        /**
         * <p>
         * The maximum number of instances that a non-pinned Lambda function can run at the same time.
         * </p>
         * 
         * @param maxInstancesCount
         *        The maximum number of instances that a non-pinned Lambda function can run at the same time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxInstancesCount(Integer maxInstancesCount);

        /**
         * <p>
         * The maximum amount of time in seconds that a non-pinned Lambda function can idle before the IoT Greengrass
         * Core software stops its process.
         * </p>
         * 
         * @param maxIdleTimeInSeconds
         *        The maximum amount of time in seconds that a non-pinned Lambda function can idle before the IoT
         *        Greengrass Core software stops its process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxIdleTimeInSeconds(Integer maxIdleTimeInSeconds);

        /**
         * <p>
         * The maximum amount of time in seconds that the Lambda function can process a work item.
         * </p>
         * 
         * @param timeoutInSeconds
         *        The maximum amount of time in seconds that the Lambda function can process a work item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInSeconds(Integer timeoutInSeconds);

        /**
         * <p>
         * The interval in seconds at which a pinned (also known as long-lived) Lambda function component sends status
         * updates to the Lambda manager component.
         * </p>
         * 
         * @param statusTimeoutInSeconds
         *        The interval in seconds at which a pinned (also known as long-lived) Lambda function component sends
         *        status updates to the Lambda manager component.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusTimeoutInSeconds(Integer statusTimeoutInSeconds);

        /**
         * <p>
         * Whether or not the Lambda function is pinned, or long-lived.
         * </p>
         * <ul>
         * <li>
         * <p>
         * A pinned Lambda function starts when IoT Greengrass starts and keeps running in its own container.
         * </p>
         * </li>
         * <li>
         * <p>
         * A non-pinned Lambda function starts only when it receives a work item and exists after it idles for
         * <code>maxIdleTimeInSeconds</code>. If the function has multiple work items, the IoT Greengrass Core software
         * creates multiple instances of the function.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>true</code>
         * </p>
         * 
         * @param pinned
         *        Whether or not the Lambda function is pinned, or long-lived.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        A pinned Lambda function starts when IoT Greengrass starts and keeps running in its own container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A non-pinned Lambda function starts only when it receives a work item and exists after it idles for
         *        <code>maxIdleTimeInSeconds</code>. If the function has multiple work items, the IoT Greengrass Core
         *        software creates multiple instances of the function.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pinned(Boolean pinned);

        /**
         * <p>
         * The encoding type that the Lambda function supports.
         * </p>
         * <p>
         * Default: <code>json</code>
         * </p>
         * 
         * @param inputPayloadEncodingType
         *        The encoding type that the Lambda function supports.</p>
         *        <p>
         *        Default: <code>json</code>
         * @see LambdaInputPayloadEncodingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaInputPayloadEncodingType
         */
        Builder inputPayloadEncodingType(String inputPayloadEncodingType);

        /**
         * <p>
         * The encoding type that the Lambda function supports.
         * </p>
         * <p>
         * Default: <code>json</code>
         * </p>
         * 
         * @param inputPayloadEncodingType
         *        The encoding type that the Lambda function supports.</p>
         *        <p>
         *        Default: <code>json</code>
         * @see LambdaInputPayloadEncodingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaInputPayloadEncodingType
         */
        Builder inputPayloadEncodingType(LambdaInputPayloadEncodingType inputPayloadEncodingType);

        /**
         * <p>
         * The list of arguments to pass to the Lambda function when it runs.
         * </p>
         * 
         * @param execArgs
         *        The list of arguments to pass to the Lambda function when it runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder execArgs(Collection<String> execArgs);

        /**
         * <p>
         * The list of arguments to pass to the Lambda function when it runs.
         * </p>
         * 
         * @param execArgs
         *        The list of arguments to pass to the Lambda function when it runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder execArgs(String... execArgs);

        /**
         * <p>
         * The map of environment variables that are available to the Lambda function when it runs.
         * </p>
         * 
         * @param environmentVariables
         *        The map of environment variables that are available to the Lambda function when it runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environmentVariables(Map<String, String> environmentVariables);

        /**
         * <p>
         * The parameters for the Linux process that contains the Lambda function.
         * </p>
         * 
         * @param linuxProcessParams
         *        The parameters for the Linux process that contains the Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linuxProcessParams(LambdaLinuxProcessParams linuxProcessParams);

        /**
         * <p>
         * The parameters for the Linux process that contains the Lambda function.
         * </p>
         * This is a convenience method that creates an instance of the {@link LambdaLinuxProcessParams.Builder}
         * avoiding the need to create one manually via {@link LambdaLinuxProcessParams#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LambdaLinuxProcessParams.Builder#build()} is called immediately
         * and its result is passed to {@link #linuxProcessParams(LambdaLinuxProcessParams)}.
         * 
         * @param linuxProcessParams
         *        a consumer that will call methods on {@link LambdaLinuxProcessParams.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linuxProcessParams(LambdaLinuxProcessParams)
         */
        default Builder linuxProcessParams(Consumer<LambdaLinuxProcessParams.Builder> linuxProcessParams) {
            return linuxProcessParams(LambdaLinuxProcessParams.builder().applyMutation(linuxProcessParams).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<LambdaEventSource> eventSources = DefaultSdkAutoConstructList.getInstance();

        private Integer maxQueueSize;

        private Integer maxInstancesCount;

        private Integer maxIdleTimeInSeconds;

        private Integer timeoutInSeconds;

        private Integer statusTimeoutInSeconds;

        private Boolean pinned;

        private String inputPayloadEncodingType;

        private List<String> execArgs = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> environmentVariables = DefaultSdkAutoConstructMap.getInstance();

        private LambdaLinuxProcessParams linuxProcessParams;

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaExecutionParameters model) {
            eventSources(model.eventSources);
            maxQueueSize(model.maxQueueSize);
            maxInstancesCount(model.maxInstancesCount);
            maxIdleTimeInSeconds(model.maxIdleTimeInSeconds);
            timeoutInSeconds(model.timeoutInSeconds);
            statusTimeoutInSeconds(model.statusTimeoutInSeconds);
            pinned(model.pinned);
            inputPayloadEncodingType(model.inputPayloadEncodingType);
            execArgs(model.execArgs);
            environmentVariables(model.environmentVariables);
            linuxProcessParams(model.linuxProcessParams);
        }

        public final List<LambdaEventSource.Builder> getEventSources() {
            List<LambdaEventSource.Builder> result = LambdaEventSourceListCopier.copyToBuilder(this.eventSources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEventSources(Collection<LambdaEventSource.BuilderImpl> eventSources) {
            this.eventSources = LambdaEventSourceListCopier.copyFromBuilder(eventSources);
        }

        @Override
        public final Builder eventSources(Collection<LambdaEventSource> eventSources) {
            this.eventSources = LambdaEventSourceListCopier.copy(eventSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventSources(LambdaEventSource... eventSources) {
            eventSources(Arrays.asList(eventSources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventSources(Consumer<LambdaEventSource.Builder>... eventSources) {
            eventSources(Stream.of(eventSources).map(c -> LambdaEventSource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxQueueSize() {
            return maxQueueSize;
        }

        public final void setMaxQueueSize(Integer maxQueueSize) {
            this.maxQueueSize = maxQueueSize;
        }

        @Override
        public final Builder maxQueueSize(Integer maxQueueSize) {
            this.maxQueueSize = maxQueueSize;
            return this;
        }

        public final Integer getMaxInstancesCount() {
            return maxInstancesCount;
        }

        public final void setMaxInstancesCount(Integer maxInstancesCount) {
            this.maxInstancesCount = maxInstancesCount;
        }

        @Override
        public final Builder maxInstancesCount(Integer maxInstancesCount) {
            this.maxInstancesCount = maxInstancesCount;
            return this;
        }

        public final Integer getMaxIdleTimeInSeconds() {
            return maxIdleTimeInSeconds;
        }

        public final void setMaxIdleTimeInSeconds(Integer maxIdleTimeInSeconds) {
            this.maxIdleTimeInSeconds = maxIdleTimeInSeconds;
        }

        @Override
        public final Builder maxIdleTimeInSeconds(Integer maxIdleTimeInSeconds) {
            this.maxIdleTimeInSeconds = maxIdleTimeInSeconds;
            return this;
        }

        public final Integer getTimeoutInSeconds() {
            return timeoutInSeconds;
        }

        public final void setTimeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
        }

        @Override
        public final Builder timeoutInSeconds(Integer timeoutInSeconds) {
            this.timeoutInSeconds = timeoutInSeconds;
            return this;
        }

        public final Integer getStatusTimeoutInSeconds() {
            return statusTimeoutInSeconds;
        }

        public final void setStatusTimeoutInSeconds(Integer statusTimeoutInSeconds) {
            this.statusTimeoutInSeconds = statusTimeoutInSeconds;
        }

        @Override
        public final Builder statusTimeoutInSeconds(Integer statusTimeoutInSeconds) {
            this.statusTimeoutInSeconds = statusTimeoutInSeconds;
            return this;
        }

        public final Boolean getPinned() {
            return pinned;
        }

        public final void setPinned(Boolean pinned) {
            this.pinned = pinned;
        }

        @Override
        public final Builder pinned(Boolean pinned) {
            this.pinned = pinned;
            return this;
        }

        public final String getInputPayloadEncodingType() {
            return inputPayloadEncodingType;
        }

        public final void setInputPayloadEncodingType(String inputPayloadEncodingType) {
            this.inputPayloadEncodingType = inputPayloadEncodingType;
        }

        @Override
        public final Builder inputPayloadEncodingType(String inputPayloadEncodingType) {
            this.inputPayloadEncodingType = inputPayloadEncodingType;
            return this;
        }

        @Override
        public final Builder inputPayloadEncodingType(LambdaInputPayloadEncodingType inputPayloadEncodingType) {
            this.inputPayloadEncodingType(inputPayloadEncodingType == null ? null : inputPayloadEncodingType.toString());
            return this;
        }

        public final Collection<String> getExecArgs() {
            if (execArgs instanceof SdkAutoConstructList) {
                return null;
            }
            return execArgs;
        }

        public final void setExecArgs(Collection<String> execArgs) {
            this.execArgs = LambdaExecArgsListCopier.copy(execArgs);
        }

        @Override
        public final Builder execArgs(Collection<String> execArgs) {
            this.execArgs = LambdaExecArgsListCopier.copy(execArgs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder execArgs(String... execArgs) {
            execArgs(Arrays.asList(execArgs));
            return this;
        }

        public final Map<String, String> getEnvironmentVariables() {
            if (environmentVariables instanceof SdkAutoConstructMap) {
                return null;
            }
            return environmentVariables;
        }

        public final void setEnvironmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = LambdaEnvironmentVariablesCopier.copy(environmentVariables);
        }

        @Override
        public final Builder environmentVariables(Map<String, String> environmentVariables) {
            this.environmentVariables = LambdaEnvironmentVariablesCopier.copy(environmentVariables);
            return this;
        }

        public final LambdaLinuxProcessParams.Builder getLinuxProcessParams() {
            return linuxProcessParams != null ? linuxProcessParams.toBuilder() : null;
        }

        public final void setLinuxProcessParams(LambdaLinuxProcessParams.BuilderImpl linuxProcessParams) {
            this.linuxProcessParams = linuxProcessParams != null ? linuxProcessParams.build() : null;
        }

        @Override
        public final Builder linuxProcessParams(LambdaLinuxProcessParams linuxProcessParams) {
            this.linuxProcessParams = linuxProcessParams;
            return this;
        }

        @Override
        public LambdaExecutionParameters build() {
            return new LambdaExecutionParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
