/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object that describes a spectral <code>Config</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpectrumConfig implements SdkPojo, Serializable, ToCopyableBuilder<SpectrumConfig.Builder, SpectrumConfig> {
    private static final SdkField<FrequencyBandwidth> BANDWIDTH_FIELD = SdkField
            .<FrequencyBandwidth> builder(MarshallingType.SDK_POJO).memberName("bandwidth")
            .getter(getter(SpectrumConfig::bandwidth)).setter(setter(Builder::bandwidth))
            .constructor(FrequencyBandwidth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bandwidth").build()).build();

    private static final SdkField<Frequency> CENTER_FREQUENCY_FIELD = SdkField.<Frequency> builder(MarshallingType.SDK_POJO)
            .memberName("centerFrequency").getter(getter(SpectrumConfig::centerFrequency))
            .setter(setter(Builder::centerFrequency)).constructor(Frequency::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("centerFrequency").build()).build();

    private static final SdkField<String> POLARIZATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("polarization").getter(getter(SpectrumConfig::polarizationAsString))
            .setter(setter(Builder::polarization))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("polarization").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BANDWIDTH_FIELD,
            CENTER_FREQUENCY_FIELD, POLARIZATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final FrequencyBandwidth bandwidth;

    private final Frequency centerFrequency;

    private final String polarization;

    private SpectrumConfig(BuilderImpl builder) {
        this.bandwidth = builder.bandwidth;
        this.centerFrequency = builder.centerFrequency;
        this.polarization = builder.polarization;
    }

    /**
     * <p>
     * Bandwidth of a spectral <code>Config</code>. AWS Ground Station currently has the following bandwidth
     * limitations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>AntennaDownlinkDemodDecodeconfig</code>, valid values are between 125 kHz to 650 MHz.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>AntennaDownlinkconfig</code> valid values are between 10 kHz to 54 MHz.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>AntennaUplinkConfig</code>, valid values are between 10 kHz to 54 MHz.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Bandwidth of a spectral <code>Config</code>. AWS Ground Station currently has the following bandwidth
     *         limitations:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>AntennaDownlinkDemodDecodeconfig</code>, valid values are between 125 kHz to 650 MHz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>AntennaDownlinkconfig</code> valid values are between 10 kHz to 54 MHz.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>AntennaUplinkConfig</code>, valid values are between 10 kHz to 54 MHz.
     *         </p>
     *         </li>
     */
    public final FrequencyBandwidth bandwidth() {
        return bandwidth;
    }

    /**
     * <p>
     * Center frequency of a spectral <code>Config</code>. Valid values are between 2200 to 2300 MHz and 7750 to 8400
     * MHz for downlink and 2025 to 2120 MHz for uplink.
     * </p>
     * 
     * @return Center frequency of a spectral <code>Config</code>. Valid values are between 2200 to 2300 MHz and 7750 to
     *         8400 MHz for downlink and 2025 to 2120 MHz for uplink.
     */
    public final Frequency centerFrequency() {
        return centerFrequency;
    }

    /**
     * <p>
     * Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
     * <code>"LEFT_HAND"</code> polarization requires two separate configs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #polarization} will
     * return {@link Polarization#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #polarizationAsString}.
     * </p>
     * 
     * @return Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
     *         <code>"LEFT_HAND"</code> polarization requires two separate configs.
     * @see Polarization
     */
    public final Polarization polarization() {
        return Polarization.fromValue(polarization);
    }

    /**
     * <p>
     * Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
     * <code>"LEFT_HAND"</code> polarization requires two separate configs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #polarization} will
     * return {@link Polarization#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #polarizationAsString}.
     * </p>
     * 
     * @return Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
     *         <code>"LEFT_HAND"</code> polarization requires two separate configs.
     * @see Polarization
     */
    public final String polarizationAsString() {
        return polarization;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bandwidth());
        hashCode = 31 * hashCode + Objects.hashCode(centerFrequency());
        hashCode = 31 * hashCode + Objects.hashCode(polarizationAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpectrumConfig)) {
            return false;
        }
        SpectrumConfig other = (SpectrumConfig) obj;
        return Objects.equals(bandwidth(), other.bandwidth()) && Objects.equals(centerFrequency(), other.centerFrequency())
                && Objects.equals(polarizationAsString(), other.polarizationAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpectrumConfig").add("Bandwidth", bandwidth()).add("CenterFrequency", centerFrequency())
                .add("Polarization", polarizationAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bandwidth":
            return Optional.ofNullable(clazz.cast(bandwidth()));
        case "centerFrequency":
            return Optional.ofNullable(clazz.cast(centerFrequency()));
        case "polarization":
            return Optional.ofNullable(clazz.cast(polarizationAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpectrumConfig, T> g) {
        return obj -> g.apply((SpectrumConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpectrumConfig> {
        /**
         * <p>
         * Bandwidth of a spectral <code>Config</code>. AWS Ground Station currently has the following bandwidth
         * limitations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>AntennaDownlinkDemodDecodeconfig</code>, valid values are between 125 kHz to 650 MHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>AntennaDownlinkconfig</code> valid values are between 10 kHz to 54 MHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>AntennaUplinkConfig</code>, valid values are between 10 kHz to 54 MHz.
         * </p>
         * </li>
         * </ul>
         * 
         * @param bandwidth
         *        Bandwidth of a spectral <code>Config</code>. AWS Ground Station currently has the following bandwidth
         *        limitations:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>AntennaDownlinkDemodDecodeconfig</code>, valid values are between 125 kHz to 650 MHz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>AntennaDownlinkconfig</code> valid values are between 10 kHz to 54 MHz.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>AntennaUplinkConfig</code>, valid values are between 10 kHz to 54 MHz.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bandwidth(FrequencyBandwidth bandwidth);

        /**
         * <p>
         * Bandwidth of a spectral <code>Config</code>. AWS Ground Station currently has the following bandwidth
         * limitations:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>AntennaDownlinkDemodDecodeconfig</code>, valid values are between 125 kHz to 650 MHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>AntennaDownlinkconfig</code> valid values are between 10 kHz to 54 MHz.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>AntennaUplinkConfig</code>, valid values are between 10 kHz to 54 MHz.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link FrequencyBandwidth.Builder} avoiding the need to
         * create one manually via {@link FrequencyBandwidth#builder()}.
         *
         * When the {@link Consumer} completes, {@link FrequencyBandwidth.Builder#build()} is called immediately and its
         * result is passed to {@link #bandwidth(FrequencyBandwidth)}.
         * 
         * @param bandwidth
         *        a consumer that will call methods on {@link FrequencyBandwidth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bandwidth(FrequencyBandwidth)
         */
        default Builder bandwidth(Consumer<FrequencyBandwidth.Builder> bandwidth) {
            return bandwidth(FrequencyBandwidth.builder().applyMutation(bandwidth).build());
        }

        /**
         * <p>
         * Center frequency of a spectral <code>Config</code>. Valid values are between 2200 to 2300 MHz and 7750 to
         * 8400 MHz for downlink and 2025 to 2120 MHz for uplink.
         * </p>
         * 
         * @param centerFrequency
         *        Center frequency of a spectral <code>Config</code>. Valid values are between 2200 to 2300 MHz and 7750
         *        to 8400 MHz for downlink and 2025 to 2120 MHz for uplink.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder centerFrequency(Frequency centerFrequency);

        /**
         * <p>
         * Center frequency of a spectral <code>Config</code>. Valid values are between 2200 to 2300 MHz and 7750 to
         * 8400 MHz for downlink and 2025 to 2120 MHz for uplink.
         * </p>
         * This is a convenience that creates an instance of the {@link Frequency.Builder} avoiding the need to create
         * one manually via {@link Frequency#builder()}.
         *
         * When the {@link Consumer} completes, {@link Frequency.Builder#build()} is called immediately and its result
         * is passed to {@link #centerFrequency(Frequency)}.
         * 
         * @param centerFrequency
         *        a consumer that will call methods on {@link Frequency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #centerFrequency(Frequency)
         */
        default Builder centerFrequency(Consumer<Frequency.Builder> centerFrequency) {
            return centerFrequency(Frequency.builder().applyMutation(centerFrequency).build());
        }

        /**
         * <p>
         * Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
         * <code>"LEFT_HAND"</code> polarization requires two separate configs.
         * </p>
         * 
         * @param polarization
         *        Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
         *        <code>"LEFT_HAND"</code> polarization requires two separate configs.
         * @see Polarization
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Polarization
         */
        Builder polarization(String polarization);

        /**
         * <p>
         * Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
         * <code>"LEFT_HAND"</code> polarization requires two separate configs.
         * </p>
         * 
         * @param polarization
         *        Polarization of a spectral <code>Config</code>. Capturing both <code>"RIGHT_HAND"</code> and
         *        <code>"LEFT_HAND"</code> polarization requires two separate configs.
         * @see Polarization
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Polarization
         */
        Builder polarization(Polarization polarization);
    }

    static final class BuilderImpl implements Builder {
        private FrequencyBandwidth bandwidth;

        private Frequency centerFrequency;

        private String polarization;

        private BuilderImpl() {
        }

        private BuilderImpl(SpectrumConfig model) {
            bandwidth(model.bandwidth);
            centerFrequency(model.centerFrequency);
            polarization(model.polarization);
        }

        public final FrequencyBandwidth.Builder getBandwidth() {
            return bandwidth != null ? bandwidth.toBuilder() : null;
        }

        @Override
        public final Builder bandwidth(FrequencyBandwidth bandwidth) {
            this.bandwidth = bandwidth;
            return this;
        }

        public final void setBandwidth(FrequencyBandwidth.BuilderImpl bandwidth) {
            this.bandwidth = bandwidth != null ? bandwidth.build() : null;
        }

        public final Frequency.Builder getCenterFrequency() {
            return centerFrequency != null ? centerFrequency.toBuilder() : null;
        }

        @Override
        public final Builder centerFrequency(Frequency centerFrequency) {
            this.centerFrequency = centerFrequency;
            return this;
        }

        public final void setCenterFrequency(Frequency.BuilderImpl centerFrequency) {
            this.centerFrequency = centerFrequency != null ? centerFrequency.build() : null;
        }

        public final String getPolarization() {
            return polarization;
        }

        @Override
        public final Builder polarization(String polarization) {
            this.polarization = polarization;
            return this;
        }

        @Override
        public final Builder polarization(Polarization polarization) {
            this.polarization(polarization == null ? null : polarization.toString());
            return this;
        }

        public final void setPolarization(String polarization) {
            this.polarization = polarization;
        }

        @Override
        public SpectrumConfig build() {
            return new SpectrumConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
