/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.groundstation.internal.GroundStationServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.groundstation.model.CancelContactRequest;
import software.amazon.awssdk.services.groundstation.model.CancelContactResponse;
import software.amazon.awssdk.services.groundstation.model.CreateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.CreateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.CreateEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.CreateEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DependencyException;
import software.amazon.awssdk.services.groundstation.model.DescribeContactRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeContactResponse;
import software.amazon.awssdk.services.groundstation.model.DescribeEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.GetAgentConfigurationRequest;
import software.amazon.awssdk.services.groundstation.model.GetAgentConfigurationResponse;
import software.amazon.awssdk.services.groundstation.model.GetConfigRequest;
import software.amazon.awssdk.services.groundstation.model.GetConfigResponse;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageRequest;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageResponse;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteRequest;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteResponse;
import software.amazon.awssdk.services.groundstation.model.GroundStationException;
import software.amazon.awssdk.services.groundstation.model.InvalidParameterException;
import software.amazon.awssdk.services.groundstation.model.ListConfigsRequest;
import software.amazon.awssdk.services.groundstation.model.ListConfigsResponse;
import software.amazon.awssdk.services.groundstation.model.ListContactsRequest;
import software.amazon.awssdk.services.groundstation.model.ListContactsResponse;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsRequest;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsResponse;
import software.amazon.awssdk.services.groundstation.model.ListEphemeridesRequest;
import software.amazon.awssdk.services.groundstation.model.ListEphemeridesResponse;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsRequest;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsResponse;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesRequest;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesResponse;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesRequest;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesResponse;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.groundstation.model.RegisterAgentRequest;
import software.amazon.awssdk.services.groundstation.model.RegisterAgentResponse;
import software.amazon.awssdk.services.groundstation.model.ReserveContactRequest;
import software.amazon.awssdk.services.groundstation.model.ReserveContactResponse;
import software.amazon.awssdk.services.groundstation.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.groundstation.model.ResourceNotFoundException;
import software.amazon.awssdk.services.groundstation.model.TagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.TagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UntagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.UntagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateAgentStatusRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateAgentStatusResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.transform.CancelContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DescribeContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DescribeEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetAgentConfigurationRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMinuteUsageRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetSatelliteRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListConfigsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListDataflowEndpointGroupsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListEphemeridesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListGroundStationsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListMissionProfilesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListSatellitesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.RegisterAgentRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ReserveContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateAgentStatusRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.waiters.GroundStationAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link GroundStationAsyncClient}.
 *
 * @see GroundStationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGroundStationAsyncClient implements GroundStationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultGroundStationAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultGroundStationAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Cancels a contact with a specified contact ID.
     * </p>
     *
     * @param cancelContactRequest
     * @return A Java Future containing the result of the CancelContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.CancelContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CancelContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelContactResponse> cancelContact(CancelContactRequest cancelContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelContactRequest, CancelContactResponse>()
                            .withOperationName("CancelContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelContactRequest));
            CompletableFuture<CancelContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>Config</code> with the specified <code>configData</code> parameters.
     * </p>
     * <p>
     * Only one type of <code>configData</code> can be specified.
     * </p>
     *
     * @param createConfigRequest
     * @return A Java Future containing the result of the CreateConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceLimitExceededException Account limits for this resource have been exceeded.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.CreateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigResponse> createConfig(CreateConfigRequest createConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigRequest, CreateConfigResponse>()
                            .withOperationName("CreateConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigRequest));
            CompletableFuture<CreateConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>DataflowEndpoint</code> group containing the specified list of <code>DataflowEndpoint</code>
     * objects.
     * </p>
     * <p>
     * The <code>name</code> field in each endpoint is used in your mission profile <code>DataflowEndpointConfig</code>
     * to specify which endpoints to use during a contact.
     * </p>
     * <p>
     * When a contact uses multiple <code>DataflowEndpointConfig</code> objects, each <code>Config</code> must match a
     * <code>DataflowEndpoint</code> in the same group.
     * </p>
     *
     * @param createDataflowEndpointGroupRequest
     * @return A Java Future containing the result of the CreateDataflowEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.CreateDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataflowEndpointGroupResponse> createDataflowEndpointGroup(
            CreateDataflowEndpointGroupRequest createDataflowEndpointGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataflowEndpointGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataflowEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataflowEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataflowEndpointGroupRequest, CreateDataflowEndpointGroupResponse>()
                            .withOperationName("CreateDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataflowEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataflowEndpointGroupRequest));
            CompletableFuture<CreateDataflowEndpointGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Ephemeris with the specified <code>EphemerisData</code>.
     * </p>
     *
     * @param createEphemerisRequest
     * @return A Java Future containing the result of the CreateEphemeris operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.CreateEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEphemerisResponse> createEphemeris(CreateEphemerisRequest createEphemerisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEphemeris");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEphemerisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEphemerisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEphemerisRequest, CreateEphemerisResponse>()
                            .withOperationName("CreateEphemeris").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEphemerisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEphemerisRequest));
            CompletableFuture<CreateEphemerisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a mission profile.
     * </p>
     * <p>
     * <code>dataflowEdges</code> is a list of lists of strings. Each lower level list of strings has two elements: a
     * <i>from</i> ARN and a <i>to</i> ARN.
     * </p>
     *
     * @param createMissionProfileRequest
     * @return A Java Future containing the result of the CreateMissionProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.CreateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMissionProfileResponse> createMissionProfile(
            CreateMissionProfileRequest createMissionProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMissionProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMissionProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMissionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMissionProfileRequest, CreateMissionProfileResponse>()
                            .withOperationName("CreateMissionProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMissionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMissionProfileRequest));
            CompletableFuture<CreateMissionProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a <code>Config</code>.
     * </p>
     *
     * @param deleteConfigRequest
     * @return A Java Future containing the result of the DeleteConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DeleteConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfigResponse> deleteConfig(DeleteConfigRequest deleteConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigRequest, DeleteConfigResponse>()
                            .withOperationName("DeleteConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConfigRequest));
            CompletableFuture<DeleteConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataflow endpoint group.
     * </p>
     *
     * @param deleteDataflowEndpointGroupRequest
     * @return A Java Future containing the result of the DeleteDataflowEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DeleteDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataflowEndpointGroupResponse> deleteDataflowEndpointGroup(
            DeleteDataflowEndpointGroupRequest deleteDataflowEndpointGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataflowEndpointGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataflowEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataflowEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataflowEndpointGroupRequest, DeleteDataflowEndpointGroupResponse>()
                            .withOperationName("DeleteDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataflowEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataflowEndpointGroupRequest));
            CompletableFuture<DeleteDataflowEndpointGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an ephemeris
     * </p>
     *
     * @param deleteEphemerisRequest
     * @return A Java Future containing the result of the DeleteEphemeris operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DeleteEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEphemerisResponse> deleteEphemeris(DeleteEphemerisRequest deleteEphemerisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEphemeris");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEphemerisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEphemerisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEphemerisRequest, DeleteEphemerisResponse>()
                            .withOperationName("DeleteEphemeris").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEphemerisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEphemerisRequest));
            CompletableFuture<DeleteEphemerisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a mission profile.
     * </p>
     *
     * @param deleteMissionProfileRequest
     * @return A Java Future containing the result of the DeleteMissionProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DeleteMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMissionProfileResponse> deleteMissionProfile(
            DeleteMissionProfileRequest deleteMissionProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMissionProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMissionProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMissionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMissionProfileRequest, DeleteMissionProfileResponse>()
                            .withOperationName("DeleteMissionProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMissionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMissionProfileRequest));
            CompletableFuture<DeleteMissionProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing contact.
     * </p>
     *
     * @param describeContactRequest
     * @return A Java Future containing the result of the DescribeContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DescribeContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DescribeContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeContactResponse> describeContact(DescribeContactRequest describeContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeContactRequest, DescribeContactResponse>()
                            .withOperationName("DescribeContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeContactRequest));
            CompletableFuture<DescribeContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing ephemeris.
     * </p>
     *
     * @param describeEphemerisRequest
     * @return A Java Future containing the result of the DescribeEphemeris operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.DescribeEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DescribeEphemeris"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEphemerisResponse> describeEphemeris(DescribeEphemerisRequest describeEphemerisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEphemeris");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEphemerisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEphemerisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEphemerisRequest, DescribeEphemerisResponse>()
                            .withOperationName("DescribeEphemeris").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEphemerisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEphemerisRequest));
            CompletableFuture<DescribeEphemerisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Gets the latest configuration information for a registered agent.
     * </p>
     *
     * @param getAgentConfigurationRequest
     * @return A Java Future containing the result of the GetAgentConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetAgentConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetAgentConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAgentConfigurationResponse> getAgentConfiguration(
            GetAgentConfigurationRequest getAgentConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAgentConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAgentConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAgentConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAgentConfigurationRequest, GetAgentConfigurationResponse>()
                            .withOperationName("GetAgentConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAgentConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAgentConfigurationRequest));
            CompletableFuture<GetAgentConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns <code>Config</code> information.
     * </p>
     * <p>
     * Only one <code>Config</code> response can be returned.
     * </p>
     *
     * @param getConfigRequest
     * @return A Java Future containing the result of the GetConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfigResponse> getConfig(GetConfigRequest getConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfigRequest, GetConfigResponse>().withOperationName("GetConfig")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetConfigRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfigRequest));
            CompletableFuture<GetConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the dataflow endpoint group.
     * </p>
     *
     * @param getDataflowEndpointGroupRequest
     * @return A Java Future containing the result of the GetDataflowEndpointGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataflowEndpointGroupResponse> getDataflowEndpointGroup(
            GetDataflowEndpointGroupRequest getDataflowEndpointGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataflowEndpointGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataflowEndpointGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataflowEndpointGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataflowEndpointGroupRequest, GetDataflowEndpointGroupResponse>()
                            .withOperationName("GetDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataflowEndpointGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataflowEndpointGroupRequest));
            CompletableFuture<GetDataflowEndpointGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of reserved minutes used by account.
     * </p>
     *
     * @param getMinuteUsageRequest
     * @return A Java Future containing the result of the GetMinuteUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetMinuteUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMinuteUsage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMinuteUsageResponse> getMinuteUsage(GetMinuteUsageRequest getMinuteUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMinuteUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMinuteUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMinuteUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMinuteUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMinuteUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMinuteUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMinuteUsageRequest, GetMinuteUsageResponse>()
                            .withOperationName("GetMinuteUsage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMinuteUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMinuteUsageRequest));
            CompletableFuture<GetMinuteUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a mission profile.
     * </p>
     *
     * @param getMissionProfileRequest
     * @return A Java Future containing the result of the GetMissionProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMissionProfileResponse> getMissionProfile(GetMissionProfileRequest getMissionProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMissionProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMissionProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMissionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMissionProfileRequest, GetMissionProfileResponse>()
                            .withOperationName("GetMissionProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMissionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMissionProfileRequest));
            CompletableFuture<GetMissionProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a satellite.
     * </p>
     *
     * @param getSatelliteRequest
     * @return A Java Future containing the result of the GetSatellite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.GetSatellite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetSatellite" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSatelliteResponse> getSatellite(GetSatelliteRequest getSatelliteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSatelliteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSatelliteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSatellite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSatelliteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSatelliteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSatelliteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSatelliteRequest, GetSatelliteResponse>()
                            .withOperationName("GetSatellite").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSatelliteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSatelliteRequest));
            CompletableFuture<GetSatelliteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>Config</code> objects.
     * </p>
     *
     * @param listConfigsRequest
     * @return A Java Future containing the result of the ListConfigs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListConfigs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigsResponse> listConfigs(ListConfigsRequest listConfigsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListConfigsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigsRequest, ListConfigsResponse>()
                            .withOperationName("ListConfigs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConfigsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigsRequest));
            CompletableFuture<ListConfigsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of contacts.
     * </p>
     * <p>
     * If <code>statusList</code> contains AVAILABLE, the request must include <code>groundStation</code>,
     * <code>missionprofileArn</code>, and <code>satelliteArn</code>.
     * </p>
     *
     * @param listContactsRequest
     * @return A Java Future containing the result of the ListContacts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListContacts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListContactsResponse> listContacts(ListContactsRequest listContactsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListContactsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContactsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                            .withOperationName("ListContacts").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListContactsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listContactsRequest));
            CompletableFuture<ListContactsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>DataflowEndpoint</code> groups.
     * </p>
     *
     * @param listDataflowEndpointGroupsRequest
     * @return A Java Future containing the result of the ListDataflowEndpointGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListDataflowEndpointGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListDataflowEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataflowEndpointGroupsResponse> listDataflowEndpointGroups(
            ListDataflowEndpointGroupsRequest listDataflowEndpointGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataflowEndpointGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataflowEndpointGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataflowEndpointGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataflowEndpointGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataflowEndpointGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataflowEndpointGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataflowEndpointGroupsRequest, ListDataflowEndpointGroupsResponse>()
                            .withOperationName("ListDataflowEndpointGroups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataflowEndpointGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataflowEndpointGroupsRequest));
            CompletableFuture<ListDataflowEndpointGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List existing ephemerides.
     * </p>
     *
     * @param listEphemeridesRequest
     * @return A Java Future containing the result of the ListEphemerides operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListEphemerides
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListEphemerides" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEphemeridesResponse> listEphemerides(ListEphemeridesRequest listEphemeridesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEphemeridesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEphemeridesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEphemerides");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEphemeridesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEphemeridesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEphemeridesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEphemeridesRequest, ListEphemeridesResponse>()
                            .withOperationName("ListEphemerides").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEphemeridesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEphemeridesRequest));
            CompletableFuture<ListEphemeridesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ground stations.
     * </p>
     *
     * @param listGroundStationsRequest
     * @return A Java Future containing the result of the ListGroundStations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListGroundStations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListGroundStations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGroundStationsResponse> listGroundStations(ListGroundStationsRequest listGroundStationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroundStationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroundStationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroundStations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGroundStationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGroundStationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListGroundStationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGroundStationsRequest, ListGroundStationsResponse>()
                            .withOperationName("ListGroundStations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListGroundStationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listGroundStationsRequest));
            CompletableFuture<ListGroundStationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of mission profiles.
     * </p>
     *
     * @param listMissionProfilesRequest
     * @return A Java Future containing the result of the ListMissionProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListMissionProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListMissionProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMissionProfilesResponse> listMissionProfiles(
            ListMissionProfilesRequest listMissionProfilesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMissionProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMissionProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMissionProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMissionProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMissionProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMissionProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMissionProfilesRequest, ListMissionProfilesResponse>()
                            .withOperationName("ListMissionProfiles").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMissionProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMissionProfilesRequest));
            CompletableFuture<ListMissionProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of satellites.
     * </p>
     *
     * @param listSatellitesRequest
     * @return A Java Future containing the result of the ListSatellites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListSatellites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListSatellites" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSatellitesResponse> listSatellites(ListSatellitesRequest listSatellitesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSatellitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSatellitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSatellites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSatellitesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSatellitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSatellitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSatellitesRequest, ListSatellitesResponse>()
                            .withOperationName("ListSatellites").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSatellitesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSatellitesRequest));
            CompletableFuture<ListSatellitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Registers a new agent with AWS Ground Station.
     * </p>
     *
     * @param registerAgentRequest
     * @return A Java Future containing the result of the RegisterAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.RegisterAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/RegisterAgent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterAgentResponse> registerAgent(RegisterAgentRequest registerAgentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerAgentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RegisterAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterAgentRequest, RegisterAgentResponse>()
                            .withOperationName("RegisterAgent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RegisterAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(registerAgentRequest));
            CompletableFuture<RegisterAgentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reserves a contact using specified parameters.
     * </p>
     *
     * @param reserveContactRequest
     * @return A Java Future containing the result of the ReserveContact operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.ReserveContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ReserveContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ReserveContactResponse> reserveContact(ReserveContactRequest reserveContactRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(reserveContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reserveContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReserveContact");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReserveContactResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReserveContactResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReserveContactResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReserveContactRequest, ReserveContactResponse>()
                            .withOperationName("ReserveContact").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ReserveContactRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(reserveContactRequest));
            CompletableFuture<ReserveContactResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deassigns a resource tag.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Update the status of the agent.
     * </p>
     *
     * @param updateAgentStatusRequest
     * @return A Java Future containing the result of the UpdateAgentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.UpdateAgentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateAgentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAgentStatusResponse> updateAgentStatus(UpdateAgentStatusRequest updateAgentStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAgentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAgentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAgentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAgentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAgentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAgentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAgentStatusRequest, UpdateAgentStatusResponse>()
                            .withOperationName("UpdateAgentStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAgentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAgentStatusRequest));
            CompletableFuture<UpdateAgentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>Config</code> used when scheduling contacts.
     * </p>
     * <p>
     * Updating a <code>Config</code> will not update the execution parameters for existing future contacts scheduled
     * with this <code>Config</code>.
     * </p>
     *
     * @param updateConfigRequest
     * @return A Java Future containing the result of the UpdateConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.UpdateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigResponse> updateConfig(UpdateConfigRequest updateConfigRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigRequest, UpdateConfigResponse>()
                            .withOperationName("UpdateConfig").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigRequest));
            CompletableFuture<UpdateConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing ephemeris
     * </p>
     *
     * @param updateEphemerisRequest
     * @return A Java Future containing the result of the UpdateEphemeris operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.UpdateEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEphemerisResponse> updateEphemeris(UpdateEphemerisRequest updateEphemerisRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEphemeris");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEphemerisResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEphemerisResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEphemerisRequest, UpdateEphemerisResponse>()
                            .withOperationName("UpdateEphemeris").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEphemerisRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEphemerisRequest));
            CompletableFuture<UpdateEphemerisResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a mission profile.
     * </p>
     * <p>
     * Updating a mission profile will not update the execution parameters for existing future contacts.
     * </p>
     *
     * @param updateMissionProfileRequest
     * @return A Java Future containing the result of the UpdateMissionProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException One or more parameters are not valid.</li>
     *         <li>DependencyException Dependency encountered an error.</li>
     *         <li>ResourceNotFoundException Resource was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>GroundStationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample GroundStationAsyncClient.UpdateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMissionProfileResponse> updateMissionProfile(
            UpdateMissionProfileRequest updateMissionProfileRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMissionProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMissionProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMissionProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMissionProfileRequest, UpdateMissionProfileResponse>()
                            .withOperationName("UpdateMissionProfile").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMissionProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMissionProfileRequest));
            CompletableFuture<UpdateMissionProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public GroundStationAsyncWaiter waiter() {
        return GroundStationAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final GroundStationServiceClientConfiguration serviceClientConfiguration() {
        return new GroundStationServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(GroundStationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(431).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(434).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyException")
                                .exceptionBuilderSupplier(DependencyException::builder).httpStatusCode(531).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        GroundStationServiceClientConfigurationBuilder serviceConfigBuilder = new GroundStationServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
