/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Ephemeris data in Orbit Ephemeris Message (OEM) format.
 * </p>
 * <p>
 * AWS Ground Station processes OEM Customer Provided Ephemerides according to the <a
 * href="https://public.ccsds.org/Pubs/502x0b3e1.pdf">CCSDS standard</a> with some extra restrictions. OEM files should
 * be in KVN format. For more detail about the OEM format that AWS Ground Station supports, see <a href=
 * "https://docs.aws.amazon.com/ground-station/latest/ug/providing-custom-ephemeris-data.html#oem-ephemeris-format">OEM
 * ephemeris format</a> in the AWS Ground Station user guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OEMEphemeris implements SdkPojo, Serializable, ToCopyableBuilder<OEMEphemeris.Builder, OEMEphemeris> {
    private static final SdkField<String> OEM_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("oemData").getter(getter(OEMEphemeris::oemData)).setter(setter(Builder::oemData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("oemData").build()).build();

    private static final SdkField<S3Object> S3_OBJECT_FIELD = SdkField.<S3Object> builder(MarshallingType.SDK_POJO)
            .memberName("s3Object").getter(getter(OEMEphemeris::s3Object)).setter(setter(Builder::s3Object))
            .constructor(S3Object::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Object").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OEM_DATA_FIELD,
            S3_OBJECT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String oemData;

    private final S3Object s3Object;

    private OEMEphemeris(BuilderImpl builder) {
        this.oemData = builder.oemData;
        this.s3Object = builder.s3Object;
    }

    /**
     * <p>
     * The data for an OEM ephemeris, supplied directly in the request rather than through an S3 object.
     * </p>
     * 
     * @return The data for an OEM ephemeris, supplied directly in the request rather than through an S3 object.
     */
    public final String oemData() {
        return oemData;
    }

    /**
     * <p>
     * Identifies the S3 object to be used as the ephemeris.
     * </p>
     * 
     * @return Identifies the S3 object to be used as the ephemeris.
     */
    public final S3Object s3Object() {
        return s3Object;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(oemData());
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OEMEphemeris)) {
            return false;
        }
        OEMEphemeris other = (OEMEphemeris) obj;
        return Objects.equals(oemData(), other.oemData()) && Objects.equals(s3Object(), other.s3Object());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OEMEphemeris").add("OemData", oemData()).add("S3Object", s3Object()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "oemData":
            return Optional.ofNullable(clazz.cast(oemData()));
        case "s3Object":
            return Optional.ofNullable(clazz.cast(s3Object()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OEMEphemeris, T> g) {
        return obj -> g.apply((OEMEphemeris) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OEMEphemeris> {
        /**
         * <p>
         * The data for an OEM ephemeris, supplied directly in the request rather than through an S3 object.
         * </p>
         * 
         * @param oemData
         *        The data for an OEM ephemeris, supplied directly in the request rather than through an S3 object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder oemData(String oemData);

        /**
         * <p>
         * Identifies the S3 object to be used as the ephemeris.
         * </p>
         * 
         * @param s3Object
         *        Identifies the S3 object to be used as the ephemeris.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * Identifies the S3 object to be used as the ephemeris.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Object.Builder} avoiding the need to
         * create one manually via {@link S3Object#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().applyMutation(s3Object).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String oemData;

        private S3Object s3Object;

        private BuilderImpl() {
        }

        private BuilderImpl(OEMEphemeris model) {
            oemData(model.oemData);
            s3Object(model.s3Object);
        }

        public final String getOemData() {
            return oemData;
        }

        public final void setOemData(String oemData) {
            this.oemData = oemData;
        }

        @Override
        public final Builder oemData(String oemData) {
            this.oemData = oemData;
            return this;
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        @Override
        public OEMEphemeris build() {
            return new OEMEphemeris(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
