/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.groundstation.internal.GroundStationServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.groundstation.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.groundstation.model.CancelContactRequest;
import software.amazon.awssdk.services.groundstation.model.CancelContactResponse;
import software.amazon.awssdk.services.groundstation.model.CreateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.CreateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupV2Request;
import software.amazon.awssdk.services.groundstation.model.CreateDataflowEndpointGroupV2Response;
import software.amazon.awssdk.services.groundstation.model.CreateEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.CreateEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.CreateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteConfigResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.DeleteMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.DependencyException;
import software.amazon.awssdk.services.groundstation.model.DescribeContactRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeContactResponse;
import software.amazon.awssdk.services.groundstation.model.DescribeEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.DescribeEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.GetAgentConfigurationRequest;
import software.amazon.awssdk.services.groundstation.model.GetAgentConfigurationResponse;
import software.amazon.awssdk.services.groundstation.model.GetAgentTaskResponseUrlRequest;
import software.amazon.awssdk.services.groundstation.model.GetAgentTaskResponseUrlResponse;
import software.amazon.awssdk.services.groundstation.model.GetConfigRequest;
import software.amazon.awssdk.services.groundstation.model.GetConfigResponse;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupRequest;
import software.amazon.awssdk.services.groundstation.model.GetDataflowEndpointGroupResponse;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageRequest;
import software.amazon.awssdk.services.groundstation.model.GetMinuteUsageResponse;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.GetMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteRequest;
import software.amazon.awssdk.services.groundstation.model.GetSatelliteResponse;
import software.amazon.awssdk.services.groundstation.model.GroundStationException;
import software.amazon.awssdk.services.groundstation.model.InvalidParameterException;
import software.amazon.awssdk.services.groundstation.model.ListConfigsRequest;
import software.amazon.awssdk.services.groundstation.model.ListConfigsResponse;
import software.amazon.awssdk.services.groundstation.model.ListContactsRequest;
import software.amazon.awssdk.services.groundstation.model.ListContactsResponse;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsRequest;
import software.amazon.awssdk.services.groundstation.model.ListDataflowEndpointGroupsResponse;
import software.amazon.awssdk.services.groundstation.model.ListEphemeridesRequest;
import software.amazon.awssdk.services.groundstation.model.ListEphemeridesResponse;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsRequest;
import software.amazon.awssdk.services.groundstation.model.ListGroundStationsResponse;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesRequest;
import software.amazon.awssdk.services.groundstation.model.ListMissionProfilesResponse;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesRequest;
import software.amazon.awssdk.services.groundstation.model.ListSatellitesResponse;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.groundstation.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.groundstation.model.RegisterAgentRequest;
import software.amazon.awssdk.services.groundstation.model.RegisterAgentResponse;
import software.amazon.awssdk.services.groundstation.model.ReserveContactRequest;
import software.amazon.awssdk.services.groundstation.model.ReserveContactResponse;
import software.amazon.awssdk.services.groundstation.model.ResourceInUseException;
import software.amazon.awssdk.services.groundstation.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.groundstation.model.ResourceNotFoundException;
import software.amazon.awssdk.services.groundstation.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.groundstation.model.TagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.TagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UntagResourceRequest;
import software.amazon.awssdk.services.groundstation.model.UntagResourceResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateAgentStatusRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateAgentStatusResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateConfigResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateEphemerisRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateEphemerisResponse;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileRequest;
import software.amazon.awssdk.services.groundstation.model.UpdateMissionProfileResponse;
import software.amazon.awssdk.services.groundstation.transform.CancelContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateDataflowEndpointGroupV2RequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.CreateMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DeleteMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DescribeContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.DescribeEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetAgentConfigurationRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetAgentTaskResponseUrlRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetDataflowEndpointGroupRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMinuteUsageRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.GetSatelliteRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListConfigsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListContactsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListDataflowEndpointGroupsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListEphemeridesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListGroundStationsRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListMissionProfilesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListSatellitesRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.RegisterAgentRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.ReserveContactRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateAgentStatusRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateConfigRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateEphemerisRequestMarshaller;
import software.amazon.awssdk.services.groundstation.transform.UpdateMissionProfileRequestMarshaller;
import software.amazon.awssdk.services.groundstation.waiters.GroundStationWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link GroundStationClient}.
 *
 * @see GroundStationClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultGroundStationClient implements GroundStationClient {
    private static final Logger log = Logger.loggerFor(DefaultGroundStationClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultGroundStationClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "GroundStation" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels a contact with a specified contact ID.
     * </p>
     *
     * @param cancelContactRequest
     * @return Result of the CancelContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CancelContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CancelContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelContactResponse cancelContact(CancelContactRequest cancelContactRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelContactResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelContact");

            return clientHandler.execute(new ClientExecutionParams<CancelContactRequest, CancelContactResponse>()
                    .withOperationName("CancelContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>Config</code> with the specified <code>configData</code> parameters.
     * </p>
     * <p>
     * Only one type of <code>configData</code> can be specified.
     * </p>
     *
     * @param createConfigRequest
     * @return Result of the CreateConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceLimitExceededException
     *         Account limits for this resource have been exceeded.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateConfigResponse createConfig(CreateConfigRequest createConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceLimitExceededException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfig");

            return clientHandler.execute(new ClientExecutionParams<CreateConfigRequest, CreateConfigResponse>()
                    .withOperationName("CreateConfig").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataflowEndpoint</code> group containing the specified list of <code>DataflowEndpoint</code>
     * objects.
     * </p>
     * <p>
     * The <code>name</code> field in each endpoint is used in your mission profile <code>DataflowEndpointConfig</code>
     * to specify which endpoints to use during a contact.
     * </p>
     * <p>
     * When a contact uses multiple <code>DataflowEndpointConfig</code> objects, each <code>Config</code> must match a
     * <code>DataflowEndpoint</code> in the same group.
     * </p>
     *
     * @param createDataflowEndpointGroupRequest
     * @return Result of the CreateDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataflowEndpointGroupResponse createDataflowEndpointGroup(
            CreateDataflowEndpointGroupRequest createDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataflowEndpointGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataflowEndpointGroupRequest, CreateDataflowEndpointGroupResponse>()
                            .withOperationName("CreateDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a <code>DataflowEndpointGroupV2</code> containing the specified list of <code>DataflowEndpoint</code>
     * objects.
     * </p>
     * <p>
     * The <code>name</code> field in each endpoint is used in your mission profile <code>DataflowEndpointConfig</code>
     * to specify which endpoints to use during a contact.
     * </p>
     * <p>
     * When a contact uses multiple <code>DataflowEndpointConfig</code> objects, each <code>Config</code> must match a
     * <code>DataflowEndpoint</code> in the same group.
     * </p>
     *
     * @param createDataflowEndpointGroupV2Request
     * @return Result of the CreateDataflowEndpointGroupV2 operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws ServiceQuotaExceededException
     *         Request would cause a service quota to be exceeded.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateDataflowEndpointGroupV2
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateDataflowEndpointGroupV2"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataflowEndpointGroupV2Response createDataflowEndpointGroupV2(
            CreateDataflowEndpointGroupV2Request createDataflowEndpointGroupV2Request) throws InvalidParameterException,
            ServiceQuotaExceededException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataflowEndpointGroupV2Response> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataflowEndpointGroupV2Response::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataflowEndpointGroupV2Request,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataflowEndpointGroupV2Request.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataflowEndpointGroupV2");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataflowEndpointGroupV2Request, CreateDataflowEndpointGroupV2Response>()
                            .withOperationName("CreateDataflowEndpointGroupV2").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDataflowEndpointGroupV2Request)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataflowEndpointGroupV2RequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Create an ephemeris with your specified <a>EphemerisData</a>.
     * </p>
     *
     * @param createEphemerisRequest
     * @return Result of the CreateEphemeris operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEphemerisResponse createEphemeris(CreateEphemerisRequest createEphemerisRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEphemerisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEphemeris");

            return clientHandler.execute(new ClientExecutionParams<CreateEphemerisRequest, CreateEphemerisResponse>()
                    .withOperationName("CreateEphemeris").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createEphemerisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEphemerisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a mission profile.
     * </p>
     * <p>
     * <code>dataflowEdges</code> is a list of lists of strings. Each lower level list of strings has two elements: a
     * <i>from</i> ARN and a <i>to</i> ARN.
     * </p>
     *
     * @param createMissionProfileRequest
     * @return Result of the CreateMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.CreateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/CreateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMissionProfileResponse createMissionProfile(CreateMissionProfileRequest createMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMissionProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateMissionProfileRequest, CreateMissionProfileResponse>()
                    .withOperationName("CreateMissionProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a <code>Config</code>.
     * </p>
     *
     * @param deleteConfigRequest
     * @return Result of the DeleteConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteConfigResponse deleteConfig(DeleteConfigRequest deleteConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfig");

            return clientHandler.execute(new ClientExecutionParams<DeleteConfigRequest, DeleteConfigResponse>()
                    .withOperationName("DeleteConfig").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a dataflow endpoint group.
     * </p>
     *
     * @param deleteDataflowEndpointGroupRequest
     * @return Result of the DeleteDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataflowEndpointGroupResponse deleteDataflowEndpointGroup(
            DeleteDataflowEndpointGroupRequest deleteDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataflowEndpointGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataflowEndpointGroupRequest, DeleteDataflowEndpointGroupResponse>()
                            .withOperationName("DeleteDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete an ephemeris.
     * </p>
     *
     * @param deleteEphemerisRequest
     * @return Result of the DeleteEphemeris operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws ResourceInUseException
     *         The specified resource is in use by non-terminal state contacts and cannot be modified or deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteEphemerisResponse deleteEphemeris(DeleteEphemerisRequest deleteEphemerisRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, ResourceInUseException,
            AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEphemerisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEphemeris");

            return clientHandler.execute(new ClientExecutionParams<DeleteEphemerisRequest, DeleteEphemerisResponse>()
                    .withOperationName("DeleteEphemeris").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteEphemerisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEphemerisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a mission profile.
     * </p>
     *
     * @param deleteMissionProfileRequest
     * @return Result of the DeleteMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DeleteMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DeleteMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMissionProfileResponse deleteMissionProfile(DeleteMissionProfileRequest deleteMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteMissionProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteMissionProfileRequest, DeleteMissionProfileResponse>()
                    .withOperationName("DeleteMissionProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing contact.
     * </p>
     *
     * @param describeContactRequest
     * @return Result of the DescribeContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DescribeContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DescribeContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeContactResponse describeContact(DescribeContactRequest describeContactRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeContactResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContactRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContact");

            return clientHandler.execute(new ClientExecutionParams<DescribeContactRequest, DescribeContactResponse>()
                    .withOperationName("DescribeContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve information about an existing ephemeris.
     * </p>
     *
     * @param describeEphemerisRequest
     * @return Result of the DescribeEphemeris operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.DescribeEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/DescribeEphemeris"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeEphemerisResponse describeEphemeris(DescribeEphemerisRequest describeEphemerisRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEphemerisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEphemeris");

            return clientHandler.execute(new ClientExecutionParams<DescribeEphemerisRequest, DescribeEphemerisResponse>()
                    .withOperationName("DescribeEphemeris").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeEphemerisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEphemerisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Gets the latest configuration information for a registered agent.
     * </p>
     * </p>
     *
     * @param getAgentConfigurationRequest
     * @return Result of the GetAgentConfiguration operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetAgentConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetAgentConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAgentConfigurationResponse getAgentConfiguration(GetAgentConfigurationRequest getAgentConfigurationRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAgentConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAgentConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetAgentConfigurationRequest, GetAgentConfigurationResponse>()
                    .withOperationName("GetAgentConfiguration").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAgentConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAgentConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Gets a presigned URL for uploading agent task response logs.
     * </p>
     * </p>
     *
     * @param getAgentTaskResponseUrlRequest
     * @return Result of the GetAgentTaskResponseUrl operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetAgentTaskResponseUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetAgentTaskResponseUrl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAgentTaskResponseUrlResponse getAgentTaskResponseUrl(GetAgentTaskResponseUrlRequest getAgentTaskResponseUrlRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAgentTaskResponseUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAgentTaskResponseUrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentTaskResponseUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentTaskResponseUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentTaskResponseUrl");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAgentTaskResponseUrlRequest, GetAgentTaskResponseUrlResponse>()
                            .withOperationName("GetAgentTaskResponseUrl").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getAgentTaskResponseUrlRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAgentTaskResponseUrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns <code>Config</code> information.
     * </p>
     * <p>
     * Only one <code>Config</code> response can be returned.
     * </p>
     *
     * @param getConfigRequest
     * @return Result of the GetConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetConfigResponse getConfig(GetConfigRequest getConfigRequest) throws InvalidParameterException, DependencyException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfig");

            return clientHandler.execute(new ClientExecutionParams<GetConfigRequest, GetConfigResponse>()
                    .withOperationName("GetConfig").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getConfigRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the dataflow endpoint group.
     * </p>
     *
     * @param getDataflowEndpointGroupRequest
     * @return Result of the GetDataflowEndpointGroup operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetDataflowEndpointGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetDataflowEndpointGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataflowEndpointGroupResponse getDataflowEndpointGroup(
            GetDataflowEndpointGroupRequest getDataflowEndpointGroupRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataflowEndpointGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataflowEndpointGroupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataflowEndpointGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataflowEndpointGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataflowEndpointGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataflowEndpointGroupRequest, GetDataflowEndpointGroupResponse>()
                            .withOperationName("GetDataflowEndpointGroup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDataflowEndpointGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataflowEndpointGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the number of reserved minutes used by account.
     * </p>
     *
     * @param getMinuteUsageRequest
     * @return Result of the GetMinuteUsage operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetMinuteUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMinuteUsage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMinuteUsageResponse getMinuteUsage(GetMinuteUsageRequest getMinuteUsageRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMinuteUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMinuteUsageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMinuteUsageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMinuteUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMinuteUsage");

            return clientHandler.execute(new ClientExecutionParams<GetMinuteUsageRequest, GetMinuteUsageResponse>()
                    .withOperationName("GetMinuteUsage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMinuteUsageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMinuteUsageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a mission profile.
     * </p>
     *
     * @param getMissionProfileRequest
     * @return Result of the GetMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMissionProfileResponse getMissionProfile(GetMissionProfileRequest getMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMissionProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<GetMissionProfileRequest, GetMissionProfileResponse>()
                    .withOperationName("GetMissionProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a satellite.
     * </p>
     *
     * @param getSatelliteRequest
     * @return Result of the GetSatellite operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.GetSatellite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/GetSatellite" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSatelliteResponse getSatellite(GetSatelliteRequest getSatelliteRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSatelliteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSatelliteResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSatelliteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSatelliteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSatellite");

            return clientHandler.execute(new ClientExecutionParams<GetSatelliteRequest, GetSatelliteResponse>()
                    .withOperationName("GetSatellite").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getSatelliteRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSatelliteRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>Config</code> objects.
     * </p>
     *
     * @param listConfigsRequest
     * @return Result of the ListConfigs operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListConfigs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListConfigs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListConfigsResponse listConfigs(ListConfigsRequest listConfigsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConfigsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigs");

            return clientHandler.execute(new ClientExecutionParams<ListConfigsRequest, ListConfigsResponse>()
                    .withOperationName("ListConfigs").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listConfigsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConfigsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of contacts.
     * </p>
     * <p>
     * If <code>statusList</code> contains AVAILABLE, the request must include <code>groundStation</code>,
     * <code>missionprofileArn</code>, and <code>satelliteArn</code>.
     * </p>
     *
     * @param listContactsRequest
     * @return Result of the ListContacts operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListContacts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListContacts" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListContactsResponse listContacts(ListContactsRequest listContactsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContactsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContactsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContacts");

            return clientHandler.execute(new ClientExecutionParams<ListContactsRequest, ListContactsResponse>()
                    .withOperationName("ListContacts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listContactsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of <code>DataflowEndpoint</code> groups.
     * </p>
     *
     * @param listDataflowEndpointGroupsRequest
     * @return Result of the ListDataflowEndpointGroups operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListDataflowEndpointGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListDataflowEndpointGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataflowEndpointGroupsResponse listDataflowEndpointGroups(
            ListDataflowEndpointGroupsRequest listDataflowEndpointGroupsRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataflowEndpointGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataflowEndpointGroupsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataflowEndpointGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataflowEndpointGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataflowEndpointGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataflowEndpointGroupsRequest, ListDataflowEndpointGroupsResponse>()
                            .withOperationName("ListDataflowEndpointGroups").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listDataflowEndpointGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataflowEndpointGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List your existing ephemerides.
     * </p>
     *
     * @param listEphemeridesRequest
     * @return Result of the ListEphemerides operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListEphemerides
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListEphemerides" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEphemeridesResponse listEphemerides(ListEphemeridesRequest listEphemeridesRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEphemeridesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEphemeridesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEphemeridesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEphemeridesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEphemerides");

            return clientHandler.execute(new ClientExecutionParams<ListEphemeridesRequest, ListEphemeridesResponse>()
                    .withOperationName("ListEphemerides").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEphemeridesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEphemeridesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ground stations.
     * </p>
     *
     * @param listGroundStationsRequest
     * @return Result of the ListGroundStations operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListGroundStations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListGroundStations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListGroundStationsResponse listGroundStations(ListGroundStationsRequest listGroundStationsRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGroundStationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListGroundStationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGroundStationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGroundStationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGroundStations");

            return clientHandler.execute(new ClientExecutionParams<ListGroundStationsRequest, ListGroundStationsResponse>()
                    .withOperationName("ListGroundStations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listGroundStationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGroundStationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of mission profiles.
     * </p>
     *
     * @param listMissionProfilesRequest
     * @return Result of the ListMissionProfiles operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListMissionProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListMissionProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListMissionProfilesResponse listMissionProfiles(ListMissionProfilesRequest listMissionProfilesRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMissionProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMissionProfilesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMissionProfilesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMissionProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMissionProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListMissionProfilesRequest, ListMissionProfilesResponse>()
                    .withOperationName("ListMissionProfiles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listMissionProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMissionProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of satellites.
     * </p>
     *
     * @param listSatellitesRequest
     * @return Result of the ListSatellites operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListSatellites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListSatellites" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSatellitesResponse listSatellites(ListSatellitesRequest listSatellitesRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSatellitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSatellitesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSatellitesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSatellitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSatellites");

            return clientHandler.execute(new ClientExecutionParams<ListSatellitesRequest, ListSatellitesResponse>()
                    .withOperationName("ListSatellites").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSatellitesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSatellitesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags for a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Registers a new agent with AWS Ground Station.
     * </p>
     * </p>
     *
     * @param registerAgentRequest
     * @return Result of the RegisterAgent operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.RegisterAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/RegisterAgent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterAgentResponse registerAgent(RegisterAgentRequest registerAgentRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RegisterAgentResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(registerAgentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterAgent");

            return clientHandler.execute(new ClientExecutionParams<RegisterAgentRequest, RegisterAgentResponse>()
                    .withOperationName("RegisterAgent").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(registerAgentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterAgentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Reserves a contact using specified parameters.
     * </p>
     *
     * @param reserveContactRequest
     * @return Result of the ReserveContact operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceLimitExceededException
     *         Account limits for this resource have been exceeded.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.ReserveContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/ReserveContact" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ReserveContactResponse reserveContact(ReserveContactRequest reserveContactRequest) throws InvalidParameterException,
            DependencyException, ResourceLimitExceededException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReserveContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ReserveContactResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(reserveContactRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reserveContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReserveContact");

            return clientHandler.execute(new ClientExecutionParams<ReserveContactRequest, ReserveContactResponse>()
                    .withOperationName("ReserveContact").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(reserveContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ReserveContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns a tag to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deassigns a resource tag.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * For use by AWS Ground Station Agent and shouldn't be called directly.
     * </p>
     * </note>
     * <p>
     * Update the status of the agent.
     * </p>
     * </p>
     *
     * @param updateAgentStatusRequest
     * @return Result of the UpdateAgentStatus operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateAgentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateAgentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAgentStatusResponse updateAgentStatus(UpdateAgentStatusRequest updateAgentStatusRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAgentStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAgentStatusResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAgentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAgentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAgentStatus");

            return clientHandler.execute(new ClientExecutionParams<UpdateAgentStatusRequest, UpdateAgentStatusResponse>()
                    .withOperationName("UpdateAgentStatus").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAgentStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAgentStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the <code>Config</code> used when scheduling contacts.
     * </p>
     * <p>
     * Updating a <code>Config</code> will not update the execution parameters for existing future contacts scheduled
     * with this <code>Config</code>.
     * </p>
     *
     * @param updateConfigRequest
     * @return Result of the UpdateConfig operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateConfigResponse updateConfig(UpdateConfigRequest updateConfigRequest) throws InvalidParameterException,
            DependencyException, ResourceNotFoundException, AwsServiceException, SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateConfigResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfig");

            return clientHandler.execute(new ClientExecutionParams<UpdateConfigRequest, UpdateConfigResponse>()
                    .withOperationName("UpdateConfig").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an existing ephemeris.
     * </p>
     *
     * @param updateEphemerisRequest
     * @return Result of the UpdateEphemeris operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateEphemeris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateEphemeris" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateEphemerisResponse updateEphemeris(UpdateEphemerisRequest updateEphemerisRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEphemerisResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEphemerisResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEphemerisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEphemerisRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEphemeris");

            return clientHandler.execute(new ClientExecutionParams<UpdateEphemerisRequest, UpdateEphemerisResponse>()
                    .withOperationName("UpdateEphemeris").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateEphemerisRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEphemerisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a mission profile.
     * </p>
     * <p>
     * Updating a mission profile will not update the execution parameters for existing future contacts.
     * </p>
     *
     * @param updateMissionProfileRequest
     * @return Result of the UpdateMissionProfile operation returned by the service.
     * @throws InvalidParameterException
     *         One or more parameters are not valid.
     * @throws DependencyException
     *         Dependency encountered an error.
     * @throws ResourceNotFoundException
     *         Resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws GroundStationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample GroundStationClient.UpdateMissionProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/groundstation-2019-05-23/UpdateMissionProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateMissionProfileResponse updateMissionProfile(UpdateMissionProfileRequest updateMissionProfileRequest)
            throws InvalidParameterException, DependencyException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, GroundStationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateMissionProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateMissionProfileResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(431)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceInUseException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceInUseException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ResourceInUseException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(434)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DependencyException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DependencyException").httpStatusCode(531)
                        .exceptionBuilderSupplier(DependencyException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMissionProfileRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMissionProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "GroundStation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMissionProfile");

            return clientHandler.execute(new ClientExecutionParams<UpdateMissionProfileRequest, UpdateMissionProfileResponse>()
                    .withOperationName("UpdateMissionProfile").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateMissionProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateMissionProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link GroundStationWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link GroundStationWaiter}
     */
    @Override
    public GroundStationWaiter waiter() {
        return GroundStationWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        GroundStationServiceClientConfigurationBuilder serviceConfigBuilder = new GroundStationServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(GroundStationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final GroundStationServiceClientConfiguration serviceClientConfiguration() {
        return new GroundStationServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
