/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Azimuth elevation ephemeris data.
 * </p>
 * <p>
 * Use this ephemeris type to provide pointing angles directly, rather than satellite orbital elements. Use this when
 * you need precise antenna pointing but have imprecise or unknown satellite trajectory information.
 * </p>
 * <p>
 * The azimuth elevation data specifies the antenna pointing direction at specific times relative to a ground station
 * location. AWS Ground Station uses 4th order Lagrange interpolation to compute pointing angles between the provided
 * data points.
 * </p>
 * <p>
 * AWS Ground Station automatically filters interpolated pointing angles, including only those that are above the site
 * mask elevation of the specified ground station.
 * </p>
 * <p>
 * For more detail about providing azimuth elevation ephemerides to AWS Ground Station, see the <a
 * href="https://docs.aws.amazon.com/ground-station/latest/ug/providing-azimuth-elevation-ephemeris-data.html">azimuth
 * elevation ephemeris section</a> of the AWS Ground Station User Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AzElEphemeris implements SdkPojo, Serializable, ToCopyableBuilder<AzElEphemeris.Builder, AzElEphemeris> {
    private static final SdkField<String> GROUND_STATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("groundStation").getter(getter(AzElEphemeris::groundStation)).setter(setter(Builder::groundStation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("groundStation").build()).build();

    private static final SdkField<AzElSegmentsData> DATA_FIELD = SdkField.<AzElSegmentsData> builder(MarshallingType.SDK_POJO)
            .memberName("data").getter(getter(AzElEphemeris::data)).setter(setter(Builder::data))
            .constructor(AzElSegmentsData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("data").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GROUND_STATION_FIELD,
            DATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String groundStation;

    private final AzElSegmentsData data;

    private AzElEphemeris(BuilderImpl builder) {
        this.groundStation = builder.groundStation;
        this.data = builder.data;
    }

    /**
     * <p>
     * The ground station name for which you're providing azimuth elevation data.
     * </p>
     * <p>
     * This ephemeris is specific to this ground station and can't be used at other locations.
     * </p>
     * 
     * @return The ground station name for which you're providing azimuth elevation data.</p>
     *         <p>
     *         This ephemeris is specific to this ground station and can't be used at other locations.
     */
    public final String groundStation() {
        return groundStation;
    }

    /**
     * <p>
     * Azimuth elevation segment data.
     * </p>
     * <p>
     * You can provide data inline in the request or through an Amazon S3 object reference.
     * </p>
     * 
     * @return Azimuth elevation segment data.</p>
     *         <p>
     *         You can provide data inline in the request or through an Amazon S3 object reference.
     */
    public final AzElSegmentsData data() {
        return data;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(groundStation());
        hashCode = 31 * hashCode + Objects.hashCode(data());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AzElEphemeris)) {
            return false;
        }
        AzElEphemeris other = (AzElEphemeris) obj;
        return Objects.equals(groundStation(), other.groundStation()) && Objects.equals(data(), other.data());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AzElEphemeris").add("GroundStation", groundStation()).add("Data", data()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "groundStation":
            return Optional.ofNullable(clazz.cast(groundStation()));
        case "data":
            return Optional.ofNullable(clazz.cast(data()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("groundStation", GROUND_STATION_FIELD);
        map.put("data", DATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AzElEphemeris, T> g) {
        return obj -> g.apply((AzElEphemeris) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AzElEphemeris> {
        /**
         * <p>
         * The ground station name for which you're providing azimuth elevation data.
         * </p>
         * <p>
         * This ephemeris is specific to this ground station and can't be used at other locations.
         * </p>
         * 
         * @param groundStation
         *        The ground station name for which you're providing azimuth elevation data.</p>
         *        <p>
         *        This ephemeris is specific to this ground station and can't be used at other locations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groundStation(String groundStation);

        /**
         * <p>
         * Azimuth elevation segment data.
         * </p>
         * <p>
         * You can provide data inline in the request or through an Amazon S3 object reference.
         * </p>
         * 
         * @param data
         *        Azimuth elevation segment data.</p>
         *        <p>
         *        You can provide data inline in the request or through an Amazon S3 object reference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder data(AzElSegmentsData data);

        /**
         * <p>
         * Azimuth elevation segment data.
         * </p>
         * <p>
         * You can provide data inline in the request or through an Amazon S3 object reference.
         * </p>
         * This is a convenience method that creates an instance of the {@link AzElSegmentsData.Builder} avoiding the
         * need to create one manually via {@link AzElSegmentsData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AzElSegmentsData.Builder#build()} is called immediately and its
         * result is passed to {@link #data(AzElSegmentsData)}.
         * 
         * @param data
         *        a consumer that will call methods on {@link AzElSegmentsData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #data(AzElSegmentsData)
         */
        default Builder data(Consumer<AzElSegmentsData.Builder> data) {
            return data(AzElSegmentsData.builder().applyMutation(data).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String groundStation;

        private AzElSegmentsData data;

        private BuilderImpl() {
        }

        private BuilderImpl(AzElEphemeris model) {
            groundStation(model.groundStation);
            data(model.data);
        }

        public final String getGroundStation() {
            return groundStation;
        }

        public final void setGroundStation(String groundStation) {
            this.groundStation = groundStation;
        }

        @Override
        public final Builder groundStation(String groundStation) {
            this.groundStation = groundStation;
            return this;
        }

        public final AzElSegmentsData.Builder getData() {
            return data != null ? data.toBuilder() : null;
        }

        public final void setData(AzElSegmentsData.BuilderImpl data) {
            this.data = data != null ? data.build() : null;
        }

        @Override
        public final Builder data(AzElSegmentsData data) {
            this.data = data;
            return this;
        }

        @Override
        public AzElEphemeris build() {
            return new AzElEphemeris(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
