/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A time segment containing azimuth elevation pointing data.
 * </p>
 * <p>
 * Each segment defines a continuous time period with pointing angle data points. AWS Ground Station uses 4th order
 * Lagrange interpolation between the provided points, so each segment must contain at least five data points.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AzElSegment implements SdkPojo, Serializable, ToCopyableBuilder<AzElSegment.Builder, AzElSegment> {
    private static final SdkField<Instant> REFERENCE_EPOCH_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("referenceEpoch")
            .getter(getter(AzElSegment::referenceEpoch))
            .setter(setter(Builder::referenceEpoch))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("referenceEpoch").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<ISO8601TimeRange> VALID_TIME_RANGE_FIELD = SdkField
            .<ISO8601TimeRange> builder(MarshallingType.SDK_POJO).memberName("validTimeRange")
            .getter(getter(AzElSegment::validTimeRange)).setter(setter(Builder::validTimeRange))
            .constructor(ISO8601TimeRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validTimeRange").build()).build();

    private static final SdkField<List<TimeAzEl>> AZ_EL_LIST_FIELD = SdkField
            .<List<TimeAzEl>> builder(MarshallingType.LIST)
            .memberName("azElList")
            .getter(getter(AzElSegment::azElList))
            .setter(setter(Builder::azElList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("azElList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TimeAzEl> builder(MarshallingType.SDK_POJO)
                                            .constructor(TimeAzEl::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REFERENCE_EPOCH_FIELD,
            VALID_TIME_RANGE_FIELD, AZ_EL_LIST_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant referenceEpoch;

    private final ISO8601TimeRange validTimeRange;

    private final List<TimeAzEl> azElList;

    private AzElSegment(BuilderImpl builder) {
        this.referenceEpoch = builder.referenceEpoch;
        this.validTimeRange = builder.validTimeRange;
        this.azElList = builder.azElList;
    }

    /**
     * <p>
     * The reference time for this segment in ISO 8601 format in Coordinated Universal Time (UTC).
     * </p>
     * <p>
     * All time values within the segment's <a>AzElSegment$azElList</a> are specified as offsets in atomic seconds from
     * this reference epoch.
     * </p>
     * <p>
     * Example: <code>2024-01-15T12:00:00.000Z</code>
     * </p>
     * 
     * @return The reference time for this segment in ISO 8601 format in Coordinated Universal Time (UTC).</p>
     *         <p>
     *         All time values within the segment's <a>AzElSegment$azElList</a> are specified as offsets in atomic
     *         seconds from this reference epoch.
     *         </p>
     *         <p>
     *         Example: <code>2024-01-15T12:00:00.000Z</code>
     */
    public final Instant referenceEpoch() {
        return referenceEpoch;
    }

    /**
     * <p>
     * The valid time range for this segment.
     * </p>
     * <p>
     * Specifies the start and end timestamps in ISO 8601 format in Coordinated Universal Time (UTC). The segment's
     * pointing data must cover this entire time range.
     * </p>
     * 
     * @return The valid time range for this segment.</p>
     *         <p>
     *         Specifies the start and end timestamps in ISO 8601 format in Coordinated Universal Time (UTC). The
     *         segment's pointing data must cover this entire time range.
     */
    public final ISO8601TimeRange validTimeRange() {
        return validTimeRange;
    }

    /**
     * For responses, this returns true if the service returned a value for the AzElList property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAzElList() {
        return azElList != null && !(azElList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of time-tagged azimuth elevation data points.
     * </p>
     * <p>
     * Must contain at least five points to support 4th order Lagrange interpolation. Points must be in chronological
     * order with no duplicates.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAzElList} method.
     * </p>
     * 
     * @return List of time-tagged azimuth elevation data points.</p>
     *         <p>
     *         Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
     *         chronological order with no duplicates.
     */
    public final List<TimeAzEl> azElList() {
        return azElList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(referenceEpoch());
        hashCode = 31 * hashCode + Objects.hashCode(validTimeRange());
        hashCode = 31 * hashCode + Objects.hashCode(hasAzElList() ? azElList() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AzElSegment)) {
            return false;
        }
        AzElSegment other = (AzElSegment) obj;
        return Objects.equals(referenceEpoch(), other.referenceEpoch())
                && Objects.equals(validTimeRange(), other.validTimeRange()) && hasAzElList() == other.hasAzElList()
                && Objects.equals(azElList(), other.azElList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AzElSegment").add("ReferenceEpoch", referenceEpoch()).add("ValidTimeRange", validTimeRange())
                .add("AzElList", hasAzElList() ? azElList() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "referenceEpoch":
            return Optional.ofNullable(clazz.cast(referenceEpoch()));
        case "validTimeRange":
            return Optional.ofNullable(clazz.cast(validTimeRange()));
        case "azElList":
            return Optional.ofNullable(clazz.cast(azElList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("referenceEpoch", REFERENCE_EPOCH_FIELD);
        map.put("validTimeRange", VALID_TIME_RANGE_FIELD);
        map.put("azElList", AZ_EL_LIST_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AzElSegment, T> g) {
        return obj -> g.apply((AzElSegment) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AzElSegment> {
        /**
         * <p>
         * The reference time for this segment in ISO 8601 format in Coordinated Universal Time (UTC).
         * </p>
         * <p>
         * All time values within the segment's <a>AzElSegment$azElList</a> are specified as offsets in atomic seconds
         * from this reference epoch.
         * </p>
         * <p>
         * Example: <code>2024-01-15T12:00:00.000Z</code>
         * </p>
         * 
         * @param referenceEpoch
         *        The reference time for this segment in ISO 8601 format in Coordinated Universal Time (UTC).</p>
         *        <p>
         *        All time values within the segment's <a>AzElSegment$azElList</a> are specified as offsets in atomic
         *        seconds from this reference epoch.
         *        </p>
         *        <p>
         *        Example: <code>2024-01-15T12:00:00.000Z</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceEpoch(Instant referenceEpoch);

        /**
         * <p>
         * The valid time range for this segment.
         * </p>
         * <p>
         * Specifies the start and end timestamps in ISO 8601 format in Coordinated Universal Time (UTC). The segment's
         * pointing data must cover this entire time range.
         * </p>
         * 
         * @param validTimeRange
         *        The valid time range for this segment.</p>
         *        <p>
         *        Specifies the start and end timestamps in ISO 8601 format in Coordinated Universal Time (UTC). The
         *        segment's pointing data must cover this entire time range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validTimeRange(ISO8601TimeRange validTimeRange);

        /**
         * <p>
         * The valid time range for this segment.
         * </p>
         * <p>
         * Specifies the start and end timestamps in ISO 8601 format in Coordinated Universal Time (UTC). The segment's
         * pointing data must cover this entire time range.
         * </p>
         * This is a convenience method that creates an instance of the {@link ISO8601TimeRange.Builder} avoiding the
         * need to create one manually via {@link ISO8601TimeRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ISO8601TimeRange.Builder#build()} is called immediately and its
         * result is passed to {@link #validTimeRange(ISO8601TimeRange)}.
         * 
         * @param validTimeRange
         *        a consumer that will call methods on {@link ISO8601TimeRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validTimeRange(ISO8601TimeRange)
         */
        default Builder validTimeRange(Consumer<ISO8601TimeRange.Builder> validTimeRange) {
            return validTimeRange(ISO8601TimeRange.builder().applyMutation(validTimeRange).build());
        }

        /**
         * <p>
         * List of time-tagged azimuth elevation data points.
         * </p>
         * <p>
         * Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
         * chronological order with no duplicates.
         * </p>
         * 
         * @param azElList
         *        List of time-tagged azimuth elevation data points.</p>
         *        <p>
         *        Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
         *        chronological order with no duplicates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder azElList(Collection<TimeAzEl> azElList);

        /**
         * <p>
         * List of time-tagged azimuth elevation data points.
         * </p>
         * <p>
         * Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
         * chronological order with no duplicates.
         * </p>
         * 
         * @param azElList
         *        List of time-tagged azimuth elevation data points.</p>
         *        <p>
         *        Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
         *        chronological order with no duplicates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder azElList(TimeAzEl... azElList);

        /**
         * <p>
         * List of time-tagged azimuth elevation data points.
         * </p>
         * <p>
         * Must contain at least five points to support 4th order Lagrange interpolation. Points must be in
         * chronological order with no duplicates.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.groundstation.model.TimeAzEl.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.groundstation.model.TimeAzEl#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.groundstation.model.TimeAzEl.Builder#build()} is called immediately
         * and its result is passed to {@link #azElList(List<TimeAzEl>)}.
         * 
         * @param azElList
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.groundstation.model.TimeAzEl.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #azElList(java.util.Collection<TimeAzEl>)
         */
        Builder azElList(Consumer<TimeAzEl.Builder>... azElList);
    }

    static final class BuilderImpl implements Builder {
        private Instant referenceEpoch;

        private ISO8601TimeRange validTimeRange;

        private List<TimeAzEl> azElList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AzElSegment model) {
            referenceEpoch(model.referenceEpoch);
            validTimeRange(model.validTimeRange);
            azElList(model.azElList);
        }

        public final Instant getReferenceEpoch() {
            return referenceEpoch;
        }

        public final void setReferenceEpoch(Instant referenceEpoch) {
            this.referenceEpoch = referenceEpoch;
        }

        @Override
        public final Builder referenceEpoch(Instant referenceEpoch) {
            this.referenceEpoch = referenceEpoch;
            return this;
        }

        public final ISO8601TimeRange.Builder getValidTimeRange() {
            return validTimeRange != null ? validTimeRange.toBuilder() : null;
        }

        public final void setValidTimeRange(ISO8601TimeRange.BuilderImpl validTimeRange) {
            this.validTimeRange = validTimeRange != null ? validTimeRange.build() : null;
        }

        @Override
        public final Builder validTimeRange(ISO8601TimeRange validTimeRange) {
            this.validTimeRange = validTimeRange;
            return this;
        }

        public final List<TimeAzEl.Builder> getAzElList() {
            List<TimeAzEl.Builder> result = TimeAzElListCopier.copyToBuilder(this.azElList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAzElList(Collection<TimeAzEl.BuilderImpl> azElList) {
            this.azElList = TimeAzElListCopier.copyFromBuilder(azElList);
        }

        @Override
        public final Builder azElList(Collection<TimeAzEl> azElList) {
            this.azElList = TimeAzElListCopier.copy(azElList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder azElList(TimeAzEl... azElList) {
            azElList(Arrays.asList(azElList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder azElList(Consumer<TimeAzEl.Builder>... azElList) {
            azElList(Stream.of(azElList).map(c -> TimeAzEl.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public AzElSegment build() {
            return new AzElSegment(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
