/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.groundstation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Two-line element set (TLE) ephemeris.
 * </p>
 * <p>
 * For more detail about providing Two-line element sets to AWS Ground Station, see the <a
 * href="https://docs.aws.amazon.com/ground-station/latest/ug/providing-tle-ephemeris-data.html">TLE section</a> of the
 * AWS Ground Station user guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TLEEphemeris implements SdkPojo, Serializable, ToCopyableBuilder<TLEEphemeris.Builder, TLEEphemeris> {
    private static final SdkField<S3Object> S3_OBJECT_FIELD = SdkField.<S3Object> builder(MarshallingType.SDK_POJO)
            .memberName("s3Object").getter(getter(TLEEphemeris::s3Object)).setter(setter(Builder::s3Object))
            .constructor(S3Object::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Object").build()).build();

    private static final SdkField<List<TLEData>> TLE_DATA_FIELD = SdkField
            .<List<TLEData>> builder(MarshallingType.LIST)
            .memberName("tleData")
            .getter(getter(TLEEphemeris::tleData))
            .setter(setter(Builder::tleData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tleData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TLEData> builder(MarshallingType.SDK_POJO)
                                            .constructor(TLEData::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_OBJECT_FIELD,
            TLE_DATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final S3Object s3Object;

    private final List<TLEData> tleData;

    private TLEEphemeris(BuilderImpl builder) {
        this.s3Object = builder.s3Object;
        this.tleData = builder.tleData;
    }

    /**
     * <p>
     * The Amazon S3 object that contains the ephemeris data.
     * </p>
     * 
     * @return The Amazon S3 object that contains the ephemeris data.
     */
    public final S3Object s3Object() {
        return s3Object;
    }

    /**
     * For responses, this returns true if the service returned a value for the TleData property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTleData() {
        return tleData != null && !(tleData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * TLE data that you provide directly instead of using an Amazon S3 object.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTleData} method.
     * </p>
     * 
     * @return TLE data that you provide directly instead of using an Amazon S3 object.
     */
    public final List<TLEData> tleData() {
        return tleData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Object());
        hashCode = 31 * hashCode + Objects.hashCode(hasTleData() ? tleData() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TLEEphemeris)) {
            return false;
        }
        TLEEphemeris other = (TLEEphemeris) obj;
        return Objects.equals(s3Object(), other.s3Object()) && hasTleData() == other.hasTleData()
                && Objects.equals(tleData(), other.tleData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TLEEphemeris").add("S3Object", s3Object()).add("TleData", hasTleData() ? tleData() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "s3Object":
            return Optional.ofNullable(clazz.cast(s3Object()));
        case "tleData":
            return Optional.ofNullable(clazz.cast(tleData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("s3Object", S3_OBJECT_FIELD);
        map.put("tleData", TLE_DATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TLEEphemeris, T> g) {
        return obj -> g.apply((TLEEphemeris) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TLEEphemeris> {
        /**
         * <p>
         * The Amazon S3 object that contains the ephemeris data.
         * </p>
         * 
         * @param s3Object
         *        The Amazon S3 object that contains the ephemeris data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Object(S3Object s3Object);

        /**
         * <p>
         * The Amazon S3 object that contains the ephemeris data.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Object.Builder} avoiding the need to
         * create one manually via {@link S3Object#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Object.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Object(S3Object)}.
         * 
         * @param s3Object
         *        a consumer that will call methods on {@link S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Object(S3Object)
         */
        default Builder s3Object(Consumer<S3Object.Builder> s3Object) {
            return s3Object(S3Object.builder().applyMutation(s3Object).build());
        }

        /**
         * <p>
         * TLE data that you provide directly instead of using an Amazon S3 object.
         * </p>
         * 
         * @param tleData
         *        TLE data that you provide directly instead of using an Amazon S3 object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tleData(Collection<TLEData> tleData);

        /**
         * <p>
         * TLE data that you provide directly instead of using an Amazon S3 object.
         * </p>
         * 
         * @param tleData
         *        TLE data that you provide directly instead of using an Amazon S3 object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tleData(TLEData... tleData);

        /**
         * <p>
         * TLE data that you provide directly instead of using an Amazon S3 object.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.groundstation.model.TLEData.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.groundstation.model.TLEData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.groundstation.model.TLEData.Builder#build()} is called immediately and
         * its result is passed to {@link #tleData(List<TLEData>)}.
         * 
         * @param tleData
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.groundstation.model.TLEData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tleData(java.util.Collection<TLEData>)
         */
        Builder tleData(Consumer<TLEData.Builder>... tleData);
    }

    static final class BuilderImpl implements Builder {
        private S3Object s3Object;

        private List<TLEData> tleData = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TLEEphemeris model) {
            s3Object(model.s3Object);
            tleData(model.tleData);
        }

        public final S3Object.Builder getS3Object() {
            return s3Object != null ? s3Object.toBuilder() : null;
        }

        public final void setS3Object(S3Object.BuilderImpl s3Object) {
            this.s3Object = s3Object != null ? s3Object.build() : null;
        }

        @Override
        public final Builder s3Object(S3Object s3Object) {
            this.s3Object = s3Object;
            return this;
        }

        public final List<TLEData.Builder> getTleData() {
            List<TLEData.Builder> result = TLEDataListCopier.copyToBuilder(this.tleData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTleData(Collection<TLEData.BuilderImpl> tleData) {
            this.tleData = TLEDataListCopier.copyFromBuilder(tleData);
        }

        @Override
        public final Builder tleData(Collection<TLEData> tleData) {
            this.tleData = TLEDataListCopier.copy(tleData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tleData(TLEData... tleData) {
            tleData(Arrays.asList(tleData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tleData(Consumer<TLEData.Builder>... tleData) {
            tleData(Stream.of(tleData).map(c -> TLEData.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public TLEEphemeris build() {
            return new TLEEphemeris(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
